{ Form Template - Source and Destination Choices Lists }
unit Dllstdlg;
{
  DllstDlg.pas

  (c) 1996 Dwayne Mercredi and Borland International

  declares classes TDualListForm and TDualListDialog

  TDualListForm is the dual list form created with the form template,
  with the following added behaviour:
    Double clicking on a list moves that element to the other list.

  TDualListDialog is a component that wraps the TDualListForm
}
interface

uses WinTypes, WinProcs, Classes, Graphics, Forms, Controls, Buttons,
  StdCtrls;

type
  TDualListForm = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    HelpBtn: TBitBtn;
    SrcListBox: TListBox;
    DstListBox: TListBox;
    SrcLabel: TLabel;
    DstLabel: TLabel;
    IncludeBtn: TSpeedButton;
    IncAllBtn: TSpeedButton;
    ExcludeBtn: TSpeedButton;
    ExAllBtn: TSpeedButton;
    procedure IncludeBtnClick(Sender: TObject);
    procedure ExcludeBtnClick(Sender: TObject);
    procedure IncAllBtnClick(Sender: TObject);
    procedure ExcAllBtnClick(Sender: TObject);
    procedure MoveSelected(List: TCustomListBox; Items: TStrings);
    procedure SetItem(List: TListBox; Index: Integer);
    function GetFirstSelection(List: TCustomListBox): Integer;
    procedure SetButtons;
    procedure ListBoxDblClick(Sender: TObject);
    procedure FormShow(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

  TDualListDialog = class(TComponent)
  private
    FFormCaption: String;

    FSrcListCaption: String;
    FDstListCaption: String;

    FSrcList: TStrings;
    FDstList: TStrings;

    procedure SetSrcList(NewList: TStrings);
    procedure SetDstList(NewList: TStrings);

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    {
      function Execute
        shows dual list dialog modally
        returns true iff Ok button was used to end form
    }
    function Execute: Boolean;

  published
    {
      property FormCaption
        holds the caption of the form
    }
    property FormCaption: String
        read FFormCaption
        write FFormCaption;

    {
      property SrcListCaption
        holds the text of the label just above the Source List Box
    }
    property SrcListCaption: String
        read FSrcListCaption
        write FSrcListCaption;

    {
      property DstListCaption
        holds the text of the label just above the Destination List Box
    }
    property DstListCaption: String
        read  FDstListCaption
        write FDstListCaption;

    {
      property SrcList
        holds the list of strings and objects associated with the Source
        List Box.
        Is only updated from form results if Execute returns True.
    }
    property SrcList: TStrings
        read FSrcList
        write SetSrcList;

    {
      property DstList
        holds the list of strings and objects associated with the Destination
        List Box.
        Is only updated from form results if Execute returns True.
    }
    property DstList: TStrings
        read FDstList
        write SetDstList;

  end;

var
  DualListForm: TDualListForm;

implementation

{$R *.DFM}

{------------------------------------------------------------------
                      class TDualListForm
 ------------------------------------------------------------------}
procedure TDualListForm.IncludeBtnClick(Sender: TObject);
var
  Index: Integer;
begin
  Index := GetFirstSelection(SrcListBox);
  MoveSelected(SrcListBox, DstListBox.Items);
  SetItem(SrcListBox, Index);
end;

procedure TDualListForm.ExcludeBtnClick(Sender: TObject);
var
  Index: Integer;
begin
  Index := GetFirstSelection(DstListBox);
  MoveSelected(DstListBox, SrcListBox.Items);
  SetItem(DstListBox, Index);
end;

procedure TDualListForm.IncAllBtnClick(Sender: TObject);
var
  I: Integer;
begin
  for I := 0 to SrcListBox.Items.Count - 1 do
    DstListBox.Items.AddObject(SrcListBox.Items[I],
      SrcListBox.Items.Objects[I]);
  SrcListBox.Items.Clear;
  SetItem(SrcListBox, 0);
end;

procedure TDualListForm.ExcAllBtnClick(Sender: TObject);
var
  I: Integer;
begin
  for I := 0 to DstListBox.Items.Count - 1 do
    SrcListBox.Items.AddObject(DstListBox.Items[I], DstListBox.Items.Objects[I]);
  DstListBox.Items.Clear;
  SetItem(DstListBox, 0);
end;

procedure TDualListForm.MoveSelected(List: TCustomListBox; Items: TStrings);
var
  I: Integer;
begin
  for I := List.Items.Count - 1 downto 0 do
    if List.Selected[I] then
    begin
      Items.AddObject(List.Items[I], List.Items.Objects[I]);
      List.Items.Delete(I);
    end;
end;

procedure TDualListForm.SetButtons;
var
  SrcEmpty, DstEmpty: Boolean;
begin
  SrcEmpty := SrcListBox.Items.Count = 0;
  DstEmpty := DstListBox.Items.Count = 0;
  IncludeBtn.Enabled := not SrcEmpty;
  IncAllBtn.Enabled := not SrcEmpty;
  ExcludeBtn.Enabled := not DstEmpty;
  ExAllBtn.Enabled := not DstEmpty;
end;

function TDualListForm.GetFirstSelection(List: TCustomListBox): Integer;
begin
  for Result := 0 to List.Items.Count - 1 do
    if List.Selected[Result] then Exit;
  Result := LB_ERR;
end;

procedure TDualListForm.SetItem(List: TListBox; Index: Integer);
var
  MaxIndex: Integer;
begin
  with List do
  begin
    SetFocus;
    MaxIndex := List.Items.Count - 1;
    if Index = LB_ERR then Index := 0
    else if Index > MaxIndex then Index := MaxIndex;
    Selected[Index] := True;
  end;
  SetButtons;
end;

procedure TDualListForm.ListBoxDblClick(Sender: TObject);
var
  Index: Integer;
  ListFrom, ListTo: TListBox;
begin
  { set up from and to vars depending on caller }
  if (Sender = SrcListBox) then begin
    ListFrom := SrcListBox;
    ListTo   := DstListBox;
  end
  else begin
    ListFrom := DstListBox;
    ListTo   := SrcListBox;
  end;

  { get first index }
  Index := GetFirstSelection(ListFrom);
  { move all selected }
  MoveSelected(ListFrom, ListTo.Items);
  { set index again }
  SetItem(ListFrom, Index);
  { sync buttons with self }
  SetButtons;

end;

procedure TDualListForm.FormShow(Sender: TObject);
begin
  SetButtons;
end;

{------------------------------------------------------------------
                    end class TDualListForm
 ------------------------------------------------------------------}
{------------------------------------------------------------------
                  class TDualListDialog
 ------------------------------------------------------------------}
constructor TDualListDialog.Create(AOwner: TComponent);
begin
  { create me }
  inherited Create(AOwner);

  { set captions to empty }
  FFormCaption    := '';
  FSrcListCaption := '';
  FDstListCaption := '';

  { create string lists }
  FSrcList := TStringList.Create;
  FDstList := TStringList.Create;
end;

destructor TDualListDialog.Destroy;
begin
  { free string lists }
  FSrcList.Free;
  FDstList.Free;

  { destroy me }
  inherited Destroy;
end;

procedure TDualListDialog.SetSrcList(NewList: TStrings);
begin
  { if different, assign to src list }
  if (FSrcList <> NewList) then
    FSrcList.Assign(NewList);
end;

procedure TDualListDialog.SetDstList(NewList: TStrings);
begin
  { if different, assign to dst list }
  if (FDstList <> NewList) then
    FDstList.Assign(NewList);
end;

function TDualListDialog.Execute: Boolean;
var
  DualListForm: TDualListForm;
begin
  { create form }
  DualListForm := TDualListForm.Create(Self);
  try
    with (DualListForm) do begin
      { assign properties to form }
      Caption := FormCaption;

      SrcLabel.Caption := SrcListCaption;
      DstLabel.Caption := DstListCaption;

      SrcListBox.Items.Assign(SrcList);
      DstListBox.Items.Assign(DstList);

      { show form }
      ShowModal;

      { if OK was pressed }
      if (ModalResult = mrOk) then begin
        { assign list boxes back to strings }
        SrcList.Assign(SrcListBox.Items);
        DstList.Assign(DstListBox.Items);
        Result := True;
      end
      else
        Result := False;

      { return True IFF Ok was pressed }

    end; { with dialog }
  finally
    { free form }
    DualListForm.Free;
  end;
end;

{------------------------------------------------------------------
                  end class TDualListDialog
 ------------------------------------------------------------------}

end.
