unit Commncmd;
{
  CommnCmd.pas

  (c) 1996 Dwayne Mercredi

  declares classes TCommonPropertyGroup, TCommonCommand

  class TCommonPropertyGroup:
    base class for any group of components on a single form that share
    a number of properties or events; you can simply place the shared
    properties in a descendant class of TCommonPropertyGroup, override
    a few methods, and setting the properties in the common property
    group will set the properties of all contained components

  class TCommonCommand:
    implements a common command with Enabled, Checked, and OnClick.
    Allows only TButton, TSpeedButton, and TMenuItem components to
    be added to the group.

    Enabled controls the enable state of all contained components

    OnClick points to the event handler for OnClick for all contained
      components

    Checked is reflected in the Checked state of TMenuItems and the
      Down state of TSpeedButtons.  It is ignored for TButtons.
}

interface

uses
  SysUtils, Classes, DBMUtils, Buttons, StdCtrls, Menus;
type

  TCommonPropertyGroup = class(TComponent)
  private
    FComponentList: TStrings;

    function GetComponentAt(Index: Integer): TComponent;
    function GetCount: Integer;
  protected
    {
      procedure Notification
        Removes component from list if the notification is of Removal
    }
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    {
      procedure DefineProperties
        sets up ReadGroup and WriteGroup to read and write the list
        of components to the given filer
    }
    procedure DefineProperties(Filer: TFiler); override;

    {
      procedure ReadGroup
        reads the list of component names from the given Reader
    }
    procedure ReadGroup(Reader: TReader);

    {
      procedure WriteGroup
        writes the list of component names to the given writer
    }
    procedure WriteGroup(Writer: TWriter);

    {
      procedure Loaded
        sets up the list of components given the names.  Searches
        the Owner's list of components for the names
    }
    procedure Loaded; override;

    {
      procedure Syncronize
        used to syncronize the given component with the state
        of the common property group.  Called for each component
        when it is added to the list and whenever SyncronizeAll
        is called
    }
    procedure Synchronize(AComponent: TComponent); virtual;

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    {
      procedure Assign
        allows assignment of PropertyGroups to one another
    }
    procedure Assign(Source: TPersistent); override;

    {
      procedure SynchronizeAll
        calls Synchronize with all components in list
    }
    procedure SynchronizeAll;

    {
      function CanAdd
        returns whether or not the given component can be added to
        the list.  Should return True for only appropriate components.
        Returns true if not already in list
    }
    function CanAdd(AComponent: TComponent): Boolean; virtual;

    {
      procedure Add
        adds component to list if CanAdd returns true for the given
        component
    }
    procedure Add(Item: TComponent);

    {
      procedure Clear
        clears all components from the list
    }
    procedure Clear;

    {
      function Contains
        returns true iff the given component is in the list
    }
    function Contains(Item: TComponent): Boolean;

    {
      procedure Remove
        removes the given component from the list if it is in the list
    }
    procedure Remove(Item: TComponent);

    {
      property Count
        returns the number of components in list
    }
    property Count: Integer
        read GetCount;

    {
      property Group
        zero based array of components within this common property group
        range of Index: 0 .. Count - 1
    }
    property Group[Index: Integer]: TComponent
        read GetComponentAt; default;

  end;

  TCommonCommand = class(TCommonPropertyGroup)
  private
    FEnabled: Boolean;
    FChecked: Boolean;
    FOnClick: TNotifyEvent;

    procedure SetEnabled(NewValue: Boolean);
    procedure SetChecked(NewValue: Boolean);
    procedure SetOnClick(NewValue: TNotifyEvent);

  protected
    {
      procedure Syncronize
        sets the Enabled, Checked and OnClick properties of the given
        component.
    }
    procedure Synchronize(AComponent: TComponent); override;

  public
    constructor Create(AOwner: TComponent); override;

    {
      function CanAdd
        returns True iff the given component is a TSpeedButton, TButton,
        or TMenuItem, along with all of the conditions of TCommonPropertyGroup
    }
    function CanAdd(AComponent: TComponent): Boolean; override;

  published
    {
      property Enabled
        controls the Enabled state of all contained components
    }
    property Enabled: Boolean
      read FEnabled
      write SetEnabled
      default True;

    {
      property Checked
        controls the Checked state of all contained TMenuItems,
        and the Down state of all contained TSpeedButtons.
        if (Checked) then Down;
        Is ignored for all contained TButtons.
    }
    property Checked: Boolean
      read FChecked
      write SetChecked
      default False;

    {
      property OnClick
        Points to the event handler for the OnClick event of all contained
        components
    }
    property OnClick: TNotifyEvent
      read FOnClick
      write SetOnClick;

  end;

  {
    represents an assertion error within the common property group
    code
  }
  ECommonPropertyError = class (Exception);

implementation

{---------------------------------------------------------------------
                    class TCommonPropertyGroup
 ---------------------------------------------------------------------}
constructor TCommonPropertyGroup.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  FComponentList := TStringList.Create;
end;

destructor TCommonPropertyGroup.Destroy;
begin
  FComponentList.Free;

  inherited Destroy;
end;

procedure TCommonPropertyGroup.Notification(AComponent: TComponent;
                                                      Operation: TOperation);
begin
  if ((Operation = opRemove) and (AComponent <> Self)) then
    Remove(AComponent);

  inherited Notification(AComponent, Operation);
end;

procedure TCommonPropertyGroup.Assign(Source: TPersistent);
var
  OtherGroup: TCommonPropertyGroup;
  i: Integer;
begin
  { if its a common property group }
  if (Source is TCommonPropertyGroup) then begin

    OtherGroup := Source as TCommonPropertyGroup;

    { if different from Source }
    if (Self <> OtherGroup) then begin
      { clear this group }
      Clear;

      { add each item of the other group to self }
      for i := 0 to OtherGroup.Count - 1 do
        Add(OtherGroup[i]);
    end;
  end
  { otherwise }
  else
    { default action }
    inherited Assign(Source);
end;

function TCommonPropertyGroup.CanAdd(AComponent: TComponent): Boolean;
begin
  { only can add if it isn't already in list }
  Result := not Contains(AComponent);
end;

procedure TCommonPropertyGroup.Synchronize(AComponent: TComponent);
begin
  { no synchronization to do }
end;

procedure TCommonPropertyGroup.SynchronizeAll;
var
  i: Integer;
begin
  { for each component }
  for i := 0 to Count - 1 do
    { synchronize }
    Synchronize(Group[i]);
end;

procedure TCommonPropertyGroup.Add(Item: TComponent);
begin
  { if can add }
  if (CanAdd(Item)) then begin
    { add and synchronize }
    FComponentList.AddObject(Item.Name, Item);
    Synchronize(Item);
  end;
end;

procedure TCommonPropertyGroup.Clear;
begin
  FComponentList.Clear;
end;

function TCommonPropertyGroup.Contains(Item: TComponent): Boolean;
begin
  Result := (FComponentList.IndexOfObject(Item) <> -1);
end;

procedure TCommonPropertyGroup.Remove(Item: TComponent);
var
  Index: Integer;
begin
  Index := FComponentList.IndexOfObject(Item);
  if (Index <> -1) then
    FComponentList.Delete(Index);
end;

function TCommonPropertyGroup.GetComponentAt(Index: Integer): TComponent;
begin
  Result := TComponent(FComponentList.Objects[Index]);
end;

function TCommonPropertyGroup.GetCount: Integer;
begin
  Result := FComponentList.Count;
end;

procedure TCommonPropertyGroup.DefineProperties(Filer: TFiler);
begin
  inherited DefineProperties(Filer);

  Filer.DefineProperty('Group', ReadGroup, WriteGroup, True);
end;

procedure TCommonPropertyGroup.ReadGroup(Reader: TReader);
begin
  Reader.ReadListBegin;

  { for each item in the list }
  while (not Reader.EndOfList) do begin
    { read the components name and add to internal list }
    FComponentList.Add(Reader.ReadString);
  end;

  Reader.ReadListEnd;
end;

procedure TCommonPropertyGroup.WriteGroup(Writer: TWriter);
var
  i: Integer;
begin
  Writer.WriteListBegin;

  { for each item in the list }
  for i := 0 to Count - 1 do
    { write the components name }
    Writer.WriteString(Group[i].Name);

  Writer.WriteListEnd;
end;

procedure TCommonPropertyGroup.Loaded;
var
  Name: String;
  Component: TComponent;
  i: Integer;
begin
  { for each item in the list }
  for i := 0 to FComponentList.Count - 1 do begin
    { grab the component that belongs to the name }
    Name := FComponentList[i];
    Component := Owner.FindComponent(Name);

    Assert(Component <> Nil, 'Internal Consistancy error in property group',
            ECommonPropertyError);

    { add that component to the position corresponding to its name }
    FComponentList.Objects[i] := Component;
  end;
end;


{---------------------------------------------------------------------
                     end class TCommonPropertyGroup
 ---------------------------------------------------------------------}
{---------------------------------------------------------------------
                     class TCommonCommand
 ---------------------------------------------------------------------}
constructor TCommonCommand.Create(AOwner: TComponent);
begin
  { create }
  inherited Create(AOwner);

  { set defaults }
  FEnabled := True;
  FChecked := False;
  FOnClick := Nil;
end;

procedure TCommonCommand.SetEnabled(NewValue: Boolean);
begin
  { if different }
  if (NewValue <> FEnabled) then begin
    { set and sync all grouped components }
    FEnabled := NewValue;
    SynchronizeAll;
  end;
end;

procedure TCommonCommand.SetChecked(NewValue: Boolean);
begin
  { if different }
  if (NewValue <> FChecked) then begin
    { set and sync all grouped components }
    FChecked := NewValue;
    SynchronizeAll;
  end;
end;

procedure TCommonCommand.SetOnClick(NewValue: TNotifyEvent);
begin
  { set and sync all grouped components }
  FOnClick := NewValue;
  SynchronizeAll;
end;

function TCommonCommand.CanAdd(AComponent: TComponent): Boolean;
begin
  {
    can only add if all previous conditions are met and component is
    one of TMenuItem, TSpeedButton, or TButton derivitive
  }
  Result := (inherited CanAdd(AComponent)) and
            ( (AComponent is TMenuItem) or (AComponent is TSpeedButton) or
              (AComponent is TButton) );
end;

procedure TCommonCommand.Synchronize(AComponent: TComponent);
begin
  {
    syncronize for each of the three allowed component types
  }
  if (AComponent is TMenuItem) then
    with (AComponent as TMenuItem) do begin
      Enabled := FEnabled;
      Checked := FChecked;
      OnClick := FOnClick;
    end
  else if (AComponent is TSpeedButton) then
    with (AComponent as TSpeedButton) do begin
      Enabled := FEnabled;
      Down    := FChecked;
      OnClick := FOnClick;
    end
  else if (AComponent is TButton) then
    with (AComponent as TButton) do begin
      Enabled := FEnabled;
      OnClick := FOnClick;
    end;
end;
{---------------------------------------------------------------------
                     end class TCommonCommand
 ---------------------------------------------------------------------}


end.
