unit DBDateEd;

(*********************************************
tDbDateEdit -> TEdit

A date edit field with drop down calendar.

PROPERTIES:

Date - TDateTime that contains the date value of the control.

ValidDateColor - The color that "valid dates" will be rendered.

METHODS:

procedure AddValidDate - Adds a datetime value to a list of "valid dates" maintained by the
control.  These dates will be drawn in the color specified by ValidDateColor.

procedure ClearValidDates - Clears all "valid dates" from the list.

function DateInList - Checks if the specified date is in the list of "valid dates".

EVENTS:

OnDateChange - Triggered whenever the Date property is updated.
*********************************************)

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, StdCtrls, DBCalpop, Buttons, IniFiles, Mask, Menus, DB, DBTables;

type

  PTDateTime = ^TDateTime;

  TDateButton = class( TBitBtn )
  private
  protected
     procedure Click; override;
  public
  published
  end;

  tDbDateEdit = class( TCustomMaskEdit )
  private
     hBitmap: HBitmap;
     FButton: TDateButton;
     FOnDateChange: TNotifyEvent;
     FValColor: TColor;
     lstDates: TList;
     Token: integer;
     {added by tDBEdit}
     FDataLink: TFieldDataLink;
     FCanvas: TControlCanvas;
     FAlignment: TAlignment;
     FFocused: Boolean;
     FTextMargin: Integer;

     procedure SetToken;
     procedure SelectToken;
     procedure SetSeperators;
     {added by tDBEdit}
     procedure CalcTextMargin;
     procedure DataChange(Sender: TObject);
     procedure EditingChange(Sender: TObject);
     function GetDataField: string;
     function GetDataSource: TDataSource;
     function GetField: TField;
     function GetReadOnly: Boolean;
     procedure SetDataField(const Value: String);
     procedure SetDataSource(Value: TDataSource);
     procedure SetFocused(Value: Boolean);
     procedure SetReadOnly(Value: Boolean);
     procedure UpdateData(Sender: TObject);
     procedure WMCut(var Message: TMessage); message WM_CUT;
     procedure WMPaste(var Message: TMessage); message WM_PASTE;
     procedure CMEnter(var Message: TCMEnter); message CM_ENTER;
     procedure CMExit(var Message: TCMExit); message CM_EXIT;
     procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
     procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
  protected
     nSep1, nSep2: integer;
     procedure WMSize( var Message: TWMSize ); message WM_SIZE;
     function GetDate: TDateTime;
     procedure SetDate( dtArg: TDateTime );
     procedure KeyPress( var Key: char ); override;
     procedure MouseUp( Button: TMouseButton; ShiftState: TShiftState; X, Y: integer ); override;
     procedure DoEnter; override;
     {added by tDBEdit}
     procedure Change; override;
     function EditCanModify: Boolean; override;
     procedure KeyDown(var Key: Word; Shift: TShiftState); override;
     procedure Notification(AComponent: TComponent;
       Operation: TOperation); override;
     procedure Reset; override;
     procedure Click; override;
  public
     constructor Create( AOwner: TComponent ); override;
     destructor Destroy; override;
     procedure CreateParams( var Params: TCreateParams ); override;
     property Date: TDateTime read GetDate write SetDate;
     function DateInList( dt: TDateTime ): boolean;
     procedure AddValidDate( dt: TDateTime );
     procedure ClearValidDates;
     {added by tDBEdit}
     property Field: TField read GetField;
  published
     property OnDateChange: TNotifyEvent read FOnDateChange write FOnDateChange;
     property ValidDateColor: TColor read FValColor write FValColor default clMaroon;
     {added by tDBEdit}
     property AutoSelect;
     property AutoSize;
     property BorderStyle;
     property CharCase;
     property Color;
     property Ctl3D;
     property DataField: string read GetDataField write SetDataField;
     property DataSource: TDataSource read GetDataSource write SetDataSource;
     property DragCursor;
     property DragMode;
     property Enabled;
     property Font;
     property MaxLength;
     property ParentColor;
     property ParentCtl3D;
     property ParentFont;
     property ParentShowHint;
     property PasswordChar;
     property PopupMenu;
     property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
     property ShowHint;
     property TabOrder;
     property TabStop;
     property Visible;
     property OnChange;
     property OnClick;
     property OnDblClick;
     property OnDragDrop;
     property OnDragOver;
     property OnEndDrag;
     property OnEnter;
     property OnExit;
     property OnKeyDown;
     property OnKeyPress;
     property OnKeyUp;
     property OnMouseDown;
     property OnMouseMove;
     property OnMouseUp;
  end;

procedure Register;

var
  frmCalendar: TfrmDbCalPop;

implementation

{$R DBDateEd}

{--- TDateButton ---}
procedure TDateButton.Click;
var
  editParent: tDbDateEdit;
begin
  editParent := tDbDateEdit( Parent );
  editParent.Click;
  frmCalendar := TfrmDbCalPop.Create( editParent );
  frmCalendar.ShowModal;
  frmCalendar.Free;
  inherited Click;
end;

{--- tDbDateEdit ---}

constructor tDbDateEdit.Create( AOwner: TComponent );
begin
  inherited Create( AOwner );
  inherited ReadOnly := True; {added by tDBEdit}

{ Get international time seperator }

  Token := 1;
{  FDate := 0.0;}
  FButton := TDateButton.Create( self );
  FButton.Visible := TRUE;
  FButton.Parent := self;
  FButton.Glyph.Handle := LoadBitmap( hInstance, 'DBCALPOPUP' );
  ControlStyle := ControlStyle - [csSetCaption];
  lstDates := TList.Create;
  FValColor := clBlue;
  {added by tDBEdit}
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnEditingChange := EditingChange;
  FDataLink.OnUpdateData := UpdateData;
  CalcTextMargin;
end;

procedure tDbDateEdit.CreateParams( var Params: TCreateParams );
begin
  inherited CreateParams( Params );
  Params.Style := Params.Style or WS_CLIPCHILDREN;
end;

destructor tDbDateEdit.Destroy;
begin
  FButton := nil;
  ClearValidDates;
  lstDates.Free;
  {added by tDBEdit}
  FDataLink.Free;
  FDataLink := nil;
  FCanvas.Free;
  inherited Destroy;
end;

procedure tDbDateEdit.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure tDbDateEdit.WMSize( var Message: TWMSize );
begin
  FButton.Height := Height;
  FButton.Width := Height;
  FButton.Left := Width - Height;
  FButton.Refresh;
{  if Date = 0.0 then
     Date := Now;}
end;

function tDbDateEdit.GetDate: TDateTime;
begin
  GetDate := Field.AsDateTime;
end;

procedure tDbDateEdit.SetDate( dtArg: TDateTime );
var
	FormattedDate : String;
begin
  if Field.AsDateTime <> dtArg then
     begin
        Field.AsDateTime := dtArg;
        Modified := TRUE;
        if Assigned( FOnDateChange ) then
           FOnDateChange( self );
     end;
end;

procedure tDbDateEdit.DoEnter;
begin
  inherited DoEnter;
  Token := 1;
  SetSeperators;
  SelectToken;
end;

(*********************************************
Is the supplied data in the date list?
*********************************************)
function tDbDateEdit.DateInList( dt: TDateTime ): boolean;
var
  pDate: PTDateTime;
  i: integer;
begin
  Result := FALSE;
  for i := 0 to lstDates.Count - 1 do
     begin
        pDate := lstDates[i];
        if pDate^ = dt then
           begin
              Result := TRUE;
              Break;
           end;
     end;
end;

(*********************************************
Maintain list of valid dates.
*********************************************)
procedure tDbDateEdit.AddValidDate( dt: TDateTime );
var
  pDate: PTDateTime;
begin
  New( pDate );
  pDate^ := dt;
  lstDates.Add( PDate );
end;

procedure tDbDateEdit.ClearValidDates;
var
  pDate: PTDateTime;
begin
  while lstDates.Count > 0 do
     begin
        pDate := lstDates[0];
        Dispose( pDate );
        lstDates.Delete( 0 );
     end;
end;

function tDbDateEdit.EditCanModify: Boolean;
begin
  Result := FDataLink.Edit;
end;

procedure tDbDateEdit.Reset;
begin
  FDataLink.Reset;
  SelectAll;
end;

procedure tDbDateEdit.Click;
begin
  FDataLink.Edit;
  inherited Click;
  FDataLink.Modified;
end;

procedure tDbDateEdit.SetFocused(Value: Boolean);
begin
  if FFocused <> Value then
  begin
    FFocused := Value;
    if (FAlignment <> taLeftJustify) and not IsMasked then Invalidate;
    FDataLink.Reset;
  end;
end;

procedure tDbDateEdit.Change;
begin
  FDataLink.Modified;
  inherited Change;
end;

function tDbDateEdit.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure tDbDateEdit.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

function tDbDateEdit.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure tDbDateEdit.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function tDbDateEdit.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure tDbDateEdit.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function tDbDateEdit.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure tDbDateEdit.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
  begin
    if FAlignment <> FDataLink.Field.Alignment then
    begin
      EditText := '';  {forces update}
      FAlignment := FDataLink.Field.Alignment;
    end;
    EditMask := FDataLink.Field.EditMask;
    if FDataLink.Field.DataType = ftString then
      MaxLength := FDataLink.Field.Size else
      MaxLength := 0;
    if FFocused and FDataLink.CanModify then
      Text := FDataLink.Field.Text
    else
      EditText := FDataLink.Field.DisplayText;
  end else
  begin
    FAlignment := taLeftJustify;
    EditMask := '';
    MaxLength := 0;
    if csDesigning in ComponentState then
      EditText := Name else
      EditText := '';
  end;
end;

procedure tDbDateEdit.EditingChange(Sender: TObject);
begin
  inherited ReadOnly := not FDataLink.Editing;
end;

procedure tDbDateEdit.UpdateData(Sender: TObject);
begin
  try
    ValidateEdit;
  except
    on EDBEditError do
    begin
      Text := '';
    end;
  end;
  Try
    FDataLink.Field.Text := Text;
  Except
    FDataLink.Field.AsDateTime := Date;
  End;
end;

procedure tDbDateEdit.WMPaste(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure tDbDateEdit.WMCut(var Message: TMessage);
begin
  FDataLink.Edit;
  inherited;
end;

procedure tDbDateEdit.CMEnter(var Message: TCMEnter);
begin
  SetFocused(True);
  inherited;
end;

procedure tDbDateEdit.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  SetFocused(False);
  SetCursor(0);
  DoExit;
end;

procedure tDbDateEdit.WMPaint(var Message: TWMPaint);
var
  Width, Indent, Left, I: Integer;
  R: TRect;
  DC: HDC;
  PS: TPaintStruct;
  S: string;
begin
  if (FAlignment = taLeftJustify) or FFocused then
  begin
    inherited;
    Exit;
  end;
{ Since edit controls do not handle justification unless multi-line (and
  then only poorly) we will draw right and center justify manually unless
  the edit has the focus. }
  if FCanvas = nil then
  begin
    FCanvas := TControlCanvas.Create;
    FCanvas.Control := Self;
  end;
  DC := Message.DC;
  if DC = 0 then DC := BeginPaint(Handle, PS);
  FCanvas.Handle := DC;
  try
    FCanvas.Font := Font;
    with FCanvas do
    begin
      R := ClientRect;
      if (BorderStyle = bsSingle) then
      begin
        Brush.Color := clWindowFrame;
        FrameRect(R);
        InflateRect(R, -1, -1);
      end;
      Brush.Color := Color;
      S := EditText;
      if PasswordChar <> #0 then
      begin
        for I := 1 to Length(S) do
          S[I] := PasswordChar;
      end;
      Width := TextWidth(S);
      if BorderStyle = bsNone then Indent := 0 else Indent := FTextMargin;
      if FAlignment = taRightJustify then
        Left := R.Right - Width - Indent else
        Left := (R.Left + R.Right - Width) div 2;
      TextRect(R, Left, Indent, S);
    end;
  finally
    FCanvas.Handle := 0;
    if Message.DC = 0 then EndPaint(Handle, PS);
  end;
end;

procedure tDbDateEdit.CMFontChanged(var Message: TMessage);
begin
  inherited;
  CalcTextMargin;
end;

procedure tDbDateEdit.CalcTextMargin;
var
  DC: HDC;
  SaveFont: HFont;
  I: Integer;
  SysMetrics, Metrics: TTextMetric;
begin
  DC := GetDC(0);
  GetTextMetrics(DC, SysMetrics);
  SaveFont := SelectObject(DC, Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  I := SysMetrics.tmHeight;
  if I > Metrics.tmHeight then I := Metrics.tmHeight;
  FTextMargin := I div 4;
end;

procedure tDbDateEdit.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if (Key = VK_DELETE) or ((Key = VK_INSERT) and (ssShift in Shift)) then
    FDataLink.Edit;
end;

procedure tDbDateEdit.KeyPress( var Key: char );
begin
  {altered by tDBEdit}
  if ((Key in ['0'..'9']) or (Key=DateSeparator))and (FDataLink.Field <> nil)
    and not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, '0'..'9':
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
    else begin
      If Key=DateSeparator
      then begin
        if Token < 3
        then begin
          Inc( Token );
          SetSeperators;
          SelectToken;
          Key := #0;
        end
        else
          Key := #0;
      end;
    end;
  end;
  inherited KeyPress( Key );
end;

(*********************************************
Determine which token the user is on and highlight
the entire text of that token.
*********************************************)
procedure tDbDateEdit.MouseUp( Button: TMouseButton; ShiftState: TShiftState; X, Y: integer );
begin
  SetToken;
  SelectToken;
  inherited MouseUp( Button, ShiftState, X, Y );
end;

(*********************************************
Set the positions of the seperators in text.
*********************************************)
procedure tDbDateEdit.SetSeperators;
var
  i: integer;
begin
  nSep1 := Pos( DateSeparator, Text );
  for i := nSep1 + 1 to Length( Text ) do
     if Text[i] = DateSeparator then
        begin
           nSep2 := i;
           Break;
        end;
end;

(*********************************************
Determine which token the cursor is over;
*********************************************)
procedure tDbDateEdit.SetToken;
var
  nPos: integer;
begin
  nPos := SendMessage( Handle, cb_GetEditSel, 0, 0 ) div 65536;
  SetSeperators;
  if nPos <= nSep1 then
     Token := 1
  else if nPos <= nSep2 then
     Token := 2
  else
     Token := 3;
end;

(*********************************************
Select the token the cursor is on.
*********************************************)
procedure tDbDateEdit.SelectToken;
begin
  case Token of
     1:
        SendMessage( Handle, em_SetSel, 0, ( nSep1 - 1 ) * 65536 );
     2:
        SendMessage( Handle, em_SetSel, 0, ( nSep1 + ( nSep2 - 1 ) * 65536 ) );
     3:
        SendMessage( Handle, em_SetSel, 0, nSep2 + ( ( Length( Text ) ) * 65536 ) );
  end;
end;

procedure Register;
begin
  RegisterComponents('Freeware', [tDbDateEdit]);
end;

end.

