unit TablPack;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, DBTables, DbiTypes, DbiProcs, DbiErrs;

type
  { Exceptions for TablePack }
  EPackError = class(Exception);
  EPack_InvalidParam = class(EPackError);
  EPack_InvalidHndl = class(EPackError);
  EPack_InvalidDBSpec = class(EPackError);
  EPack_NoConfigFile = class(EPackError);
  EPack_NoSuchTable = class(EPackError);
  EPack_NotSupported = class(EPackError);
  EPack_UnknownTblType = class(EPackError);
  EPack_UnknownDB = class(EPackError);
  EPack_NeedExclAccess = class(EPackError);
  EPack_IncorrectTblType = class(EPackError);
  EPack_DBLimit = class(EPackError);

  TTablePack = class(TTable)
  private
    { Private declarations }
    function GetTableType: PChar;
    procedure PackDBaseTable;
    procedure PackParadoxTable;
  protected
    { Protected declarations }
    function Chk(rslt: DbiResult): DbiResult; virtual;
  public
    { Public declarations }
    procedure Pack;
  published
    { Published declarations }
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Data Access', [TTablePack]);
end;

function TTablePack.GetTableType: PChar;
var
  { FCurProp Holds information about the structure of the table }
  FCurProp: CurProps;

begin
  { Find out what type of table is currently opened.  NOTE: This is
    different than TTablePack.TableType }
  Chk(DbiGetCursorProps(Handle, FCurProp));
  GetTableType := FCurProp.szTableType;
end;

procedure TTablePack.Pack;
var
  TType: array[0..40] of char;

begin
  { The table must be opened to get directory information about the table
    before closing }
  if Active <> True then
    raise EPackError.Create('Table must be opened to be packed');
  { Get the type of table }
  strcopy(TType, GetTableType);
  if strcomp(TType, szParadox) = 0 then
    { Call PackParadoxTable procedure if PARADOX table }
    PackParadoxTable
  else
    if strcomp(TType, szDBase) = 0 then
      { Call PackDBaseTable procedure if dBase table }
      PackDBaseTable
    else
      { PARADOX and dBase table are the only types that can be packed }
      raise EPack_IncorrectTblType.Create('Incorrect table type: ' +
                                          StrPas(TType));
end;

procedure TTablePack.PackParadoxTable;
var
  { Specific information about the table structure, indexes, etc. }
  TblDesc: CRTblDesc;
  { Uses as a handle to the database }
  hDb: hDbiDb;
  { Path to the currently opened table }
  TablePath: array[0..dbiMaxPathLen] of char;

begin
  { Initialize the table descriptor }
  FillChar(TblDesc, SizeOf(CRTblDesc), 0);
  with TblDesc do
  begin
    { Place the table name in descriptor }
    StrPCopy(szTblName, TableName);
    { Place the table type in descriptor }
    StrCopy(szTblType, GetTableType);
    { Set the packing option to true }
    bPack := True;
  end;
  { Initialize the DB handle }
  hDb := nil;
  { Get the current table's directory.  This is why the table MUST be
    opened until now }
  Chk(DbiGetDirectory(DBHandle, True, TablePath));
  { Close the table }
  Close;
  { NOW: since the DbiDoRestructure call needs a valid DB handle BUT the
    table cannot be opened, call DbiOpenDatabase to get a valid handle.
    Just leaving Active = FALSE does not give you a valid handle }
  Chk(DbiOpenDatabase(nil, 'STANDARD', dbiReadWrite, dbiOpenExcl, nil,
                        0, nil, nil, hDb));
  { Set the table's directory to the old directory }
  Chk(DbiSetDirectory(hDb, TablePath));
  { Pack the PARADOX table }
  Chk(DbiDoRestructure(hDb, 1, @TblDesc, nil, nil, nil, FALSE));
  { Re-Open the table }
  Open;
end;

procedure TTablePack.PackDBaseTable;
begin
  { Pack the dBase Table }
  Chk(DbiPackTable(DBHandle, Handle, nil, nil, True));
end;

function TTablePack.Chk(rslt: DbiResult): DbiResult;
var
  ErrInfo: DbiErrInfo;
  ErrStr: string;
  pErr: array[0..dbiMaxMsgLen] of char;

begin
  { Only enter the error routine if an error has occured }
  if rslt <> dbiErr_None then
  begin
    { Get information on the error that occured.  ALWAYS DO THIS FIRST!! }
    DbiGetErrorInfo(False, ErrInfo);
    if ErrInfo.iError = rslt then
    begin
      ErrStr := Format('%s  ', [ErrInfo.szErrCode]);
      if StrComp(ErrInfo.szContext[1], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[1]]);
      if StrComp(ErrInfo.szContext[2], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[2]]);
      if StrComp(ErrInfo.szContext[3], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[3]]);
      if StrComp(ErrInfo.szContext[4], '') = 0 then
        ErrStr := Format('%s  %s', [ErrStr, ErrInfo.szContext[4]]);
    end
    else
    begin
      DbiGetErrorString(rslt, pErr);
      ErrStr := StrPas(pErr);
    end;
    ErrStr := Format('Table Pack Error: %d.  %s', [rslt, ErrStr]);

    MessageBeep(mb_IconExclamation);
    { Raise the corrisponding exception }
    case rslt of
      dbiErr_InvalidParam:
        raise EPack_InvalidParam.Create(ErrStr);
      dbiErr_InvalidHndl:
        raise EPack_InvalidHndl.Create(ErrStr);
      dbiErr_InvalidDBSpec:
        raise EPack_InvalidDBSpec.Create(ErrStr);
      dbiErr_NoSuchTable:
        raise EPack_NoSuchTable.Create(ErrStr);
      dbiErr_NoConfigFile:
        raise EPack_NoConfigFile.Create(ErrStr);
      dbiErr_NotSupported:
        raise EPack_NotSupported.Create(ErrStr);
      dbiErr_UnknownTblType:
        raise EPack_UnknownTblType.Create(ErrStr);
      dbiErr_UnknownDB:
        raise EPack_UnknownDB.Create(ErrStr);
      dbiErr_NeedExclAccess:
        raise EPack_NeedExclAccess.Create(ErrStr);
      dbiErr_DBLimit:
        raise EPack_DBLimit.Create(ErrStr);
      else
        raise EPackError.Create(ErrStr);
    end;
  end;
end;

end.
