{
+----------------------------------------------------------------------------+
|                                                                          |
|                                                                       |
{
|                                                                      |
|                                                                       |
|                                                                       |
|                                                                    |
|                                                             |
|                                                        |
|                                                      |
|                       Copyright  1996-1997 by:  |
|                                                  |
|                           WHITE ANTS SYSTEMHOUSE BV  |
|                            Geleen 12                  |
|                                  8032 GB Zwolle             |
|                                        Netherlands                |
|                                                               |
|                                         Tel. +31 38 453 86 31      |
|                                              Fax. +31 38 453 41 22      |
|                                                                        |
|                                             www.whiteants.com          |
|                                            support@whiteants.com      |
|                                                                           |
+----------------------------------------------------------------------------+
  file     : Filters
  version  : 1.0
  comment  : Implements and extends the BP 7.0 Streaming mechanism in Delphi
  author   : G. Beuze
  compiler : Delphi 1.0
+----------------------------------------------------------------------------+
| DISCLAIMER:                                                                |
| THIS SOURCE IS FREEWARE. YOU ARE ALLOWED TO USE IT IN YOUR OWN PROJECTS    |
| WITHOUT ANY RESTRICTIONS. YOU ARE NOT ALLOWED TO SELL THE SOURCE CODE.     |
| THERE IS NO WARRANTY AT ALL - YOU USE IT ON YOUR OWN RISC. WHITE ANTS DOES |
| NOT ASSUME ANY RESPONSIBILITY FOR ANY DAMAGE OR ANY LOSS OF TIME OR MONEY  |
| DUE THE USE OF ANY PART OF THIS SOURCE CODE.                               |
+----------------------------------------------------------------------------+
}

{ TFilter is a TStream descendant which add's the Put / Get object streaming
  capabilities to a TStream instance. TFilter does not actually read or write data
  but delegates this to an attached stream. This stream can be owned or not.

  Filters adhere to the decorator pattern.
  Filters can be cascaded:
     Filter1 => Filter2 => Filter3 => TStream. (such as TFileStream)

  Some additional filters are included:
  - BufferedFilter buffers it's IO
  - BufFileFilter is a combination of a TBufferedFilter with a TFileStream
  - TCycleFilter is a filter that is able to read and write cyclic object structures
    like: Parent saves children and children save their parent.
  - TClipBrdFilter reads and writes is a binary clipboard stream, allowing you
    to stream classes to the clipboard the same way you stream them to a file
  - TOffsetFilter which lets you treat a section in an other stream as
    new complete stream, including Position := 0, Size etc.

  Utility procedures include
  - Reading and writing of PChar and TStrings from/to a TFilter
  - Copying a TStreamable instance by first putting it to memory and then
    getting it back.

  For classes to be streamable, they MUST be a TStreamable descendant AND
  be registered as streamable using the RegisterStreamable procedure.
}

unit Filters;

interface

uses Classes, SysUtils;

type
  TStreamable = class;

  TFilter = class(TStream)
  private
    FOwnsStream: Boolean;
    FStream: TStream;
  protected
    procedure RegisterInstance(Instance: TStreamable); virtual;
    procedure SetStream(Value: TStream); virtual;
  public
    constructor Create(AStream: TStream);
    destructor Destroy; override;
    { Get returns an TStreamable instance which was read from the stream }
    function Get: TStreamable; virtual;
    { Put writes the instance to the stream }
    procedure Put(Instance: TStreamable); virtual;
    { Read wraps the Stream's read method }
    function Read(var Buffer; Count: Longint): Longint; override;
    { Reads a PASCAL style string (length byte followed by length chars)
      from Stream's using the read method }
    function ReadStr: String;
    { Seek wraps the Stream's seek method }
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    { Truncate is not implemented, but gives a common interface to truncate
      at the current position }
    procedure Truncate; virtual;
    { Write wraps the Stream's Write method }
    function Write(const Buffer; Count: Longint): Longint; override;

    procedure WriteStr(const Value: string);
    property OwnsStream: Boolean read FOwnsStream write FOwnsStream;
    property Stream: TStream read FStream write SetStream;
 end;


  TStreamable = class(TPersistent)
    constructor Load(S: TFilter); virtual;
    procedure Store(S: TFilter); virtual;
  end;

  TStreamableClass = class of TStreamable;

  { Streamable registration exception class }

  EStreamableException = class(EStreamError);

type
  TCycleFilter = class(TFilter)
  private
    FReferences: TList;
  protected
    procedure RegisterInstance(Instance: TStreamable); override;
  public
    constructor Create(AStream: TStream);
    destructor Destroy; override;
    function Get: TStreamable; override;
    procedure Put(Instance: TStreamable); override;
  end;

type
  TBufferedFilter = class (TFilter)
  private
    FBasePos: LongInt;
    FBaseSize: LongInt;
    FBufCnt: Word;
    FBufEnd: LongInt;
    FBuffer: PByteArray;
    FBufModified: Boolean;
    FBufSize: Word;
    FBufStart: LongInt;
    FCurPos: LongInt;
    procedure SetBufStart(Value: LongInt);
  protected
    procedure LoadBuffer; virtual;
    procedure RepositionBuffer;
    procedure SetStream(Value: TStream); override;
    property BufStart: LongInt read FBufStart write SetBufStart;
  public
    constructor Create(AStream: TStream; BufSize: Word);
    destructor Destroy; override;
    procedure Flush; virtual;
    function Read(var Buffer; Count: LongInt): LongInt; override;
    function Seek(Offset: LongInt; Origin: Word): LongInt; override;
    function Write(const Buffer; Count: LongInt): LongInt; override;
  end;

type
  TBufFileStream = class(TBufferedFilter)
    constructor Create(const FileName: string; Mode, ABufSize: Word);
  end;

const
  CF_STREAMABLE : Word = 0; { Clipboard format for streamables }

type
  TClipbrdFilterMode = (cmRead, cmWrite);

  EClipbrdFilterError = class (EStreamError);

  TClipbrdFilter = class(TFilter)
  private
    FMode: TClipbrdFilterMode;
    procedure ReadFromClipboard;
    procedure WriteToClipboard;
  public
    constructor Create(AMode: TClipbrdFilterMode);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
  end;

const
  UnlimitedRange = -1;

type
  TOffsetFilter = class (TFilter)
  private
    FBaseOffset: LongInt;
    FCurPos: LongInt;
    FRange: LongInt;
  protected
    function GetLimitedRange: Boolean;
    procedure RangeError;
    procedure SetBaseOffset(Value: LongInt);
    procedure SetRange(Value: LongInt);
    procedure SetStream(Value: TStream);
    procedure Validate;
  public
    constructor Create(AStream: TStream; AOffset, ARange: LongInt);
    function Read(var Buffer; Count: LongInt): LongInt; override;
    function Seek(Offset:LongInt; Origin: Word): LongInt; override;
    function Write(const Buffer; Count: LongInt): LongInt; override;
    property BaseOffset: LongInt read FBaseOffset write SetBaseOffset;
    property LimitedRange: Boolean read GetLimitedRange;
    property Range: LongInt read FRange write SetRange;
  end;


function CopyStreamable(Instance: TStreamable): TStreamable;

procedure LoadStrings(Filter: TFilter; List: TStrings);

procedure StoreStrings(Filter: TFilter; List: TStrings);

function LoadPChar(Stream: TStream): PChar;

procedure StorePChar(Stream: TStream; Source: PChar);

function FindStreamable(const ClassName: String): TStreamableClass;

function GetStreamable(const ClassName: String): TStreamableClass;

procedure RegisterStreamable(AClass: TStreamableClass);

procedure RegisterStreamables(AClasses: array of TStreamableClass);

procedure UnregisterStreamables; far;

implementation

uses WinTypes, WinProcs, Clipbrd, NumUtils;

{
*********************** Streamable registration ******************************
}
var
  StreamableList: TList;

procedure StreamableNotFound(const ClassName: string);
begin
  raise EStreamableException.Create(ClassName + ' not registered as streamable');
end;

function FindStreamable(const ClassName: string): TStreamableClass;
begin
  Result := GetStreamable(ClassName);
  if Result = nil then StreamableNotFound(ClassName);
end;

function GetStreamable(const ClassName: string): TStreamableClass;
var
  I: Integer;
begin
  for I := 0 to StreamableList.Count - 1 do
  begin
    Result := StreamableList[I];
    if CompareText(Result.ClassName, ClassName) = 0 then Exit;
  end;
  Result := nil;
end;

procedure RegisterStreamable(AClass: TStreamableClass);
var
  ClassName: string[63];
begin
  if not Assigned(StreamableList) then
    StreamableList := TList.Create;
  while StreamableList.IndexOf(AClass) = -1 do
  begin
    ClassName := AClass.ClassName;
    if GetStreamable(ClassName) = nil then
      StreamableList.Add(AClass);
    if AClass = TStreamable then Break;
    AClass := TStreamableClass(AClass.ClassParent);
  end;
end;

procedure RegisterStreamables(AClasses: array of TStreamableClass);
var
  I: Integer;
begin
  for I := Low(AClasses) to High(AClasses) do RegisterStreamable(AClasses[I]);
end;

procedure UnregisterStreamables;
begin
  StreamableList.Free;
  StreamableList := nil;
end;

function CopyStreamable(Instance: TStreamable): TStreamable;
var S: TStream;
    F: TFilter;
begin
  Result := nil;
  S := TMemoryStream.Create;
  try
    F := TFilter.Create(S);
    try
      F.Put(Instance);
      F.Seek(0, 0);
      Result := F.Get;
    finally
      F.Free;
    end;
  finally
    S.Free;
  end;
end;

procedure LoadStrings(Filter: TFilter; List: TStrings);
var I, LineCnt: Integer;
begin
  List.Clear;
  Filter.ReadBuffer(LineCnt, SizeOf(LineCnt));
  if (LineCnt < 0) or (LineCnt > MaxListSize) then
    raise EStreamError.Create('Error reading TStrings from stream');
  for I := 0 to LineCnt - 1 do
    List.Add(Filter.ReadStr);
end;

procedure StoreStrings(Filter: TFilter; List: TStrings);
var I, LineCnt: Integer;
begin
  LineCnt := List.Count;
  Filter.WriteBuffer(LineCnt, SizeOf(LineCnt));
  for I := 0 to LineCnt - 1 do
    Filter.WriteStr(List[I]);
end;

function LoadPChar(Stream: TStream): PChar;
var Cnt: Word;
begin
  Stream.ReadBuffer(Cnt, SizeOf(Cnt));
  if Cnt > 0 then
  begin
    Inc(Cnt);
    Result := StrAlloc(Cnt);
    Stream.Read(Result[0], Cnt);
  end
  else
    if Cnt < 0 then
      raise EStreamError.Create('Error reading PChar from stream')
    else
      Result := nil;
end;

procedure StorePChar(Stream: TStream; Source: PChar);
var Cnt: Word;
begin
  if Assigned(Source) then Cnt := StrLen(Source) else Cnt := 0;
  Stream.WriteBuffer(Cnt, SizeOf(Cnt));
  if Cnt > 0 then
    Stream.WriteBuffer(Source[0], Cnt + 1);
end;

{
****************************** TFilter ***************************************
}
constructor TFilter.Create(AStream: TStream);
begin
  inherited Create;
  Stream := AStream;
  FOwnsStream := False;
end;

destructor TFilter.Destroy;
begin
  Stream := nil; { calls SetStream }
  inherited Destroy;
end;

function TFilter.Get: TStreamable;
begin
  Result := FindStreamable(ReadStr).Load(Self);
end;

procedure TFilter.Put(Instance: TStreamable);
begin
  WriteStr(Instance.ClassName);
  Instance.Store(Self);
end;

function TFilter.Read(var Buffer; Count: Longint): Longint;
begin
  if Assigned(FStream) then
    Result := FStream.Read(Buffer, Count)
  else
    Read := 0;
end;

function TFilter.ReadStr: string;
begin
  ReadBuffer(Result[0], 1);
  ReadBuffer(Result[1], Ord(Result[0]));
end;

procedure TFilter.RegisterInstance(Instance: TStreamable);
begin
  if Assigned(FStream) and (FStream is TFilter) then
    TFilter(FStream).RegisterInstance(Instance);
end;

function TFilter.Seek(Offset: Longint; Origin: Word): Longint;
begin
  if Assigned(FStream) then
    Result := Stream.Seek(Offset, Origin)
  else
    Result := 0;
end;

procedure TFilter.SetStream(Value: TStream);
begin
  if FOwnsStream then FStream.Free; { Allright even if FStream = nil }
  FStream := Value;
end;

procedure TFilter.Truncate;
begin
end;

function TFilter.Write(const Buffer; Count: Longint): Longint;
begin
  if Assigned(FStream) then
    Result := Stream.Write(Buffer, Count)
  else
    Result := 0;
end;

procedure TFilter.WriteStr(const Value: string);
begin
  WriteBuffer(Value[0], Ord(Value[0]) + 1);
end;

{
******************************** TStreamable *********************************
}
constructor TStreamable.Load(S: TFilter);
begin
  Create;
  S.RegisterInstance(Self);
end;

procedure TStreamable.Store(S: TFilter);
begin
end;

{
********************************* TCycleFilter *********************************
}
constructor TCycleFilter.Create(AStream: TStream);
begin
  inherited Create(AStream);
  FReferences := TList.Create;
end;

destructor TCycleFilter.Destroy;
begin
  FReferences.Free;
  inherited Destroy;
end;

function TCycleFilter.Get: TStreamable;
var
  Idx: Integer;
begin
  ReadBuffer(Idx, SizeOf(Idx));
  if Idx = -1 then
  begin
    Idx := FReferences.Count;
    FReferences.Add(nil);
    FindStreamable(ReadStr).Load(Self);
  end;
  if Idx = -2 then
    Result := nil
  else
    Result := FReferences.Items[Idx];
end;

procedure TCycleFilter.Put(Instance: TStreamable);
var
  Idx: Integer;
begin
  if Assigned(Instance) then
    Idx := FReferences.IndexOf(Instance)
  else
    Idx := -2;
  WriteBuffer(Idx, SizeOf(Idx));
  if Idx = -1 then
  begin
    FReferences.Add(Instance);
    inherited Put(Instance)
  end;
end;

procedure TCycleFilter.RegisterInstance(Instance: TStreamable);
begin
  FReferences.Items[FReferences.Count - 1] := Instance;
  inherited RegisterInstance(Instance);
end;

{
************************************ TBufferedFilter ************************************
}
constructor TBufferedFilter.Create(AStream: TStream; BufSize: Word);
begin
  inherited Create(AStream);
  FBufSize := BufSize;
  if FBufSize = 0 then
    raise EStreamError.Create('Buffer size of 0 not allowed');
  GetMem(FBuffer, FBufSize);
end;

destructor TBufferedFilter.Destroy;
begin
  Flush;
  FreeMem(FBuffer, FBufSize);
  FBufSize := 0;
  FBuffer := nil;
  inherited Destroy;
end;

procedure TBufferedFilter.Flush;
begin
  if FBufModified and (FBufCnt > 0) then
  begin
    if FBasePos <> FBufStart then Stream.Position := FBufStart;
    if (Stream.Write(FBuffer^, FBufCnt) = FBufCnt) then
    begin
      FBufModified := False;
      FBasePos := FBufStart + FBufCnt;
    end
    else
    begin
      FBasePos := Stream.Position;
      raise EStreamError.Create('Unable to flush buffer.');
    end;
  end;
end;

procedure TBufferedFilter.LoadBuffer;
var CurBufPos: LongInt;
begin
  CurBufPos := FBufStart + FBufCnt;
  if CurBufPos < FBaseSize then
  begin
    if FBasePos <> CurBufPos then Stream.Position := CurBufPos;
    FBufCnt := FBufCnt + Stream.Read(FBuffer^[FBufCnt],
                         MinLong(FBufSize - FBufCnt, FBaseSize - CurBufPos));
    FBasePos := FBufStart + FBufCnt;
  end;
end;

function TBufferedFilter.Read(var Buffer; Count: LongInt): LongInt;
var
  Rest: LongInt;
  M: Word;
begin
  if InRange(FCurPos, FBufStart, FBufEnd - 1) then
  begin
    if FBufCnt < FBufSize then LoadBuffer; { Make sure buffer is filled completely }
  end
  else
  begin
    Flush;
    RepositionBuffer;
    LoadBuffer;
  end;
  Rest := Count;
  while Rest > 0 do
  begin
    M := MinLong(FBufStart + FBufCnt - FCurPos, Rest);
    SYSTEM.Move(FBuffer^[FCurPos - FBufStart], TByteArray(Buffer)[Count - Rest] , M);
    Inc(FCurPos, M);
    if (FCurPos >= FBufStart + FBufCnt) then
    begin
      Flush;
      RepositionBuffer;
      LoadBuffer;
    end;
    Dec(Rest, M);
    if (Rest > 0) and (FBufCnt = 0) then Break; { Read at EOF }
  end;
  Result := Count - Rest;
end;

procedure TBufferedFilter.RepositionBuffer;
begin
  FBufCnt := 0;
  SetBufStart(FCurPos);
end;

function TBufferedFilter.Seek(Offset: LongInt; Origin: Word): LongInt;
begin
  case Origin of
    0 : FCurPos := Offset;
    1 : FCurPos := FCurPos + Offset;
    2 : FCurPos := FBaseSize + Offset;
  end;
  FCurPos := LimitToRange(FCurPos, 0, FBaseSize);
  Result := FCurPos;
end;

procedure TBufferedFilter.SetBufStart(Value: LongInt);
begin
  FBufStart := Value;
  FBufEnd := FBufStart + FBufSize;
end;

procedure TBufferedFilter.SetStream(Value: TStream);
var ReadTestBuf: Byte;
begin
  if Stream <> nil then Flush;
  inherited SetStream(Value);
  FBufModified := False;
  if Stream <> nil then
  begin
    FBaseSize := Stream.Size;
    FCurPos := Stream.Position;
    { test on read only streams }
    if FBaseSize > 0 then
    begin
      Stream.Position := 0;
      if Stream.Read(ReadTestBuf, 1) = 1 then
        Stream.Position := FCurPos
      else
        raise EStreamError.Create('Buffering of write only stream might cause problems');
    end;
  end
  else
  begin
    FCurPos := 0;
    FBaseSize := 0;
  end;
  FBasePos := FCurPos;
  RepositionBuffer;
end;

function TBufferedFilter.Write(const Buffer; Count: LongInt): LongInt;
var
  Rest: LongInt;
  M: Word;
begin
  if InRange(FCurPos, FBufStart, FBufEnd - 1) then
  begin
    if (FCurPos > FBufStart + FBufCnt) then LoadBuffer; { to avoid creating a gap }
  end
  else
  begin
    Flush;
    RepositionBuffer;
  end;
  Rest := Count;
  while Rest > 0 do
  begin
    FBufModified := True;
    M := MinLong(FBufEnd - FCurPos, Rest);
    Move(TByteArray(Buffer)[Count - Rest], FBuffer^[FCurPos - FBufStart], M);
    Inc(FCurPos, M);
    FBaseSize := MaxLong(FBaseSize, FCurPos);
    FBufCnt := MaxWord(FBufCnt, FCurPos - FBufStart);
    if (FCurPos >= FBufEnd) then
    begin
      Flush;
      RepositionBuffer;
    end;
    Dec(Rest, M);
  end;
  Result := Count;
end;

{
****************************** TObjectFileStream *****************************
}
constructor TBufFileStream.Create(const FileName: string; Mode, ABufSize: Word);
begin
  if ((Mode and $000F) = fmOpenWrite) then Mode := (Mode and $FFF0) or fmOpenReadWrite;
  try
    inherited Create(TFileStream.Create(FileName, Mode), ABufSize);
  finally
    OwnsStream := True;
  end;
end;

{
************************* TClipbrdFilter **********************************
}
constructor TClipbrdFilter.Create(AMode: TClipbrdFilterMode);
var ASize: LongInt;
begin
  inherited Create(TMemoryStream.Create);
  OwnsStream := True;
  FMode := AMode;
  if FMode = cmRead then ReadFromClipboard;
end;

destructor TClipbrdFilter.Destroy;
begin
  if FMode = cmWrite then WriteToClipboard;
  inherited Destroy;
end;

function TClipbrdFilter.Read(var Buffer; Count: Longint): Longint;
begin
  if FMode = cmRead then
    Result := inherited Read(Buffer, Count)
  else
    raise EClipbrdFilterError.Create('Can not read from write only stream')
end;

procedure TClipbrdFilter.ReadFromClipboard;
var Data: THandle;
    DataPtr: Pointer;
    ASize: Longint;
begin
  if Clipboard.HasFormat(CF_STREAMABLE) then
  begin
    Data := Clipboard.GetAsHandle(CF_STREAMABLE);
    if Data = 0 then Exit;
    DataPtr := GlobalLock(Data);  { Returns pointer referred to by Data }
    try
      ASize := GlobalSize(Data);
      if ASize >= SizeOf(ASize) then ASize := LongInt(DataPtr^) else ASize := 0;
      if ASize > 0 then
      begin
        Stream.Position := 0;
        Stream.WriteBuffer(PByteArray(DataPtr)^[SizeOf(ASize)], ASize);
        Stream.Position := 0;
      end;
    finally
      GlobalUnlock(Data);  { Always unlock Data }
    end;
  end;
end;

function TClipbrdFilter.Write(const Buffer; Count: Longint): Longint;
begin
  if FMode = cmWrite then
    Result := inherited Write(Buffer, Count)
  else
    raise EClipbrdFilterError.Create('Can not write to read only stream')
end;

procedure TClipbrdFilter.WriteToClipboard;
var
  Data: THandle;
  DataPtr: Pointer;
  ASize: LongInt;
begin
  Clipboard.Open;
  try
    { Write streamables }
    if Stream.Size > 0 then
    begin
      ASize := Stream.Size;
      Data := GlobalAlloc(GMEM_MOVEABLE, Stream.Size + SizeOf(ASize));
      try
        DataPtr := GlobalLock(Data);
        try
          hmemcpy(DataPtr, @ASize, SizeOf(ASize));
          hmemcpy(@(PByteArray(DataPtr)^[SizeOf(ASize)]),
                  (Stream as TMemoryStream).Memory, ASize);
          Clipboard.SetAsHandle(CF_STREAMABLE, Data);
        finally
          GlobalUnlock(Data);
        end;
      except
        GlobalFree(Data);
        raise;
      end;
    end;
  finally
    Clipboard.Close;
  end;
end;

{
************************************* TOffsetFilter *************************************
}
constructor TOffsetFilter.Create(AStream: TStream; AOffset, ARange: LongInt);
begin
  inherited Create(AStream);
  FBaseOffset := AOffset;
  FRange := ARange;
  Validate;
end;

function TOffsetFilter.GetLimitedRange: Boolean;
begin
  Result := FRange <> UnlimitedRange;
end;

procedure TOffsetFilter.RangeError;
begin
  raise EStreamError.Create('Encountered invalid position in OffsetFilter');
end;

function TOffsetFilter.Read(var Buffer; Count: LongInt): LongInt;
begin
  if LimitedRange and (Count > FRange - FCurPos) then Count := FRange - FCurPos;
  Result := Stream.Read(Buffer, Count);
  Inc(FCurPos, Result);
end;

function TOffsetFilter.Seek(Offset:LongInt; Origin: Word): LongInt;
begin
  case Origin of
    0 : begin
          if LimitedRange then
            if not InRange(Offset, 0, FRange) then RangeError else
          else
            if Offset < 0 then RangeError;
          FCurPos := Stream.Seek(Offset + FBaseOffset, 0);
        end;
    1 : begin
          if LimitedRange then
            if not InRange(Offset + FCurPos, 0, FRange) then RangeError else
          else
            if Offset + FCurPos < 0 then RangeError;
          FCurPos := Stream.Seek(Offset, 1);
        end;
    2 : begin
          if LimitedRange then
          begin
            if not InRange(-Offset, 0, FRange) then RangeError;
            FCurPos := Stream.Seek(FBaseOffset + FRange + Offset, 0);
          end
          else
          begin
            if -Offset > FRange then RangeError;
            FCurPos := Stream.Seek(Offset, 2);
          end;
        end;
  end;
  Dec(FCurPos, FBaseOffset);
  Result := FCurPos;
end;

procedure TOffsetFilter.SetBaseOffset(Value: LongInt);
begin
  FBaseOffset := Value;
  Validate;
end;

procedure TOffsetFilter.SetRange(Value: LongInt);
begin
  if Value < UnlimitedRange then raise EStreamError.Create('Negative ranges not allowed');
  FRange := Value;
  Validate;
end;

procedure TOffsetFilter.SetStream(Value: TStream);
begin
  inherited SetStream(Value);
  Validate;
end;

procedure TOffsetFilter.Validate;
var
  ASize: LongInt;
begin
  if Stream <> nil then
  begin
    ASize := Stream.Size;
    if ASize < FBaseOffset then
      raise EStreamError.Create('Illegal base offset (OffsetFilter)');
    if LimitedRange and (ASize < FBaseOffset + FRange) then
      raise EStreamError.Create('Illegal range (OffsetFilter)');
    Stream.Position := FBaseOffset;
    FCurPos := 0;
  end;
end;

function TOffsetFilter.Write(const Buffer; Count: LongInt): LongInt;
begin
  if LimitedRange and (Count > FRange - FCurPos) then Count := FRange - FCurPos;
  Result := Stream.Write(Buffer, Count);
  FCurPos := Result;
end;

initialization
  CF_STREAMABLE := RegisterClipboardFormat('Streamable Object');
  StreamableList := TList.Create;
  AddExitProc(UnregisterStreamables);
end.
