{
+----------------------------------------------------------------------------+
|                                                                          |
|                                                                       |
|                                                                      |
|                                                                       |
|                                                                       |
|                                                                    |
|                                                             |
|                                                        |
|                                                      |
|                       Copyright  1996-1997 by:  |
|                                                  |
|                           WHITE ANTS SYSTEMHOUSE BV  |
|                            Geleen 12                  |
|                                  8032 GB Zwolle             |
|                                        Netherlands                |
|                                                               |
|                                         Tel. +31 38 453 86 31      |
|                                              Fax. +31 38 453 41 22      |
|                                                                        |
|                                             www.whiteants.com          |
|                                            support@whiteants.com      |
|                                                                           |
+----------------------------------------------------------------------------+
  file     : CONTAINR
  version  : 1.0
  comment  : Replaces and extends BP 7.0 collections
  author   : G. Beuze
  compiler : Delphi 1.0
+----------------------------------------------------------------------------+
| DISCLAIMER:                                                                |
| THIS SOURCE IS FREEWARE. YOU ARE ALLOWED TO USE IT IN YOUR OWN PROJECTS    |
| WITHOUT ANY RESTRICTIONS. YOU ARE NOT ALLOWED TO SELL THE SOURCE CODE.     |
| THERE IS NO WARRANTY AT ALL - YOU USE IT ON YOUR OWN RISC. WHITE ANTS DOES |
| NOT ASSUME ANY RESPONSIBILITY FOR ANY DAMAGE OR ANY LOSS OF TIME OR MONEY  |
| DUE THE USE OF ANY PART OF THIS SOURCE CODE.                               |
+----------------------------------------------------------------------------+
}
unit Containr;

interface

uses Classes, SysUtils, Filters;

type
  EContainerError = class(Exception);

  TContainerDuplicates = (dupIgnore, dupAccept, dupDestroy, dupError);

  TCheckNewKeyEvent = function(CurKey, NewKey: Pointer): Boolean of object;

  TContainer = class(TStreamable)
  private
    FCanSort: Boolean;
    FDuplicates: TContainerDuplicates;
    FOnChange: TNotifyEvent;
    FOnStyleChange: TNotifyEvent;
    FOwnesItems: Boolean;
    FSorted: Boolean;
    FUpdateCount: Integer;
    function CheckDuplicates(Item: Pointer): Pointer;
    procedure DeleteDuplicates;
    procedure QuickSort(L, R: Integer);
    procedure SetSorted(Value: Boolean);
    procedure SetCanSort(Value: Boolean);
    procedure SetDuplicates(Value: TContainerDuplicates);
  protected
    procedure Changed; virtual;
      { Called whenever the list is changed. Calls FOnChanged if it is assigned
        Change mechanism is locked when FUpdateCount > 0. Note: Changed does NOT
        check for changes in the contents of the list. Changes in items should
        be reflected by replacing them in the Container using the Put method.
        This will also take care of correct sorting if a key field was changed }
    function CheckNewKey(CurKey, NewKey: Pointer): Boolean; virtual;
      { Could be called by items in a duplicate restricted list in order to
        validate a change of their key. }
    function Compare(Key1, Key2: Pointer): Integer; virtual;
      { Returns 0 if Key1 = Key2. Descendants should override this method
        in order to create some sorting order if required }
    procedure FreeItem(Item: Pointer); virtual;
      { Called whenever an owned item has to be disposed of. Assumes Item to be
        a TObject descendnat and calls it's Free method }
    procedure FreeDataStructure; virtual;
      { Called form Destroy ONLY. Calls Clear if FOwnesItems is True else RemoveAll.
        Descendants should free their data structure and also free their items
        if FOwnesItems = True }
    procedure FreeDuplicate(Item: Pointer); virtual;
      { Called if an attempt was made to insert a duplicate item, in dupDestroy
        mode. Implemented as calling FreeItem. Descendants could generate an
        appropriate error / warning }
    function Get(Index: Integer): Pointer; virtual; abstract;
      { Should give acces to indexed property Items }
    function GetCount: Integer; virtual; abstract;
      { Should returns number of items in container }
    function GetItem(S: TFilter): Pointer; virtual;
      { Used to retrieve stored items from S. Assumes TStreamable descendants
        to be stored on S and calls S.Get to retrive it }
    function GetString(Index: Integer): String; virtual; abstract;
      { Should give acces to string representation of Items[Index], this
        could e.g. be used in displaying the container in a TListBox object }
    function KeyOf(Item: Pointer): Pointer; virtual;
      { Returns Key for Comparison. Duplicate handling and sorting is based on
        the KeyOf / Compare methods. Implemented as returning Item }
    procedure ListInsert(Index: Integer; Item: Pointer); virtual; abstract;
      { Performs actual insertion in container called by Add and Insert after
        duplicates and sorting order have been checked }
    procedure ListPut(Index: Integer; Item: Pointer); virtual; abstract;
      { Should put Item at Index. All checking has been done }
    procedure ListRemove(Index: Integer); virtual; abstract;
      { Should remove item from container without destroying it. Called by
        Delete and Remove methods }
    procedure Put(Index: Integer; Item: Pointer); virtual;
      { Gives acces to indexed property Items. Calls Remove and Insert to
        implement behaviour. Also safe for sorted containers }
    procedure PutItem(Item: Pointer; S: TFilter); virtual;
      { Used to store items to stream. assumes Item to be a TStreamable
        descendant and Put's it on S. Descendants could override this method in
        order to store items different, e.g. when items are strings }
    procedure SetOwnesItems(Value: Boolean); virtual;
      { Simply sets FOwnesItems to Value }
    procedure SetUpdateState(Updating: Boolean); virtual;
      { Calls Changed if Updating is False }
    procedure StyleChanged;
      { Calls FOnStyleChanged if it is assigned }
    property CanSort: Boolean read FCanSort write SetCanSort;
      { Stores wheter container is able to sort or not. To be set by Create }
  public
    constructor Load(S: TFilter); override;
      { Loads properties from S and then reads items calling GetItem. Add is
        used to add items to container. Note: Descendants should therefore make
        sure that the data structure in which items are stored is initialised
        before inherited load is called, See TCollection.Load. Inherited Load
        should ALWAYS be called before any loading from s is done however. }
    destructor Destroy; override;
      { Sets FOnChange to nil, preventing change message's to
        be send during destruction, then calls FreeDataStructire and inherited Destroy }
    function Add(Item: Pointer): Integer; virtual;
      { Checkes for duplicates and sorting order, then calls ListInsert to
        insert item at correct position. Add returns the position at which the
        item was added or -1 if it was not added at all }
    procedure Assign(Container: TContainer); virtual;
      { List is cleared. Adds all items in Container by repeatedly calling Add.
        If Container = nil, list is cleared anyway }
    procedure BeginUpdate;
      { Increments UpdateCount and calls SetupdateState(True). Locking
        mechanism. Always lock and unlock updates with pairs of BeginUpdate and
        EndUpdate method calls }
    procedure Clear; virtual;
      { Clears all items in container. Disposes of items if OwnedItems is True.
        Clear is identical to 'DeleteAll' }
    procedure Delete(Index: Integer); virtual;
      { Deletes item at Index from container, calls FreeItem if OwnesItem is
        True }
    procedure DeleteItem(Item: Pointer); virtual;
      { Deletes item from container, uses Find to locate it, then calls Delete }
    procedure Exchange(Index1, Index2: Integer); virtual;
      { Exchages items at Index1 and Index2. Safe when container is Sorted. }
    procedure EndUpdate;
      { Decrements the UpdateCounter. Update locking mechanism.
        If EndUpdate hits UpdateCount = 0, SetUpdateState(False) is called }
    function First: Pointer;
      { returns pointer to first item or nil if no such item exists }
    function Find(Item: Pointer; var Index: Integer): Boolean; virtual;
      { Returns FindKey(KeyOf(Item) }
    function FindKey(Key: Pointer; var Index: Integer): Boolean; virtual;
      { Returns True if Item is found. Index then contains the index at which
        Key was found. If False is returned Index containes the value at which
        Key would be inserted }
    function IndexOf(Item: Pointer): Integer; virtual;
      { Calls Find and returns index of Item, or -1 if it was not found }
    function Insert(Index: Integer; Item: Pointer): Integer; virtual;
      { Checkes for sorting order and duplicates, then inserts item calling
        ListInsert. Note: if Sorted is True, calling Insert has the same result
        as calling Add. Insert always returns the position at which the item was
        inserted or -1 if it was not inserted at all }
    function Last: Pointer;
      { returns pointer to last item or nil if no such item exists }
    procedure Move(CurIndex, NewIndex: Integer); virtual;
      { Moves the item at CurIndex to NewIndex. Implemented with Remove and
        Insert methods. Also safe for sorted containers in which case nothing
        will happen (except when duplicates are accepted in which case the
        order of the duplicate items might be changed }
    procedure Remove(Index: Integer); virtual;
      { Removes the item from the container without destroying it even if
        OwnesItems is true. Implemented as calling ListRemove. }
    procedure RemoveAll; virtual;
      { Removes all items by iterating over container and calling Remove }
    procedure RemoveItem(Item: Pointer);
      { Removes Item by calling Find and Remove, does not destroy the item }
    procedure Store(S: TFilter); override;
      { Stores properties to S. Then stores each item to S calling PutItem }
    procedure Sort; virtual;
      { Calls quicksort if nessecary }
    function ValidIndex(Index: Integer): Boolean;
      { Returns True if Index in range of [0.. Count> }
    property Count: Integer read GetCount;
      { returns number of items }
    property Duplicates: TContainerDuplicates read FDuplicates write SetDuplicates;
      { Controls duplicates behavior }
    property Items[Index: Integer]: Pointer read Get write Put; default;
      { Items in containr , default property }
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
      { Dispatched each time the list container changes }
    property OnStyleChange: TNotifyEvent read FOnChange write FOnChange;
      { Dispatched each time the list style (sorted etc) changes }
    property OwnesItems: Boolean read FOwnesItems write SetOwnesItems;
      { Controls ownership of items in list. If set True FreeItem is called
        whenever an item is Deleted from the list }
    property Sorted: Boolean read FSorted write SetSorted;
      { Controls the sorting behavior. If Sorted is set from False to True
        the container will (re-) sort it's contents }
    property Strings[Index: Integer]: String read GetString;
      { Returns a string for each item in the container. Depends on getStrings
        which is abstract }
  end;


    { TCollection is a BP7.0 TCollection like class with extended features.
      It is also streamable. Main behaviour defined by TContainer, TCollection
      just implements a (Sorted) collection stored in a TList }
  TCollection = class(TContainer)
  private
    FList: TList;
  protected
    procedure FreeDataStructure; override;
      { Calls inherited, then disposes of FList }
    function Get(Index: Integer): Pointer; override;
      { Gets item from FList }
    function GetCount: Integer; override;
      { Returns FList.Count }
    procedure ListInsert(Index: Integer; Item: Pointer); override;
      { Insert Item at position Index in FList }
    procedure ListPut(Index: Integer; Item: Pointer); override;
      { Puts Item at Index in FList }
    procedure ListRemove(Index: Integer); override;
      { Removes item at Index from FList }
  public
    constructor Create;
      { Calls inherited Create then instantiates FList }
    constructor Load(S: TFilter); override;
      { Creates FList then calls inherited load which will fill it }
    procedure Assign(Container: TContainer); override;
      { Overrides the inherited Assign to improve performance }
    procedure RemoveAll; override;
      { Overrides inherited method to implement more economic algoritm }
  end;


implementation

uses NumUtils;

{$B-}

const
  SDuplicateItem = 0;
  SContainerSortError = 1;

procedure ContainerError(Ident: Word);
begin
  raise EContainerError.Create('Container error: ' + IntToStr(Ident));
end;

constructor TContainer.Load(S: TFilter);
var I, C: Integer;
begin
  inherited Load(S);
  BeginUpdate;
  S.Read(FCanSort, SizeOf(FCanSort));
  S.Read(FDuplicates, SizeOf(FDuplicates));
  S.Read(FOwnesItems, SizeOf(FOwnesItems));
  S.Read(FSorted, SizeOf(FSorted));
  S.Read(C, SizeOf(C));
  for I := 0 to C - 1 do
    Add(GetItem(S));
  EndUpdate;
end;

destructor TContainer.Destroy;
begin
  FOnChange := nil;
  FreeDataStructure;
  inherited Destroy;
end;

function TContainer.Add(Item: Pointer): Integer;
begin
  Item := CheckDuplicates(Item);
  if Assigned(Item) then
  begin
    if Sorted then
      Find(Item, Result)
    else
      Result := Count;
    ListInsert(Result, Item);
    Changed;
  end
  else
    Result := -1;
end;

procedure TContainer.Assign(Container: TContainer);
var I: Integer;
begin
  BeginUpdate;
  Clear;
  if Assigned(Container) then
    for I := 0 to Container.Count - 1 do
      Add(Container[I]);
  EndUpdate;
end;

procedure TContainer.BeginUpdate;
begin
  if FUpdateCount = 0 then SetUpdateState(True);
  Inc(FUpdateCount);
end;

procedure TContainer.Changed;
begin
  if Assigned(FOnChange) and (FUpdateCount = 0) then FOnChange(Self);
end;

function TContainer.CheckDuplicates(Item: Pointer): Pointer;
var I: Integer;
begin
  Result := Item;
  if (Duplicates <> dupAccept) and Find(Item, I) then
  begin
    case Duplicates of
      dupIgnore: ;
      dupDestroy: FreeDuplicate(Item);
      dupError: ContainerError(SDuplicateItem);
    end;
    Result := nil;
  end;
end;

function TContainer.CheckNewKey(CurKey, NewKey: Pointer): Boolean;
var I: Integer;
begin
  Result := (Duplicates = dupAccept) or (Compare(CurKey, NewKey) = 0) or
            not FindKey(NewKey, I);
end;

procedure TContainer.Clear;
var I: Integer;
begin
  BeginUpdate;
  I := GetCount - 1;
  while I >= 0 do
  begin
    Delete(I);
    I := Min2Int(I, GetCount);
    Dec(I);
  end;
  EndUpdate;
end;

function TContainer.Compare(Key1, Key2: Pointer): Integer;
begin
  if Key1 = Key2 then Result := 0 else Result := -1;
end;

procedure TContainer.Delete(Index: Integer);
var Item: Pointer;
begin
  Item := Items[Index];
  ListRemove(Index);
  { This items being removed might have caused related items to be removed.
    Therefore do not count on Index here anymore, but instead refer to item }
  if OwnesItems then FreeItem(Item);
  Changed;
end;

procedure TContainer.DeleteDuplicates;
var First, Last, I: Integer;
    Key: Pointer;
begin
  if GetCount < 2 then Exit;
  BeginUpdate;
  if Sorted then
  begin { Sorted }
    Last := GetCount - 1;
    while Last > 0 do
    begin
      Key := KeyOf(Items[Last]);
      First := Last - 1;
      while (First >= 0) and (Compare(Key, KeyOf(Items[First])) = 0) do Dec(First);
      for I := Last downto First + 2 do Delete(I);
      Last := First;
    end;
  end { Sorted }
  else
  begin { not sorted }
    First := 0;
    while First < GetCount do
    begin
      Last := GetCount - 1;
      Key := KeyOf(Items[First]);
      while Last > First do
      begin
        if Compare(KeyOf(Items[Last]), Key) = 0 then
          Delete(Last);
        Dec(Last);
      end;
      Inc(First);
    end;
  end; { not sorted }
  EndUpdate;
end;

procedure TContainer.DeleteItem(Item: Pointer);
var Index: Integer;
begin
  if Find(Item, Index) then
    Delete(Index);
end;

procedure TContainer.EndUpdate;
begin
  Dec(FUpdateCount);
  if FUpdateCount = 0 then SetUpdateState(False);
end;

procedure TContainer.Exchange(Index1, Index2: Integer);
var
  Item1, Item2: Pointer;
begin
  Item1 := Items[Index1];
  Item2 := Items[Index2];
  ListPut(Index1, Item2);
  ListPut(Index2, Item1);
  Changed;
end;

function TContainer.Find(Item: Pointer; var Index: Integer): Boolean;
begin
  Result := FindKey(KeyOf(Item), Index);
end;

function TContainer.FindKey(Key: Pointer; var Index: Integer): Boolean;
var
  L, H, I, C: Integer;
begin
  if Sorted then
  begin
    Result := False;
    L := 0;
    H := Count - 1;
    while L <= H do
    begin
      I := (L + H) shr 1;
      C := Compare(KeyOf(Items[I]), Key);
      if C < 0 then L := I + 1 else
      begin
        H := I - 1;
        if C = 0 then
        begin
          Result := True;
          if Duplicates <> dupAccept then L := I;
        end;
      end;
    end;
    Index := L;
  end
  else
  begin
    Index := -1;
    for I := 0 to Count - 1 do
      if Compare(KeyOf(Items[I]), Key) = 0 then
      begin
        Index := I;
        Break;
      end;
    Result := Index <> -1;
    if not Result then Index := Count;
  end
end;

function TContainer.First: Pointer;
begin
  if Count > 0 then
    Result := Items[0]
  else
    Result := nil;
end;

procedure TContainer.FreeDataStructure;
begin
  if FOwnesItems then
    Clear
  else
    RemoveAll;
end;

procedure TContainer.FreeItem(Item: Pointer);
begin
  TObject(Item).Free;
end;

procedure TContainer.FreeDuplicate(Item: Pointer);
begin
  FreeItem(Item);
  ContainerError(SDuplicateItem);
end;

function TContainer.GetItem(S: TFilter): Pointer;
begin
  Result := S.Get;
end;

function TContainer.IndexOf(Item: Pointer): Integer;
begin
  if not Find(Item, Result) then Result := -1;
end;

function TContainer.Insert(Index: Integer; Item: Pointer): Integer;
begin
  Item := CheckDuplicates(Item);
  if Assigned(Item) then
  begin
    if Sorted then
      Find(Item, Index);
    Result := Index;
    ListInsert(Result, Item);
    Changed;
  end
  else
    Result := -1;
end;

function TContainer.KeyOf(Item: Pointer): Pointer;
begin
  Result := Item;
end;

function TContainer.Last: Pointer;
begin
  if Count > 0 then
    Result := Items[Count - 1]
  else
    Result := nil;
end;

procedure TContainer.Move(CurIndex, NewIndex: Integer);
var
  Temp: Pointer;
begin
  if CurIndex <> NewIndex then
  begin
    BeginUpdate;
    Temp := Get(CurIndex);
    Remove(CurIndex);
    Insert(NewIndex, Temp);
    EndUpdate;
  end;
end;

procedure TContainer.Put(Index: Integer; Item: Pointer);
begin
  BeginUpdate;
  Remove(Index);
  Insert(Index, Item);
  EndUpdate;
end;

procedure TContainer.PutItem(Item: Pointer; S: TFilter);
begin
  S.Put(TStreamable(Item));
end;

procedure TContainer.QuickSort(L, R: Integer);
var
  I, J: Integer;
  P: Pointer;
begin
  I := L;
  J := R;
  P := Items[(L + R) shr 1];
  repeat
    while Compare(KeyOf(Items[I]), KeyOf(P)) < 0 do Inc(I);
    while Compare(KeyOf(Items[J]), KeyOf(P)) > 0 do Dec(J);
    if I <= J then
    begin
      Exchange(I, J);
      Inc(I);
      Dec(J);
    end;
  until I > J;
  if L < J then QuickSort(L, J);
  if I < R then QuickSort(I, R);
end;

procedure TContainer.Remove(Index: Integer);
begin
  ListRemove(Index);
  Changed;
end;

procedure TContainer.RemoveAll;
var I: Integer;
begin
  BeginUpdate;
  for I := Count - 1 downto 0 do Remove(I);
  EndUpdate;
end;

procedure TContainer.RemoveItem(Item: Pointer);
var Index: Integer;
begin
  if Find(Item, Index) then
    Remove(Index);
end;

procedure TContainer.SetCanSort(Value: Boolean);
begin
  FCanSort := Value;
  if not FCanSort then Sorted := False;
end;

procedure TContainer.SetDuplicates(Value: TContainerDuplicates);
begin
  if (FDuplicates <> Value) then
  begin
    if FDuplicates = dupAccept then
      DeleteDuplicates;
    FDuplicates := Value;
    StyleChanged;
  end;
end;

procedure TContainer.SetOwnesItems(Value: Boolean);
begin
  FOwnesItems := Value;
end;

procedure TContainer.SetSorted(Value: Boolean);
begin
  if (FSorted <> Value) then
  begin
    if Value and CanSort then Sort;
    FSorted := Value and CanSort;
    StyleChanged;
  end;
end;

procedure TContainer.SetUpdateState(Updating: Boolean);
begin
  if not Updating then Changed;
end;

procedure TContainer.Store(S: TFilter);
var I, C: Integer;
begin
  inherited Store(S);
  S.Write(FCanSort, SizeOf(FCanSort));
  S.Write(FDuplicates, SizeOf(FDuplicates));
  S.Write(FOwnesItems, SizeOf(FOwnesItems));
  S.Write(FSorted, SizeOf(FSorted));
  C := Count;
  S.Write(C, SizeOf(C));
  for I := 0 to Count - 1 do
    PutItem(Items[I], S);
end;

procedure TContainer.Sort;
begin
  if not Sorted and CanSort and (Count > 1) then
  begin
    BeginUpdate;
    QuickSort(0, Count - 1);
    EndUpdate;
  end;
end;

procedure TContainer.StyleChanged;
begin
  if Assigned(FOnStyleChange) and (FUpdateCount = 0) then FOnChange(Self);
end;

function TContainer.ValidIndex(Index: Integer): Boolean;
begin
  Result := (Index >= 0) and (Index < Count);
end;

{ TCollection }

constructor TCollection.Create;
begin
  inherited Create;
  FList := TList.Create;
end;

constructor TCollection.Load(S: TFilter);
begin
  FList := TList.Create;
  inherited Load(S);
end;

procedure TCollection.Assign(Container: TContainer);
var I: Integer;
begin
  if ((FDuplicates = dupAccept) or (Container.Duplicates <> dupAccept)) and
     (not FSorted) then
  begin
    BeginUpdate;
    try
      FList.Count := Container.Count;
      if Container is TCollection then
        SYSTEM.Move(TCollection(Container).FList.List^, FList.List^,
                    SizeOf(Pointer) * Container.Count)
      else
        for I := 0 to Container.Count - 1 do FList[I] := Container[I];
    finally
      EndUpdate;
    end;
  end
  else
    inherited Assign(Container);
end;

procedure TCollection.FreeDataStructure;
begin
  inherited FreeDataStructure;
  FList.Free;
end;

function TCollection.Get(Index: Integer): Pointer;
begin
  if Assigned(FList) then
     Result := FList[Index]
  else
    Result := nil;
end;

function TCollection.GetCount: Integer;
begin
  if Assigned(FList) then
    Result := FList.Count
  else
    Result := 0;
end;

procedure TCollection.ListInsert(Index: Integer; Item: Pointer);
begin
  if Assigned(FList) then FList.Insert(Index, Item);
end;

procedure TCollection.ListPut(Index: Integer; Item: Pointer);
begin
  if Assigned(FList) then FList.Items[Index] := Item;
end;

procedure TCollection.ListRemove(Index: Integer);
begin
  if Assigned(FList) then FList.Delete(Index);
end;

procedure TCollection.RemoveAll;
begin
  BeginUpdate;
  if Assigned(FList) then FList.Clear;
  EndUpdate;
end;


end.
