(*-----------------------------------------------------------------------*
 | Slider component for Borland Delphi.                                  |
 |                                                                       |
 | Programmed by Colin Wilson - woozle@cix.compulink.co.uk               |
 |                                                                       |
 | Adapted & Edited by Anton Bil 26-5-1997 - abil@Mail.HZeeland.nl       |
 |                                                                       |
 | Added tickmarks, colours, focus ability                               |
 | Fixed some minor errors                                               |
 *-----------------------------------------------------------------------*)

unit Slider;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls, stdctrls;

type
  TSliderOrientation = (slHoriz, slVertical);
  TTickmarks=(tmBottom,tmTop,tmNone);
  TSlider = class(TCustomControl)
  private
    Thumb : TRect;               { Current thumb position             }
    MemDC : HDC;                 { DC for saving area under the thumb }
    Bitmap : HBitmap;            { Bitmap handle for saved area.      }

    capture : boolean;           { Whether it's currently being moved }
    capturePoint : TPoint;       { Position at start of capture.      }
    captureValue : longint;      { Value at start of capture.         }

                                 { Property values...                 }
    fTrackWidth : Integer;
    fNotchLength: byte;
    fTickmarks  :TTickmarks;
    fNumTickmarks  :word;
    fTrackColor : TColor;
    fNotchColor : TColor;
    fOrientation : TSliderOrientation;
    fThumbHeight : Integer;
    fSizeTickmarks : byte;
    fThumbColor : TColor;
    fMin : longint;
    fMax : longint;
    fValue : longint;
    fValueChange : TNotifyEvent;
    fCtl3D : boolean;
    firsttime:boolean;
    fhasmousedown:boolean;

    procedure SetTrackWidth (value : Integer);
    procedure SetSizeTickmarks (value : byte);
    procedure SetNotchLength (value : byte);
    procedure SetNumTickmarks (value : word);
    procedure SetTickmarks (value : TTickmarks);
    procedure SetTrackColor (value : TColor);
    procedure SetNotchColor (value : TColor);
    procedure SetOrientation (value : TSliderOrientation);
    procedure SetThumbHeight (value : Integer);
    procedure SetThumbColor (value : TColor);
    procedure SetMin (v : longint);
    procedure SetMax (v : longint);
    procedure SetValue (value : longint);
    procedure SetCtl3D (value : boolean);
    procedure CMDialogChar(var Msg:TCMDialogChar);message CM_DIALOGCHAR;
    procedure WMSysChar(var Msg:TWMSysChar);message WM_SYSCHAR;
    procedure CMEnter(var message:TCMGotFocus);message CM_ENTER;
    procedure CMExit(var Message:TCMExit);message CM_EXIT;
    Procedure WMGetDlgCode(var Msg:TWMGetDlgCode);
              message wm_GetDlgCode;

  protected
    procedure Paint; override;
    procedure drawouter;
    procedure MouseDown (Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp (Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove (Shift: TShiftState; X, Y: Integer); override;
    procedure DrawThumb; virtual;
    procedure DrawTicks; virtual;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: char); override;
  public
    constructor Create (AOwner : TComponent); override;
    destructor Destroy; override;
  published
    property HasMouseDown:boolean read fHasMouseDown;
    property iscaptured:boolean read capture;
    property TrackWidth : Integer read fTrackWidth write SetTrackWidth;
    property NotchLength : byte read fNotchLength write SetNotchLength;
    property NumTickmarks : word read fNumTickmarks write SetNumTickmarks;
    property SizeTickmarks : byte read fSizeTickmarks write SetSizeTickmarks;
    property Tickmarks : TTickmarks read fTickmarks write SetTickMarks;
    property NotchColor : TColor read fNotchColor write SetNotchColor;
    property TrackColor : TColor read fTrackColor write SetTrackColor;
    property ThumbHeight : Integer read fThumbHeight write SetThumbHeight;
    property ThumbColor : TColor read fThumbColor write SetThumbColor;
    property Orientation : TSliderOrientation read fOrientation write SetOrientation;
    property Minimum : longint read fMin write SetMin;
    property Maximum : longint read fMax write SetMax;
    property Value : longint read fValue write SetValue;
    property Ctl3D : boolean read fCtl3D write SetCtl3D;
    property OnValueChange : TNotifyEvent read fValueChange write fValueChange;

    property Color;
    property Enabled;
    property HelpContext;
    property Hint;
    property ParentShowHint;
    property ShowHint;
    property Tag;
    property Visible;

    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnExit;
    property OnEnter;
    property OnKeydown;
    property OnKeyUp;

  end;

procedure Register;

implementation

(*-------------------------------------------------------------------------*
 | constructor TSlider.Create (AOwner);                                    |
 |                                                                         |
 | Create the slider and set initial property values.                      |
 |                                                                         |
 | parameters:                                                             |
 |   AOwner : TComponent              The owner                            |
 *-------------------------------------------------------------------------*)
constructor TSlider.Create (AOwner : TComponent);
begin
  inherited Create (AOwner);
  fhasmousedown:=false;
  firsttime:=true;
  bitmap:=0;
  tabstop:=true;
  taborder:=1;
  Width := 50;
  Height := 200;
  fTrackWidth := 10;
  fNotchLength:=5;
  fTickmarks := tmNone;
  fNumTickmarks:=10;
  fSizeTickmarks :=5;
  fOrientation := slVertical;
  fNotchColor := clBtnFace;
  fTrackColor := clBtnFace;
  fThumbColor := clBtnFace;
  fMin := 0;
  fMax := 100;
  fValue := 0;
  fThumbHeight := 20;
  fValueChange := Nil;
  fCtl3D := True;
  capture := False;
  thumb.left := -1;
end;

(*-------------------------------------------------------------------------*
 | destructor TSlider.Destroy                                              |
 |                                                                         |
 | Destroy the slider component.                                           |
 *-------------------------------------------------------------------------*)
destructor TSlider.Destroy;
begin
  if Bitmap <> 0 then DeleteObject (Bitmap);
  if MemDC <> 0 then DeleteDC (MemDC);
  inherited Destroy
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetTrackWidth (value)                                 |
 |                                                                         |
 | Write procedure for TrackWidth property.  Sets the track width (the bit |
 | the thumb slides up and down.                                           |
 |                                                                         |
 | parameters:                                                             |
 |   value : Integer        The new track width value                      |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetTrackWidth (value : Integer);
begin
  if fTrackWidth <> value then
  begin
    fTrackWidth := value;
    Invalidate
  end
end;

Procedure TSlider.WMGetDlgCode(var Msg:TWMGetDlgCode);
begin
inherited;
Msg.Result:=dlgc_WantArrows;
end;

procedure TSlider.CMDialogChar(var Msg:TCMDialogChar);
begin
end;

procedure TSlider.WMSysChar(var Msg:TWMSysChar);
begin
end;

procedure TSlider.SetSizeTickmarks (value : byte);
begin                  
  if fSizeTickmarks <> value then
  begin
    fSizeTickmarks := value;
    Invalidate
  end
end;

procedure TSlider.SetNotchLength (value : byte);
begin                  
  if fNotchLength <> value then
  begin
    fNotchLength := value;
    Invalidate
  end
end;

procedure TSlider.SetNumTickmarks (value : word);
begin
  if fNumTickmarks <> value then
  begin
    fNumTickmarks := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetTickmarks (value)                                 |
 |                                                                         |
 | Write procedure for Tickmarks property.  Sets                           |
 |                                                                         |
 | parameters:                                                             |
 |   value : TTickmarks  The new tick mark kind                            |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetTickmarks (value : TTickmarks);
begin
  if fTickmarks <> value then
  begin
    fTickmarks := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetOrientation (value)                                |
 |                                                                         |
 | Write procedure for Orientation property.  Sets the slider orientation: |
 | slHorizontal or slVertical.                                             |
 |                                                                         |
 | parameters:                                                             |
 |   value : TSliderOrientation The new orientation                        |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetOrientation (value : TSliderOrientation);
begin
  if value <> fOrientation then
  begin
{    if csDesigning in ComponentState then
    begin
      hw:=width;
      width:=height;
      height:=hw;
    end;}
    fOrientation := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetTrackColor (value)                                 |
 |                                                                         |
 | Write procedure for TrackColor property.  Sets the track colour.        |
 |                                                                         |
 | parameters:                                                             |
 |   value : TColor       The new track colour                             |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetTrackColor (value : TColor);
begin
  if value <> fTrackColor then
  begin
    fTrackColor := value;
    Invalidate
  end
end;

procedure TSlider.SetNotchColor (value : TColor);
begin
  if value <> fNotchColor then
  begin
    fNotchColor := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetThumbHeight (value)                                |
 |                                                                         |
 | Write procedure for ThumbHeight property.  Sets the thumb height (or    |
 | width for horizontal sliders).  nb the thumb is always as wide as the   |
 | component itself.                                                       |
 |                                                                         |
 | parameters:                                                             |
 |   value : Integer       The new thumb height                            |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetThumbHeight (value : Integer);
begin
  if value <> fThumbHeight then
  begin
    fThumbHeight := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetThumbColor (value)                                 |
 |                                                                         |
 | Write procedure for ThumbColor property.  Sets the thumb colour         |
 |                                                                         |
 | parameters:                                                             |
 |   value : TColor     The new thumb colour                               |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetThumbColor (value : TColor);
begin
  if value <> fThumbColor then
  begin
    fThumbColor := value;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetMin (value)                                        |
 |                                                                         |
 | Write procedure for Minimum property.  Sets the minimum value.          |
 |                                                                         |
 | parameters:                                                             |
 |   value : longint  The new minimum                                      |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetMin (v : longint);
begin
  if v <> fMin then
  begin
    fMin := V;
    if Value < fMin then Value := fMin;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetMax (value)                                        |
 |                                                                         |
 | Write procedure for Maximum property.  Sets the maximum value.          |
 |                                                                         |
 | parameters:                                                             |
 |   value : longint  The new maximum                                      |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetMax (v : longint);
begin
  if v <> fMax then
  begin
    fMax := V;
    if Value > fMax then Value := fMax;
    Invalidate
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetValue (value)                                      |
 |                                                                         |
 | Write procedure for Value property.  Sets the value, updates the thumb, |
 | and generates an OnValueChange event                                    |
 |                                                                         |
 | parameters:                                                             |
 |   value : longint  The new value                                        |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetValue (value : longint);
begin
  if value < Minimum then value := Minimum else if value > Maximum
  then value := Maximum;
  if value <> fValue then
  begin
    fValue := Value;
    if Assigned (fValueChange) then OnValueChange (self);
    if firsttime then
    invalidate else
    if showing then
    DrawThumb;
  end
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.SetCtl3D (value)                                      |
 |                                                                         |
 | Write procedure for Ctl3D property.  Setting this property gives both   |
 | th thumb and the track a 3D look.                                       |
 |                                                                         |
 | parameters:                                                             |
 |   value : Boolean    The new ctl3D value                                |
 *-------------------------------------------------------------------------*)
procedure TSlider.SetCtl3D (value : boolean);
begin
  if value <> fCtl3D then
  begin
    fCtl3D := value;
    Invalidate
  end
end;

procedure TSlider.drawouter;
var rect:trect;
begin
  with rect,canvas do
  begin
    top := 0;
    bottom:=height;
    left:=0;
    right:=width;
    pen.style:=psdot;
    pen.color:=clblack;
    brush.style:=bsclear;
    Rectangle (left, top, right, bottom);
    pen.style:=pssolid;
    brush.style:=bssolid;
  end;
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.Paint                                                 |
 |                                                                         |
 | Paint the entire slider control.                                        |
 *-------------------------------------------------------------------------*)
procedure TSlider.Paint;
var Rect : TRect;
begin
  if focused then drawouter;
  with Canvas do
  begin                             { Create memory DC for save bitmap }
    if MemDC = 0 then MemDC := CreateCompatibleDC (Canvas.Handle);

                                    { Create thumb & bitmap for vertical slider }
    if fOrientation = slVertical then
    begin
      if Bitmap = 0 then
      begin
        Bitmap := CreateCompatibleBitmap (Canvas.Handle, Width, ThumbHeight);
      end;

      if fTickmarks=tmBottom then
      Rect.left := (width - TrackWidth) div 2 - fSizeTickmarks div 2 else
      if fTickmarks=tmTop then
      Rect.left := (width - TrackWidth) div 2 + fSizeTickmarks div 2 else
      Rect.left := (width - TrackWidth) div 2;
      Rect.top := 0;
      Rect.bottom := Height;
{      Rect.left := (Width - TrackWidth) div 2;}
      Rect.Right := Rect.Left + TrackWidth
    end
    else
    begin
                                    { Create thumb & bitmap for horiz slider }
      if Bitmap = 0 then
        Bitmap := CreateCompatibleBitmap (Canvas.Handle, ThumbHeight, Height);

      if fTickmarks=tmBottom then
      Rect.top := (Height - TrackWidth) div 2 - fSizeTickmarks div 2 else
      if fTickmarks=tmTop then
      Rect.top := (Height - TrackWidth) div 2 + fSizeTickmarks div 2 else
      Rect.top := (Height - TrackWidth) div 2;
      Rect.bottom := Rect.Top + TrackWidth - fSizeTickmarks div 2;
      Rect.left := 0;
      Rect.Right := Width
    end;
                                     { Draw track.                            }
    Brush.Color := TrackColor;
    if Ctl3D then
    begin
      Pen.Color := clBtnHighlight;
      with Rect do
      begin
        Rectangle (left, top, right, bottom);
        Pen.Color := clBtnShadow;
        MoveTo (left, top);
        LineTo (right, top);
        MoveTo (left, top);
        LineTo (left, bottom)
      end
    end
    else FillRect (Rect);            { Not Ctl3D - do a simple rectangle      }

    if fTickmarks<>tmNone then
    DrawTicks;                       { Now draw the ticks                     }

    DrawThumb;                       { Now draw the thumb.                    }

  end;
    firsttime:=false;
end;

procedure TSlider.DrawTicks;
var i:integer;
    he:word;
begin
  with Canvas do
  begin
    pen.color:=fNotchColor;
    if fOrientation = slVertical then
    begin
      if fTickmarks=tmBottom then
      he:=width else
      if fTickmarks=tmTop then
      he:=fSizeTickmarks-2 else he:=fSizeTickmarks div 2;
      for i:=0 to fNumTickMarks do
      begin
        moveto(he,fThumbHeight div 2+round(((height-fThumbHeight) / fNumTickMarks)*i)
          );
        lineto(he-fSizeTickmarks,fThumbHeight div 2+round(((height-fThumbHeight)
          / fNumTickMarks)*i));
      end;
    end else
    begin
      if fTickmarks=tmBottom then
      he:=height else
      if fTickmarks=tmTop then
      he:=fSizeTickmarks-2 else he:=fSizeTickmarks div 2;
      for i:=0 to fNumTickMarks do
      begin
        moveto(fThumbHeight div 2+round(((width-fThumbHeight) / fNumTickMarks)*i)
          ,he);
        lineto(fThumbHeight div 2+round(((width-fThumbHeight) / fNumTickMarks)*i)
           ,he-fSizeTickmarks);
      end;
    end;
  end;
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.DrawThumb                                             |
 |                                                                         |
 | Draw the thumb at the correct position for the current value.           |
 *-------------------------------------------------------------------------*)
procedure TSlider.DrawThumb;
var
  basePos : Integer;
  oldBmp : HBitmap;
  oldThumb : TRect;
begin
  if csLoading in ComponentState then Exit;

  oldBmp := SelectObject (MemDC, Bitmap);

  { Set thumb color & border color }
  if Enabled then Canvas.Brush.Color := ThumbColor else Canvas.Brush.Color := clGray;
  if Ctl3D then Canvas.Pen.Color := clBtnHighlight else Canvas.Pen.Color := clBlack;

  { Save current thum rectangle }
  oldThumb := Thumb;

  if Orientation = slVertical then
  begin

  { Calculate new thumb rectangle }
    basePos := LongInt (Height - ThumbHeight) * LongInt (Value - Minimum) div (Maximum - Minimum);
{    Thumb.left := 0;
    Thumb.right := Width;}
    Thumb.Bottom := Height - BasePos;
    Thumb.top := Thumb.Bottom - ThumbHeight;
    if fTickmarks=tmBottom then
    Thumb.left := 1 else
    if fTickmarks=tmTop then
    thumb.left:=fSizeTickmarks else
    thumb.left:=fSizeTickmarks div 2;
    Thumb.right := thumb.left+width - fSizeTickmarks-1;

  { Rub out old thumb}
  if (oldThumb.left <> -1) then with oldThumb do
      BitBlt (Canvas.Handle, Left, Top, right-left, ThumbHeight, MemDC, 0, 0, SRCCOPY);

  if focused then drawouter;

  { Save what's underneath         }
   if (oldThumb.left = -1) then
    with Thumb do
      BitBlt (MemDC, 0, 0, right-left, ThumbHeight, Canvas.Handle, Left, {Top}1, SRCCOPY);
  end
  else
  begin
  { Calculate new thumb rectangle for horiz slider}
    basePos := LongInt (Width - ThumbHeight) * LongInt(Value - Minimum) div (Maximum - Minimum);
    Thumb.left := basePos;
    Thumb.Right := Thumb.left + ThumbHeight;
    if fTickmarks=tmBottom then
    Thumb.Top := 1 else
    if fTickmarks=tmTop then
    thumb.top:=fSizeTickmarks else
    thumb.top:=fSizeTickmarks div 2;
    Thumb.Bottom := thumb.top+Height - fSizeTickmarks-1;

  { Rub out old thumb             }
    if (oldThumb.left <> -1) then with oldThumb do
      BitBlt (Canvas.Handle, Left, Top, ThumbHeight, bottom-top, MemDC, 0, 0, SRCCOPY);

  if focused then drawouter;

  { Save what's underneath         }
   if (oldThumb.left=-1) then
   begin
    with Thumb do
      BitBlt (MemDC, 0, 0, ThumbHeight, bottom-top, Canvas.Handle, {Left+1}1, Top, SRCCOPY);
   end;
  end;

  { Draw the thumb                 }
  with Canvas do
  begin
    with Thumb do if Ctl3D then
    begin
      Rectangle (left, top, right-1, bottom-1);
      Pen.Color := clBtnShadow;
      MoveTo (Left + 1, Bottom - 3);
      LineTo (Left + 1, Top+1);
      LineTo (Right - 2, Top+1);
      MoveTo (Left, Bottom - 1);
      LineTo (Right-1, Bottom - 1);
      LineTo (Right-1, Top - 1)
    end
    else
      Rectangle (left, top, right, bottom);

    {draw notch}

  Pen.Color := fNotchColor;
  with thumb do
  if Orientation = slVertical then
  begin
    if fTickmarks=tmBottom then
    begin
      MoveTo (right-4, top+(bottom-top) div 2);
      LineTo (right-4-fNotchLength,top+(bottom-top) div 2);
    end else
    if fTickmarks=tmTop then
    begin
      MoveTo (left+3,top+(bottom-top) div 2);
      LineTo (left+3+fNotchLength,top+(bottom-top) div 2);
    end else {do nothing}
  end else
  begin
    if fTickmarks=tmBottom then
    begin
      MoveTo (left+(right-Left) div 2, Bottom - 4);
      LineTo (left+(right-Left) div 2, Bottom - 4-fNotchLength);
    end else
    if fTickmarks=tmTop then
    begin
      MoveTo (left+(right-Left) div 2, top + 3);
      LineTo (left+(right-Left) div 2, top + 3+fNotchLength);
    end else {do nothing}
  end;

  Pen.Color := clBtnShadow;


  end;

  SelectObject (MemDC, OldBmp);
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.MouseDown (button, Shift, X, Y)                       |
 |                                                                         |
 | Respond to 'MouseDown' events.  Record the position of press for use in |
 | the MouseMove event handler.                                            |
 |                                                                         |
 | Parameters:                                                             |
 |   button : TMouseButton         Set of mouse buttons currently pressed. |
 |   Shift  : TShiftState          Set of shift, ctrl, alt keys            |
 |   X, Y   : Integer              Mouse position relative to top of slider|
 *-------------------------------------------------------------------------*)
procedure TSlider.MouseDown (Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
var hr:trect;
begin
  inherited MouseDown (Button, Shift, X, Y);
  if not visible then exit;
  setfocus;
  if (Button = mbLeft) and PtInRect (Thumb, Point (X, Y)) then
  begin
    fhasmousedown:=true;
    capture := True;
    capturePoint := Point (X, Y);
    captureValue := value;
  end else
    with hr do
    begin
      if forientation=slHoriz then
      begin
        right:=thumb.left;
        bottom:=height;
        left:=0;
        top:=0;
      end else
      begin
        top:=thumb.bottom;
        right:=width;
        bottom:=height;
        left:=0;
      end;
      if (Button = mbLeft) and PtInRect (hr, Point (X, Y)) then
      begin
        value:=value-1;
        exit;
      end;
      if forientation=slHoriz then
      begin
        left:=thumb.right;
        bottom:=height;
        right:=width;
        top:=0;
      end else
      begin
        bottom:=thumb.top;
        right:=width;
        top:=0;
        left:=0;
      end;
      if (Button = mbLeft) and PtInRect (hr, Point (X, Y)) then
      begin
        value:=value+1;
      end;
    end;
  begin

  end;
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.MouseUp (button, Shift, X, Y)                         |
 |                                                                         |
 | Respond to 'MouseUp' events                                             |
 |                                                                         |
 | Parameters:                                                             |
 |   button : TMouseButton         Set of mouse buttons currently pressed. |
 |   Shift  : TShiftState          Set of shift, ctrl, alt keys            |
 |   X, Y   : Integer              Mouse position relative to top of slider|
 *-------------------------------------------------------------------------*)
procedure TSlider.MouseUp (Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseUp (Button, Shift, X, Y);
  if (Button = mbLeft) then
  begin
    capture := False;
    fhasmousedown:=false;
  end;
end;

(*-------------------------------------------------------------------------*
 | procedure TSlider.MouseMove (Shift, X, Y)                               |
 |                                                                         |
 | Respond to 'MouseMove' events                                           |
 |                                                                         |
 | Parameters:                                                             |
 |   Shift  : TShiftState          Set of shift, ctrl, alt keys            |
 |   X, Y   : Integer              Mouse position relative to top of slider|
 *-------------------------------------------------------------------------*)
procedure TSlider.MouseMove (Shift: TShiftState; X, Y: Integer);
begin
  inherited MouseMove (shift, X, Y);
  if capture then
    if Orientation = slVertical then
      value := captureValue {+ Minimum} + LongInt (Maximum - Minimum) *
          LongInt (capturePoint.Y - Y) div (Height - ThumbHeight)
    else
      value := captureValue{ + Minimum} + LongInt (Maximum - Minimum) * LongInt
         (X - capturePoint.X) div (Width - ThumbHeight);
end;

procedure TSlider.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited keydown(key,shift);
  fhasmousedown:=true;
Case Key Of
     vk_End:value:=FMax;
     vk_Home:value:=FMin;
     vk_Left,vk_down: value:=value-1;
     vk_Right,vk_up:value:=value+1;
    end;
end;

procedure TSlider.KeyUp(var Key: Word; Shift: TShiftState);
begin
  inherited keyup(key,shift);
  fhasmousedown:=false;
end;

procedure TSlider.KeyPress(var Key: char);
begin
  case key of
    '-':value:=value-1;
    '+':value:=value+1;
  else
    inherited keypress(key);
  end;
end;

procedure TSlider.CMEnter(var message:TCMGotFocus);
begin
  invalidate;
  inherited;
end;
procedure TSlider.CMExit(var Message:TCMExit);
begin
  invalidate;invalidate;
  inherited;
end;

procedure Register;
begin
  RegisterComponents('Samples', [TSlider]);
end;

end.
