(******************************************************************************
**
**  TFILTERCALLBACK Component.
**
**  Author:   Andy Strong     e-mail: ANDREWS@NBAQSL.CO.UK
**                        CompuServe: 100716,3015
**
**  A component to be dropped on a form, to allow low-level BDE callback
**  on a table independently of any other indexes, ranges set.
**  Encapsulates the BDE callback in a standard Delphi event passing
**  adequate information to allow selective filtering using Pascal code.
**
**  Taken from an item on the BDE Compuserve forum.
**
**  This unit consists of two objects and one component.
**  The component is used to encapsulate the useage of the BDE callback filter,
**  and raw data handling.
**  The TFilterCallback Component sets up the callback filter for the BDE;
**  the TFilterRecord object, is the object passed to the users Delphi callback
**  routine to enable easy access to the raw data to work on;
**  the TRaw object is an internal sanitiser of the raw BDE data for the Delphi
**  users event handler.
**
**  Ideally I would like to have the ACTIVE property available at design time,
**  however since I cannot find a way of ensuring that our linked table is
**  created first, I have had to provide calls to start and end the filtering.
**  If this component is put on the form before its linked table, and the
**  auto active set to true, it references the table whilst its internal state
**  is still set to csLoading, and is unstable.
**
*******************************************************************************)
unit Fltcback;

interface

{$C FIXED}

uses
  Classes, DBTables, DB, DbiTypes, DbiProcs, SysUtils;

const
  MAXRAWSIZE = SizeOf(String);              { Max Pascal String size }

type
  ECallBackFilter = class(Exception);       { Special Filter Exception }

  {
    Object to handle the conversion of RAW data to a known Pascal type
  }
  TRawData = class(TObject)
  public
    RawData: array [0..MAXRAWSIZE] of byte; { Repository for BDE field raw data }
    function AsBoolean: boolean;
    function AsCurrency: Double;
    function AsDateTime: TDateTime;
    function AsFloat: Double;
    function AsInteger: LongInt;
    function AsSmallInt: Integer;
    function AsString: string;
  end;

  {
    Object which is passed to the users Filter callback, giving access to the
    BDEs raw data for a record in a useable state.
  }
  TFilterRecord = class(TObject)
  private
    FRawTable: TTable;                      { Pointer to Table being filtered }
    FFieldList: TStringList;                { List of All Field names in Table }
    FRaw: TRawData;                         { Pointer to RAW data handling object }
    FpRecBuf: pointer;                      { Pointer to BDE record structure }
    FPhysRecNum: Longint;                   { Physical record number }

    procedure LoadFields(Table: TTable);    { Loads all Field Names into list }

  public

    property PhysicalRecordNum: Longint read FPhysRecNum;

    constructor Create;
    destructor Destroy; override;

    function GetFieldData(FieldName: openstring): TRawData;
    function FieldIsBlank(FieldName: openstring): boolean;
    function IsValidField(FieldName: openstring): boolean;
  end;

  {
    Function prototype for our sanitised Delphi filter event.
  }
  TFilterEvent = procedure(FilterRec: TFilterRecord; var Allow: boolean) of object;

  {
    Actual component doing the work.
  }
  TFilterCallback = class(TComponent)
  private
    FActive: boolean;                       { Whether or not the filter is active }
    FFilterRecord: TFilterRecord;           { Pointer to our record handler }
    FTable: TTable;                         { Table Filter to be applied to }
    hFilter: hDbiFilter;                    { Handle of Filter when applied }
    FOnFilter: TFilterEvent;                { Users callback event handler }

    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SetActive(Value: boolean);
    procedure SetTable(Value: TTable);

  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure StartFilter;
    procedure EndFilter;

  published
    property Active: boolean read FActive default False;
    property Table: TTable read FTable write SetTable;
    property OnFilter: TFilterEvent read FOnFilter write FOnFilter;
  end;

  procedure Register;

implementation

(******************************************************************************
**  TRAWDATA Object functions/Methods/Procedures
******************************************************************************)

(*
**  Returns Raw data formatted as a boolean type
*)
function TRawData.AsBoolean: Boolean;
begin
  Result := Boolean(@RawData);
end;

(*
**  Returns Raw data formatted as a Currency type
*)
function TRawData.AsCurrency: Double;
begin
  Result := AsFloat;
end;

(*
**  Returns Raw data formatted as DateTime type
*)
function TRawData.AsDateTime: TDateTime;
begin
  Result := AsFloat;
end;

(*
**  Returns Raw data formatted as a float type
*)
function TRawData.AsFloat: Double;
var
  PFloat: ^Double;
begin
  PFloat := @RawData;
  Result := PFloat^;
end;

(*
**  Returns Raw data formatted as a LongInt type
*)
function TRawData.AsInteger: LongInt;
var
  PInteger: ^LongInt;
begin
  PInteger := @RawData;
  Result := PInteger^;
end;

(*
**  Returns Raw data formatted as a pascal string type
*)
function TRawData.AsString: string;
begin
  Result := StrPas(@RawData);
end;

(*
**  Returns Raw data formatted as an integer type
*)
function TRawData.AsSmallInt: Integer;
var
  PInteger: ^Integer;
begin
  PInteger := @RawData;
  Result := PInteger^;
end;


(******************************************************************************
**  TFILTERRECORD Object functions/Methods/Procedures
******************************************************************************)

(*
**  Constructor for client callback object
*)
constructor TFilterRecord.Create;
begin
  inherited Create;                         { Create base object }
  FFieldList := TStringList.Create;         { Create Field Names Stringlist }
  FRaw := TRawData.Create;                  { Create Raw data handling object }
  FpRecBuf := nil;                          { initialise pointers to nil }
  FRawTable := nil;
end;

(*
**  Destructor for client callback object.  Release all associated objects
**  resources.
*)
destructor TFilterRecord.Destroy;
begin
  FpRecBuf := nil;
  FRawTable := nil;
  FFieldList.Free;
  FRaw.Free;
  inherited Destroy;
end;

(*
**  Loads internal list with all field names in passed table.  The list is in
**  the same order as the fields are in the file definition, allowing the
**  list item number plus one to be used in the DbiGetField function.
*)
procedure TFilterRecord.LoadFields(Table: TTable);
var
  CProps: CurProps;                         { Storage for table properties }
  FDescs: pFieldDescList;                   { Pointer to list of field descs }
  i: integer;
begin
  FFieldList.Clear;                         { Clear any old field descriptions }
  Check(DbiGetCursorProps(
    FRawTable.Handle, CProps));             { Get various info about table }
  GetMem(FDescs,
    CProps.iFields * SizeOf(FLDDesc));      { Create space for all field defs }
  try
    Check(DbiGetFieldDescs(FRawTable.Handle,
      pfldDesc(FDescs)));                   { Get all Field definitions }
    for i := 0 to CProps.iFields -1 do      { Iterate down all fields returned }
      FFieldList.Add(
        StrPas(FDescs^[i].szName));         { Add to list in field defined order }
  finally
    FreeMem(FDescs,
      CProps.iFields * SizeOf(FLDDesc));    { Free up field structure }
  end;
end;

(*
**  Function to decide whether passed field name exists in the table
*)
function TFilterRecord.IsValidField(FieldName: openstring): boolean;
var
  Locn: integer;
begin
  Locn := FFieldList.IndexOf(FieldName);    { Find location in field list }
  if Locn < 0 then                          { Returns -1 if not in list }
    raise ECallBackFilter.Create(FieldName + ' not in Table')
  else
    Result := Locn >= 0;                    { in list if index >= zero }
end;

(*
**  Function to decide whether the passed fields data is blank
*)
function TFilterRecord.FieldIsBlank(FieldName: openstring): boolean;
var
  Blank: wordbool;
begin
  Result := True;                           { Assume it is blank }
  if IsValidField(FieldName) and
    (FRawTable <> nil) then                 { Check it exists! }
  begin
    Check(DbiGetField(FRawTable.Handle,
      FFieldList.IndexOf(FieldName) + 1,
      FpRecBuf, nil, Blank));               { Get status with exception checking }
    Result := Blank;                        { Return result }
  end;
end;

(*
**  Function to get Raw data for passed field name.
*)
function TFilterRecord.GetFieldData(FieldName: openstring): TRawData;
var
  Blank: wordbool;
begin
  Result := FRaw;                           { Point at our Raw data handler }
  if (FRawTable = nil) then
    FillChar(FRaw, MAXRAWSIZE, 0)           { Clear out raw data }
  else if IsValidField(FieldName) then      { If valid field name }
  begin
    Check(DbiGetField(FRawTable.Handle,
      FFieldList.IndexOf(FieldName) + 1,
      FpRecBuf, @(Result.RawData), Blank)); { Get data with exception handling }
  end;
end;


(******************************************************************************
**  TFILTERCALLBACK Component functions/Methods/Procedures
******************************************************************************)

(*
**  Function which is called by the low level BDE routine for each record to be
**  filtered.  This function must return True to allow Delphi to see the record.
**  When we setup the callback in SETACTIVE, we passed a pointer to our
**  instantiation of TFILTERCALLBACK.  This is used in this isolated function
**  to access the users callback function, and our data handling object.
*)
function BDEFilterCallback(ulClientdata: Longint; pRecBuff: Pointer;
          iPhyRecNum: Longint): Integer; export;
var
  Allow: boolean;
begin
  Allow := True;                            { Allow through this filter? }
  with TFilterCallBack(ulClientdata) do     { Pointer to our TFilterCallback Obj }
  if Assigned(FOnFilter) then               { User callback function assigned? }
  begin                                     { Yes... }
    FFilterRecord.FpRecBuf := pRecBuff;     { Pass current record to handler obj }
    FFilterRecord.FPhysRecNum := iPhyRecNum;{ Pass physical record No. to handler obj }
    FOnFilter(FFilterRecord, Allow);        { Call user filter function }
  end;
  Result := Integer(Allow);                 { Function has decided to allow record }
end;

(*
**  Create component which handles the BDE callbacks.
*)
constructor TFilterCallBack.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FOnFilter := nil;
  FTable := nil;
  hFilter := nil;
  FActive := False;
  FFilterRecord := TFilterRecord.Create;    { Create client record handler }
end;

(*
**  Destroy resources used by component
*)
destructor TFilterCallBack.Destroy;
begin
  FOnFilter := nil;                         { Stop GPFs }
  FFilterRecord.Free;
  inherited Destroy;
end;

(*
**  Starts filter active
*)
procedure TFilterCallBack.StartFilter;
begin
  SetActive(True);
end;

(*
**  Terminates filter
*)
procedure TFilterCallBack.EndFilter;
begin
  SetActive(False);
end;

(*
**  Activate or Deactivate the filter callback.
*)
procedure TFilterCallBack.SetActive(Value: boolean);
begin
  if (Value <> FActive) then
  begin
    FActive := Value;                         { Set value }
    if not (csDesigning in ComponentState)
      and (FTable <> nil) then
    with FTable do
    begin
      DisableControls;                        { Un-link any data controls }
      if FFilterRecord.FFieldList.Count < 1 then{ Fields not yet loaded... }
        FFilterRecord.LoadFields(FTable);     { Load Table Fields }
      if Value then                           { Activating }
      begin
        Check(DbiAddFilter(Handle, Longint(Self),
          1, False, nil, BDEFilterCallBack,
          hFilter));                          { Set filter to pass pointer to us }
        Check(DbiActivateFilter(
          Handle, hFilter));                  { Initiate filter }
      end else
      begin
        Check(DbiDeActivateFilter(
          Handle, hFilter));                  { Deactivate our filter }
        Check(DbiDropFilter(Handle, hFilter));{ Remove it from table }
      end;
      First;                                  { Reset table }
      EnableControls;                         { Re-Link data controls }
    end;
  end;
end;

(*
**  When a table name is entered, read in all field names.
**  Whilst then component is created, the LoadFields method cannot be called
**  because the dependant variables and objects have not been completely
**  created.  This eventuality is taken care of by the SetActive method
*)
procedure TFilterCallBack.SetTable(Value: TTable);
begin
  if (Value <> FTable) then
  begin
    FTable := Value;                        { Save New Table }
    FFilterRecord.FRawTable := Value;       { Save user referenced pointer }
    FFilterRecord.FFieldList.Clear;         { Clear old Field definitions }
    if not ((csDesigning in ComponentState) or
      (csLoading in ComponentState)) then
      FFilterRecord.LoadFields(Value);      { Only Fill fields at run-time }
  end;
end;

(*
**  Removes our reference to the selected table if it is being removed
*)
procedure TFilterCallBack.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if (AComponent = FTable) and (Operation = opRemove) then
  begin
    SetActive(False);                       { Disable filter first }
    FFilterRecord.FFieldList.Clear;         { Clear old Field definitions }
    FTable := nil;
    FFilterRecord.FRawTable := nil;         { Save user referenced pointer }
  end
end;

(*
**  Standard Component Palette register proceedure
*)
procedure Register;
begin
  RegisterComponents('Custom', [TFilterCallBack]);
end;

end.
