(*
 *  TExplorerButton version 3.0
 *  TExplorerPopup version 1.0
 *  (c)1996-1997 Fabrice Deville
 *  delphinarium@technologist.com
 *  http://www.tornado.be/~fdev/
 *
 *  Shareware source - compiles under Delphi 1.0 & 2.0
 *
 *  The source code can not be redistributed
 *
 *)

{$IFDEF WIN32}
{$ObjExportAll On} {export all objects (for C++Builder 3.0)}
{$ENDIF}

unit ExplBtn;

interface

uses
  WinTypes, WinProcs, Messages,
{$IFDEF VER120}
  ImgList,
{$ENDIF}    
  SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, dsgnIntf, ExtCtrls;

const
  WM_EXPLORERPOPUPOPEN=WM_USER+100;

type
  { TExplorerPopup declaration }

  TExplorerPopup = class(TCustomPanel)
  private
    FUndockable: Boolean;
    FOnClose: TNotifyEvent;
    FOnOpen: TNotifyEvent;
    FOnUndock: TNotifyEvent;
    FUndockThreshold: Integer;
    FHideInactive: Boolean;
    FRedockable: Boolean;
    clicked, FUndocked, titleFocused, terminated: Boolean;
    mousex, mousey: Integer;
    oldParent: TWinControl;
  protected
    procedure SetUndockable(value: Boolean);
    procedure Paint; override;
    procedure WMLButtonDown(var msg: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var msg: TWMLButtonUp); message WM_LBUTTONUP;
    procedure WMMouseMove(var msg: TWMMouseMove); message WM_MOUSEMOVE;
    procedure Loaded; override;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure DockedFormClose(Sender: TObject; var Action: TCloseAction);
    procedure ShiftControlsUpDown(offset: Integer);
  public
    constructor Create(AOwner: TComponent); override;
    procedure Open(btn: TControl);
    procedure Close;
    procedure CloseIfDocked;
  published
    property Undockable: Boolean read FUndockable write SetUndockable stored True default True;
    property UndockThreshold: Integer read FUndockThreshold write FUndockThreshold stored True default 10;
    property Undocked: Boolean read FUndocked;
    property HideInactive: Boolean read FHideInactive write FHideInactive stored True default True;
    property Redockable: Boolean read FRedockable write FRedockable stored True default True;
    property OnClose: TNotifyEvent read FOnClose write FOnClose;
    property OnOpen: TNotifyEvent read FOnOpen write FOnOpen;
    property OnUndock: TNotifyEvent read FOnUndock write FOnUndock;
    property Caption;
  end;

  { TExplorerButton declaration }

  TExplorerButtonLayout = (blBitmapTop, blBitmapBottom, blBitmapLeft, blBitmapRight);
  TExplorerButtonOption = (boMonoDisplay, boIconOnly, boPopupMark, boShowBevel,
                          boShowDownPattern, boTransparent, boWordWrap, boNoDepress);
  TExplorerButtonOptions = set of TExplorerButtonOption;
  TExplorerButtonShadingType = (stLight, stMedium, stDark);
  TExplorerButtonDropDownStyle = (ddsIExplorer, ddsOffice);
  TExplorerButtonButtonSize = (bsCustom, bsSmall, bsLarge);

  TExplorerButton = class(TCustomControl)
  private
    FDefault: Boolean;
    FCancel: Boolean;
    FActive: Boolean;
    FAlignment: TAlignment;
    FAllowAllUp: Boolean;
    FBevelStyle: TBevelStyle;
    FBitmap, FNoFocusBitmap, FDisabledBitmap, IBitmap: TBitmap;
    FCaption: TCaption;
    FDown: Boolean;
    FDropDown: TPopupMenu;
    FDropDownStyle: TExplorerButtonDropDownStyle;
    FEnabled: Boolean;
    FExplorerPopup: TExplorerPopup;
    FGroupIndex: Integer;
    FLayout: TExplorerButtonLayout;
    FOnDropDownClick: TNotifyEvent;
    FOptions: TExplorerButtonOptions;
    FShadingType: TExplorerButtonShadingType;
    FUnselectedFontColor: TColor;
    FOnEnter, FOnExit: TNotifyEvent;
    (* state flags *)
    popupPushed, Pushed, MouseIn, popupOpened: Boolean;
    painted, backBitmapCreated, initialized, wasObscured, regenerating: Boolean;
    backBitmap: TBitmap;

    FSmallWidth, FSmallHeight, FLargeWidth, FLargeHeight, FCustomWidth, FCustomHeight: Integer;
    FButtonSize: TExplorerButtonButtonSize;
    updatingSize, fakeResize: Boolean;
  protected
    procedure SetButtonStyle(ADefault: Boolean); virtual;
    procedure SetDefault(Value: Boolean);
    procedure CreateWnd; override;
    procedure CMDialogKey(var Message: TCMDialogKey); message CM_DIALOGKEY;
    procedure CMFocusChanged(var Message: TCMFocusChanged); message CM_FOCUSCHANGED;
    procedure CMDialogChar(var Message: TCMDialogChar); message CM_DIALOGCHAR;
    procedure ComputeExtent(TempCaption: PChar; var TempRect: TRect; theCanvas: TCanvas);
    procedure CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DefineProperties(Filer: TFiler); override;
    procedure ReadIBitmap(Stream: TStream);
    procedure WriteIBitmap(Stream: TStream);
    procedure DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
    procedure DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
    procedure DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
    procedure DrawOutline(theCanvas: TCanvas; pushed: Boolean);
    procedure GetLost;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override ;
    function  Obscured: boolean;
    procedure BitmapChange(Sender: TObject);
    procedure NoFocusBitmapChange(Sender: TObject);
    procedure DisabledBitmapChange(Sender: TObject);
    procedure Paint; override;
    procedure SetAlignment(value: TAlignment);
    procedure SetAllowAllUp(value: Boolean);
    procedure SetBevelStyle(value: TBevelStyle);
    procedure SetBitmap(value: TBitmap);
    procedure SetCaption(value: TCaption);
    procedure SetDisabledBitmap(value: TBitmap);
    procedure SetDown(value: Boolean);
    procedure SetDropDown(value: TPopupMenu);
    procedure SetDropDownStyle(value: TExplorerButtonDropDownStyle);
    procedure SetEnabled(value: Boolean); {$IFDEF VER120} override; {$ENDIF}
    procedure SetExplorerPopup(value: TExplorerPopup);
    procedure SetGroupIndex(value: Integer);
    procedure SetLayout(value: TExplorerButtonLayout);
    procedure SetNoFocusBitmap(value: TBitmap);
    procedure SetOptions(value: TExplorerButtonOptions);
    procedure SetShadingType(value: TExplorerButtonShadingType);
    procedure SetUnselectedFontColor(value: TColor);
    procedure WMRButtonUp(var msg: TWMRButtonUp); message WM_RBUTTONUP;
    procedure WMLButtonDblClk(var msg: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
    procedure WMLButtonDown(var msg: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonUp(var msg: TWMLButtonUp); message WM_LBUTTONUP;
    procedure WMSize(var msg: TWMSize); message WM_SIZE;
    procedure WMEraseBkgnd(var msg: TWMEraseBkgnd); message WM_ERASEBKGND;
    procedure WMMove(var msg: TWMMove); message WM_MOVE;
    procedure CMMouseEnter(var msg: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var msg: TMessage); message CM_MOUSELEAVE;
    procedure CMSysColorChange(var msg: TMessage); message CM_SYSCOLORCHANGE;

    procedure SetSmallWidth(value: Integer);
    procedure SetSmallHeight(value: Integer);
    procedure SetLargeWidth(value: Integer);
    procedure SetLargeHeight(value: Integer);
    procedure SetCustomWidth(value: Integer);
    procedure SetCustomHeight(value: Integer);
    procedure SetButtonSize(value: TExplorerButtonButtonSize);
    procedure DrawTransparentBitmap(theCanvas: TCanvas; bmp: TBitmap; xStart, yStart: Integer);
    procedure DrawDisabledBitmap(Canvas: TCanvas; x, y: Integer; bmp: TBitmap);

  public
    constructor Create(aOwner: TComponent); override;
    destructor Destroy; override;
    procedure Click; override;
    procedure Regenerate;
    procedure WMExplorerPopupOpen(var msg: TMessage); message WM_EXPLORERPOPUPOPEN;
  published
    (* new properties *)
    property Cancel: Boolean read FCancel write FCancel default False;
    property Default: Boolean read FDefault write SetDefault default False;
    property Alignment: TAlignment read FAlignment write SetAlignment stored True default taCenter;
    property AllowAllUp: Boolean read FAllowAllUp write SetAllowAllUp stored True default False;
    property BevelStyle: TBevelStyle read FBevelStyle write SetBevelStyle stored True default bsRaised;
    property NoFocusBitmap: TBitmap read FNoFocusBitmap write SetNoFocusBitmap stored True;
    property Bitmap: TBitmap read FBitmap write SetBitmap stored True;
    property Caption: TCaption read FCaption write SetCaption stored True;
    property GroupIndex: Integer read FGroupIndex write SetGroupIndex stored True default 0;
    property DisabledBitmap: TBitmap read FDisabledBitmap write SetDisabledBitmap stored True;
    property Down: Boolean read FDown write SetDown stored True default False;
    property DropDownMenu: TPopupMenu read FDropDown write SetDropDown stored True;
    property DropDownStyle: TExplorerButtonDropDownStyle read FDropDownStyle write SetDropDownStyle stored True;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property ExplorerPopup: TExplorerPopup read FExplorerPopup write SetExplorerPopup stored True;
    property Layout: TExplorerButtonLayout read FLayout write SetLayout stored True default blBitmapTop;
    property Options: TExplorerButtonOptions read FOptions write SetOptions stored True;
    property ShadingType: TExplorerButtonShadingType read FShadingType write SetShadingType stored True default stMedium;
    property UnselectedFontColor: TColor read FUnselectedFontColor write SetUnselectedFontColor stored True
             default clWindowText;
    property OnEnter: TNotifyEvent read FOnEnter write FOnEnter;
    property OnExit: TNotifyEvent read FOnExit write FOnExit;
    property OnDropDownClick: TNotifyEvent read FOnDropDownClick write FOnDropDownClick;

    property SmallWidth: Integer read FSmallWidth write SetSmallWidth stored True default 23;
    property SmallHeight: Integer read FSmallHeight write SetSmallHeight stored True default 22;
    property LargeWidth: Integer read FLargeWidth write SetLargeWidth stored True default 39;
    property LargeHeight: Integer read FLargeHeight write SetLargeHeight stored True default 38;
    property CustomWidth: Integer read FCustomWidth write SetCustomWidth stored True;
    property CustomHeight: Integer read FCustomHeight write SetCustomHeight stored True;
    property ButtonSize: TExplorerButtonButtonSize read FButtonSize write SetButtonSize stored True default bsCustom;

    (* republished properties *)
    property Font;
    property OnClick;
    property OnDblClick;
    property ParentFont;
    property ParentShowHint;
    property ShowHint;
    property Visible;
    property OnDragDrop;
    property OnDragOver;
    property PopupMenu;
    property HelpContext;
    property OnMouseMove;
    property OnMouseDown;
    property OnMouseUp;
    property TabStop;
    property TabOrder;
 end;

procedure Register;

var
   bool_Version95 : Boolean;
   pattern: TBitmap;

implementation

const SMALLBITMAPWIDTH=16;
     SMALLBITMAPHEIGHT=16;
      LARGEBITMAPWIDTH=32;
      LARGEBITMAPHEIGHT=32;

var
  activeExplorerPopup: TExplorerPopup;
  activeExplorerButton: TExplorerButton;

{
  *********************************************************
  *
  *  TExplorerPopup implementation
  *
  *********************************************************
}

{$IFNDEF WIN32}
{
  16 bit: no small caption style, so this is my own implementation of the form.
  Tries to respect the style of Windows 3.1/NT 3.51 or Windows 95/NT4.
  No depress effect is implemented.
}
type
    TExplorerPopupForm = class(TForm)
    protected
        topDock, leftDock: integer;
        leftHome, redocked, checkRedock, hideInactive: Boolean;
        procedure WMNCHitTest(var msg: TMessage); message WM_NCHITTEST;
        procedure WMNCLButtonDown(var msg: TWMNCLButtonDown); message WM_NCLBUTTONDOWN;
       procedure WMActivateApp(var msg: TWMActivateApp); message WM_ACTIVATEAPP;
        procedure Paint; override;
    published
        property Caption;
    end;

procedure TExplorerPopupForm.WMNCHitTest(var msg: TMessage);
begin
     msg.Result := HTCAPTION;
end;

procedure TExplorerPopupForm.Paint;
begin
     with Canvas do
     begin
          Brush.Style := bsSolid;
          Brush.Color := clBtnFace;
          Pen.Style := psClear;
          Rectangle(0, 0, Width, Height);
          Pen.Color := clBtnHighlight;
          Pen.Style := psSolid;
          MoveTo(1, Height - 2);
          LineTo(1, 1);
          LineTo(Width - 2, 1);
          Pen.Color := clBtnShadow;
          LineTo(Width - 2, Height - 2);
          LineTo(1, Height - 2);
          Pen.Color := clBlack;
          MoveTo(0, Height - 1);
          LineTo(Width - 1, Height - 1);
          LineTo(Width - 1, 0);

          Font.Name := 'MS Sans Serif';
          Font.Size := 6;
          Font.Color := clCaptionText;
          Brush.Color := clActiveCaption;
          Brush.Style := bsSolid;
          Pen.Style := psClear;
          Rectangle(3, 3, Width - 2, 19);

          if not bool_Version95 then
          begin
              (* Win 31 style close button *)
              Brush.Color := clBtnFace;
              Pen.Style := psSolid;
              Pen.Color := clBtnHighLight;
              Rectangle(5, 5, 17, 17);
              Pen.Color := clBtnShadow;
              MoveTo(5, 16);
              LineTo(16, 16);
              LineTo(16,5);
              Pen.Style := psClear;
              Brush.Color := clBtnShadow;
              Rectangle(8, 10, 16, 14);
              Brush.Color := clBlack;
              Rectangle(7, 9, 15, 13);
              Pen.Style := psSolid;
              Pen.Color := clWhite;
              MoveTo(8,10);
              LineTo(13,10);
          end
          else
          begin
              (* Win 95 style close button *)
              Brush.Color := clBtnFace;
              Rectangle(Width - 17, 6, Width - 6, 15);
              Pen.Style := psSolid;
              Pen.Color := clBlack;
              MoveTo(Width - 18, 15);
              LineTo(Width - 6, 15);
              LineTo(Width - 6, 4);
              Pen.Color := clBtnShadow;
              MoveTo(Width - 17, 14);
              LineTo(Width - 7, 14);
              LineTo(Width - 7, 5);
              Pen.Color := clBtnHighlight;
              MoveTo(Width - 18, 14);
              Lineto(Width - 18, 5);
              LineTo(Width - 6, 5);
              Pen.Color := clBlack;
              MoveTo(Width - 15, 8);
              LineTo(Width - 10, 13);
              MoveTo(Width - 14, 8);
              LineTo(Width - 9, 13);
              MoveTo(Width - 15, 12);
              LineTo(Width - 10, 7);
              MoveTo(Width - 14, 12);
              LineTo(Width - 9, 7);
          end;

          Pen.Style := psSolid;
          Brush.Style := bsClear;
          if bool_Version95 then
              TextRect(Rect(6, 3, Width - 19, 18), 6, 4, Caption)
          else
              TextRect(Rect(20, 3, Width - 4, 18), 20, 4, Caption)
     end;
end;

procedure TExplorerPopupForm.WMNCLButtonDown(var msg: TWMNCLButtonDown);
var
  p: TPoint;
begin
  p := ScreenToClient(Point(msg.XCursor, msg.YCursor));
  if not bool_Version95 then
  begin
    if (p.x > 5) and (p.y > 4) and (p.y < 15) and (p.x < 15) then
      Close
     else
       inherited;
  end
  else
  begin
    if (p.x > Width - 19) and (p.y > 4) and (p.y < 18) and (p.x < Width - 4) then
      Close
     else
       inherited
  end;
end;

{$ELSE}
  type
    TExplorerPopupForm = class(TForm)
    protected
     topDock, leftDock: integer;
     leftHome, redocked, checkRedock, hideInactive: Boolean;
      procedure WMMoving(var msg: TMessage); message WM_MOVING;
     procedure WMActivateApp(var msg: TWMActivateApp); message WM_ACTIVATEAPP;
    end;

procedure TExplorerPopupForm.WMMoving(var msg: TMessage);
var
  rect: TRect;
begin
  if checkRedock then
  begin
    rect := PRECT(msg.lParam)^;
    if (not leftHome) and ((Abs(rect.Top - topDock) > 40) or (Abs(rect.Left - leftDock) > 40)) then
      leftHome := True;
    if leftHome and (Abs(rect.Top - topDock) < 10) and (Abs(rect.Left - leftDock) < 50) then
    begin
      redocked := True;
      SendMessage(Handle, WM_CANCELMODE, 0, 0);
    end
  end
end;
{$ENDIF}

procedure TExplorerPopupForm.WMActivateApp(var msg: TWMActivateApp);
begin
  if hideInactive then
  begin
    if msg.Active then
      Show
    else
      Hide
  end
end;

{ TExplorerPopup implementation }
constructor TExplorerPopup.Create(AOwner: TComponent);
begin
     inherited Create(AOwner);
     BevelOuter := bvRaised;
     clicked := False;
     FUndocked := False;
     FUndockable := True;
     FUndockThreshold := 10;
     FHideInactive := True;
     FRedockable := True;
end;

procedure TExplorerPopup.Loaded;
begin
     inherited Loaded;
     Visible := False;
end;

procedure TExplorerPopup.SetUndockable(value: Boolean);
begin
     if value <> FUndockable then
     begin
          FUndockable := value;
          Refresh;
     end;
end;

{ Adjusts the vertical position of the components of the popup }
procedure TExplorerPopup.ShiftControlsUpDown(offset: Integer);
var
  i: Integer;
begin
  Height := Height + offset;
  for i := 0 to ControlCount - 1 do
    Controls[i].Top := Controls[i].Top + offset;
end;

procedure TExplorerPopup.Paint;
begin
   with Canvas do
   begin
     Brush.Style := bsSolid;
     Brush.Color := clBtnFace;
     if not FUndocked then
       Pen.Style := psSolid
     else
       Pen.Style := psClear;
     Pen.Color := clBtnShadow;
    if not FUndocked then
      Rectangle(1, 1, Width-1, Height-1)
    else
      Rectangle(2, 2, Width-2, Height-2);
    if not FUndocked then
    begin
      Pen.Color := clBtnHighlight;
      MoveTo(1, Height - 2);
      LineTo(1, 1);
      LineTo(Width - 2, 1);
      Pen.Color := clBlack;
      MoveTo(Width - 1, 0);
      LineTo(Width - 1, Height-1);
      LineTo(0, Height-1);
    end;
  end;

  if FUndockable and (not FUndocked) and not (csDesigning in ComponentState) then
  begin
    with Canvas do
    begin
       if titleFocused then
        Brush.Color := clActiveCaption
        else
        Brush.Color := clInactiveCaption;
      Brush.Style := bsSolid;
      Pen.Style := psClear;
      Rectangle(4, 2, Width - 3, 10);
    end;
   end;
end;

procedure TExplorerPopup.WMLButtonUp(var msg: TWMLButtonUp);
begin
  clicked := False;
  MouseCapture := False;
end;

procedure TExplorerPopup.WMLButtonDown(var msg: TWMLButtonDown);
begin
  if (msg.XPos > 3) and (msg.YPos > 1) and (msg.YPos < 9) and (msg.XPos < Width - 3) then
  begin
    clicked := True;
    mousex := msg.XPos;
     mousey := msg.YPos;
     MouseCapture := True;
  end;
end;

{ Manage the undocking of the popup }
const SC_DRAGMOVE = $F012;
procedure TExplorerPopup.WMMouseMove(var msg: TWMMouseMove);
var
   f: TExplorerPopupForm;
   p: TPoint;
   dockedForm: TForm;
begin
  if FUndockable and (not FUndocked) then
  begin
    if clicked or ((msg.XPos > 3) and (msg.YPos > 1) and (msg.YPos < 9) and (msg.XPos < Width - 3)) then
     begin
       if not titleFocused then
        begin
        with Canvas do
        begin
          Brush.Color := clActiveCaption;
          Brush.Style := bsSolid;
          Pen.Style := psClear;
          Rectangle(4, 2, Width - 3, 10);
        end
        end;
        titleFocused := True;
     end
     else if titleFocused then
     begin
      with Canvas do
      begin
        Brush.Color := clInactiveCaption;
        Brush.Style := bsSolid;
        Pen.Style := psClear;
        Rectangle(4, 2, Width - 3, 10);
      end;
        titleFocused := False;
     end;
     if clicked and
        (sqrt((msg.Xpos - mousex)*(msg.Xpos - mousex)+(msg.Ypos - mousey)*(msg.Ypos - mousey)) > FUndockThreshold) then
     begin
       clicked := False;
       if Assigned(FOnUndock) then
            FOnUndock(Self);
        Visible := False;
       MouseCapture := False;
         f := TExplorerPopupForm.CreateNew(Self);
         f.OnClose := FormClose;
       GetCursorPos(p);
         f.Left := p.x - Width div 2;
         f.Top := p.y - 5;
         f.topDock := Parent.Top;
         f.leftDock := Parent.Left;
         f.leftHome := False;
         f.redocked := False;
         f.checkRedock := FRedockable;
         f.hideInactive := FHideInactive;
         (Parent as TForm).Close;

         if FUndockable then
           ShiftControlsUpDown(-10);
         f.Caption := Caption;
         f.BorderIcons := [biSystemMenu];
         f.FormStyle := fsStayOnTop;
         f.Font.Assign(Font);
         Parent := f;
    {$IFNDEF WIN32}
         f.ClientWidth := Width + 4;
         f.BorderStyle := bsNone;
         f.ClientHeight := ClientHeight + 20;
         Top := 18;
         Left := 2;
     {$ELSE}
         f.ClientWidth := Width;
         f.BorderStyle := bsToolWindow;
         f.ClientHeight := ClientHeight;
        Top := 0;
         Left := 0;
     {$ENDIF}
         Visible := True;
         f.Show;

         (* Again set the client width to force it
           * (when the width is small, sometimes Delphi
           *  increases it (caption length ?))
           *)
    {$IFNDEF WIN32}
         f.ClientWidth := Width + 4;
     {$ELSE}
         f.ClientWidth := Width;
     {$ENDIF}
         FUndocked := True;

         SendMessage(f.Handle, WM_SYSCOMMAND, SC_DRAGMOVE, 0);

         if f.redocked then
         begin
        FUndocked := False;
        Visible := False;
           dockedForm := TForm.CreateNew(Self);
           if FUndockable then
           ShiftControlsUpDown(10);
         dockedForm.OnClose := DockedFormClose;
         dockedForm.FormStyle := fsStayOnTop;
         dockedForm.Font.Assign(Font);
         Parent := dockedForm;
         dockedForm.BorderStyle := bsNone;
        dockedForm.ClientWidth := Width;
        dockedForm.ClientHeight := Height;

         dockedForm.Left := f.leftDock;
         dockedForm.Top := f.topDock;
         Left := 0;
         Top := 0;

         dockedForm.Show;

        dockedForm.ClientWidth := Width;

        f.Free;
         Visible := True;
         FUndocked := False;
         titleFocused := False;
         dockedForm.Show;
         end
         else
           f.checkRedock := False;
     end;
  end;
end;

procedure TExplorerPopup.FormClose(Sender: TObject; var Action: TCloseAction);
begin
     Action := caFree;
     Visible := False;
     Parent := oldParent;
     FUndocked := False;
     clicked := False;
end;

procedure TExplorerPopup.DockedFormClose(Sender: TObject; var Action: TCloseAction);
begin
     Action := caFree;
     Parent := oldParent;
end;

procedure TExplorerPopup.Open(btn: TControl);
var
  p, cPos: TPoint;
  Msg: TMsg;
  f: TForm;
begin
     Terminated := False;
     if Visible then
     begin
          if FUndocked then
             (Parent as TForm).Close
          else
              Exit;
     end;

     if Assigned(FOnOpen) then
       FOnOpen(Self);

     if FUndockable then
       ShiftControlsUpDown(10);

     oldParent := Parent;
     f := TForm.CreateNew(Self);
     f.OnClose := DockedFormClose;
     f.FormStyle := fsStayOnTop;
     f.Font.Assign(Font);
     Parent := f;
     f.BorderStyle := bsNone;
    f.ClientWidth := Width;
    f.ClientHeight := Height;

     p := Point(btn.Left, btn.Top);
     p := btn.Parent.ClientToScreen(p);
     p.y := p.y + btn.Height;

     (* Adjust with the screen limits *)
     if p.y + Height > Screen.Height then
       p.y := p.y - btn.Height - Height;
     if p.x + Width > Screen.Width then
       p.x := Screen.Width - Width;

     f.Left := p.x;
     f.Top := p.y;
     Left := 0;
     Top := 0;

     f.Show;

    f.ClientWidth := Width;
     Visible := True;
     FUndocked := False;
     titleFocused := False;

     while (not terminated) and (not FUndocked) do
     begin
       if PeekMessage(Msg, 0, 0, 0, PM_NOREMOVE) then
        begin
          if Msg.Message = WM_QUIT then
           begin
                 terminated := True;
             if not FUndocked then
               Close;
           end;
            if (Msg.message = WM_LBUTTONDOWN) or (Msg.message = WM_NCLBUTTONDOWN) then
           begin
             p := ClientToScreen(Point(0, 0));
               GetCursorPos(cPos);
              if (cpos.x < p.x) or (cpos.x > p.x + Width) or (cpos.y < p.y) or (cpos.y > p.y + Height) then
              begin
                 terminated := True;
             if not FUndocked then
               Close;
              end;
           end;
        end;
        Application.HandleMessage;
     end;
end;

procedure TExplorerPopup.Close;
begin
  if not Visible then
    Exit;
     if FUndocked then
          (Parent as TForm).Close
     else
     begin
          Visible := False;
         (Parent as TForm).Close;
          clicked := False;
          Terminated := True;
         if FUndockable then
             ShiftControlsUpDown(-10);
     end;
     if Assigned(FOnClose) then
       FOnClose(Self);
end;

procedure TExplorerPopup.CloseIfDocked;
begin
  if not FUndocked then
    Close;
end;

{
  *********************************************************
  *
  *  TExplorerButton implementation
  *
  *********************************************************
}

(* Thanks to Stefano Rustioni *)
procedure GetSystemVersion;
{$IFDEF WIN32}
var
   VersionInfo : TOsVersionInfo;
begin
  bool_Version95 := True;
  try
    VersionInfo.dwOSVersionInfoSize := sizeof (VersionInfo);
    GetVersionEx (VersionInfo);
    if (VersionInfo.dwPlatformId = VER_PLATFORM_WIN32_WINDOWS) or
        ((VersionInfo.dwPlatformId = VER_PLATFORM_WIN32_NT)
        and (VersionInfo.dwMajorVersion >= 4)) then
      bool_Version95 := True
    else
      bool_Version95 := False;
    except
      bool_Version95 := True;
  end;
{$ELSE}
begin
  if LOWORD(GetVersion) <= 3 then
    bool_Version95 := False
  else
    bool_Version95 := True;
{$ENDIF}
end;

procedure CreatePattern;
var
  X, Y: Integer;
begin
  Pattern := TBitmap.Create;
  Pattern.Width := 8;
  Pattern.Height := 8;
  with Pattern.Canvas do
  begin
    Brush.Style := bsSolid;
    Brush.Color := clBtnFace;
    FillRect(Rect(0, 0, Pattern.Width, Pattern.Height));
    for Y := 0 to 7 do
      for X := 0 to 7 do
      begin
        if (X mod 2) = (Y mod 2) then
          Pixels[X, Y] := clBtnHighlight;
      end;
  end;
end;

{$B-} (* Quick evaluation *)
(* Won't work properly in the case of 'text&' caption, but the one
   who defines such a caption has a really sick mind :)*)
function RemoveAmpersand(input: String): String;
var i: Integer;
begin
  Result := input;
  i := 1;
  while i < Length(Result) do
  begin
    if (Result[i] = '&') then
    begin
      if (Result[i+1] = '&') then
        Delete(Result, i + 1, 1)
      else
        Delete(Result, i, 1);
    end
    else
      Inc(i);
  end;
end;

(*
 * TExplorerButton implementation
 *)

constructor TExplorerButton.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);

  FBitmap := TBitmap.Create;
  FBitmap.OnChange := BitmapChange;
  FNoFocusBitmap := TBitmap.Create;
  FNoFocusBitmap.OnChange := NoFocusBitmapChange;
  FDisabledBitmap := TBitmap.Create;
  FDisabledBitmap.OnChange := DisabledBitmapChange;
  IBitmap := TBitmap.Create;
   backBitmap := TBitmap.Create;

  ControlStyle := [csClickEvents, csCaptureMouse, csSetCaption, csDoubleClicks];
  Pushed := False;
  popupPushed := False;
  MouseIn := False;
  backBitmapCreated := False;
  pushed := False;
  painted := False;
  initialized := False;
  updatingSize := False;
  fakeResize := False;
  Alignment := taCenter;
  Width := 50;
  Height := 40;
  wasObscured := False;
  regenerating := False;

  if (csDesigning in ComponentState) and not (csLoading in TControl(Owner).ComponentState) then
    Caption := 'ExplorerButton';

  FAllowAllUp := False;
  FBevelStyle := bsRaised;
  FDown := False;
  FDropDown := nil;
  FDropDownStyle := ddsIExplorer;
  FEnabled := True;
  FExplorerPopup := nil;
  FGroupIndex := 0;
  FLayout := blBitmapTop;
  FOptions := [boPopupMark, boShowBevel, boShowDownPattern];
  FShadingType := stMedium;
  FUnselectedFontColor := clWindowText;

  FButtonSize := bsCustom;
  FSmallWidth := 23;
  FSmallHeight := 22;
  FLargeWidth := 39;
  FLargeHeight := 38;

  TabStop := True;
end;

destructor TExplorerButton.Destroy;
begin
  FBitmap.Free;
  FNoFocusBitmap.Free;
  FDisabledBitmap.Free;
  IBitmap.Free;
  backBitmap.Free;
  if pattern <> nil then
  begin
    pattern.Free;
    pattern := nil;
  end;
  inherited Destroy;
end;

procedure TExplorerButton.BitmapChange(Sender: TObject);
begin
  if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
    CreateGrayscaleBitmap(IBitmap, FBitmap);

  if not FBitmap.Empty then
    FBitmap.Dormant;
  Repaint;
end;

procedure TExplorerButton.NoFocusBitmapChange(Sender: TObject);
begin
  if not FBitmap.Empty and FNoFocusBitmap.Empty and (csDesigning in ComponentState) then
  begin
    CreateGrayscaleBitmap(IBitmap, FBitmap);
  end;

  if not FNoFocusBitmap.Empty then
   begin
     IBitmap.Free;
      IBitmap := TBitmap.Create;
    FNoFocusBitmap.Dormant;
   end;
  Repaint;
end;

procedure TExplorerButton.CreateParams(var Params: TCreateParams);
begin
     inherited CreateParams(Params);
     if (boTransparent in FOptions) and not (csDesigning in ComponentState) then
       Params.ExStyle := Params.ExStyle + WS_EX_TRANSPARENT;
     painted := False;
end;

procedure TExplorerButton.DefineProperties(Filer: TFiler);
begin
  Filer.DefineBinaryProperty('IBitmap', ReadIBitmap, WriteIBitmap, True);
end;

procedure TExplorerButton.ReadIBitmap(Stream: TStream);
begin
  IBitmap.LoadFromStream(Stream);
end;

procedure TExplorerButton.WriteIBitmap(Stream: TStream);
begin
  if not IBitmap.Empty then
    IBitmap.SaveToStream(Stream)
end;

procedure TExplorerButton.DisabledBitmapChange(Sender: TObject);
begin
  if not FDisabledBitmap.Empty then
    FDisabledBitmap.Dormant;
  if not FEnabled then
    Repaint;
end;

procedure TExplorerButton.CreateWnd;
begin
  inherited CreateWnd;
  FActive := FDefault;
end;

procedure TExplorerButton.SetButtonStyle(ADefault: Boolean);
const
  BS_MASK = $000F;
var
  Style: Word;
begin
  if HandleAllocated then
  begin
    if ADefault then Style := BS_DEFPUSHBUTTON else Style := BS_PUSHBUTTON;
    if GetWindowLong(Handle, GWL_STYLE) and BS_MASK <> Style then
      SendMessage(Handle, BM_SETSTYLE, Style, 1);
  end;
end;

procedure TExplorerButton.SetDefault(Value: Boolean);
begin
  FDefault := Value;
  if HandleAllocated then
    with GetParentForm(Self) do
      Perform(CM_FOCUSCHANGED, 0, Longint(ActiveControl));
end;

procedure TExplorerButton.CMDialogKey(var Message: TCMDialogKey);
begin
  with Message do
    if  (((CharCode = VK_RETURN) and FActive) or
      ((CharCode = VK_ESCAPE) and FCancel)) and
      (KeyDataToShiftState(Message.KeyData) = []) and CanFocus then
    begin
      Click;
      Result := 1;
    end else
      inherited;
end;

procedure TExplorerButton.CMFocusChanged(var Message: TCMFocusChanged);
begin
  with Message do
    if Sender is TExplorerButton then
      FActive := Sender = Self
    else
      FActive := FDefault;
  SetButtonStyle(FActive);
  inherited;
end;

procedure TExplorerButton.CMDialogChar(var Message: TCMDialogChar);
var lpPoint : TPoint;
begin
  with Message do
    if IsAccel(CharCode, Caption)and Enabled and Visible then
  begin
    MouseIn := True;
    Pushed := False;
    Repaint;
    Application.ProcessMessages;
    WMLButtonDown( TWMLBUTTONDOWN(Message));
    Application.ProcessMessages;
    WMLButtonUp( TWMLBUTTONUP(Message));
    Application.ProcessMessages;
    GetCursorPos(lpPoint);
    lpPoint := GetParentForm(self).ScreenToClient(lpPoint);
    if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
      and (lpPoint.x > left) and (lpPoint.x < left + width)) then
    begin
      MouseIn := False;
      Repaint;
    end;
    Result := 1;
  end;
end;

procedure TExplorerButton.ComputeExtent(TempCaption: PChar; var TempRect: TRect; theCanvas: TCanvas);
var Flags: Integer;
begin
  if Alignment = taLeftJustify then
    Flags := DT_LEFT
  else if Alignment = taCenter then
    Flags := DT_CENTER
  else Flags := DT_RIGHT;

  if boWordWrap in FOptions then
  begin
    Flags := Flags or DT_WORDBREAK;
    (* Sometimes DrawText looses the last word, except when there's a space character. Uh ? *)
    StrCat(TempCaption, ' ');
  end;

  DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, DT_CALCRECT or Flags);
end;

(*
 * These thresholds are used for the grayscaling and were experimentaly
 * determined
 *)
const THRESHOLD1_LIGHT = 205;
      THRESHOLD2_LIGHT = 127;
      THRESHOLD3_LIGHT = 68;
      THRESHOLD1_MEDIUM = 553;
      THRESHOLD2_MEDIUM = 231;
      THRESHOLD3_MEDIUM = 57;
      THRESHOLD1_DARK = 335;
      THRESHOLD2_DARK = 274;
      THRESHOLD3_DARK = 175;

procedure TExplorerButton.CreateGrayscaleBitmap(outputbmp, bmp: TBitmap);
var x, y: Integer;
    TransparentColor, col: LongInt;
    r, g, b, sum, threshold1, threshold2, threshold3: SmallInt;
begin
  outputbmp.Assign(bmp);
  TransparentColor := ColorToRGB(bmp.Canvas.Pixels[0,0]);
  if FShadingType = stLight then
  begin
    threshold1 := THRESHOLD1_LIGHT;
    threshold2 := THRESHOLD2_LIGHT;
    threshold3 := THRESHOLD3_LIGHT;
  end
  else
  if FShadingType = stMedium then
  begin
    threshold1 := THRESHOLD1_MEDIUM;
    threshold2 := THRESHOLD2_MEDIUM;
    threshold3 := THRESHOLD3_MEDIUM;
  end
  else
  begin
    threshold1 := THRESHOLD1_DARK;
    threshold2 := THRESHOLD2_DARK;
    threshold3 := THRESHOLD3_DARK;
  end;
  for x := 0 to bmp.Width do
    for y := 0 to bmp.Height do
    begin
      col := ColorToRGB(bmp.Canvas.Pixels[x, y]);
      if col <> TransparentColor then
      begin
        r := col shr 16;
        g := (col shr 8) and $00FF;
        b := col and $0000FF;
        sum := r + g + b;
        if sum > THRESHOLD1 then
          outputbmp.Canvas.Pixels[x, y] := clWhite
        else if sum > THRESHOLD2 then
          outputbmp.Canvas.Pixels[x, y] := clBtnHighlight
        else if sum > THRESHOLD3 then
          outputbmp.Canvas.Pixels[x, y] := clBtnShadow
        else
          outputbmp.Canvas.Pixels[x, y] := clBlack;
      end;
   end;
   if not bmp.Empty then
     bmp.Dormant;
   {$IFNDEF WIN32}
   if not outputbmp.Empty then
     outputbmp.Dormant;
   {$ENDIF}
end;

procedure TExplorerButton.DrawTheText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
var Flags: Integer;
begin
  if Alignment = taLeftJustify then
    Flags := DT_LEFT
  else if Alignment = taCenter then
    Flags := DT_CENTER
  else Flags := DT_RIGHT;

  if boWordWrap in FOptions then
           Flags := Flags or DT_WORDBREAK;

  if bool_Version95 then
  begin
    {$IFDEF WIN32}
    DrawTextEx(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect,
            DT_END_ELLIPSIS or Flags, nil);
    {$ELSE}
    DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
    {$ENDIF}
  end
  else
    (* NT 3.51 users *)
    DrawText(theCanvas.handle, TempCaption, StrLen(TempCaption), TempRect, Flags);
end;

(*
 * Drawing of a disabled text (Win95 style)
 *)
procedure TExplorerButton.DrawDisabledText(theCanvas: TCanvas; TempRect: TRect; TempCaption: PChar);
begin
  theCanvas.Brush.Style := bsClear;
  theCanvas.Font.Color := clBtnHighlight;
  with TempRect do
  begin
    left := left + 1;
    top  := top + 1;
    right:= right + 1;
    bottom:= bottom + 1;
  end;
  DrawTheText(theCanvas, TempRect, TempCaption);
  theCanvas.Font.Color := clBtnShadow;
  with TempRect do
  begin
    left := left - 1;
    top  := top - 1;
    right:= right - 1;
    bottom:= bottom - 1;
  end;
  DrawTheText(theCanvas, TempRect, TempCaption);
end;

procedure TExplorerButton.DrawOutline(theCanvas: TCanvas; pushed: Boolean);
var
  buttonWidth: Integer;
begin
  if boShowBevel in FOptions then
  begin
    buttonWidth := Width;
    if  FDropDownStyle = ddsOffice then
      Dec(buttonWidth, 11);

     if BevelStyle = bsRaised then
     begin
       if (pushed) then
         theCanvas.Pen.Color := clBtnShadow
        else
         theCanvas.Pen.Color := clBtnHighlight;
     end
     else
     begin
       if (pushed) then
         theCanvas.Pen.Color := clBtnHighlight
        else
         theCanvas.Pen.Color := clBtnShadow;
     end;

    theCanvas.MoveTo(0, Height-1);
    theCanvas.LineTo(0, 0);
    theCanvas.LineTo(buttonWidth-1, 0);

     if BevelStyle = bsRaised then
     begin
       if (pushed) then
         theCanvas.Pen.Color := clBtnHighlight
        else
         theCanvas.Pen.Color := clBtnShadow;
     end
     else
     begin
       if (pushed) then
         theCanvas.Pen.Color := clBtnShadow
        else
         theCanvas.Pen.Color := clBtnHighlight;
     end;

    theCanvas.LineTo(buttonWidth-1, Height-1);
    theCanvas.LineTo(0, Height-1);

    if FDropDownStyle = ddsOffice then
     begin
       if ((popupPushed or popupOpened) and (BevelStyle = bsLowered)) or (((not popupPushed)
            and (not popupOpened)) and (BevelStyle = bsRaised)) then
         theCanvas.Pen.Color := clBtnHighLight
       else
         theCanvas.Pen.Color := clBtnShadow;

      theCanvas.MoveTo(buttonWidth, Height-1);
      theCanvas.LineTo(buttonWidth, 0);
      theCanvas.LineTo(Width-1, 0);

       if ((popupPushed or popupOpened)  and (BevelStyle = bsLowered))
           or (((not popupPushed) and (not popupOpened)) and (BevelStyle = bsRaised)) then
         theCanvas.Pen.Color := clBtnShadow
       else
         theCanvas.Pen.Color := clBtnHighLight;

      theCanvas.LineTo(Width-1, Height-1);
      theCanvas.LineTo(buttonWidth-1, Height-1);
     end
  end;
end;

procedure TExplorerButton.DrawPopupMark(theCanvas: TCanvas; x, y: Integer);
var theColor: TColor;
begin
  theColor := theCanvas.Font.Color;
  if FDropDownStyle = ddsIExplorer then
  begin
     theCanvas.Pixels[x    , y - 1] := theColor;
     theCanvas.Pixels[x + 1, y - 1] := theColor;
     theCanvas.Pixels[x + 2, y - 1] := theColor;
     theCanvas.Pixels[x + 3, y - 1] := theColor;
     theCanvas.Pixels[x + 4, y - 1] := theColor;
     theCanvas.Pixels[x + 5, y - 1] := theColor;
     theCanvas.Pixels[x + 6, y - 1] := theColor;
  end;

  theCanvas.Pixels[x + 1, y    ] := theColor;
  theCanvas.Pixels[x + 2, y    ] := theColor;
  theCanvas.Pixels[x + 3, y    ] := theColor;
  theCanvas.Pixels[x + 4, y    ] := theColor;
  theCanvas.Pixels[x + 5, y    ] := theColor;

  theCanvas.Pixels[x + 2, y + 1] := theColor;
  theCanvas.Pixels[x + 3, y + 1] := theColor;
  theCanvas.Pixels[x + 4, y + 1] := theColor;

  theCanvas.Pixels[x + 3, y + 2] := theColor;
end;

procedure TExplorerButton.GetLost;
begin
  if FGroupIndex = 0 then
    FDown := False;
  Pushed := False;
  MouseIn := False;
  Repaint;
end;

(*
 * After the loading of the properties, the grayscale version can be
 * safely created
 *)
procedure TExplorerButton.Loaded;
begin
  inherited Loaded;
{//  if not FBitmap.Empty and FNoFocusBitmap.Empty and not (csLoading in ComponentState) then}
{//    CreateGrayscaleBitmap(IBitmap, FBitmap);}
end;

procedure TExplorerButton.Notification(AComponent: TComponent; Operation: TOperation);
begin
  if Operation = opRemove then
  begin
    if AComponent = FDropDown then
      FDropDown := nil;
     if AComponent = FExplorerPopup then
       FExplorerPopup := nil;
  end;
end ;

(*
 * Painting of the button
 *)
const PADDING = 2; (* Space between the bitmap and the caption *)

function TExplorerButton.Obscured: boolean;
var
  p: TPoint;
begin
  p := ClientToScreen(Point(1, 1));

   if WindowFromPoint(p) <> Handle then
      result := True
   else
   begin
     p.x := p.x + Width - 3;
     if WindowFromPoint(p) <> Handle then
        result := True
     else
     begin
       p.y := p.y + Height - 3;
       if WindowFromPoint(p) <> Handle then
          result := True
       else
       begin
         p.x := p.x - Width + 3;
         if WindowFromPoint(p) <> Handle then
            result := True
         else
           result := False
         end
      end
   end
end;

procedure TExplorerButton.Regenerate;
begin
  (* Transparency with TWinControl-derived controls is a real headache.
    * This is the *only* way the transparency is correctly regenerated....
    * And I can ensure you I spent tens of hours on this particular point :)
    *)
   regenerating := True;
   try
    Width := Width + 1;
     Application.ProcessMessages;
    Width := Width - 1;
   finally
     regenerating := False;
   end;
end;

procedure TExplorerButton.Paint;
var xt, yt, wt, _wt, ht, xb, yb, wb, hb, thePadding, buttonWidth: Integer;
  TempCaption : Array[0..256] of char;
  TempRect    : TRect;
  OffScreen: TBitmap;
  p: TPoint;
  screenDC: HDC;
   drawCanvas: TCanvas;
   depressOffset: Integer;
begin
  if (not regenerating) or visible or (csDesigning in ComponentState) then
  begin
    if (boTransparent in FOptions) and (not backBitmapCreated) and not (csDesigning in ComponentState) then
    begin
        if wasObscured and not Obscured then
         begin
          (* the button was obscured and now is fully visible -> let's regenerate the
             * transparency
             *)
           wasObscured := false;
           Regenerate;
            Exit
         end;
      p := ClientToScreen(Point(0, 0));
      backBitmap.Width := Width;
      backBitmap.Height := Height;

      screenDC := GetDC(0);

      try
        BitBlt(backBitmap.Canvas.Handle, 0, 0, Width, Height, screenDC, p.x, p.y, SRCCOPY);
      finally
        ReleaseDC(0, screenDC);
         end;
         if not Obscured then
         begin
            (* the button is visible on screen-> the back bitmap is okay *)
        backBitmapCreated := true;
            wasObscured := false
         end
         else
           (* the button is (partially) obscured (maybe a stay-on-op window was over)
             * -> the back bitmap is not created and we mark a flag to try to generate
             * the back bitmap again the next time a redraw occurs
             *)
           wasObscured := true
    end;

    OffScreen := TBitmap.Create;
    try
      OffScreen.Width := Width;
      OffScreen.Height := Height;
      drawCanvas := OffScreen.Canvas;
      if (boTransparent in FOptions) and not (csDesigning in ComponentState) then
      begin
      if not backBitmapCreated then
            drawCanvas := Canvas;
         drawCanvas.CopyRect(Rect(0, 0, Width, Height), backBitmap.Canvas, Rect(0,0,Width,Height))
      end
      else
      begin
         drawCanvas.Brush.Color := Parent.Brush.Color;
         drawCanvas.FillRect(Rect(0,0,Width,Height));
      end;

      (* Just to stop compiler warnings *)
      xt := 0;
      yb := 0;
      xb := 0;
      wb := 0;
      hb := 0;

      buttonWidth := Width;

      if FDropDownStyle = ddsOffice then
         Dec(buttonWidth, 11);

      if FDown and (boShowDownPattern in FOptions) then
      begin
         if pattern = nil then
            CreatePattern;
         drawCanvas.Brush.Bitmap := pattern;
         drawCanvas.FillRect(Rect(0, 0, buttonWidth, Height));
      end;

      drawCanvas.Brush.Style := bsClear;
      drawCanvas.Font := Self.Font;

      if not (boIconOnly in FOptions) and (Length(FCaption) > 0) then
      begin
         thePadding := PADDING;
         TempRect.Top := 0;
         TempRect.Left := 0;
         TempRect.Right := buttonWidth - 4;
         TempRect.Bottom := 0;
         StrPCopy(TempCaption, Caption);
         ComputeExtent(TempCaption, TempRect, drawCanvas);
         wt := TempRect.Right;
         ht := TempRect.Bottom;
      end
      else
      begin
         thePadding := 0;
         wt := 0;
         ht := 0;
      end;

      if wt > buttonWidth - 4 then
         wt := buttonWidth - 4;

      case FButtonSize of
        bsCustom:
         begin
          wb := FBitmap.Width;
          hb := FBitmap.Height;
         end;
        bsSmall:
         begin
          wb := SMALLBITMAPWIDTH;
          hb := SMALLBITMAPHEIGHT;
         end;
        bsLarge:
         begin
          wb := LARGEBITMAPWIDTH;
          hb := LARGEBITMAPHEIGHT;
         end;
      end;

      { Reserve place for the drawing of the popup mark }
      if ((((boPopupMark in FOptions) and Assigned(FDropDown))
         or Assigned(FExplorerPopup)) and (FDropDownStyle = ddsIExplorer)) then
         wb := wb + 10;

      if FBitmap.Empty then
      begin
         if Assigned(FDropDown) and (boPopupMark in FOptions) then
         begin
            _wt := wt + 10;
            if wt > buttonWidth - 14 then
               wt := buttonWidth - 14;
         end
         else
            _wt := wt;

         yt := (Height - ht) div 2;
         case Alignment of
            taLeftJustify:    xt := 3;
            taRightJustify:  xt := buttonWidth - _wt - 3;
            taCenter:      xt := (buttonWidth - _wt) div 2
         end;
      end
      else if (Layout = blBitmapTop) or (Layout = blBitmapBottom) then
      begin
         if Layout = blBitmapTop then
         begin
            yb := (Height - (ht + hb + thePadding)) div 2;
            yt := yb + hb + thePadding
         end
         else
         begin
            yt := (Height - (ht + hb + thePadding)) div 2;
            yb := yt + ht + thePadding
         end;
         case Alignment of
            taLeftJustify:
               begin
                  xt := 3;
                  xb := 3
               end;
            taRightJustify:
               begin
                  xt := buttonWidth - wt - 3;
                  xb := buttonWidth - wb - 3
               end;
            taCenter:
               begin
                  xb := (buttonWidth - wb) div 2;
                  xt := (buttonWidth - wt) div 2
               end;
         end;
      end
      else
      if Layout = blBitmapLeft then
      begin
         if wt + wb + thePadding > buttonWidth - 4 then
            wt := buttonWidth - 4 - thePadding - wb;
         yb := (Height - hb) div 2;
         yt := (Height - ht) div 2;

         case Alignment of
         taLeftJustify:
            begin
               xb := 3;
               xt := xb + wb + thePadding
            end;
         taRightJustify:
            begin
               xt := buttonWidth - wt - 3;
               xb := xt - wb - thePadding
            end;
         taCenter:
            begin
               xb := (buttonWidth - (wb + wt + thePadding)) div 2;
               xt := xb + wb + thePadding
            end;
         end;
      end
      else (* blBitmapRight *)
      begin
         if wt + wb + thePadding > buttonWidth - 4 then
            wt := buttonWidth - 4 - thePadding - wb;
         yb := (Height - hb) div 2;
         yt := (Height - ht) div 2;
         case Alignment of
         taLeftJustify:
            begin
               xt := 3;
               xb := xt + wt + thePadding
            end;
         taRightJustify:
            begin
               xb := buttonWidth - wb - 3;
               xt := xb - wt - thePadding
            end;
         taCenter:
            begin
               xt := (buttonWidth - (wb + wt + thePadding)) div 2;
               xb := xt + wt + thePadding
            end;
         end;
      end;

      if csDesigning in ComponentState then
      begin
         drawCanvas.Pen.Color := clBlack;
         drawCanvas.Pen.Style := psSolid;
         drawCanvas.Brush.Style := bsClear;
         drawCanvas.Rectangle(0, 0, buttonWidth, Height);
         if FDropDownStyle = ddsOffice then
            drawCanvas.Rectangle(buttonWidth, 0, Width, Height);
      end;

      if (boWordWrap in Options) and (xt + wt > buttonWidth - 5) then
         wt := buttonWidth - xt - 5;

      with TempRect do
      begin
         left := xt;
         top  := yt;
         right:= xt + wt;
         bottom:= yt + ht;
      end;

      if Enabled then
      begin
         if not ((Pushed and MouseIn) or (popupOpened and (FDropDownStyle = ddsIExplorer))) and (not Down) then
         begin
            (* Unpushed state - Mouse in or out *)
            if MouseIn or popupOpened then
               DrawOutline(drawCanvas, False)
            else
               drawCanvas.Font.Color := UnselectedFontColor;
            if BevelStyle = bsLowered then
            begin
               Inc(TempRect.Left);
               Inc(TempRect.Top);
               Inc(TempRect.Right);
               Inc(TempRect.Bottom);
            end;
            if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
               DrawTheText(drawCanvas, TempRect, TempCaption);
            if not FBitmap.Empty then
            begin
               (* Draw the normal or shaded bitmap.
                * Transparency color is at (0,0)
                *)
               if MouseIn or popupOpened or not (boMonoDisplay in FOptions) then
                  DrawTransparentBitmap(drawCanvas, FBitmap, xb, yb)
               else if FNoFocusBitmap.Empty then
                  DrawTransparentBitmap(drawCanvas, IBitmap, xb, yb)
               else
                  DrawTransparentBitmap(drawCanvas, FNoFocusBitmap, xb, yb);
               if (boPopupMark in FOptions) and (Assigned(FDropDown)
                  or Assigned(FExplorerPopup)) and (FDropDownStyle = ddsIExplorer) then
               begin
                  DrawPopupMark(drawCanvas, xb + FBitmap.Width + 3, yb + (hb div 2));
               end;
            end
            else if (boPopupMark in FOptions) and (Assigned(FDropDown)
                 or Assigned(FExplorerPopup)) and (FDropDownStyle = ddsIExplorer) then
                        DrawPopupMark(drawCanvas, xt + wt + 3, yt + (ht div 2));
         end
         else
         begin
            if boNoDepress in FOptions then
                depressOffset := 0
            else
                 depressOffset := 1;

            (* Pushed state *)
            DrawOutline(drawCanvas, True);
            if (BevelStyle = bsRaised) and not(boNoDepress in FOptions) then
            begin
               Inc(TempRect.Left);
               Inc(TempRect.Top);
               Inc(TempRect.Right);
               Inc(TempRect.Bottom);
            end;
            if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
               DrawTheText(drawCanvas, TempRect, TempCaption);
            if not FBitmap.Empty then
            begin
               DrawTransparentBitmap(drawCanvas, FBitmap, xb+depressOffset, yb+depressOffset);
               if (boPopupMark in FOptions) and (Assigned(FDropDown)
                  or Assigned(FExplorerPopup)) and (FDropDownStyle = ddsIExplorer) then
                  DrawPopupMark(drawCanvas, xb + FBitmap.Width + 3 + depressOffset, yb + (hb div 2) + depressOffset);
            end
            else if (boPopupMark in FOptions) and (Assigned(FDropDown)
                 or Assigned(FExplorerPopup)) and (FDropDownStyle = ddsIExplorer) then
               DrawPopupMark(drawCanvas, xt + wt + 3 + depressOffset, yt + (ht div 2) + depressOffset);
         end
      end
      else
      begin
         (* Disabled state *)
         if Down then
            DrawOutline(drawCanvas, True);
         if not FDisabledBitmap.Empty then
            DrawTransparentBitmap(drawCanvas, FDisabledBitmap, xb, yb)
         else
            DrawDisabledBitmap(drawCanvas, xb, yb, FBitmap);
         if not (boIconOnly in FOptions) and (Length(Caption) > 0) then
            DrawDisabledText(drawCanvas, TempRect, TempCaption);
      end;

      if FDropDownStyle = ddsOffice then
      begin
         if popupPushed or popupOpened then
            DrawPopupMark(drawCanvas, Width - 8, Height div 2 + 1)
         else
            DrawPopupMark(drawCanvas, Width - 9, Height div 2)
      end;

      if OffScreen.Canvas = drawCanvas then
        Canvas.CopyRect(Rect(0,0,Width,Height), drawCanvas, Rect(0,0,Width,Height));

    finally
       OffScreen.Free;
    end;

    if not FBitmap.Empty then
      FBitmap.Dormant;
    if not FDisabledBitmap.Empty then
      FDisabledBitmap.Dormant;
    if not FNoFocusBitmap.Empty then
       FNoFocusBitmap.Dormant;
    if not IBitmap.Empty then
       IBitmap.Dormant;
    if not backBitmap.Empty then
       backBitmap.Dormant;
  end;
  painted := True;
end;

procedure TExplorerButton.SetAlignment(value: TAlignment);
begin
  FAlignment := value;
  Repaint;
end;

procedure TExplorerButton.SetAllowAllUp(value: Boolean);
var i: Integer;
  otherbutton: TExplorerButton;
begin
  FAllowAllUp := value;
  if FGroupIndex <> 0 then
  begin
    for i := 0 to Parent.ControlCount - 1 do
    begin
      if Parent.Controls[i] is TExplorerButton then
      begin
        otherbutton := (Parent.Controls[i] as TExplorerButton);
        if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex) then
          otherbutton.FAllowAllUp := value;
      end
    end
  end
end;

procedure TExplorerButton.SetBevelStyle(value: TBevelStyle);
begin
  FBevelStyle := value;
  Repaint;
end;

procedure TExplorerButton.SetBitmap(value: TBitmap);
begin
  FBitmap.Assign(value);
  if not FBitmap.Empty then
    FBitmap.Dormant;
end;

procedure TExplorerButton.SetCaption(value: TCaption);
begin
  FCaption := value;
  if boIconOnly in FOptions then
    Hint := RemoveAmpersand(FCaption)
  else
    Repaint;
end;

procedure TExplorerButton.SetDisabledBitmap(value: TBitmap);
begin
  FDisabledBitmap.Assign(value);
  if not FDisabledBitmap.Empty then
    FDisabledBitmap.Dormant;
end;

procedure TExplorerButton.SetDown(value: Boolean);
var i: Integer;
  otherbutton: TExplorerButton;
begin
  if FDown = value then
     Exit;
  if FGroupIndex <> 0 then
  begin
    FDown := value;
    Repaint;
       if FDown = True then
    begin
      for i := 0 to Parent.ControlCount - 1 do
         begin
        if Parent.Controls[i] is TExplorerButton then
        begin
          otherbutton := (Parent.Controls[i] as TExplorerButton);
          if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
            and (otherbutton.FDown = True) then
          begin
            otherbutton.FDown := False;
            otherbutton.Repaint;
          end
        end
      end
    end
  end
end;

procedure TExplorerButton.SetDropDown(value: TPopupMenu);
begin
  FDropDown := value;
  if  FDropDown <> nil then
    ExplorerPopup := nil;
  if boPopupMark in FOptions then
    Invalidate;
end;

procedure TExplorerButton.SetDropDownStyle(value: TExplorerButtonDropDownStyle);
begin
  if FDropDownStyle = value then
     Exit;
  FDropDownStyle := value;
  GetLost;
end;

procedure TExplorerButton.SetEnabled(value: Boolean);
begin
  if FEnabled = value then
     Exit;
  FEnabled := value;
  GetLost;
end;

procedure TExplorerButton.SetExplorerPopup(value: TExplorerPopup);
begin
  if value <> FExplorerPopup then
  begin
    FExplorerPopup := value;
     if value <> nil then
        DropDownMenu := nil;
  end
end;

procedure TExplorerButton.SetGroupIndex(value: Integer);
begin
  if FGroupIndex = value then
    Exit;
  FGroupIndex := value;
  if (FGroupIndex = 0) and FDown then
  begin
    FDown := False;
    Repaint;
  end;
end;

procedure TExplorerButton.SetLayout(value: TExplorerButtonLayout);
begin
  if FLayout = value then
     Exit;
  FLayout := value;
  Repaint;
end;

procedure TExplorerButton.SetNoFocusBitmap(value: TBitmap);
begin
  FNoFocusBitmap.Assign(value);
  if not FNoFocusBitmap.Empty then
    FNoFocusBitmap.Dormant;
end;

procedure TExplorerButton.SetOptions(value: TExplorerButtonOptions);
var
  wasTransparent: Boolean;
begin
  if boTransparent in FOptions then
    wasTransparent := True
  else
    wasTransparent := False;
  FOptions := value;
  if (boIconOnly in FOptions) and (Hint = '') then
  begin
    Hint := RemoveAmpersand(FCaption);
    ShowHint := True;
  end
  else if (not(boIconOnly in FOptions)) and (Hint = RemoveAmpersand(FCaption)) then
  begin
    Hint := '';
    ShowHint := False;
  end;
  if (((boTransparent in FOptions) and not wasTransparent)
      or (not (boTransparent in FOptions) and wasTransparent))
      and not (csDesigning in ComponentState) then
  begin
    backbitmapCreated := False;
     RecreateWnd;
     Parent.Refresh;
  end
  else
    Repaint;
end;

procedure TExplorerButton.SetShadingType(value: TExplorerButtonShadingType);
begin
  if value <> FShadingType then
  begin
    FShadingType := value;
    if (not FBitmap.Empty) and FNoFocusBitmap.Empty and not (csLoading in ComponentState) then
    begin
      CreateGrayscaleBitmap(IBitmap, FBitmap);
      Repaint;
    end;
  end;
end;

procedure TExplorerButton.SetUnselectedFontColor(value: TColor);
begin
  if FUnselectedFontColor <> value then
  begin
    FUnselectedFontColor := value;
    Repaint;
  end;
end;

procedure TExplorerButton.WMLButtonDown(var msg: TWMLButtonDown);
var p: TPoint;
  theMsg: TMsg;
  lpPoint: TPoint;
begin
  if not GetParentForm(Self).Active then
    GetParentForm(Self).BringToFront;

  if Enabled and Visible then
  begin
     if FDropDownStyle = ddsIExplorer then
      Pushed := True
     else
     begin
       if msg.XPos < Width - 11 then
        Pushed := True
        else
          popupPushed := True
     end;

    MouseIn := True;
    Repaint;
    if ((FDropDown <> nil) or popupPushed or ((FExplorerPopup <> nil) and (FDropDownStyle = ddsIExplorer)))
      and Assigned(FOnDropDownClick) then
      FOnDropDownClick(Self);

     if (not popupOpened) and Assigned(FExplorerPopup)
       and (((FDropDownStyle = ddsOffice) and popupPushed) or (FDropDownStyle = ddsIExplorer)) then
     begin
          PostMessage(Handle, WM_EXPLORERPOPUPOPEN, 0, 0);
     end
     else
    if (not popupOpened) and (FDropDown <> nil) and ((FDropDownStyle = ddsIExplorer) or popupPushed) then
    begin
        p := ClientToScreen(Point(0, Height));

      FDropDown.Popup(p.x, p.y);
      while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
        ;
      if GetCapture <> 0 then
        SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
      GetCursorPos(lpPoint);
      lpPoint := Parent.ScreenToClient(lpPoint);
      if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
        and (lpPoint.x > left) and (lpPoint.x < left + width)) then
        MouseIn := False;

       if FDropDownStyle = ddsIExplorer then
        Pushed := False
        else
          popupPushed := False;
      Repaint;
    end
     else
       SetCaptureControl(Self);
  end;
   MouseDown(mbLeft, KeysToShiftState(msg.Keys), msg.XPos, msg.YPos);
end;

procedure TExplorerButton.WMLButtonUp(var msg: TWMLButtonUp);
var i: Integer;
  otherbutton: TExplorerButton;
  lpPoint: TPoint;
begin
  if Enabled and Visible and (Pushed or popupPushed) then
  begin
    if Pushed and MouseIn and (GroupIndex <> 0) then
    begin
      (* Was Down -> allow up state if AllowAllUp is True *)
      if FDown then
      begin
        if FAllowAllUp then
          FDown := False
      end
      else
      (* Was Up *)
      begin
        (* Set 'up' all buttons having the same parent
          and GroupIndex *)
        for i := 0 to Parent.ControlCount - 1 do
        begin
          if Parent.Controls[i] is TExplorerButton then
          begin
            otherbutton := (Parent.Controls[i] as TExplorerButton);
            if (otherbutton <> Self) and (otherbutton.GroupIndex = GroupIndex)
              and (otherbutton.FDown = True) then
            begin
              otherbutton.Down := False;
              otherbutton.Repaint;
            end
          end
        end;
        FDown := True;
      end;
    end;

    Pushed := False;

     if MouseCapture then
       SetCaptureControl(nil);
    if MouseIn then
    begin
      GetCursorPos(lpPoint);
      lpPoint := Parent.ScreenToClient(lpPoint);
      if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
        and (lpPoint.x > left) and (lpPoint.x < left + width)) then
        MouseIn := False;
      Invalidate;
      (* If the popup is not defined, activate the click event *)
        if ((FDropDownStyle = ddsOffice) and (not popupPushed))
           or ((FDropDownStyle = ddsIExplorer) and (FDropDown = nil) and (FExplorerPopup = nil)) then
        Click;
    end;
    popupPushed := False;
     MouseUp(mbLeft, KeysToShiftState(msg.Keys), msg.XPos, msg.YPos);
  end;
end;

procedure TExplorerButton.CMMouseEnter(var msg: TMessage);
begin
  MouseIn := True;
  if not FDown and FEnabled then
    Repaint;
        if Assigned(FOnEnter) then
                FOnEnter(Self);
  (*
   * Automatic opening of the popup panel if there is already
   * an opened popup
   *)
  if Assigned(activeExplorerPopup) and (activeExplorerButton <> Self) then
  begin
    if Parent <> activeExplorerPopup then
     begin
     activeExplorerButton.PopupOpened := False;
    activeExplorerPopup.Close;
    activeExplorerButton.Repaint;
     activeExplorerPopup := nil;
     if Assigned(FExplorerPopup) then
     begin
          {
            Needed to avoir recursive opens (could quickly lead to
            stack space exhaustion)
          }
          PostMessage(Handle, WM_EXPLORERPOPUPOPEN, 0, 0);
     end
     end
  end;
end;

procedure TExplorerButton.CMSysColorChange(var msg: TMessage);
begin
  if FNoFocusBitmap.Empty then
    CreateGrayscaleBitmap(IBitmap, FBitmap);
  Regenerate;
end;

procedure TExplorerButton.CMMouseLeave(var msg: TMessage);
begin
  MouseIn := False;
  if not FDown and FEnabled then
    Repaint;
        if Assigned(FOnExit) then
                FOnExit(Self);
end;

procedure TExplorerButton.WMRButtonUp(var msg: TWMRButtonUp);
var
  theMsg: TMsg;
  lpPoint: TPoint;
begin
  inherited;
  if (PopupMenu <> nil) or (TForm(GetParentForm(Self)).PopupMenu <> nil) then
  begin
    while PeekMessage(theMsg, HWND(0), WM_MOUSEFIRST, WM_MOUSELAST, PM_REMOVE) do
      ;
    if GetCapture <> 0 then
      SendMessage(GetCapture, WM_CANCELMODE, 0, 0);
    GetCursorPos(lpPoint);
    lpPoint := Parent.ScreenToClient(lpPoint);
    if not  ((lpPoint.y > top) and (lpPoint.y < top + height)
      and (lpPoint.x > left) and (lpPoint.x < left + width)) then
      MouseIn := False;
    Pushed := False;
     popupPushed := False;
    Repaint;
  end;
end;

procedure TExplorerButton.WMLButtonDblClk(var msg: TWMLButtonDblClk);
begin
  if ((FDropDownStyle = ddsOffice) and (msg.XPos < Width - 11))
     or ((FDropDownStyle = ddsIExplorer) and (FExplorerPopup = nil)) then
    DblClick;
end;

procedure TExplorerButton.WMSize(var msg: TWMSize);
begin
     inherited;
     if (boTransparent in FOptions) and painted and not (csDesigning in ComponentState) then
     begin
        backBitmapCreated := False;
        RecreateWnd;
     end;

     if FakeResize then
     begin
       FakeResize := False;
       SetBounds(Left, Top, Width - 1, Height);
      Exit;
     end;

     if FButtonSize = bsCustom then
     begin
         updatingSize := True;
        CustomWidth := Width;
        CustomHeight := Height;
        updatingSize := False;
     end else if FButtonSize = bsLarge then
     begin
         updatingSize := True;
        LargeWidth := Width;
        LargeHeight := Height;
        updatingSize := False;
     end else if FButtonSize = bsSmall then
     begin
         updatingSize := True;
        SmallWidth := Width;
        SmallHeight := Height;
        updatingSize := False;
     end
end;

procedure TExplorerButton.WMMove(var msg: TWMMove);
begin
     inherited;
     if (boTransparent in FOptions) and painted and not (csDesigning in ComponentState) then
     begin
        backBitmapCreated := False;
        RecreateWnd;
     end;
end;

procedure TExplorerButton.WMEraseBkgnd(var msg: TWMEraseBkgnd);
begin
  Msg.Result := 1;
end;

procedure TExplorerButton.Click;
begin
  inherited Click;
end;

procedure TExplorerButton.WMExplorerPopupOpen(var msg: TMessage);
begin
     popupOpened := True;
     activeExplorerPopup := FExplorerPopup;
     activeExplorerButton := Self;
     Repaint;
     FExplorerPopup.Open(Self);
     activeExplorerPopup := nil;
     popupOpened := False;
     Repaint;
end;

procedure TExplorerButton.SetSmallWidth(value: Integer);
begin
  if value <> FSmallWidth then
  begin
    FSmallWidth := value;
   if (not updatingSize) and (FButtonSize = bsSmall) then
       Width := FSmallWidth
  end
end;

procedure TExplorerButton.SetSmallHeight(value: Integer);
begin
  if value <> FSmallHeight then
  begin
    FSmallHeight := value;
     if (not updatingSize) and (FButtonSize = bsSmall) then
       Height := FSmallHeight
  end
end;

procedure TExplorerButton.SetLargeWidth(value: Integer);
begin
  if value <> FLargeWidth then
  begin
    FLargeWidth := value;
     if (not updatingSize) and (FButtonSize = bsLarge) then
       Width := FLargeWidth
  end
end;

procedure TExplorerButton.SetLargeHeight(value: Integer);
begin
  if value <> FLargeHeight then
  begin
    FLargeHeight := value;
     if (not updatingSize) and (FButtonSize = bsLarge) then
       Height := FLargeHeight
  end
end;

procedure TExplorerButton.SetCustomWidth(value: Integer);
begin
  if value <> FCustomWidth then
  begin
    FCustomWidth := value;
   if (not updatingSize) and (FButtonSize = bsCustom) then
       Width := FCustomWidth
  end
end;

procedure TExplorerButton.SetCustomHeight(value: Integer);
begin
  if value <> FCustomHeight then
  begin
    FCustomHeight := value;
     if (not updatingSize) and (FButtonSize = bsCustom) then
       Height := FCustomHeight
  end
end;

procedure TExplorerButton.SetButtonSize(value: TExplorerButtonButtonSize);
begin
  if value <> FButtonSize then
  begin
    FButtonSize := value;
     case FButtonSize of
       bsSmall: SetBounds(Left, Top, SmallWidth, SmallHeight);
       bsLarge: SetBounds(Left, Top, LargeWidth, LargeHeight);
       bsCustom: SetBounds(Left, Top, CustomWidth, CustomHeight);
     end
  end
end;

procedure TExplorerButton.DrawTransparentBitmap(theCanvas: TCanvas; bmp: TBitmap; xStart, yStart: Integer);
var
   ImageList: TImageList;
   Index: Integer;
   TransparentColor: TColor;
   saveBmp: TBitmap;
begin
  if not (boTransparent in FOptions) then
   begin
      theCanvas.Brush.Color := parent.brush.Color;
     if FButtonSize = bsSmall then
      theCanvas.BrushCopy(Rect(xStart, yStart, xStart + SMALLBITMAPWIDTH, yStart + SMALLBITMAPHEIGHT), bmp,
                    Rect(0, 0, SMALLBITMAPWIDTH, SMALLBITMAPHEIGHT), bmp.Canvas.Pixels[0, 0])
      else if FButtonSize = bsLarge then
      theCanvas.BrushCopy(Rect(xStart, yStart, xStart + LARGEBITMAPWIDTH, yStart + LARGEBITMAPHEIGHT), bmp,
                    Rect(0, 0, SMALLBITMAPWIDTH, SMALLBITMAPHEIGHT), bmp.Canvas.Pixels[0, 0])
      else
      theCanvas.BrushCopy(Rect(xStart, yStart, xStart + bmp.Width, yStart + bmp.Height), bmp,
                    Rect(0, 0, bmp.Width, bmp.Height), bmp.Canvas.Pixels[0, 0])
   end
   else
  begin
      theCanvas.brush.style := bsClear;
      {$IFDEF WIN32}
       ImageList := TImageList.CreateSize(bmp.Width, bmp.Height);
      {$ELSE}
       ImageList := TImageList.Create(bmp.Width, bmp.Height);
      {$ENDIF}
     {$IFNDEF VER80}
        ImageList.DrawingStyle := dsTransparent;
        ImageList.BkColor := clNone;
      {$ENDIF}
{      // Work with a copy of the bitmap as it is modified by addmasked}
      saveBmp := TBitmap.Create;
      saveBmp.Assign(bmp);
      TransparentColor := saveBmp.Canvas.Pixels[0, 0];
      try
        Index := ImageList.AddMasked(saveBmp, TransparentColor);
        ImageList.Draw(theCanvas, xStart, yStart, Index);
      finally
        ImageList.Free;
         saveBmp.Free;
      end;
   end;
end;

procedure TExplorerButton.DrawDisabledBitmap(Canvas: TCanvas; x, y: Integer; bmp: TBitmap);
var MonoBmp: TBitmap;
begin
  MonoBmp := TBitmap.Create;
  try
    MonoBmp.Assign(bmp);
    MonoBmp.Canvas.Brush.Color := clBlack;
    MonoBmp.Monochrome := True;
    Canvas.Brush.Color := clBtnHighlight;
    SetTextColor(Canvas.Handle, clBlack);
    SetBkColor(Canvas.Handle, clWhite);
     case FButtonSize of
       bsCustom: BitBlt(Canvas.Handle, x+1, y+1, bmp.Width, bmp.Height,
                 MonoBmp.Canvas.Handle, 0, 0, $00E20746);
        bsSmall: BitBlt(Canvas.Handle, x+1, y+1, SMALLBITMAPWIDTH, SMALLBITMAPWIDTH,
                 MonoBmp.Canvas.Handle, 0, 0, $00E20746);
        bsLarge: StretchBlt(Canvas.Handle, x+1, y+1, LARGEBITMAPWIDTH, LARGEBITMAPWIDTH,
                 MonoBmp.Canvas.Handle, 0, 0, SMALLBITMAPWIDTH, SMALLBITMAPWIDTH, $00E20746);
     end;
    Canvas.Brush.Color := clBtnShadow;
    SetTextColor(Canvas.Handle, clBlack);
    SetBkColor(Canvas.Handle, clWhite);
     case FButtonSize of
       bsCustom: BitBlt(Canvas.Handle, x, y, bmp.Width, bmp.Height,
                 MonoBmp.Canvas.Handle, 0, 0, $00E20746);
        bsSmall: BitBlt(Canvas.Handle, x, y, SMALLBITMAPWIDTH, SMALLBITMAPWIDTH,
                 MonoBmp.Canvas.Handle, 0, 0, $00E20746);
        bsLarge: StretchBlt(Canvas.Handle, x, y, LARGEBITMAPWIDTH, LARGEBITMAPWIDTH,
                 MonoBmp.Canvas.Handle, 0, 0, SMALLBITMAPWIDTH, SMALLBITMAPWIDTH, $00E20746);
     end;
  finally
    MonoBmp.Free;
  end
end;

procedure Register;
begin
   RegisterComponents('Freeware', [TExplorerButton]);
   RegisterComponents('Freeware', [TExplorerPopup]);
end;

initialization
   pattern := nil;
   activeExplorerPopup := nil;
   GetSystemVersion;

end.

