unit AMHDrvComboBox;

interface

uses
  Windows, Messages, Classes, Graphics, Controls, StdCtrls;

type

  TAMHDrvComboBox = class(TCustomComboBox)
    private
      Drives: TStrings;
      DriveItemIndex: Integer;
      Images: TImagelist;
      FDrive: Char;
      FOnChange: TNotifyEvent;
      procedure CMFontChanged(var Msg:TMessage); message CM_FONTCHANGED;
      procedure ResetItemHeight;
    protected
      procedure Change; override;
      procedure CreateWnd; override;
      procedure DrawItem(Index: Integer; Rect: TRect; State: TOwnerDrawState); override;
      procedure CNCommand(var Msg: TWMCommand); message CN_COMMAND;
      procedure BuildList; virtual;
      procedure SetDrive(Value:char);
    public
      constructor Create(AOwner: TComponent); override;
      destructor Destroy; override;
      procedure UpdateDrives;
    published
      property Drive: char read FDrive write SetDrive;
      property OnChange: TNotifyEvent read FOnChange write FOnChange;
  end;

procedure Register;

implementation

uses
  ShellAPI;

const
  SHGFI = SHGFI_SYSICONINDEX or SHGFI_SMALLICON;

function GetItemHeight(Font:TFont):Integer;
var
  DC       : HDC;
  SaveFont : HFont;
  Metrics  : TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC,Font.Handle);
  GetTextMetrics(DC, Metrics);
  SelectObject(DC, SaveFont);
  ReleaseDC(0, DC);
  Result := Metrics.tmHeight;
end;

constructor TAMHDrvComboBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  Style := csOwnerDrawFixed;
  ImeMode := imDisable;
  Drives := TStringList.Create;
  Images := TImagelist.CreateSize(GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON));
  with Images do begin
       DrawingStyle := dsTransparent;
       ShareImages := True;
  end;
  ResetItemHeight;
end;

destructor TAMHDrvComboBox.Destroy;
begin
  Drives.Free;
  Images.Free;
  inherited Destroy;
end;

procedure TAMHDrvComboBox.BuildList;
var
  Info      : TSHFileInfo;
  DriveChar : Char;
  CurrDrive : string;
  DriveType:Integer;
begin
  if Items.Count > 0 then begin
     if ItemIndex > -1 then
        DriveItemIndex := ItemIndex;
        Items.Clear;
  end;
  Images.Handle := SHGetFileInfo('', 0, Info, SizeOf(TShFileInfo), SHGFI);
  for DriveChar:='A' to 'Z' do begin
      CurrDrive := DriveChar + ':\';
      DriveType := GetDriveType(PChar(CurrDrive));
      if DriveType in [0,1] then
         Continue;
      SHGetFileInfo(PChar(CurrDrive), 0, Info, SizeOf(TShFileInfo), SHGFI_DISPLAYNAME or SHGFI);
      Items.AddObject(Info.szDisplayName, TObject(Info.iIcon));
      Drives.Add(DriveChar);
  end;
  SetDrive(Drives[DriveItemIndex][1]);
  Update;
end;

procedure TAMHDrvComboBox.CreateWnd;
begin
  inherited CreateWnd;
  BuildList;
end;

procedure TAMHDrvComboBox.DrawItem(Index:Integer;Rect:TRect;State:TOwnerDrawState);
var
  ImageTop: Integer;
begin
  with Canvas do begin
       FillRect(Rect);
       if Images.Count > 0 then begin
          ImageTop := Rect.Top + ((Rect.Bottom-Rect.Top-Images.Height) div 2);
          Images.Draw(Canvas, Rect.Left + 4, ImageTop, Integer(Items.Objects[Index]));
          Rect.Left := Rect.Left + Images.Width + 8;
       end;
       DrawText(Canvas.Handle, PChar(Items[Index]), -1, Rect, DT_SINGLELINE or DT_VCENTER or DT_NOPREFIX or DT_NOCLIP);
  end;
end;

procedure TAMHDrvComboBox.CMFontChanged(var Msg:TMessage);
begin
  inherited;
  ResetItemHeight;
  RecreateWnd;
end;

procedure TAMHDrvComboBox.ResetItemHeight;
var
  NewHeight : Integer;
  MinHeight : Integer;
begin
  MinHeight := Images.Height + 2;
  NewHeight := GetItemHeight(Font);
  if NewHeight < MinHeight then
     NewHeight := MinHeight;
     ItemHeight := NewHeight;
end;

procedure TAMHDrvComboBox.SetDrive(Value:Char);
var
  i : Integer;
  j : Integer;
begin
  j := 0;
  if DriveItemIndex <> -1 then
     j := DriveItemIndex;
  Value := UpCase(Value);
  if FDrive <> Value then begin
     for i := 0 to Items.Count - 1 do
         if Drives[i][1] = Value then begin
            FDrive := Value;
            DriveItemIndex := i;
            ItemIndex := i;
            Exit;
         end;
     end else
     ItemIndex := j;
end;

procedure TAMHDrvComboBox.Change;
begin
 if ItemIndex <> -1 then
    DriveItemIndex := ItemIndex;
 SetDrive(Drives[DriveItemIndex][1]);
 if Assigned(FOnChange) then
    FOnChange(Self);
end;

procedure TAMHDrvComboBox.CNCommand(var Msg: TWMCommand);
begin
  inherited;
  if Msg.NotifyCode = CBN_SELCHANGE then
     Change;
end;

procedure TAMHDrvComboBox.UpdateDrives;
var
  Info : TSHFileInfo;
begin
  LockWindowUpdate(Handle);
  if Assigned(Images) then
     Images.Free;
  Images := TImagelist.CreateSize(GetSystemMetrics(SM_CXSMICON), GetSystemMetrics(SM_CYSMICON));
  with Images do begin
       DrawingStyle := dsTransparent;
       ShareImages := True;
  end;
  Images.Handle := SHGetFileInfo('', 0, Info, SizeOf(TShFileInfo), SHGFI);
  BuildList;
  LockWindowUpdate(0);
end;

procedure Register;
begin
  RegisterComponents('AMH', [TAMHDrvComboBox]);
end;

end.
