{****************************
 *                          *
 * Autor: Adrian M. Hanslik *
 * Date 02. April 1999      *
 * Version: 1.4             *
 *                          *
 ****************************}

unit AMHExtMonthCal;

interface

uses Classes, Controls, Messages, Windows, Forms, Graphics, StdCtrls,
  Grids, SysUtils;

type
  TDayOfWeek = 0..6;

  TAMHExtMonthCal = class(TCustomGrid)
  private
    FDate: TDateTime;
    FMonthOffset: Integer;
    FOnChange: TNotifyEvent;
    FReadOnly: Boolean;
    FStartOfWeek: TDayOfWeek;
    FUpdating: Boolean;
    FUseCurrentDate: Boolean;
    FColorDefault: TColor;
    FColorHeader: TColor;
    FColorSundays: TColor;
    FColorToday: TColor;
    FFontDefault: TFont;
    FFontHeader: TFont;
    FFontSundays: TFont;
    FFontToday: TFont;
    function GetCellText(ACol, ARow: Integer): string;
    function GetDateElement(Index: Integer): Integer;
    procedure SetCalendarDate(Value: TDateTime);
    procedure SetDateElement(Index: Integer; Value: Integer);
    procedure SetStartOfWeek(Value: TDayOfWeek);
    procedure SetUseCurrentDate(Value: Boolean);
    function StoreCalendarDate: Boolean;
    procedure SetColorDefault(Value: TColor);
    procedure SetColorHeader(Value: TColor);
    procedure SetColorSundays(Value: TColor);
    procedure SetColorToday(Value: TColor);
    procedure SetFontDefault(Value: TFont);
    procedure SetFontHeader(Value: TFont);
    procedure SetFontSundays(Value: TFont);
    procedure SetFontToday(Value: TFont);
  protected
    procedure Change; dynamic;
    procedure ChangeMonth(Delta: Integer);
    procedure Click; override;
    function DaysPerMonth(AYear, AMonth: Integer): Integer; virtual;
    function DaysThisMonth: Integer; virtual;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    function IsLeapYear(AYear: Integer): Boolean; virtual;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public
    constructor Create(AOwner: TComponent); override;
    property CalendarDate: TDateTime  read FDate write SetCalendarDate stored StoreCalendarDate;
    property CellText[ACol, ARow: Integer]: string read GetCellText;
    procedure NextMonth;
    procedure NextYear;
    procedure PrevMonth;
    procedure PrevYear;
    procedure UpdateCalendar; virtual;
  published
    property Align;
    property BorderStyle;
    property ColorDefault: TColor read FColorDefault write SetColorDefault;
    property ColorHeader: TColor read FColorHeader write SetColorHeader;
    property ColorSundays: TColor read FColorSundays write SetColorSundays;
    property ColorToday: TColor read FColorToday write SetColorToday;
    {property Color;}
    property Ctl3D;
    property Day: Integer index 3  read GetDateElement write SetDateElement stored False;
    property DefaultColWidth;
    property DefaultRowHeight;
    property Enabled;
    property FixedCols;
    property FixedRows;
    property FontDefault: TFont read FFontDefault write SetFontDefault;
    property FontHeader: TFont read FFontHeader write SetFontHeader;
    property FontSundays: TFont read FFontSundays write SetFontSundays;
    property FontToday: TFont read FFontToday write SetFontToday;
    {property Font;}
    property GridLineWidth;
    property Month: Integer index 2  read GetDateElement write SetDateElement stored False;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property ScrollBars;
    property ShowHint;
    property StartOfWeek: TDayOfWeek read FStartOfWeek write SetStartOfWeek;
    property TabOrder;
    property TabStop;
    property UseCurrentDate: Boolean read FUseCurrentDate write SetUseCurrentDate default True;
    property Visible;
    property Year: Integer index 1  read GetDateElement write SetDateElement stored False;
    property OnClick;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnStartDrag;
  end;

implementation

constructor TAMHExtMonthCal.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { defaults }
  FUseCurrentDate := True;
  FixedCols := 0;
  FixedRows := 0;
  ColCount := 7;
  RowCount := 7;
  FColorDefault := clNone;
  FColorHeader := clNone;
  FColorSundays := clNone;
  FColorToday := clNone;
  FFontDefault := TFont.Create;
  FFontHeader := TFont.Create;
  FFontSundays := TFont.Create;
  FFontToday := TFont.Create;
  ScrollBars := ssNone;
  Options := Options - [goRangeSelect] + [goDrawFocusSelected];
  FDate := Date;
  UpdateCalendar;
end;

procedure TAMHExtMonthCal.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TAMHExtMonthCal.Click;
var
  TheCellText: string;
begin
  inherited Click;
  TheCellText := CellText[Col, Row];
  if TheCellText <> '' then Day := StrToInt(TheCellText);
end;

function TAMHExtMonthCal.IsLeapYear(AYear: Integer): Boolean;
begin
  Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

function TAMHExtMonthCal.DaysPerMonth(AYear, AMonth: Integer): Integer;
const
  DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
  Result := DaysInMonth[AMonth];
  if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

function TAMHExtMonthCal.DaysThisMonth: Integer;
begin
  Result := DaysPerMonth(Year, Month);
end;

procedure TAMHExtMonthCal.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  TheText: string;
  Present: TDateTime;
  PYear, PMonth, PDay: Word;
begin
  TheText := CellText[ACol, ARow];
  Present:= Now;
  DecodeDate(Present, PYear, PMonth, PDay);
  with ARect, Canvas do begin
       if ARow = 0 then begin
          Brush.Color := ColorHeader;
          Font := FontHeader;
       end else
       if ((FStartOfWeek = 0) and  (ACol = 0)) or (ACol = 7 - FStartOfWeek) then begin
          Brush.Color := ColorSundays;
          Font := FontSundays;
       end else
       if (PYear = Year) and (PMonth = Month) and (PDay = Day) and (GetCellText(ACol, ARow) = IntToStr(Day)) then begin
          Brush.Color := ColorToday;
          Font := FontToday;
       end else begin
           Brush.Color := ColorDefault;
           Font := FontDefault;
       end;
       TextRect(ARect, Left + (Right - Left - TextWidth(TheText)) div 2,
                Top + (Bottom - Top - TextHeight(TheText)) div 2, TheText);
  end;
end;

function TAMHExtMonthCal.GetCellText(ACol, ARow: Integer): string;
var
  DayNum: Integer;
begin
  if ARow = 0 then  { day names at tops of columns }
    Result := ShortDayNames[(StartOfWeek + ACol) mod 7 + 1]
  else
  begin
    DayNum := FMonthOffset + ACol + (ARow - 1) * 7;
    if (DayNum < 1) or (DayNum > DaysThisMonth) then Result := ''
    else Result := IntToStr(DayNum);
  end;
end;

function TAMHExtMonthCal.SelectCell(ACol, ARow: Longint): Boolean;
begin
  if ((not FUpdating) and FReadOnly) or (CellText[ACol, ARow] = '') then
    Result := False
  else Result := inherited SelectCell(ACol, ARow);
end;

procedure TAMHExtMonthCal.SetCalendarDate(Value: TDateTime);
begin
  FDate := Value;
  UpdateCalendar;
  Change;
end;

function TAMHExtMonthCal.StoreCalendarDate: Boolean;
begin
  Result := not FUseCurrentDate;
end;

function TAMHExtMonthCal.GetDateElement(Index: Integer): Integer;
var
  AYear, AMonth, ADay: Word;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  case Index of
    1: Result := AYear;
    2: Result := AMonth;
    3: Result := ADay;
    else Result := -1;
  end;
end;

procedure TAMHExtMonthCal.SetDateElement(Index: Integer; Value: Integer);
var
  AYear, AMonth, ADay: Word;
begin
  if Value > 0 then
  begin
    DecodeDate(FDate, AYear, AMonth, ADay);
    case Index of
      1: if AYear <> Value then AYear := Value else Exit;
      2: if (Value <= 12) and (Value <> AMonth) then AMonth := Value else Exit;
      3: if (Value <= DaysThisMonth) and (Value <> ADay) then ADay := Value else Exit;
      else Exit;
    end;
    FDate := EncodeDate(AYear, AMonth, ADay);
    FUseCurrentDate := False;
    UpdateCalendar;
    Change;
  end;
end;

procedure TAMHExtMonthCal.SetStartOfWeek(Value: TDayOfWeek);
begin
  if Value <> FStartOfWeek then
  begin
    FStartOfWeek := Value;
    UpdateCalendar;
  end;
end;

procedure TAMHExtMonthCal.SetUseCurrentDate(Value: Boolean);
begin
  if Value <> FUseCurrentDate then
  begin
    FUseCurrentDate := Value;
    if Value then
    begin
      FDate := Date; { use the current date, then }
      UpdateCalendar;
    end;
  end;
end;

{ Given a value of 1 or -1, moves to Next or Prev month accordingly }
procedure TAMHExtMonthCal.ChangeMonth(Delta: Integer);
var
  AYear, AMonth, ADay: Word;
  NewDate: TDateTime;
  CurDay: Integer;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  CurDay := ADay;
  if Delta > 0 then ADay := DaysPerMonth(AYear, AMonth)
  else ADay := 1;
  NewDate := EncodeDate(AYear, AMonth, ADay);
  NewDate := NewDate + Delta;
  DecodeDate(NewDate, AYear, AMonth, ADay);
  if DaysPerMonth(AYear, AMonth) > CurDay then ADay := CurDay
  else ADay := DaysPerMonth(AYear, AMonth);
  CalendarDate := EncodeDate(AYear, AMonth, ADay);
end;

procedure TAMHExtMonthCal.PrevMonth;
begin
  ChangeMonth(-1);
end;

procedure TAMHExtMonthCal.NextMonth;
begin
  ChangeMonth(1);
end;

procedure TAMHExtMonthCal.NextYear;
begin
  if IsLeapYear(Year) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year + 1;
end;

procedure TAMHExtMonthCal.PrevYear;
begin
  if IsLeapYear(Year) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year - 1;
end;

procedure TAMHExtMonthCal.UpdateCalendar;
var
  AYear, AMonth, ADay: Word;
  FirstDate: TDateTime;
begin
  FUpdating := True;
  try
    DecodeDate(FDate, AYear, AMonth, ADay);
    FirstDate := EncodeDate(AYear, AMonth, 1);
    FMonthOffset := 2 - ((DayOfWeek(FirstDate) - StartOfWeek + 7) mod 7); { day of week for 1st of month }
    if FMonthOffset = 2 then FMonthOffset := -5;
    MoveColRow((ADay - FMonthOffset) mod 7, (ADay - FMonthOffset) div 7 + 1,
      False, False);
    Invalidate;
  finally
    FUpdating := False;
  end;
end;

procedure TAMHExtMonthCal.WMSize(var Message: TWMSize);
var
  GridLines: Integer;
begin
  GridLines := 6 * GridLineWidth;
  DefaultColWidth := (Message.Width - GridLines) div 7;
  DefaultRowHeight := (Message.Height - GridLines) div 7;
end;

procedure TAMHExtMonthCal.SetColorDefault(Value: TColor);
begin
  if Value <> FColorDefault then
     FColorDefault := Value;
  Invalidate;
end;

procedure TAMHExtMonthCal.SetColorHeader(Value: TColor);
begin
  if Value <> FColorHeader then
     FColorHeader := Value;
  Invalidate;
end;

procedure TAMHExtMonthCal.SetColorSundays(Value: TColor);
begin
  if Value <> FColorSundays then
     FColorSundays := Value;
  Invalidate;
end;

procedure TAMHExtMonthCal.SetColorToday(Value: TColor);
begin
  if Value <> FColorToday then
     FColorToday := Value;
  Invalidate;
end;

procedure TAMHExtMonthCal.SetFontDefault(Value: TFont);
begin
  if Value <> FFontDefault then
     FFontDefault.Assign(Value);
  Invalidate;
end;

procedure TAMHExtMonthCal.SetFontHeader(Value: TFont);
begin
  if Value <> FFontHeader then
     FFontHeader.Assign(Value);
  Invalidate;
end;

procedure TAMHExtMonthCal.SetFontSundays(Value: TFont);
begin
  if Value <> FFontSundays then
     FFontSundays.Assign(Value);
  Invalidate;
end;

procedure TAMHExtMonthCal.SetFontToday(Value: TFont);
begin
  if Value <> FFontToday then
     FFontToday.Assign(Value);
  Invalidate;
end;

end.
