unit Udbconv;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, DB, DBTables, Gauges, StdCTRLs;

type { Custom exceptions }
     EUnknownAlias           = class(Exception);
     ESourceAliasNotSet      = class(Exception);
     EDestinationAliasNotSet = class(Exception);

const { Error messages }
      msgEUnknownAlias           = 'Unknown alias [%s]. Cannot set %s';
      msgESourceAliasNotSet      = 'Alias alias not set';
      msgEDestinationAliasNotSet = 'Destination alias not set';

type
  TDBConverter = class(TComponent)
  private
    fSourceAlias,
    fDestinationAlias : string;

    fTablesToConvert,
    fAliasList        : TStringList;

    fPattern          : string;

    fExtensions,
    fSystemTables     : boolean;

    fResultFile       : TFileName;
    fStatusLabel      : TLabel;
    fProgressGauge    : TGauge;

    SourceTable,
    DestinationTable  : TTable;
    BatchMove         : TBatchMove;

    fStrictIndexConvertion : boolean;

    { Looks if the alias name is defined }
    function ValidAlias(AliasName : string) : boolean;

    { Property validators }
    procedure SetSourceAlias(AliasName : string);
    procedure SetDestinationAlias(AliasName : string);
  protected
  public
    Stop : boolean;

    constructor Create(aOwner : TComponent);override;
    destructor Destroy;override;

    function Execute : integer;
  published
    property SourceAlias      : string read fSourceAlias write SetSourceAlias;
    property DestinationAlias : string read fDestinationAlias write SetDestinationAlias;
    property TablesToConvert  : TStringList   read fTablesToConvert write fTablesToConvert;
    property Pattern          : string read fPattern write fPattern;
    property Extensions       : boolean read fExtensions write fExtensions;
    property SystemTables     : boolean read fSystemTables write fSystemTables;
    property ProgressGauge    : TGauge read fProgressGauge write fProgressGauge;
    property StatusLabel      : TLabel read fStatusLabel write fStatusLabel;
    property ResultFile       : TFileName read fResultFile write fResultFile;
    property StrictIndexConvertion : boolean read fStrictIndexConvertion write fStrictIndexConvertion default FALSE;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('AleF', [TDBConverter]);
end;

{ Looks if the alias name is defined }
function TDBConverter.ValidAlias(AliasName : string) : boolean;
begin
  ValidAlias := (fAliasList.IndexOf(AliasName)>=0)or(AliasName='');
end;

{ Property validators }
procedure TDBConverter.SetSourceAlias(AliasName : string);
begin
  if ValidAlias(AliasName)
    then begin
           fSourceAlias := AliasName;
           TablesToConvert.Clear;
           Session.GetTableNames(AliasName, Pattern, Extensions, SystemTables, TablesToConvert);
         end
  else raise EUnknownAlias.Create(Format(msgEUnknownAlias, [AliasName, 'SourceAlias']));
end;

procedure TDBConverter.SetDestinationAlias(AliasName : string);
begin
  if ValidAlias(AliasName)
    then fDestinationAlias := AliasName
    else raise EUnknownAlias.Create(Format(msgEUnknownAlias, [AliasName, 'DestinationAlias']));
end;

{ Public methods }
constructor TDBConverter.Create(aOwner : TComponent);
var n : integer;
begin
  inherited Create(aOwner);

  { Initializes values }
  Pattern       := '*.*';
  fExtensions   := FALSE;
  fSystemTables := FALSE;

  fStatusLabel   := NIL;
  fProgressGauge := NIL;

  { Reads the names of aliases }
  fAliasList := TStringList.Create;
  Session.GetAliasNames(fAliasList);

  { Creates the components }
  fTablesToConvert := TStringList.Create;
  SourceTable      := TTable.Create(Self);
  DestinationTable := TTable.Create(Self);
  BatchMove        := TBatchMove.Create(Self);
end;

destructor TDBConverter.Destroy;
begin
  fAliasList.Free;

  fTablesToConvert.Free;
  SourceTable.Free;
  DestinationTable.Free;
  BatchMove.Free;

  inherited Destroy;
end;

function TDBConverter.Execute : integer;
const IndexOptionsStrings : array[0..4]of string=
         ('Primary','Unique','Descending','Non maintained','Case insensitive');
var n, index, errcount : integer;
    errfile  : TextFile;
    str      : string;
    iopt     : byte;
begin
  result := 0;

  if (SourceAlias='') then raise ESourceAliasNotSet.Create(msgESourceAliasNotSet);
  if (DestinationAlias='') then raise EDestinationAliasNotSet.Create(msgEDestinationAliasNotSet);

  try
    try
      if (ResultFile<>'')
        then begin
               AssignFile(errfile, ResultFile);
               ReWrite(errfile);
             end;

      SourceTable.DatabaseName      := SourceAlias;
      DestinationTable.DatabaseName := DestinationAlias;

      BatchMove.Source      := SourceTable;
      BatchMove.Destination := DestinationTable;
      BatchMove.Mode        := batCopy;

      Stop                  := FALSE;

      for n := 0 to (TablesToConvert.Count-1) do
        begin
          if Assigned(fStatusLabel)
            then fStatusLabel.Caption := 'Migrating '+TablesToConvert[n]+'...';

          if Assigned(fProgressGauge)
            then fProgressGauge.Progress := (((n+1)*100) div TablesToConvert.Count);

          Application.ProcessMessages;

          SourceTable.TableName      := TablesToConvert[n];
          DestinationTable.TableName := TablesToConvert[n];

          BatchMove.Execute;

          { Create the indexes }
          SourceTable.IndexDefs.Update;
          for index := 0 to (SourceTable.IndexDefs.Count-1)
            do try
                 case StrictIndexConvertion of
                   TRUE : DestinationTable.AddIndex( SourceTable.IndexDefs[index].Name,
                                                     SourceTable.IndexDefs[index].Fields,
                                                     SourceTable.IndexDefs[index].Options);

                   FALSE : if (index=0) then DestinationTable.AddIndex( SourceTable.IndexDefs[index].Name,
                                                                        SourceTable.IndexDefs[index].Fields,
                                                                        [ixPrimary, ixUnique])
                                        else DestinationTable.AddIndex( SourceTable.IndexDefs[index].Name,
                                                                        SourceTable.IndexDefs[index].Fields,
                                                                        [ixCaseInsensitive]);
                 end;
               except
                 if (ResultFile<>'')
                   then begin
                          Inc(result);

                          Writeln(errfile, Format('Exception raised trying to add index ["%s","%s"] to table ["%s"]',
                                                  [SourceTable.IndexDefs[index].Name,
                                                   SourceTable.IndexDefs[index].Fields,
                                                   SourceTable.TableName]));
                          str := '';
                          for iopt := 0  to 4 do
                            if (Byte(SourceTable.IndexDefs[index].Options) and iopt<>0)
                              then str := str+IndexOptionsStrings[iopt]+', ';

                          Writeln(errfile, 'The index was '+str);
                          Writeln(errfile);
                        end;
               end;

          if Stop
            then begin
                   Writeln(errfile, 'User abort');
                   Exit;
                 end;
        end;
    except
      raise
    end;
  finally
    if (ResultFile<>'') then CloseFile(errfile);
  end;
end;

end.
