{@author(Kjell Hasthi (kjehas@frisurf.no))

TStringStree class implements a non-visual tree structure
like that found in TreeView. Simplicity itself. Easy to use and
easy to modify.
The terms used are derived from the directory structure every
PC-user is familiar with.
Run demo program as introductory learning.
}

unit StringTree;

interface

uses
    Classes, SysUtils, IniFiles, Dialogs;

type
    TStringTreeError = (
        strOK,
        strDirAlreadyExists,
        strPathNotFound,
        strFileListError,
        strEmptyFileName,
        strExceptionInLoadFromFile,
        strExceptionInSaveToFile,
        strException,
        strFileNotFound
        );

    TStringTree = class( TObject)
    private
      CreationOrder: Integer;
{ Function will fill the stringList with all files for the CreationOrder argument }
      function FileListByCreationOrder( const co: integer; var stringList: TStringList): TStringTreeError;
{ Function will save FileList for CreationOrder argument }
      function SaveFiles( iniFile: TIniFile; index: Integer): TStringTreeError;
    public
    { StringList for saving dir struct }
      DirTree: TStringList;
    { StringList for saving all file names }
      FileList: TStringList;
      constructor Create;
      destructor Destroy; override;
{ Function will return a meaningful string for StringTreeError argument }
      function ErrorToStr( StringTreeError: TStringTreeError): String;
{ Function returns parent path for path argument }
      function GetParentPath( const path: String): String;
{ Function returns True if dir path is defined }
      function DirExists( const path: String): Boolean;
{ Function will defined a dir path }
      function CreateDir( const path: String): TStringTreeError;
{ Function will will fill the stringList argument with whole dir tree }
      function GetDirTree( const path: String; var stringList: TStringList): TStringTreeError;
{ Funtion will define a new file name }
      function CreateFile( const path, fileName: String): TStringTreeError;
{ Function will add a creation order + file string to the FileList strinfList }
      function AddToFileList( const co: integer; fileName: String): TStringTreeError;
{ Function will return then CreationOrder integer saved in from DirTree }
      function GetCreationOrder( index: integer): integer;
{ Function will return the stringList with all files for the path argument }
      function GetFileList( const path: String; var stringList: TStringList): TStringTreeError;
{ Function will fill the stringList will all files in both path and subpath dir }
      function GetAllFiles( const path: String; var stringList: TStringList): TStringTreeError;
{ Function will return True is path-filename is defined }
      function FileExists( const path, fileName: String): Boolean;
{ Function will fill stringList with all paths for fileName argument }
      function FindFile( const fileName: String; var stringList: TStringList): TStringTreeError;
{ Function will fill path for first fileName found}
      function FindFileFirst( const fileName: String; var path: String): TStringTreeError;
{ Function will will remove path-fileName from FileList }
      function DeleteFile( const path, fileName: String): TStringTreeError;
{ Function will will remove all files for the dir path argument }
      function DeleteFiles( const path: String): TStringTreeError;
{ Function will remove dir path from DirTree, and remove all files
 and subdirectories DirTree and FileList}
      function DeleteDir( const path: String): TStringTreeError;
{ Function will clear DirTree and FileList }
      procedure Clear;
{ Function will read inifile into DirTree and fileList }
      function LoadFromFile( fileName: String): TStringTreeError;
{ Function will save DirTree and FileList to inifile }
      function SaveToFile( fileName: String): TStringTreeError;
{ Function will save FileList for dir path argument to inifile }
      function SaveFileListToFile( const path, fileName: String): TStringTreeError;
{ Function will return (real) file count for FileList }
      function GetFileCount: Integer;
{ Function will fill stringList with subdirectories (but not
   sub-sub directories) for dir path argument }
      function GetDirList( const path: String; var stringList: TStringList):  TStringTreeError;
{ Function will return True if fileName (without path) is defined }
      function FileDefined( const fileName: String): Boolean;
      function DeleteFilesByCreationOrder( const co: integer): TStringTreeError;
      function GetValue( const path: String): String;
    end;

implementation

constructor TStringTree.Create;
begin
     inherited Create;
     CreationOrder := 0;
     DirTree := TStringList.Create;
     DirTree.Sorted := False;
     FileList := TStringList.Create;
     //FileList.Sorted := False;
     FileList.Sorted := True;
end;

destructor TStringTree.Destroy;
begin
     FileList.Free;
     DirTree.Free;
     inherited Destroy;
end;

function TStringTree.ErrorToStr( StringTreeError: TStringTreeError): String;
begin
     case StringTreeError of
     strOK: Result := 'OK';
     strDirAlreadyExists: Result := 'Dir already exists';
     strPathNotFound: Result := 'Path not found';
     strFileListError: Result := 'FileList error';
     strEmptyFileName: Result := 'Empty filename';
     strExceptionInLoadFromFile: Result := 'LoadFromFile failed';
     strExceptionInSaveToFile: Result := 'SaveToFile failed';
     strFileNotFound: Result := 'Not found in path';
     end;
end;

function TStringTree.GetParentPath( const path: String): String;
var
   i: Integer;
begin
     for i := Length( path) downto 1 do begin
         if path[ i] = '\' then begin
            Result := Copy( path, 1, i - 1);
            exit;
         end;
     end;
     Result := '';
end;

function TStringTree.DirExists( const path: String): Boolean;
begin
     if path = '' then begin
        Result := True;
        exit;
     end;
     if DirTree.IndexOf( path) < 0 then Result := False
     else Result := True;
end;

function TStringTree.CreateDir( const path: String): TStringTreeError;
var
   i,j: Integer;
   parentPath: String;

   function findLastSubDir( const path: String; index: Integer): Integer;
   var
      s: String;
   begin
        Result := index;
        s := DirTree[ Result];
        while path = Copy( s, 1, Length( path)) do begin
              Inc( Result);
              if Result > DirTree.Count - 1then begin
                 Dec( Result);
                 exit;
              end;
              s := DirTree[ Result];
        end;
        Dec( Result);
   end;

begin
     if DirExists( path) then begin
        Result := strDirAlreadyExists;
        exit;
     end;
     parentPath := GetParentPath( path);
     if not DirExists( parentPath) then begin
        Result := CreateDir( parentPath);
        if Result <> strOK then exit;
     end;
     if parentPath = '' then begin
        Inc( CreationOrder);
        DirTree.AddObject( path, TObject( CreationOrder));
        Result := strOK;
        exit;
     end;
     i := DirTree.IndexOf( parentPath);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     if i = DirTree.Count - 1 then begin
        Inc( CreationOrder);
        DirTree.AddObject( path, TObject( CreationOrder));
     end
     else begin
        j := findLastSubDir( parentPath, i);
        if j = DirTree.Count - 1 then begin
           Inc( CreationOrder);
           DirTree.AddObject( path, TObject( CreationOrder));
        end
        else begin
           Inc( CreationOrder);
           DirTree.InsertObject( j + 1, path, TObject( CreationOrder));
        end;
     end;
     Result := strOK;
end;

function TStringTree.GetDirTree( const path: String; var stringList: TStringList): TStringTreeError;
var
   i, j: integer;
begin
     if path = '' then begin
        stringList.Assign( DirTree);
        Result := strOK;
        exit;
     end;
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     for j := i to DirTree.Count - 1 do begin
         if path = Copy( DirTree[ j], 1, Length( path)) then begin
            stringList.Add( DirTree[ j]);
         end
         else break;
     end;
     Result := strOK;
end;

function TStringTree.CreateFile( const path, fileName: String): TStringTreeError;
var
   i: Integer;
begin
     if not DirExists( path) then begin
        Result := CreateDir( path);
        if Result <> strOK then exit;
     end;
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     Result := AddToFileList( GetCreationOrder( i), fileName);
end;

function TStringTree.AddToFileList( const co: integer; fileName: String): TStringTreeError;
var
   i: Integer;
begin
     i := FileList.IndexOf( IntToStr( co));
     if i < 0 then begin
        FileList.Add( IntToStr( co));
        i := FileList.IndexOf( IntToStr( co));
     end;
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     FileList.Add( IntToStr( co) + ' ' + fileName);
     Result := strOK;
end;

function TStringTree.GetCreationOrder( index: integer): integer;
begin
     Result := Integer( DirTree.Objects[ index]);
end;

function TStringTree.FileListByCreationOrder( const co: integer; var stringList: TStringList): TStringTreeError;
var
   i, j: Integer;
   s, creationOrderStr, creationOrder: String;

   function GetCreationOrder: String;
   var
      ii: Integer;
      ss: String;
   begin
        ss := FileList[ j];
        ii := Pos( ' ', ss);
        if ii <= 0 then begin
           Result := Trim( ss);
        end;
        Result := Trim( Copy( ss, 1, ii - 1));
   end;

begin
     creationOrderStr := IntToStr( co);
     i := FileList.IndexOf( creationOrderStr);
     if i < 0 then begin
        i := FileList.IndexOf( creationOrderStr + ' ');
        if i < 0 then begin
           Result := strFileListError;
           exit;
        end;
     end;
     for j := i to FileList.Count - 1 do begin
         creationOrder := GetCreationOrder;
         if creationOrder = '' then continue;
         if creationOrderStr <> Trim( creationOrder) then break;
         s := Copy( FileList[ j], Length( creationOrderStr) + 2, Length( FileList[ j]));
         s := Trim( s);
         if s = '' then continue;
         stringList.Add( s);
     end;
     Result :=  strOK;
end;

function TStringTree.GetFileList( const path: String; var stringList: TStringList): TStringTreeError;
var
   i: Integer;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     FileListByCreationOrder( GetCreationOrder( i), stringList);
     Result := strOK;
end;

function TStringTree.DeleteFilesByCreationOrder( const co: integer): TStringTreeError;
var
   i, j: Integer;
   creationOrderStr, creationOrder: String;

   function GetCreationOrder: String;
   var
      ii: Integer;
      ss: String;
   begin
        ss := FileList[ j];
        ii := Pos( ' ', ss);
        if ii <= 0 then begin
           Result := Trim( ss);
        end;
        Result := Trim( Copy( ss, 1, ii - 1));
   end;

begin
     creationOrderStr := IntToStr( co);
     i := FileList.IndexOf( creationOrderStr);
     if i < 0 then begin
        i := FileList.IndexOf( creationOrderStr + ' ');
        if i < 0 then begin
           Result := strFileListError;
           exit;
        end;
     end;
     for j := FileList.Count - 1 downto i do begin
         creationOrder := GetCreationOrder;
         if creationOrder = '' then continue;
         if creationOrderStr <> Trim( creationOrder) then break;
         FileList.Delete( j);
     end;
     Result :=  strOK;
end;

function TStringTree.DeleteFiles( const path: String): TStringTreeError;
var
   i: Integer;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     DeleteFilesByCreationOrder( GetCreationOrder( i));
     Result := strOK;
end;

function TStringTree.GetAllFiles( const path: String; var stringList: TStringList): TStringTreeError;
var
   selectedDirTree: TStringList;
   i: integer;
begin
     selectedDirTree := TStringList.Create;
     try
        Result := GetDirTree( path, selectedDirTree);
        if Result <> strOK then exit;
        for i := 0 to selectedDirTree.Count - 1 do begin
            Result := GetFileList( selectedDirTree[ i], stringList);
            if Result <> strOK then exit;
        end;
        Result := strOK;
     finally
        selectedDirTree.Free;
     end;
end;

function TStringTree.FileExists( const path, fileName: String): Boolean;
var
   i: Integer;
   creationOrderStr: String;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := False;
        exit;
     end;
     creationOrderStr := IntToStr( GetCreationOrder( i));
     i := FileList.IndexOf( creationOrderStr + ' ' + fileName);
     if i < 0 then Result := False
     else Result := True;
end;

function TStringTree.FindFileFirst( const fileName: String; var path: String): TStringTreeError;
var
   i, p: Integer;
   s: String;
   co: Integer;

   function GetPath( creationOrder: Integer): String;
   var
      i: Integer;
   begin
        for i := 0 to DirTree.Count - 1 do begin
            if Integer( DirTree.Objects[ i]) = creationOrder then begin
               Result := DirTree[ i];
               exit;
            end;
        end;
        Result := ''
   end;

begin
     path := '';
  try
     for i := 0 to FileList.Count - 1 do begin
         s := FileList[ i];
         p := Pos( ' ', s);
         if p > 0 then begin
            if Copy( s, p + 1, Length( s)) = fileName then begin
               co := StrToInt( Copy( s, 1, p - 1));
               path := GetPath( co);
               Result := strOK;
               exit;
            end;
         end;
     end;
     Result := strFileNotFound;
  except
     Result := strException;
  end;
end;

function TStringTree.FindFile( const fileName: String; var stringList: TStringList): TStringTreeError;
var
   i, p: Integer;
   s: String;
   co: Integer;

   function GetPath( creationOrder: Integer): String;
   var
      i: Integer;
   begin
        for i := 0 to DirTree.Count - 1 do begin
            if Integer( DirTree.Objects[ i]) = creationOrder then begin
               Result := DirTree[ i];
               exit;
            end;
        end;
        Result := ''
   end;

begin
  try
     for i := 0 to FileList.Count - 1 do begin
         s := FileList[ i];
         p := Pos( ' ', s);
         if p > 0 then begin
            if Copy( s, p + 1, Length( s)) = fileName then begin
               co := StrToInt( Copy( s, 1, p - 1));
               stringList.Add( GetPath( co));
            end;
         end;
     end;
     Result := strOK;
  except
     Result := strException;
  end;
end;

function TStringTree.DeleteDir( const path: String): TStringTreeError;
var
   i, j, dirCount: Integer;
   stringList: TStringList;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     stringList := TStringList.Create;
     try
        dirCount := 0;
        for j := i to DirTree.Count - 1 do begin
            if path = Copy( DirTree[ j], 1, Length( path)) then begin
               DeleteFiles( DirTree[ j]);
               Inc( dirCount);
            end
            else break;
        end;
        while dirCount > 0 do begin
              DirTree.Delete( i);
              Dec( dirCount);
        end;
        Result := strOK;
     finally
        stringList.Free;
     end;
end;

procedure TStringTree.Clear;
begin
     FileList.Clear;
     DirTree.Clear;
     CreationOrder := 0;
end;

function TStringTree.LoadFromFile( fileName: String): TStringTreeError;
var
   stringList: TStringList;
   i, j: Integer;
   IniFile: TIniFile;
   s, path: String;
begin
  try
     if fileName = '' then begin
        Result := strEmptyFileName;
        exit;
     end;
     stringList := TStringList.Create;
     IniFile :=  TIniFile.Create( fileName);
     try
        for i := 0 to 5000 do begin
            path := IniFile.ReadString( 'DirTree', IntToStr( i), '');
            if path = '' then break;
            Result := CreateDir( path);
            if Result <> strOK then exit;
        end;
        IniFile.ReadSections( stringList);
        for i := 0 to stringList.Count - 1 do begin
            path := stringList[ i];
            if path = 'DirTree' then continue;
            if not DirExists( path) then begin
               Result := CreateDir( path);
               if Result <> strOK then exit;
            end;
            for j := 0 to 5000 do begin
                s := IniFile.ReadString( path, IntToStr( j), '');
                if s = '' then break;
                Result := CreateFile( path, s);
                if Result <> strOK then exit;
            end;
        end;
     Result := strOK;
     finally
        stringList.Free;
        IniFile.Free;
     end;
  except
      Result := strExceptionInLoadFromFile;
  end;
end;

function TStringTree.SaveToFile( fileName: String): TStringTreeError;
var
   i: Integer;
   IniFile: TIniFile;
begin
  try
     if fileName = '' then begin
        Result := strEmptyFileName;
        exit;
     end;
     SysUtils.DeleteFile( fileName);
     IniFile := TIniFile.Create( fileName);
     try
        for i := 0 to DirTree.Count - 1 do begin
            IniFile.WriteString( 'DirTree', IntToStr( i), DirTree[ i]);
        end;
        for i := 0 to DirTree.Count - 1 do SaveFiles( IniFile, i);
     Result := strOK;
     finally
        IniFile.Free;
     end;
  except
     Result := strExceptionInSaveToFile;
  end;
end;

function TStringTree.SaveFileListToFile( const path, fileName: String): TStringTreeError;
var
   i: Integer;
   IniFile: TIniFile;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     IniFile :=  TIniFile.Create( fileName);
     try
        IniFile.EraseSection( path);
        Result := SaveFiles( IniFile, i);
     finally
        IniFile.Free;
     end;
end;

function TStringTree.SaveFiles( iniFile: TIniFile; index: Integer): TStringTreeError;
var
   i: Integer;
   stringList: TStringList;
begin
try
     stringList := TStringList.Create;
  try
     FileListByCreationOrder( GetCreationOrder( index), stringList);
     if stringList.Count = 0 then begin
        Result := strOK;
        exit;
     end;
     for i := 0 to stringList.Count - 1 do begin
         IniFile.WriteString( Trim( DirTree[ index]), Trim( IntToStr( i)), Trim( stringList[ i]));
     end;
     Result := strOK;
  finally
     stringList.Free;
  end;
except
     Result := strException;
end;
end;

function TStringTree.GetFileCount: Integer;
var
   i, p: Integer;
begin
     Result := 0;
     for i := 0 to FileList.Count - 1 do begin
         p := Pos( ' ', FileList[ i]);
         if p <= 0 then Inc( Result);
     end;
end;

function TStringTree.GetDirList( const path: String; var stringList: TStringList): TStringTreeError;
var
   i, j, p: Integer;
   s: String;
begin
     if stringList = nil then begin
        Result := strException;
        exit;
     end;
     i := DirTree.IndexOf( path);
     if (i < 0) and (path <> '') then begin
        Result := strPathNotFound;
        exit;
     end;
     if i + 1 > DirTree.Count - 1 then begin
        Result := strOK;
        exit;
     end;
     for j := i + 1 to DirTree.Count - 1 do begin
         if path = Copy( DirTree[ j], 1, Length( path)) then begin
            if path = '' then
               s := Copy( DirTree[ j], Length( path) + 1, 2000)
            else s := Copy( DirTree[ j], Length( path) + 2, 2000);
            p := Pos( '\', s);
            if p <= 0 then stringList.add( s);
         end
         else break;
     end;
     Result := strOK;
end;

function TStringTree.FileDefined( const fileName: String): Boolean;
var
   i, p: Integer;
   s: String;
begin
     Result := True;
     for i := 0 to FileList.Count - 1 do begin
         s := FileList[ i];
         p := Pos( ' ', s);
         if p > 0 then begin
            if Copy( s, p + 1, Length( s)) = fileName then exit;
         end;
     end;
     Result := False;
end;

function TStringTree.DeleteFile( const path, fileName: String): TStringTreeError;
var
   i: Integer;
   creationOrderStr: String;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := strPathNotFound;
        exit;
     end;
     creationOrderStr := IntToStr( GetCreationOrder( i));
     i := FileList.IndexOf( creationOrderStr + ' ' + fileName);
     if i < 0 then begin
        Result := strFileNotFound;
        exit;
     end;
     FileList.Delete( i);
     Result := strOK;
end;

function TStringTree.GetValue( const path: String): String;
var
   i, p: Integer;
   creationOrderStr, s, t: String;
begin
     i := DirTree.IndexOf( path);
     if i < 0 then begin
        Result := '';
        exit;
     end;
     creationOrderStr := IntToStr( GetCreationOrder( i));
     i := FileList.IndexOf( creationOrderStr);
     if i < 0 then Result := ''
     else begin
          s := FileList[ i + 1];
          p := Pos( ' ', s);
          t := Copy( s, 1, p - 1);
          if creationOrderStr <> t then begin
             Result := '';
             exit;
          end;
          Result := Copy( s, p + 1, 32000);
     end;
end;

end.
