{
@abstract(provides time manipulation object)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@lastmod(7 Feb 2000)

The TTime class defined herein represents a single point in time,
e.g. 31 Dec 1998 13:00:03 GMT.

TTimeOptions defines how a TTime object will be converted to a
string or initialized from a string.
}
unit Time;

{$I platform.inc}

interface

uses
  {$IFDEF PPC_DELPHI}
  SysUtils,
  {$ENDIF}
  Numbers;

type
  { pointer to @link(TTimeOptions) }
  PTimeOptions = ^TTimeOptions;
  { @abstract(stores information on how a @link(TTime) object is converted
    to String)

       In this object you specify how a @link(TTime) object is converted to
       string.

       Define the field @link(TimeString) the way you want a point in time to
    be converted to a string.

       Take the % plus character combinations to be replaced by the actual
    values in @link(TTime.GetString) - this way you have the absolute
       freedom of what you want to have included.

    %Y  Full year (1998)
    %y  Short year (98)

    %M  Two digit month number (03)
    %m  Month number (3)
  %N  Long month name (March)
  %n  Short month name (Mar)

  %W	Long day-of-the-week name (Tuesday)
  %w	Short day-of-the-week name (Tue)

  %D  Two digit day number (09)
  %d  Day number (9)

  %H  Two digit hour number(05)
  %h	Hour number (5)

  %L  Two digit minute number(05)
  %l	Minute number (5)

  %S  Two digit second number(02)
  %s	Minute number (2)

  %E  'am' or 'pm'
  %Z	Time zone 'GMT', 'EST', 'PST' etc.
  %%  a literal %
  }
  TTimeOptions = object
    { 0=Sunday, 1=Monday, ..., 6=Saturday }
    FirstDOW: Integer;
    { format of a string with date and time }
    TimeString: string;
    { difference in seconds to GMT }
    TimeZone: TSInt32;
    { Initialize fields to default values }
    constructor Init;
  end;

var
  { default options to be used in @link(TTime.GetDefaultString);
    it will be initialized in the unit initialization section;
    this option variable will create output of the form "Wed 31 Dec 1998
    13:12:34" }
  CurrentTimeOptions: TTimeOptions;

type
  { pointer to @link(TTime) }
  PTime = ^TTime;
  { Stores a single point in time and allows manipulation and conversion
    to / from string. Note: 2 author tags within object definition!
    @author(Marco Schmidt <marcoschmidt@geocities.com>)
    @author(John Doe <doe@anonymous.org>) }
  TTime = object
    { Initializes fields, by calling @link(SetCurrentDateTime). }
    constructor Init;
    { Compares the point of time stored in this object to that in DT. }
    function CompareTo (var dt: TTime): TSInt32;
    { Converts date in D, M and Y to days passed since Jan 1, 1600,
      which will be stored in NumDays; returns true if date was valid,
      false otherwise. }
    function ConvertDMYToDays (d, m, y: TSInt32; var NumDays: TSInt32): Boolean;
    { Like @link(ConvertDMYToDays), just converts in the opposite
      direction. }
    function ConvertDaysToDMY (NumDays: TSInt32; var d, m, y: TSInt32): Boolean;
    { Decreases point of time stored in this object by N days. }
    procedure DecDays (n: TSInt32);
    { Decreases point of time stored in this object by N hours. }
    procedure DecHours (n: TSInt32);
    { Decreases point of time stored in this object by N minutes. }
    procedure DecMinutes (n: TSInt32);
    { Decreases point of time stored in this object by N seconds. }
    procedure DecSeconds (n: TSInt32);
    { Returns difference in days between this object and DT. }
    function GetDiffDays (var dt: TTime): TSInt32;
    { Returns difference in seconds between this object and DT. }
    function GetDiffSeconds (var dt: TTime): TSInt32;
    { Returns day of the point of time stored in this object. }
    function GetDay: TSInt32;
    { Returns day of the week of the point of time stored in this
      object; 0=Sunday, 1=Monday, ... }
    function GetDayOfTheWeek: TSInt32;
    { Returns string representation of the point of time stored in this
      object, calling @link(GetString) with @link(CurrentTimeOptions)
      as parameter. }
    function GetDefaultString: string;
    { Returns hour of the point of time stored in this object. }
    function GetHour: TSInt32;
    { Returns milliseconds of the point of time stored in this object
      (from 0 to 999). }
    function GetMilliSeconds: TSInt32;
    { Returns minute of the point of time stored in this object (from 0
      to 59). }
    function GetMinute: TSInt32;
    { Returns month of the point of time stored in this object (from 1 to
      12). }
    function GetMonth: TSInt32;
    { Returns second of the point of time stored in this object (from 0
      to 59). }
    function GetSecond: TSInt32;
    { Returns seconds passed since the beginning of the day stored in
      this object; equal to (GetHour * 3600 + GetMinute * 60 +
      GetSecond). }
    function GetSeconds: TSInt32;
    { Returns string representation of the point of time stored in this
      object, exact format taken from the Options argument. }
    function GetString (var Options: TTimeOptions): string;
    { Returns week of the year of the point of time stored in this
      object. The first week of the year must have at least four days
      in this year, otherwise a date belongs to the last week of the
      previous year. }
    function GetWeekOfTheYear: TSInt32;
    { Returns year of the point of time stored in this object (from 1600
      to ?). }
    function GetYear: TSInt32;
    { Returns if year stored in this object is a leap year. }
    function IsLeapYear: Boolean;
    { Sets current system date and time. }
    procedure SetCurrentDateTime;
    { Sets given date. }
    procedure SetDate (d, m, y: TSInt32);
    { Sets given date and time. }
    procedure SetDateTime (d, m, y, h, Min, s: TSInt32);
    { sets }
  {		procedure SetDateTimeFromString(D: String);}
    { Sets milliseconds to M. }
    procedure SetMilliSeconds (m: TSInt32);
    { Sets given time. }
    procedure SetTime (h, m, s: TSInt32);
  private
    Days,
      Day,
      Month,
      Year,
      Hour,
      Minute,
      second,
      Millis: TSInt32;
  end;

implementation

{$I platform.inc}

{$IFNDEF PPC_DELPHI}
uses
  DOS;                                  { GetDate, GetTime }
{$ENDIF}

const
  DAYS_PER_400_YEARS = 146097;
  DAYS_PER_4_YEARS  = 1461;
  DAYS_PER_WEEK     = 7;
  FIRST_YEAR        = 1600;
  HOURS_PER_DAY     = 24;
  LAST_YEAR         = 2999;
  MONTHS_PER_YEAR   = 12;
  SECONDS_PER_DAY   = 24 * 60 * 60;
  SECONDS_PER_HOUR  = 60 * 60;
  SECONDS_PER_MINUTE = 60;
  MonthLengths      : array[0..MONTHS_PER_YEAR - 1] of Byte =
    (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
  LongDOWNames      : array[0..DAYS_PER_WEEK - 1] of string[9] =
    ('Sunday', 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday', 'Saturday');
  LongMonthNames    : array[0..MONTHS_PER_YEAR - 1] of string[9] =
    ('January', 'February', 'March', 'April', 'May', 'June',
    'July', 'August', 'September', 'October', 'November', 'December');
  ShortDOWNames     : array[0..DAYS_PER_WEEK - 1] of string[3] =
    ('Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat');
  ShortMonthNames   : array[0..MONTHS_PER_YEAR - 1] of string[3] =
    ('Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec');

  { TTime }

constructor TTime.Init;
begin
  SetCurrentDateTime;
end;

function TTime.CompareTo (var dt: TTime): TSInt32;
var
  Pass              : TUInt8;
  r                 : TSInt16;
  V1                : TSInt32;
  v2                : TSInt32;
begin
  if (Days < dt.Days) then
    begin
      CompareTo := -1;
      Exit;
    end
  else if (Days > dt.Days) then
    begin
      CompareTo := 1;
      Exit;
    end;
  { both have the same day }
  for Pass := 0 to 2 do
    begin
      case Pass of
        0:
          begin
            V1 := GetHour;
            v2 := dt.GetHour;
          end;
        1:
          begin
            V1 := GetMinute;
            v2 := dt.GetMinute;
          end;
      else
        begin
          V1 := GetSecond;
          v2 := dt.GetSecond;
        end;
      end;
      r := CompareUInt16 (V1, v2);
      if (r <> 0) then
        begin
          CompareTo := r;
          Exit;
        end;
    end;
  CompareTo := 0;
end;

function TTime.ConvertDMYToDays (d, m, y: TSInt32; var NumDays: TSInt32): Boolean;
var
  l                 : Boolean;
  {	R: TSInt32;}
begin
  ConvertDMYToDays := False;
  if (d < 1) or
    (m < 1) or (m > MONTHS_PER_YEAR) or
    (y < FIRST_YEAR) or (y > LAST_YEAR) then Exit;
  if (m = 2) then
    begin
      l := ((y mod 4) = 0) and ((y mod 100) <> 0) and
        ((y mod 400) = 0);
      if (l and (d > 29)) or ((not l) and (d > 28)) then Exit;
    end
  else
    begin
      if (d > MonthLengths[m - 1]) then Exit;
    end;
  ConvertDMYToDays := True;
  if (y = FIRST_YEAR) and (m < 3) then
    begin
      if (m = 1) then
        NumDays := d - 1
      else
        NumDays := d + 30;
      Exit;
    end;
  if (m > 2) then
    Dec (m, 3)
  else
    begin
      Inc (m, 9);
      Dec (y);
    end;
  Dec (y, FIRST_YEAR);
  NumDays := ((LongInt (y div 100) * DAYS_PER_400_YEARS) div 4) +
    ((LongInt (y mod 100) * DAYS_PER_4_YEARS) div 4) +
    (((153 * m) + 2) div 5) + d + 59;
end;

function TTime.ConvertDaysToDMY (NumDays: TSInt32; var d, m, y: TSInt32): Boolean;
begin
  ConvertDaysToDMY := False;
end;

procedure TTime.DecDays (n: TSInt32);
{var
 T1: TSInt32;
 T2: TSInt32;
 T3: TSInt32;}
begin
  if (n >= Days) then
    begin
      Days := Days - n;
      { we already know the result will be valid,
        so we ignore the result of the call }
      ConvertDaysToDMY (Days, Day, Month, Year);
    end
  else
    begin
      { ERROR: Date out of range }
    end;
end;

procedure TTime.DecHours (n: TSInt32);
var
  i                 : TSInt32;
begin
  if (n > 0) then
    begin
      i := n div HOURS_PER_DAY;
      DecDays (i);
      if (n > Hour) then
        begin
          DecDays (1);
          Hour := HOURS_PER_DAY - (n - Hour);
        end
      else
        Dec (Hour, n);
    end;
end;

procedure TTime.DecMinutes (n: TSInt32);
var
  i                 : TSInt32;
begin
  if (n > 0) then
    begin
      i := n div SECONDS_PER_MINUTE;
      DecHours (i);
    end;
end;

procedure TTime.DecSeconds (n: TSInt32);
var
  i                 : TSInt32;
begin
  if (n > 0) then
    begin
      i := n div 60;
      DecMinutes (i);
      i := n mod 60;
      if (Hour = 0) and (Minute = 0) and (second < i) then
        begin
          DecDays (1);
        end;
    end;
end;

function TTime.GetDiffDays (var dt: TTime): TSInt32;
begin
  GetDiffDays := Days - dt.Days;
end;

function TTime.GetDiffSeconds (var dt: TTime): TSInt32;
var
  s1                : TSInt32;
  s2                : TSInt32;
begin
  s1 := GetSeconds;
  s2 := dt.GetSeconds;
  GetDiffSeconds := Abs (s2 - s1);
end;

function TTime.GetDay: TSInt32;
begin
  GetDay := Day;
end;

function TTime.GetDayOfTheWeek: TSInt32;
begin
  GetDayOfTheWeek := (6 + Days) mod 7;
end;

function TTime.GetDefaultString: string;
begin
  GetDefaultString := GetString (CurrentTimeOptions);
end;

function TTime.GetHour: TSInt32;
begin
  GetHour := Hour;
end;

function TTime.GetMilliSeconds: TSInt32;
begin
  GetMilliSeconds := Millis;
end;

function TTime.GetMinute: TSInt32;
begin
  GetMinute := Minute;
end;

function TTime.GetMonth: TSInt32;
begin
  GetMonth := Month;
end;

function TTime.GetSecond: TSInt32;
begin
  GetSecond := second;
end;

function TTime.GetSeconds: TSInt32;
begin
  GetSeconds := Hour * 3600 + Minute * 60 + second; { 0 .. 86399 }
end;

function TTime.GetString (var Options: TTimeOptions): string;
var
  i                 : LongInt;
  l                 : LongInt;
  s                 : string;
  t                 : string;
begin
  i := 1;
  l := Length (CurrentTimeOptions.TimeString);
  s := '';
  while (i <= l) do
    begin
      if (CurrentTimeOptions.TimeString[i] = '%') then
        begin
          if (i = l) then Continue;
          Inc (i);
          case CurrentTimeOptions.TimeString[i] of
            'D': t := IntToStringLZ (2, Day);
            'H': t := IntToStringLZ (2, Hour);
            'L': t := IntToStringLZ (2, Minute);
            'M': t := IntToStringLZ (2, Month);
            'N': t := LongMonthNames[Month - 1];
            'S': t := IntToStringLZ (2, second);
            'W': t := LongDOWNames[GetDayOfTheWeek];
            'Y': t := IntToStr (Year);
            'd': t := IntToStr (Day);
            'h': t := IntToStr (Hour);
            'l': t := IntToStr (Minute);
            'm': t := IntToStr (Month);
            'n': t := ShortMonthNames[Month - 1];
            's': t := IntToStr (second);
            'w': t := ShortDOWNames[GetDayOfTheWeek];
            'y': t := IntToStr (Year mod 100);
            '%': t := '%';
          else
            t := 'ERROR IN OPTIONS.DATETIMESTRING - UNKNOWN ESCAPE CHARACTER: ' +
              CurrentTimeOptions.TimeString[i];
          end;
          s := s + t;
        end
      else
        s := s + CurrentTimeOptions.TimeString[i];
      Inc (i);
    end;
  GetString := s;
  {	%E  'am' or 'pm'
   %Z	Time zone 'GMT', 'EST', 'PST' etc.}
end;

function TTime.GetWeekOfTheYear: TSInt32;
begin
  GetWeekOfTheYear := -1;
end;

function TTime.GetYear: TSInt32;
begin
  GetYear := Year;
end;

function TTime.IsLeapYear: Boolean;
begin
  IsLeapYear :=
    ((Year mod 4) = 0) and
    ((Year mod 100) <> 0) and
    ((Year mod 400) = 0);

  {		  LeapYear := ((Year mod 4) = 0) and
        (not (((Year mod 100)  = 0) and
        ((Year mod 400) <> 0)));}

end;

{ TODO: test Delphi part with other versions than 5.0 }

procedure TTime.SetCurrentDateTime;
var
  w                 : array[0..3] of Word;
  {$IFDEF PPC_DELPHI}
  dt                : TDateTime;
  {$ENDIF}
begin
  {$IFDEF PPC_DELPHI}
  dt := SysUtils.Date;
  DecodeDate (dt, w[0], w[1], w[2]);
  SetDate (w[2], w[1], w[0]);
  dt := SysUtils.Time;
  DecodeTime (dt, w[0], w[1], w[2], w[3]);
  SetTime (w[0], w[1], w[2]);
  SetMilliSeconds (w[3]);
  {$ELSE}
  DOS.GetDate (w[0], w[1], w[2], w[3]);
  SetDate (w[2], w[1], w[0]);
  DOS.GetTime (w[0], w[1], w[2], w[3]);
  SetTime (w[0], w[1], w[2]);
  SetMilliSeconds (w[3] * 10);
  {$ENDIF}
end;

procedure TTime.SetDate (d, m, y: TSInt32);
var
  TempDays          : TSInt32;
begin
  if ConvertDMYToDays (d, m, y, TempDays) then
    begin
      Day := d;
      Month := m;
      Year := y;
      Days := TempDays;
    end;
end;

procedure TTime.SetDateTime (d, m, y, h, Min, s: TSInt32);
begin
  SetDate (d, m, y);
  SetTime (h, Min, s);
end;
{
procedure TTime.SetDateTimeFromString(D: String);
begin
end;
}

procedure TTime.SetMilliSeconds (m: TSInt32);
begin
  Millis := Abs (m mod 1000);
end;

procedure TTime.SetTime (h, m, s: TSInt32);
begin
  if (h >= 0) and (h <= 23) then Hour := h;
  if (m >= 0) and (m <= 59) then Minute := m;
  if (s >= 0) and (s <= 59) then second := s;
end;

{ TTimeOptions }

constructor TTimeOptions.Init;
begin
  TimeString := '%w %d %n %Y %H:%L:%S';
  FirstDOW := 0;                        { Sunday }
  TimeZone := 0;
end;

begin
  CurrentTimeOptions.Init;
end.

