{
@abstract(provides dynamic-length strings)
@created(22 Feb 1999)
@lastmod(8 Feb 2000)
@author(Marco Schmidt (marcoschmidt@geocities.com))
}
unit Texts;

{$I platform.inc}

interface

uses
  Objects;

const
  { set of characters, including all letters and the underscore }
  IdentifierStart = ['A'..'Z', 'a'..'z', '_'];
  { set of characters, including all characters from @link(IdentifierStart)
    plus the ten decimal digits }
  IdentifierOther = ['A'..'Z', 'a'..'z', '_', '0'..'9', '.'];
  { set of characters, including tab, space, line feed and carriage return
    (decimal values 9, 10, 13 and 32) }
  Whitespace = [#9, #10, #13, #32];
  { set of characters, including all lowercase and uppercase letters }
  Letters = ['A'..'Z', 'a'..'z'];

type
  { array of characters, which will be overindexed }
  TCharArray = array[0..MaxInt - 1] of Char;
  { pointer to a @link(TCharArray) }
  PCharArray = ^TCharArray;
  { pointer to @link(TText) }
  PText = ^TText;
  { @abstract(simple text buffer, capable of increasing its size dynamically)
    An object to store and manipulate an array of characters, which can
    change its size dynamically.
    Can handle pasdoc tags - a tag is considered a part of a comment which
    starts with the at character at, then has one or more letters.
    The letters make up the name of the tag, then follows an opening
    bracket, the tag content and a closing bracket. }
  TText = object(TObject)
    { number of characters actually stored in @link(TText.Data) }
    Content: LongInt;
    { character buffer }
    // Data: PCharArray;
    Data: PAnsiChar;
    { current size of character buffer Data }
    Size: LongInt;
    { Initializes all fields to default values. }
    constructor Init;
    { Releases any memory still allocated to @link(Data). }
    destructor Done; virtual;
    { Appends character C to the text buffer @link(Data), which is created
      if not already existing. }
    function AppendChar(c: Char): Boolean;
    { Appends string S to the text buffer @link(Data), which is created
      if not already existing. }
    function AppendString(s: AnsiString): Boolean;
    { Removes all trailing or leading whitespace and control characters,
      or both. Also compresses all occurances of multiple whitespace
      or control characters to one single space character. }
    procedure CompressWhiteSpace;
    { Removes N characters from field @link(Data) starting at index I. }
    procedure Delete(i, n: LongInt);
    { Deletes all characters from Offs1 to Offs2 (including these two).
      Simply calls @link(Delete) with corresponding parameters.
      Note that you have to take Offs1 and Offs3 from FindTag when
      calling this procedure, not Offs1 and Offs2. }
    procedure DeleteTag(Offs1, Offs2: LongInt);
    { }
    procedure ExtractTag(Offs1, Offs2, Offs3: LongInt; var s: AnsiString);
    { Attempts to find tag with given Name, starting at offset Offs1.
      If successful, true is returned and position of opening and closing
      parenthesis in Offs2 and Offs3, false is returned otherwise. }
    function FindTag(Name: string; var Offs1, Offs2, Offs3: LongInt): Boolean;
    { }
    function FindTagParameters(var Offs1, Offs2: LongInt): Boolean;
    { Attempts to return content as string.
      Data may be missing, as strings can only hold up to 255 characters. }
    function GetString: string;
    { returns number of free characters in buffer (Size - Content) }
    function GetNumFreeChars: LongInt;
    {  }
    procedure GetTag(Remove: Boolean; Offs1, Offs2, Offs3: LongInt; var s: AnsiString);
    { Gets a tag name, starting at position Offset. If Data^[Offset] is
      not a letter, the returned tag name will be empty! }
    function GetTagName(var Offset: LongInt): string;

    function PosCI(const Search: AnsiString): Integer;
    { }
    procedure Print;
    { Attempts to increase size of character buffer to NewSize and
      returns success or failure as a boolean value. }
    function Reallocate(NewSize: LongInt): Boolean;
    { replaces all whitespace with blanks, then removes strings of
      more than one blank and replaces them with a single blank }
    // procedure RemoveObsoleteWhitespace;
  end;

implementation

uses
  Arrays;

const
  AsciiUpperChars: array[#0..#255] of Char = (
    #000, #001, #002, #003, #004, #005, #006, #007, #008, #009, #010, #011, #012, #013, #014, #015,
    #016, #017, #018, #019, #020, #021, #022, #023, #024, #025, #026, #027, #028, #029, #030, #031,
    #032, #033, #034, #035, #036, #037, #038, #039, #040, #041, #042, #043, #044, #045, #046, #047,
    #048, #049, #050, #051, #052, #053, #054, #055, #056, #057, #058, #059, #060, #061, #062, #063,
    #064, #065, #066, #067, #068, #069, #070, #071, #072, #073, #074, #075, #076, #077, #078, #079,
    #080, #081, #082, #083, #084, #085, #086, #087, #088, #089, #090, #091, #092, #093, #094, #095,
    #096, #065, #066, #067, #068, #069, #070, #071, #072, #073, #074, #075, #076, #077, #078, #079,
    #080, #081, #082, #083, #084, #085, #086, #087, #088, #089, #090, #123, #124, #125, #126, #127,
    #128, #129, #130, #131, #132, #133, #134, #135, #136, #137, #138, #139, #140, #141, #142, #143,
    #144, #145, #146, #147, #148, #149, #150, #151, #152, #153, #138, #155, #140, #157, #142, #159,
    #160, #161, #162, #163, #164, #165, #166, #167, #168, #169, #170, #171, #172, #173, #174, #175,
    #176, #177, #178, #179, #180, #181, #182, #183, #184, #185, #186, #187, #188, #189, #190, #191,
    #192, #193, #194, #195, #196, #197, #198, #199, #200, #201, #202, #203, #204, #205, #206, #207,
    #208, #209, #210, #211, #212, #213, #214, #215, #216, #217, #218, #219, #220, #221, #222, #223,
    #192, #193, #194, #195, #196, #197, #198, #199, #200, #201, #202, #203, #204, #205, #206, #207,
    #208, #209, #210, #211, #212, #213, #214, #247, #216, #217, #218, #219, #220, #221, #222, #159);

  AsciiReverseChars: array[#0..#255] of Char = (
    #000, #001, #002, #003, #004, #005, #006, #007, #008, #009, #010, #011, #012, #013, #014, #015,
    #016, #017, #018, #019, #020, #021, #022, #023, #024, #025, #026, #027, #028, #029, #030, #031,
    #032, #033, #034, #035, #036, #037, #038, #039, #040, #041, #042, #043, #044, #045, #046, #047,
    #048, #049, #050, #051, #052, #053, #054, #055, #056, #057, #058, #059, #060, #061, #062, #063,
    #064, #097, #098, #099, #100, #101, #102, #103, #104, #105, #106, #107, #108, #109, #110, #111,
    #112, #113, #114, #115, #116, #117, #118, #119, #120, #121, #122, #091, #092, #093, #094, #095,
    #096, #065, #066, #067, #068, #069, #070, #071, #072, #073, #074, #075, #076, #077, #078, #079,
    #080, #081, #082, #083, #084, #085, #086, #087, #088, #089, #090, #123, #124, #125, #126, #127,
    #128, #129, #130, #131, #132, #133, #134, #135, #136, #137, #154, #139, #156, #141, #158, #143,
    #144, #145, #146, #147, #148, #149, #150, #151, #152, #153, #138, #155, #140, #157, #142, #255,
    #160, #161, #162, #163, #164, #165, #166, #167, #168, #169, #170, #171, #172, #173, #174, #175,
    #176, #177, #178, #179, #180, #181, #182, #183, #184, #185, #186, #187, #188, #189, #190, #191,
    #224, #225, #226, #227, #228, #229, #230, #231, #232, #233, #234, #235, #236, #237, #238, #239,
    #240, #241, #242, #243, #244, #245, #246, #215, #248, #249, #250, #251, #252, #253, #254, #223,
    #192, #193, #194, #195, #196, #197, #198, #199, #200, #201, #202, #203, #204, #205, #206, #207,
    #208, #209, #210, #211, #212, #213, #214, #247, #216, #217, #218, #219, #220, #221, #222, #159);

  { TText }

constructor TText.Init;
begin
  inherited Init;
  Content := 0;
  Data := nil;
  Size := 0;
end;

destructor TText.Done;
begin
  if Assigned(Data) then FreeMem(Data, Size);
  inherited Done;
end;

function TText.AppendChar(c: Char): Boolean;
const
  Step = 128;
begin
  if Assigned(Data) then
    begin
      if (Content < Size) then
        begin
          Data[Content] := c;
          Inc(Content);
          AppendChar := True;
        end
      else
        begin
          if Reallocate(Size + Step) then
            AppendChar := AppendChar(c)
          else
            AppendChar := False;
        end;
    end
  else
    begin
      if Reallocate(Size + Step) then
        AppendChar := AppendChar(c)
      else
        AppendChar := False;
    end;
end;

function TText.AppendString(s: AnsiString): Boolean;
var
  i: LongInt;
begin
  AppendString := True;
  if (Length(s) < 1) then Exit;
  for i := 1 to Length(s) do
    if (not AppendChar(s[i])) then
      begin
        AppendString := False;
        Exit;
      end;
end;

procedure TText.CompressWhiteSpace;
var
  l: Cardinal;
  pRead, PWrite: PAnsiChar;
begin
  pRead := Data;
  if pRead = nil then Exit;
  l := Content;
  PWrite := pRead;

  { Look for first whitespace. }
  while (l > 0) and not (pRead^ in [#1..#32]) do
    begin
      Inc(pRead);
      Dec(l);
    end;

  { End of string reached? }
  if l = 0 then Exit;

  if pRead <> PWrite then
    PWrite := pRead + 1;

  Inc(pRead);
  Dec(l);

  { Skip over first Space. }
  while (l > 0) and (pRead^ in [#1..#32]) do
    begin
      Inc(pRead);
      Dec(l);
    end;

  while l > 0 do
    begin
      while (l > 0) and not (pRead^ in [#1..#32]) do
        begin
          PWrite^ := pRead^;
          Inc(PWrite);
          Inc(pRead);
          Dec(l);
        end;

      while (l > 0) and (pRead^ in [#1..#32]) do
        begin
          Inc(pRead);
          Dec(l);
        end;

      if l > 0 then
        begin
          PWrite^ := #32;
          Inc(PWrite);
        end;
    end;

  // SetLength (s, Cardinal (pWrite) - Cardinal (s));
  Content := Cardinal(PWrite) - Cardinal(Data);
end;

procedure TText.Delete(i, n: LongInt);
begin
  if (i < 0) or (i >= Content) then Exit;
  if (i + n > Content) then n := Content - i;
  System.Move(Data[i + n], Data[i], Content - (i + n));
  Dec(Content, n);
end;

procedure TText.DeleteTag(Offs1, Offs2: LongInt);
begin
  Self.Delete(Offs1, Offs2 - Offs1 + 1);
end;

procedure TText.ExtractTag(Offs1, Offs2, Offs3: LongInt; var s: AnsiString);
begin
  GetTag(True, Offs1, Offs2, Offs3, s);
end;

{
INPUT
set Offs1 to the index where you want to start the search
OUTPUT
if result is true:
  @abstract(this is the abstract)
  ^        ^                    ^
  offs1    offs2                offs3
}
function TText.FindTag(Name: string; var Offs1, Offs2, Offs3: LongInt): Boolean;
var
  i: LongInt;
  j: LongInt;
  s: string;
begin
  {  Result := False;
    if (not Assigned(Data)) or (Offs1 < 0) or (Offs1 >= Content) or (Length(Name) < 1) then Exit;

    Arrays.StringToUpper(Name, Name);
    i := Offs1;
    repeat
      if (Data[i] = '@') then
        begin
          if (i + 1 = Content) then Exit;

          if (Data[i + 1] = '@') then
            begin
              Inc(i, 2);
              Continue;
            end;

          Offs1 := i;
          j := i + 1;
          s := GetTagName(j);
          StringToUpper(s, U);
          Offs2 := j;
          if (not FindTagParameters(Offs2, Offs3)) then Exit;
          if (Name = U) then
            begin
              Result := True;
              Exit;
            end;
          i := Offs3 + 1;
        end
      else
        Inc(i);
    until (i = Content); }
  Result := False;
  if (not Assigned(Data)) or (Offs1 < 0) or (Offs1 >= Content) or (Length(Name) < 1) then Exit;

  Arrays.StringToUpper(Name, Name);
  i := Offs1;

  if (Data[i] = '@') then
    begin
      if (i + 1 = Content) then Exit;

      Offs1 := i;
      j := i + 1;
      s := GetTagName(j);

      if Name <> s then Exit;

      Offs2 := j;
      if (not FindTagParameters(Offs2, Offs3)) then Exit;
      Result := True;
    end;
end;

function TText.FindTagParameters(var Offs1, Offs2: LongInt): Boolean;
var
  Counter: LongInt;
  i: LongInt;
begin
  Result := False;
  i := Offs1;
  if (i < 0) or (i >= Content) then Exit;

  while (i < Content) and (Data[i] <> '(') do
    Inc(i);
  if (i = Content) then Exit; { no ( found }

  Offs1 := i;
  Inc(i);
  Counter := 1;
  repeat
    case Data[i] of
      '(': Inc(Counter);
      ')': Dec(Counter);
    end;
    Inc(i);
  until (i = Content) or (Counter = 0);
  if (Counter = 0) then
    begin
      Offs2 := i - 1;
      Result := True;
    end;
end;

function TText.GetNumFreeChars: LongInt;
begin
  GetNumFreeChars := (Size - Content);
end;

function TText.GetString: string;
begin
  SetString(Result, PChar(Data), Content);
end;

procedure TText.GetTag(Remove: Boolean; Offs1, Offs2, Offs3: LongInt; var s: AnsiString);
var
  l: LongInt;
begin
  l := (Offs3 - Offs2 - 1);
  if (l < 0) then l := 0;
  // if (l > 255) then l := 255;
  // System.Move(Data[Offs2 + 1], s[1], l);
  // SetLength(s, l);
  SetString(s, Data + Offs2 + 1, l);
  if Remove then
    Self.Delete(Offs1, Offs3 - Offs1 + 1);
end;

function TText.GetTagName(var Offset: LongInt): string;
var
  s: string;
begin
  s := '';
  while (Offset < Content) and
    (((Data[Offset] >= 'A') and (Data[Offset] <= 'Z')) or
    ((Data[Offset] >= 'a') and (Data[Offset] <= 'z'))) do
    begin
      s := s + UpCase(Data[Offset]);
      Inc(Offset);
    end;
  Result := s;
end;

function TText.PosCI(const Search: AnsiString): Integer;
label
  Zero, One, Two, Three, Match, Fail, Success;
var
  pSearch, pSearchTemp, PSource, PSourceTemp: PAnsiChar;
  lSearch, lSearchTemp, lSource: Cardinal;
  c: AnsiChar;
begin
  pSearch := Pointer(Search);
  if pSearch = nil then goto Fail;

  PSource := Data;
  if PSource = nil then goto Fail;

  lSearch := Cardinal(Pointer(pSearch - 4)^);
  lSource := Content;

  if lSearch > lSource then goto Fail;

  Dec(lSearch);
  Dec(lSource, lSearch);

  c := AsciiUpperChars[pSearch^];
  Inc(pSearch);

  while lSource > 0 do
    begin

      while lSource >= 4 do
        begin
          if (AsciiUpperChars[PSource^] = c) then goto Zero;
          if (AsciiUpperChars[PSource[1]] = c) then goto One;
          if (AsciiUpperChars[PSource[2]] = c) then goto Two;
          if (AsciiUpperChars[PSource[3]] = c) then goto Three;
          Inc(PSource, 4);
          Dec(lSource, 4);
        end;

      case lSource of
        3:
          begin
            if (AsciiUpperChars[PSource^] = c) then goto Zero;
            if (AsciiUpperChars[PSource[1]] = c) then goto One;
            if (AsciiUpperChars[PSource[2]] = c) then goto Two;
          end;
        2:
          begin
            if (AsciiUpperChars[PSource^] = c) then goto Zero;
            if (AsciiUpperChars[PSource[1]] = c) then goto One;
          end;
        1:
          begin
            if (AsciiUpperChars[PSource^] = c) then goto Zero;
          end;
      end;

      Break;

      Three:
      Inc(PSource, 4); { Already Inc (pSource) here. }
      Dec(lSource, 3);
      goto Match;

      Two:
      Inc(PSource, 3); { Already Inc (pSource) here. }
      Dec(lSource, 2);
      goto Match;

      One:
      Inc(PSource, 2); { Already Inc (pSource) here. }
      Dec(lSource, 1);
      goto Match;

      Zero:
      Inc(PSource); { Already Inc (pSource) here. }

      Match:

      { The first character already matches. }
      PSourceTemp := PSource;
      pSearchTemp := pSearch;
      lSearchTemp := lSearch;

      while (lSearchTemp >= 4) and
        ((PSourceTemp^ = pSearchTemp^) or (PSourceTemp^ = AsciiReverseChars[pSearchTemp^])) and
        ((PSourceTemp[1] = pSearchTemp[1]) or (PSourceTemp[1] = AsciiReverseChars[pSearchTemp[1]])) and
        ((PSourceTemp[2] = pSearchTemp[2]) or (PSourceTemp[2] = AsciiReverseChars[pSearchTemp[2]])) and
        ((PSourceTemp[3] = pSearchTemp[3]) or (PSourceTemp[3] = AsciiReverseChars[pSearchTemp[3]])) do
        begin
          Inc(PSourceTemp, 4);
          Inc(pSearchTemp, 4);
          Dec(lSearchTemp, 4);
        end;
      if (lSearchTemp = 0) then
        goto Success;
      if (lSearchTemp = 1) and
        ((PSourceTemp^ = pSearchTemp^) or (PSourceTemp^ = AsciiReverseChars[pSearchTemp^])) then
        goto Success;
      if (lSearchTemp = 2) and
        ((PSourceTemp^ = pSearchTemp^) or (PSourceTemp^ = AsciiReverseChars[pSearchTemp^])) and
        ((PSourceTemp[1] = pSearchTemp[1]) or (PSourceTemp[1] = AsciiReverseChars[pSearchTemp[1]])) then
        goto Success;
      if (lSearchTemp = 3) and
        ((PSourceTemp^ = pSearchTemp^) or (PSourceTemp^ = AsciiReverseChars[pSearchTemp^])) and
        ((PSourceTemp[1] = pSearchTemp[1]) or (PSourceTemp[1] = AsciiReverseChars[pSearchTemp[1]])) and
        ((PSourceTemp[2] = pSearchTemp[2]) or (PSourceTemp[2] = AsciiReverseChars[pSearchTemp[2]])) then
        goto Success;

      { No match this time around: Adjust source length. Source itself is already adjusted. }
      Dec(lSource);
    end;

  Fail:
  Result := -1;
  Exit;

  Success:
  Result := Cardinal(PSource) - Cardinal(Data);
end; // Warning can be ignored.

procedure TText.Print;
var
  i: LongInt;
begin
  i := 0;
  Write('"');
  while (i < Content) do
    begin
      Write(Data[i]);
      Inc(i);
    end;
  WriteLn('"');
end;

function TText.Reallocate(NewSize: LongInt): Boolean;
{var
  Temp              : PCharArray;}
begin
  if NewSize > Size then
    begin
      ReallocMem(Data, NewSize);
      Size := NewSize;
    end;
  Result := True;
  { if (NewSize <= Size) then
     begin
       Result := True;
       Exit;
     end;
   //WriteLn('before getmem, newsize=', newsize, ', size=', size);
   GetMem (Temp, NewSize);
   //WriteLn('after getmem');
   if Assigned (Temp) then
     begin
       System.Move (Data^, Temp^, Content);
       FreeMem (Data, Size);
       Data := Temp;
       Size := NewSize;
       Result := True;
     end
   else
     Result := False; }
end;

{
procedure TText.RemoveObsoleteWhitespace;
var
  i                 : LongInt;
  j                 : LongInt;
begin
  if (not Assigned (Data)) then Exit;
  i := 0;
  while (i < Content - 1) do
    begin
      if (Data[i] in Whitespace) then
        begin
          j := 0;
          while (i + j + 1 < Content) and (Data[i + j + 1] in Whitespace) do
            Inc (j);
          if (j > 0) then Self.Delete (i, j);
          Data[i] := ' ';
        end;
      Inc (i);
    end;
  if (Content > 0) and (Data[0] in Whitespace) then Self.Delete (0, 1);
  if (Content > 0) and (Data[Content - 1] in Whitespace) then Self.Delete (Content - 1, 1);
end;
}
end.

