{
@abstract(provides basic classes for input and output streams)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@created(7 May 1998)
@lastmod(14 Aug 1999)

Text files to be read by TInputStream should stick to one of the usual
line ending formats: DOS (13 10), Unix (10) or Mac (13). Anything else,
like (13 13 10) will make counting lines impossible (though this sort
of formatting is used in some word processors).

When reading / writing raw data, function return values tell about how much
data has actually been read / written. If there were errors, -1 will be
returned. New derived stream types must keep this system.
}
unit Streams;

{$I platform.inc}

interface

uses
  Numbers;

type
  { integer type for stream offsets / lengths; must be signed so that we can return -1 on errors }
  TStreamOffset = TSInt32;
  { supported text line types }
  TTextType = (TEXT_TYPE_DOS, TEXT_TYPE_UNIX, TEXT_TYPE_MAC);
  { basic class for input and output streams }
  TStream = object
    { is true if end of stream has been reached, false otherwise }
    EndOfStream: Boolean;
    { Initializes all fields }
    constructor Init;
    { First calls @link(Close), then @link(ReleaseBuffer). }
    destructor Done; virtual;
    { Does nothing - heirs will override this procedure and actually implement some functionality. }
    procedure Close; virtual;
    { Returns if an error has occurred. }
    function Error: Boolean;
    { Returns the current size of the buffer. }
    function GetBufferSize: TUInt32;
    { Returns the current byte order. }
    function GetByteOrder: TByteOrder;
    { Returns the current line number. This will only make sense if
      the stream is a text stream which is read / written by
      @link(TInputStream.ReadLine) / @link(TOutputStream.WriteLine). }
    function GetLineNumber: TUInt32;
    function GetName: string;
    { Returns current position in stream or -1 if there were errors. }
    function GetPos: TStreamOffset;
    { Returns current size of stream or -1 if there were errors. }
    function GetSize: TStreamOffset;
    { Returns current text type, i.e. the sort of lines we are reading /
      writing. Supported are Dos, Unix and Mac types. }
    function GetTextType: TTextType;
    { Returns true if we have a buffer allocated. }
    function IsBuffered: Boolean;
    function IsOpen: Boolean;
    { Releases the buffer if one has been allocated. }
    procedure ReleaseBuffer;
    { Resets the size of the buffer. All buffer content will be lost. }
    procedure SetBufferSize(NewBufferSize: TSInt32);
    { Sets a new byte order. }
    procedure SetByteOrder(NewByteOrder: TByteOrder);
    procedure SetName(n: string);
    procedure SetOpenStatus(IsNowOpen: Boolean);
    { Sets current position in file to argument value. }
    procedure SetPos(const NewPosition: TStreamOffset);
    { Sets a new text type. }
    procedure SetTextType(NewTextType: TTextType);
  private
    { array of bytes used to speed up read / write access }
    Buffer: PByteArray;
    { number of bytes currently in buffer }
    BufferContent: TSInt32;
    { }
    BufferIndex: TSInt32;
    { number of bytes allocated to @link(Buffer) }
    BufferSize: TSInt32;

    ByteOrder: TByteOrder;
    { }
    ErrorCode: TUInt32;
    {}
    HasBeenOpened: Boolean;
    { current line number, can be determined via @link(GetLineNumber) }
    LineNumber: TUInt32;
    MyName: string;
    { current text type, can be accessed via @link(GetTextType) and @link(SetTextType) }
    TextType: TTextType;
  end;
  { pointer to @link(TInputStream) }
  PInputStream = ^TInputStream;
  { basic input stream class }
  TInputStream = object(TStream)
    function CheckBytes(n: TStreamOffset; var Data): Boolean;
    function EOS: Boolean; { end of stream }
    procedure FillBuffer;
    function HasData: Boolean; virtual;
    function PeekByte: TSInt32;
    function ReadByte: TSInt32;
    function ReadBytes(n: TStreamOffset; var Dest): TStreamOffset;
    procedure ReadChar(var c: Char);
    procedure ReadLine(var s: string);
    function ReadRawBytes(n: TStreamOffset; var Dest): TStreamOffset; virtual;
    procedure ReadSInt8(var Value: TSInt8);
    procedure ReadSInt16(var Value: TSInt16);
    procedure ReadSInt32(var Value: TSInt32);
    procedure ReadUInt8(var Value: TUInt8);
    procedure ReadUInt16(var Value: TUInt16);
    procedure ReadUInt32(var Value: TUInt32);
    procedure Skip(n: TStreamOffset);
  end;
  { pointer to @link(TOutputStream) }
  POutputStream = ^TOutputStream;
  { basic output stream class }
  TOutputStream = object(TStream)
    procedure Close; virtual;
    procedure Flush;
    procedure Truncate; virtual;
    procedure WriteByte(var Value: Byte);
    function WriteBytes(n: TStreamOffset; const Source): TStreamOffset;
    procedure WriteChar(const c: Char);
    procedure WriteLine(const s: string);
    function WriteRawBytes(n: TStreamOffset; var Source): TStreamOffset; virtual;
    procedure WriteSInt8(var Value: TSInt8);
    procedure WriteSInt16(var Value: TSInt16);
    procedure WriteSInt32(var Value: TSInt32);
    procedure WriteString(const s: string);
    procedure WriteUInt8(var Value: TUInt8);
    procedure WriteUInt16(var Value: TUInt16);
    procedure WriteUInt32(var Value: TUInt32);
  end;

const
  { buffer size to be used if not explicitly set by caller via @link(TStream.SetBufferSize) }
  DEFAULT_BUFFER_SIZE: TUInt32 = 4096; { 4 KB }
  { default text type to be used, platform-dependent }
  DEFAULT_TEXT_TYPE: TTextType =
  {$IFDEF OS_LINUX}
  TEXT_TYPE_UNIX;
  {$ELSE}
  {$IFDEF OS_MAC}
  TEXT_TYPE_MAC;
  {$ELSE}
  TEXT_TYPE_DOS;
  {$ENDIF}
  {$ENDIF}
  { minimum possible buffer size in combination with @link(TStream.SetBufferSize) }
  MINIMUM_BUFFER_SIZE: TSInt32 = 16;

implementation

uses
  Arrays;

const
  { strings representing line endings for all text types }
  TLineEndings: array[TEXT_TYPE_DOS..TEXT_TYPE_MAC] of string[2] =
  (#13#10, #10, #13);

  { TStream }

constructor TStream.Init;
begin
  { TO DO: change the following calls according to home platform!  }
  SetByteOrder(BYTE_ORDER_INTEL);
  SetTextType(DEFAULT_TEXT_TYPE);
end;

destructor TStream.Done;
begin
  Close;
  ReleaseBuffer;
end;

procedure TStream.Close;
begin
  HasBeenOpened := False;
end;

function TStream.Error: Boolean;
begin
  Error := (ErrorCode <> 0);
end;

function TStream.GetBufferSize: TUInt32;
begin
  GetBufferSize := BufferSize;
end;

function TStream.GetByteOrder: TByteOrder;
begin
  GetByteOrder := ByteOrder;
end;

function TStream.GetLineNumber: TUInt32;
begin
  GetLineNumber := LineNumber;
end;

function TStream.GetName: string;
begin
  GetName := MyName;
end;

function TStream.GetPos: TStreamOffset;
begin
  { if descendant stream type allows it, overwrite this! }
  GetPos := -1; { meaning: we don't know }
end;

function TStream.GetSize: TStreamOffset;
begin
  { if descendant stream type allows it, overwrite this! }
  GetSize := -1; { meaning: we don't know }
end;

function TStream.GetTextType: TTextType;
begin
  GetTextType := TextType;
end;

function TStream.IsBuffered: Boolean;
begin
  IsBuffered := Assigned(Buffer);
end;

function TStream.IsOpen: Boolean;
begin
  IsOpen := HasBeenOpened;
end;

procedure TStream.ReleaseBuffer;
begin
  FreeMem(Buffer, BufferSize);
  Buffer := nil;
  BufferContent := 0;
  BufferIndex := 0;
  BufferSize := 0;
end;

procedure TStream.SetBufferSize(NewBufferSize: TSInt32);
begin
  if (NewBufferSize < MINIMUM_BUFFER_SIZE) or (NewBufferSize = BufferSize) then Exit;
  ReleaseBuffer;

  GetMem(Buffer, NewBufferSize);
  BufferSize := NewBufferSize;
end;

procedure TStream.SetByteOrder(NewByteOrder: TByteOrder);
begin
  ByteOrder := NewByteOrder;
end;

procedure TStream.SetName(n: string);
begin
  MyName := n;
end;

procedure TStream.SetOpenStatus(IsNowOpen: Boolean);
begin
  HasBeenOpened := IsNowOpen;
end;

procedure TStream.SetPos(const NewPosition: TStreamOffset);
begin
  { abstract }
end;

procedure TStream.SetTextType(NewTextType: TTextType);
begin
  TextType := NewTextType;
end;

{ TInputStream }

function TInputStream.CheckBytes(n: TStreamOffset; var Data): Boolean;
var
  i: TStreamOffset;
  r: TSInt32;
begin
  i := 0;
  while (i <> n) do
    begin
      r := ReadByte;
      if (r = -1) or (r <> TByteArray(Data)[i]) then
        begin
          CheckBytes := False;
          Exit;
        end;
      Inc(i);
    end;
  CheckBytes := True;
end;

function TInputStream.EOS: Boolean;
begin
  EOS := EndOfStream;
end;

procedure TInputStream.FillBuffer;
begin
  if (not Assigned(Buffer)) then
    begin
      SetBufferSize(DEFAULT_BUFFER_SIZE);
      if (not Assigned(Buffer)) then Exit;
    end;
  BufferContent := ReadRawBytes(BufferSize, Buffer^);
  BufferIndex := 0;
end;

function TInputStream.HasData: Boolean;
begin
  HasData := (BufferIndex < BufferContent);
end;

function TInputStream.PeekByte: TSInt32;
begin
  if (BufferIndex >= BufferContent) then
    begin
      FillBuffer;
      if (BufferContent <= 0) then
        PeekByte := -1
      else
        begin
          PeekByte := Buffer^[0];
          BufferIndex := 0;
        end;
    end
  else
    begin
      PeekByte := Buffer^[BufferIndex];
    end;
end;

{ same functionality as ReadUInt8; but this one is faster, as it does not
  use ReadBytes (as ReadUInt8 does) }
function TInputStream.ReadByte: TSInt32;
begin
  if (BufferIndex >= BufferContent) then
    begin
      FillBuffer;
      if (BufferContent <= 0) then
        ReadByte := -1
      else
        begin
          ReadByte := Buffer^[0];
          BufferIndex := 1;
        end;
    end
  else
    begin
      ReadByte := Buffer^[BufferIndex];
      Inc(BufferIndex);
    end;
end;

function TInputStream.ReadBytes(n: TStreamOffset; var Dest): TStreamOffset;
var
  i: TStreamOffset;
  b: TStreamOffset;
  SaveN: TStreamOffset;
begin
  SaveN := n;
  i := 0; { index into TByteArray(Dest) }
  while (n > 0) do
    begin
      if (BufferIndex >= BufferContent) then
        begin
          FillBuffer;
          if (BufferContent <= 0) then
            begin
              ReadBytes := -1;
              Exit;
            end;
        end
      else
        begin
          { compute number of bytes left in buffer }
          b := BufferContent - BufferIndex;
          { if less byte to get than in buffer, set B to N }
          if (n < b) then b := n;
          { copy from buffer to destination }
          System.Move(Buffer^[BufferIndex], TByteArray(Dest)[i], b);
          { }
          Inc(BufferIndex, b);
          Dec(n, b);
          Inc(i, b);
        end;
    end;
  ReadBytes := SaveN;
end;

{
Reads a single character from input, regarding SizeOf(Char).
Does not deal with byte order in case SizeOf(Char) > 2.
}
procedure TInputStream.ReadChar(var c: Char);
begin
  ReadBytes(SizeOf(Char), c);
end;

{
Reads a text line from input (a complete line is read in, whether it fits
into S or not - all addtional characters will be lost.
However, it is checked how many characters S can hold, so no memory
behind S is overwritten.
Relies on Char being one byte large.
}
procedure TInputStream.ReadLine(var s: string);
var
  c: Char;
  i: TUInt32;
  StringSize: TUInt32;
begin
  i := 0;
  StringSize := SizeOf(s) - 1;
  repeat
    ReadChar(c);
    if (i < StringSize) then
      begin
        Inc(i);
        s[i] := c;
      end
    else
      begin
        Inc(i);
      end;
  until (c = #10) or (c = #13) or EOS;

  if (c = #13) and (PeekByte = 10) then
    ReadChar(c);

  SetLength(s, i - 1);
  { S[0] := Chr(I - 1); }
  Inc(LineNumber);
end;

function TInputStream.ReadRawBytes(n: TStreamOffset; var Dest): TStreamOffset;
begin
  ReadRawBytes := -1;
end;

procedure TInputStream.ReadSInt8(var Value: TSInt8);
begin
  ReadBytes(SizeOf(TSInt8), Value);
end;

procedure TInputStream.ReadSInt16(var Value: TSInt16);
var
  RawData: array[0..1] of Byte;
begin
  ReadBytes(2, RawData);
  GetSInt16(ByteOrder, RawData, Value);
end;

procedure TInputStream.ReadSInt32(var Value: TSInt32);
var
  RawData: array[0..3] of Byte;
begin
  ReadBytes(4, RawData);
  GetSInt32(ByteOrder, RawData, Value);
end;

procedure TInputStream.ReadUInt8(var Value: TUInt8);
begin
  ReadBytes(SizeOf(TUInt8), Value);
end;

procedure TInputStream.ReadUInt16(var Value: TUInt16);
var
  RawData: array[0..1] of Byte;
begin
  ReadBytes(2, RawData);
  GetUInt16(ByteOrder, RawData, Value);
end;

procedure TInputStream.ReadUInt32(var Value: TUInt32);
var
  RawData: array[0..3] of Byte;
begin
  ReadBytes(4, RawData);
  GetUInt32(ByteOrder, RawData, Value);
end;

procedure TInputStream.Skip(n: TStreamOffset);
begin
  while (n > 0) do
    begin
      ReadByte;
      Dec(n);
    end;
end;

{ TOutputStream }

procedure TOutputStream.Close;
begin
  inherited Close;
  Flush;
end;

procedure TOutputStream.Flush;
begin
  if Assigned(Buffer) and (BufferContent > 0) then
    begin
      if (WriteRawBytes(BufferContent, Buffer^) <> BufferContent) then ;
      BufferContent := 0;
    end;
end;

procedure TOutputStream.Truncate;
begin
  { abstract }
end;

procedure TOutputStream.WriteByte(var Value: Byte);
label
  1;
begin
  if Assigned(Buffer) then
    begin
      1:
      Buffer^[BufferContent] := Value;
      Inc(BufferContent);
      if (BufferContent = BufferSize) then Flush;
    end
  else
    begin
      SetBufferSize(DEFAULT_BUFFER_SIZE);
      goto 1; // if Assigned(Buffer) then WriteByte(Value);
    end;
end;

function TOutputStream.WriteBytes(n: TStreamOffset; const Source): TStreamOffset;
var
  Bytes: TStreamOffset;
  FreeBytes: TStreamOffset;
  i: TStreamOffset;
  SaveN: TStreamOffset;
begin
  i := 0;
  SaveN := n;
  while (n > 0) do
    begin
      if Assigned(Buffer) then
        begin
          FreeBytes := BufferSize - BufferContent;
          if (FreeBytes > n) then
            Bytes := n
          else
            Bytes := FreeBytes;
          System.Move(TByteArray(Source)[i], Buffer^[BufferContent], Bytes);
          Inc(BufferContent, Bytes);
          Inc(i, Bytes);
          Dec(n, Bytes);
          if (BufferContent = BufferSize) then Flush;
        end
      else
        begin
          SetBufferSize(DEFAULT_BUFFER_SIZE);
          if (not Assigned(Buffer)) then
            begin
              Result := -1;
              Exit;
            end;
        end;
    end;
  Result := SaveN;
end;

procedure TOutputStream.WriteChar(const c: Char);
begin
  WriteBytes(SizeOf(Char), c);
end;

procedure TOutputStream.WriteLine(const s: string);
var
  l: Integer;
begin
  l := Length(s);
  if l > 0 then
    WriteBytes(l * SizeOf(Char), s[1]);
  WriteBytes(Length(TLineEndings[TextType]), TLineEndings[TextType][1]);
end;

function TOutputStream.WriteRawBytes(n: TStreamOffset; var Source): TStreamOffset;
begin
  WriteRawBytes := -1;
end;

procedure TOutputStream.WriteSInt8(var Value: TSInt8);
begin
  WriteByte(Byte(Value));
end;

procedure TOutputStream.WriteSInt16(var Value: TSInt16);
var
  i: TSInt16;
begin
  SetSInt16(ByteOrder, Value, i);
  WriteBytes(2, i);
end;

procedure TOutputStream.WriteSInt32(var Value: TSInt32);
var
  i: TSInt32;
begin
  SetSInt32(ByteOrder, Value, i);
  WriteBytes(4, i);
end;

procedure TOutputStream.WriteString(const s: string);
var
  i: Word;
begin
  if (Length(s) = 0) then Exit;
  for i := 1 to Length(s) do
    WriteChar(s[i]);
end;

procedure TOutputStream.WriteUInt8(var Value: TUInt8);
begin
  WriteByte(Value);
end;

procedure TOutputStream.WriteUInt16(var Value: TUInt16);
var
  i: TUInt16;
begin
  SetUInt16(ByteOrder, Value, i);
  WriteBytes(2, i);
end;

procedure TOutputStream.WriteUInt32(var Value: TUInt32);
var
  i: TUInt32;
begin
  SetUInt32(ByteOrder, Value, i);
  WriteBytes(4, i);
end;

begin
end.

