{
@abstract(provides all the parsing functionality of pasdoc)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@lastmod(13 Apr 2000)

Parsing implements most of the functionality of the pasdoc program.

It provides the @link(TParser) object, which scans the command line parameters
for file names and switches and then starts collecting information from those
files, issueing warnings to standard out if necessary.
}

unit Parsing;

interface

{$I platform.inc}

uses
  Items,
  Objects,
  Scanning,
  Streams,
  Texts,
  Tokenizi;

const
  { indicates that method is a constructor }
  METHOD_CONSTRUCTOR = 0;
  { indicates that method is a destructor }
  METHOD_DESTRUCTOR = 1;
  { indicates that method is a function or procedure and not a constructor
    or destructor }
  METHOD_FUNCTION_PROCEDURE = 2;

type
  { pointer to @link(TParser) }
  PParser = ^TParser;
  { Parser object that will process a complete unit file and all of its
    include files, regarding directives.
    Will normally be created used @link(Init) with an input stream
    and a list of directives, then does the parsing work when its
    @link(ParseUnit) method is called.
    If no errors appear, should return a @link(PUnit) object with
    all information on the unit.
    Otherwise a description of the error should be found in
    @link(ErrorMessage). }
  TParser = object(TObject)
    { Stores description of error whenever something went wrong. }
    ErrorMessage: string;
    { Last comment found in input or nil if no comment available.
    Will be modified by @link(GetLastComment). }
    LastCommentToken: PToken;
    { The underlying scanner object. }
    Scanner: PScanner;
    { Create a parser, initialize the scanner with input stream S.
      All strings in SD are defined compiler directives. }
    constructor Init(InputStream: PInputStream; SD: PStringCollection; ifp: PCollection);
    { Release all dynamically allocated memory. }
    destructor Done; virtual;
    { Will write last comment that was found in input to T. If there was none,
      T will be set to nil. }
    procedure GetLastComment(var t: PText);
    { Get next token T from scanner that is neither whitespace nor comment.
      Return true on success. }
    function GetNextNonWCToken(var t: PToken): Boolean;
    function ParseArguments(var a: string): Boolean;
    { Parses a constructor, a destructor, a function or a procedure.
      Resulting PMethod item will be returned in M.
      CS may contain the 'class' keyword - its exact spelling is taken from
      this variable.
      CDFP contains the keyword constructor, destructor, function or procedure
      in the exact spelling as it was found in input.
      Key contains one of the KEY_xxx constants for the What field of the
      resulting method object.
      D may contain a description or nil. }
    function ParseCDFP(var m: PMethod; CS, CDFPS: string; Key: Integer; d: PText): Boolean;
    { Parses a class, an interface or an object.
      U is the unit this item will be added to on success.
      N is the name of this item.
      CIOType describes if item is class, interface or object.
      D may contain a description or nil. }
    function ParseCIO(var U: PUnit; n: string; CIOType: TCIOType; d: PText): Boolean;
    { }
    function ParseConstant(var U: PUnit; t: PToken): Boolean;
    function ParseInterfaceSection(var U: PUnit): Boolean;
    function ParseProperty(var p: PProperty): Boolean;
    function ParseType(var U: PUnit; var t: PToken): Boolean;
    function ParseUnit(var U: PUnit): Boolean;
    function ParseUses(var U: PUnit): Boolean;
    function ParseVariables(var U: PUnit; var t: PToken): Boolean;
    function SkipDeclaration: Boolean;
    { Reads tokens and throws them away as long as they are either whitespace
      or comments.
      Returns true on success, false if there were any errors. }
    function SkipWhitespaceAndComments: Boolean;
  end;

implementation

uses
  Arrays,
  Msg,
  Numbers;

{ TParser }

constructor TParser.Init(InputStream: PInputStream; SD: PStringCollection; ifp: PCollection);
begin
  inherited Init;
  LastCommentToken := nil;
  Scanner := New(PScanner, Init(InputStream));
  Scanner^.AddDirectives(SD);
  Scanner^.IncludeFilePaths := ifp;
end;

destructor TParser.Done;
begin
  if Assigned(Scanner) then Dispose(Scanner, Done);
  inherited Done;
end;

procedure TParser.GetLastComment(var t: PText);
begin
  if Assigned(LastCommentToken) then
    begin
      t := LastCommentToken^.Data;
      LastCommentToken^.Data := nil;
      Dispose(LastCommentToken, Done);
      LastCommentToken := nil;
      { remove comment characters here }
      if (t^.Content >= 1) and (t^.Data[0] = '{') then t^.Delete(0, 1);
      if (t^.Content >= 1) and (t^.Data[t^.Content - 1] = '}') then t^.Delete(t^.Content - 1, 1);
      if (t^.Content >= 2) and (t^.Data[0] = '(') and (t^.Data[1] = '*') then t^.Delete(0, 2);
      if (t^.Content >= 2) and (t^.Data[0] = '/') and (t^.Data[1] = '/') then t^.Delete(0, 2);
      if (t^.Content >= 2) and (t^.Data[t^.Content - 2] = '*') and
        (t^.Data[t^.Content - 1] = ')') then t^.Delete(t^.Content - 2, 2);
    end
  else
    t := nil;
end;

function TParser.GetNextNonWCToken(var t: PToken): Boolean;
begin
  Result := False;
  if (not SkipWhitespaceAndComments) then Exit;
  Result := Scanner^.GetToken(t);
end;

function TParser.ParseArguments(var a: string): Boolean;
var
  Finished: Boolean;
  t: PToken;
begin
  ParseArguments := False;
  Finished := False;
  a := '';
  repeat
    if (not Scanner^.GetToken(t)) then Exit;
    if (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_RIGHT_PARENTHESIS) then
      Finished := True
    else if (t^.MyType = TOK_SYMBOL) and
      ((t^.Info.SymbolType = SYM_COLON) or
      (t^.Info.SymbolType = SYM_COMMA) or
      (t^.Info.SymbolType = SYM_SEMICOLON)) then
      begin
        if (Length(a) > 0) and (a[Length(a)] = ' ') then SetLength(a, Length(a) - 1);
        a := a + t^.Data^.GetString;
      end
    else if (t^.MyType = TOK_WHITESPACE) then
      begin
        if (Length(a) > 0) and (a[Length(a)] <> ' ') then a := a + ' ';
      end
    else if (t^.MyType = TOK_COMMENT) or (t^.MyType = TOK_DIRECTIVE) then
      begin
        { ignore }
      end
    else { otherwise copy }
      a := a + t^.Data^.GetString;
    Dispose(t, Done);
  until Finished;
  ParseArguments := True;
end;

function TParser.ParseCDFP(var m: PMethod; CS, CDFPS: string; Key: Integer; d: PText): Boolean;
const
  SDSet: set of Byte = [SD_ABSTRACT, SD_ASSEMBLER, SD_CDECL, SD_DYNAMIC, SD_EXPORT,
  SD_EXTERNAL, SD_FAR, SD_FORWARD, SD_NEAR, SD_OVERLOAD, SD_OVERRIDE,
    SD_STDCALL, SD_REINTRODUCE, SD_VIRTUAL];
var
  Finished: Boolean;
  IsSemicolon: Boolean;
  pl: Integer;
  t: PToken;
begin
  Result := False;
  m := New(PMethod, Init);
  if (not Assigned(m)) then
    begin
      ErrorMessage := 'Error, could not create CDFP item.';
      Exit;
    end;
  m^.Description := d;
  case Key of
    KEY_CONSTRUCTOR:
      m^.What := METHOD_CONSTRUCTOR;
    KEY_DESTRUCTOR:
      m^.What := METHOD_DESTRUCTOR;
    KEY_FUNCTION, KEY_PROCEDURE:
      m^.What := METHOD_FUNCTION_PROCEDURE;
  else
    begin
      PrintLn(1, 'FATAL ERROR: CDFP got invalid key.');
      Halt(1);
    end;
  end;
  { next non-wc token must be the name }
  if (not GetNextNonWCToken(t)) then
    begin
      Dispose(m, Done);
      Exit;
    end;
  if (t^.MyType <> TOK_IDENTIFIER) then
    begin
      Dispose(m, Done);
      Dispose(t, Done);
      Exit;
    end;
  if (Length(CS) > 0) then CS := CS + ' ';
  m^.Name := t^.Data^.GetString;
  PrintLn(5, 'Parsing ' + CDFPS + ' ' + m^.Name);
  m^.FullDeclaration := CS + CDFPS + ' ' + m^.Name;
  Dispose(t, Done);
  { copy tokens until first semicolon with parenthesis level zero }
  pl := 0;
  repeat
    if (not Scanner^.GetToken(t)) then
      begin
        Dispose(m, Done);
        Exit;
      end;
    if (t^.MyType = TOK_COMMENT) then
    else if (t^.MyType = TOK_WHITESPACE) then
      begin
        if (m^.FullDeclaration[Length(m^.FullDeclaration)] <> ' ') then m^.FullDeclaration := m^.FullDeclaration + ' ';
      end
    else
      begin
        m^.FullDeclaration := m^.FullDeclaration + t^.Data^.GetString;
      end;
    if (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_LEFT_PARENTHESIS) then Inc(pl);
    if (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_RIGHT_PARENTHESIS) then Dec(pl);
    IsSemicolon := (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_SEMICOLON);
    Dispose(t, Done);
  until IsSemicolon and (pl = 0);

  { first get non-WC token - if it is not an identifier in SD_SET put it back
    into stream and leave; otherwise copy tokens until semicolon }
  Finished := False;
  repeat
    if (not GetNextNonWCToken(t)) then
      begin
        Dispose(m, Done);
        Exit;
      end;
    if (t^.MyType <> TOK_IDENTIFIER) then
      begin
        Scanner^.UnGetToken(t);
        Break;
      end;
    CS := t^.Data^.GetString;
    StringToUpper(CS, CS);
    pl := StandardDirectives^.IndexOf(@CS);
    if (pl < 0) or (not (pl in SDSet)) then
      begin
        Scanner^.UnGetToken(t);
        Break;
      end;
    m^.FullDeclaration := m^.FullDeclaration + ' ' + t^.Data^.GetString;
    Dispose(t, Done);

    { Apparently, the Delphi compiler does NOT enforce that
      directives must be separated and be terminated by a semicolon,
      even though Delphi help consistently uses them consistently.
      However, we take the compiler as a reference and try to mimic its behaviour. }
    if (not GetNextNonWCToken(t)) then
      begin
        Dispose(m, Done);
        Exit;
      end;
    { Is current token a semicolon? }
    if (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_SEMICOLON) then
      m^.FullDeclaration := m^.FullDeclaration + ';'
    else
      begin
        m^.FullDeclaration := m^.FullDeclaration + ' ';
        Scanner^.UnGetToken(t);
      end;

    {repeat
      if (not GetNextNonWCToken(t)) then
        begin
          Dispose(m, Done);
          Exit;
        end;
      IsSemicolon := (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_SEMICOLON);
      if (not IsSemicolon) then m^.FullDeclaration := m^.FullDeclaration + ' ';
      m^.FullDeclaration := m^.FullDeclaration + t^.Data^.GetString;
    until IsSemicolon;}
  until Finished;
  Result := True;
end;

function TParser.ParseCIO(var U: PUnit; n: string; CIOType: TCIOType;
  d: PText): Boolean;
var
  CS: string;
  CSFound: Boolean;
  f: PItem;
  Finished: Boolean;
  i: PCIO;
  Ind: Integer;
  m: PMethod;
  p: PProperty;
  s: string;
  State: Byte;
  t: PToken;
begin
  Result := False;
  PrintLn(5, 'Parsing class/interface/object "' + n + '"');
  if (not GetNextNonWCToken(t)) then Exit;

  { Test for forward class definition here:
      class MyClass = class;
    with no ancestor or class members listed after the word class. }
  if t^.IsSymbol(SYM_SEMICOLON) then
    begin
      Result := True; // No error, continue the parsing.
      Exit;
    end;

  i := New(PCIO, Init);
  i^.Name := n;
  i^.Description := d;
  i^.MyType := CIOType;
  { get all ancestors; remember, this could look like
    TNewClass = class ( Classes.TClass, MyClasses.TFunkyClass,
                MoreClasses.YAC) ... end;
    all class ancestors are supposed to be included in the docs!
  }
  if (t^.IsSymbol(SYM_LEFT_PARENTHESIS)) then
    begin
      { optional ancestor introduced by ( }
      Dispose(t, Done);
      Finished := False;
      i^.Ancestors := New(PStringCollection, Init(1, 1));
      { outer repeat loop: one ancestor per pass }
      repeat
        if (not GetNextNonWCToken(t)) then Exit;
        if (t^.MyType = TOK_IDENTIFIER) then
          begin { an ancestor }
            s := t^.Data^.GetString;
            Dispose(t, Done);
            { inner repeat loop: one part of the ancestor per name }
            repeat
              if (not Scanner^.GetToken(t)) then
                begin
                  Exit;
                end;
              if (not t^.IsSymbol(SYM_PERIOD)) then
                begin
                  Scanner^.UnGetToken(t);
                  Break; { leave inner repeat loop }
                end;
              Dispose(t, Done);
              s := s + '.';
              if (not Scanner^.GetToken(t)) or (t^.MyType <> TOK_IDENTIFIER) then
                begin
                  ErrorMessage := Scanner^.GetStreamInfo +
                    ': Error - expected class, object or interface in ancestor declaration.';
                  Exit;
                end;
              s := s + t^.Data^.GetString;
            until False;
            i^.Ancestors^.Insert(Objects.NewStr(s));
          end
        else
          if (t^.IsSymbol(SYM_COMMA)) then
            { comma, separating two ancestors }
            begin
              Dispose(t, Done)
            end
          else
            begin
              Finished := t^.IsSymbol(SYM_RIGHT_PARENTHESIS);
              Dispose(t, Done);
              if (not Finished) then
                begin
                  ErrorMessage := Scanner^.GetStreamInfo + ': Error - ")" expected.';
                  Exit;
                end;
            end;
      until Finished;
    end
  else
    if (t^.IsSymbol(SYM_LEFT_BRACKET)) then
      begin
        Dispose(t, Done);
        { for the time being, we throw away the ID itself }
        if (not GetNextNonWCToken(t)) then
          begin
            Exit;
          end;
        if (t^.MyType <> TOK_STRING) then
          begin
            ErrorMessage := Scanner^.GetStreamInfo +
              ': Error - literal string as interface ID expected.';
            Exit;
          end;
        if (not GetNextNonWCToken(t)) then
          begin
            Exit;
          end;
        if (not t^.IsSymbol(SYM_RIGHT_BRACKET)) then
          begin
            ErrorMessage := Scanner^.GetStreamInfo + ': Error - "]" expected.';
            Exit;
          end;
      end
    else
      Scanner^.UnGetToken(t);
  { now collect methods, fields and properties }
  CS := '';
  State := STATE_PUBLIC;
  Finished := False;
  repeat
    CSFound := False;
    if (not GetNextNonWCToken(t)) then
      begin
        Dispose(i, Done);
        Exit;
      end;
    if (t^.IsSymbol(SYM_SEMICOLON)) then
      begin
        { a forward declaration of type "name = class(ancestor);" - this is
          ignored, we simply leave indicating no error }
        {Dispose(i, Done);
        Dispose(t, Done);
        result := True;
        Exit;}
        Dispose(t, Done);
        U^.AddCIO(i);
        Result := True;
        Exit;
      end
    else
      if (t^.MyType = TOK_RESERVED) then
        case t^.Info.ReservedKey of
          KEY_CLASS:
            begin
              CS := t^.Data^.GetString;
              CSFound := True;
            end;
          KEY_CONSTRUCTOR,
            KEY_DESTRUCTOR,
            KEY_FUNCTION,
            KEY_PROCEDURE:
            begin
              GetLastComment(d);
              if (not ParseCDFP(m, CS, t^.Data^.GetString, t^.Info.ReservedKey, d)) then
                begin
                  Dispose(i, Done);
                  Dispose(t, Done);
                  Exit;
                end;
              m^.State := State;
              m^.InsertMethod(m, i^.Methods);
            end;
          KEY_END: Finished := True;
          KEY_PROPERTY:
            begin
              if (not ParseProperty(p)) then
                begin
                  Exit;
                end;
              p^.State := State;
              p^.InsertProperty(p, i^.Properties);
            end;
        else
          begin
            ErrorMessage := Scanner^.GetStreamInfo +
              ': Error, unexpected reserved keyword "' +
              KeyWordArray[t^.Info.ReservedKey] + '"';
            Dispose(i, Done);
            Dispose(t, Done);
            Exit;
          end;
        end
      else
        if (t^.MyType = TOK_IDENTIFIER) then
          begin
            CS := t^.Data^.GetString;
            StringToUpper(CS, CS);
            Ind := StandardDirectives^.IndexOf(@CS);
            case Ind of
              SD_DEFAULT:
                begin
                  if (not SkipDeclaration) then
                    begin
                      ErrorMessage := 'Could not skip declaration after default property';
                      Exit;
                    end;
                  PrintLn(5, 'Skipped default property keyword.');
                end;
              SD_PUBLIC: State := STATE_PUBLIC;
              SD_PUBLISHED: State := STATE_PUBLISHED;
              SD_PRIVATE: State := STATE_PRIVATE;
              SD_PROTECTED:
                State := STATE_PROTECTED;
            else
              Ind := -1;
            end;
            if (Ind = -1) then
              begin
                f := New(PItem, Init);
                if (not Assigned(f)) then
                  begin
                    ErrorMessage := 'Could not create field object.';
                    Exit;
                  end;
                f^.Name := t^.Data^.GetString;
                f^.State := State;
                GetLastComment(f^.Description);
                if (not SkipDeclaration) then
                  begin
                    Exit;
                  end;
                f^.InsertItem(f, i^.Fields);
              end;
          end;
    if (not CSFound) then CS := '';
    Dispose(t, Done);
  until Finished;
  if (not GetNextNonWCToken(t)) or (not t^.IsSymbol(SYM_SEMICOLON)) then
    begin
      ErrorMessage := Scanner^.GetStreamInfo +
        ' semicolon at the end of class/obj/interf. expected.';
      Dispose(i, Done);
      Exit;
    end;
  Dispose(t, Done);
  U^.AddCIO(i);
  Result := True;
end;

function TParser.ParseConstant(var U: PUnit; t: PToken): Boolean;
var
  i: PItem;
begin
  ParseConstant := False;
  i := New(PItem, Init);
  if (not Assigned(t)) or (not Assigned(t^.Data)) then
    begin
      PrintLn(1, 'INTERNAL ERROR in ParseConstant - token not initialized.');
      Exit;
    end;
  i^.Name := t^.Data^.GetString;
  PrintLn(5, 'Parsing constant ' + i^.Name);
  GetLastComment(i^.Description);
  if SkipDeclaration then
    begin
      U^.AddConstant(i);
      ParseConstant := True;
    end
  else
    begin
      ErrorMessage := 'Could not skip declaration of constant ' + i^.Name;
    end;
end;

function TParser.ParseInterfaceSection(var U: PUnit): Boolean;
const
  MODE_UNDEFINED = 0;
  MODE_CONST = 1;
  MODE_TYPE = 2;
  MODE_VAR = 3;
var
  d: PText;
  Finished: Boolean;
  Mode: Integer;
  m: PMethod;
  t: PToken;
begin
  PrintLn(4, 'Entering interface section of unit ' + U^.Name);
  Result := False;
  Finished := False;
  Mode := MODE_UNDEFINED;
  repeat
    if (not GetNextNonWCToken(t)) then
      begin
        PrintLn(2, Scanner^.GetStreamInfo +
          ': Error, could not get next non-whitespace, non-comment token');
        Exit;
      end;
    case t^.MyType of
      TOK_IDENTIFIER:
        begin
          // s := t^.Data^.GetString;
          case Mode of
            MODE_CONST:
              if (not ParseConstant(U, t)) then Exit;
            MODE_TYPE:
              if (not ParseType(U, t)) then Exit;
            MODE_VAR:
              if (not ParseVariables(U, t)) then Exit;
          else
            begin
              ErrorMessage := Scanner^.GetStreamInfo + ': Error, unexpected ' +
                'identifier "' + t^.Data^.GetString + '"';
              Exit;
            end;
          end;
        end;
      TOK_RESERVED:
        begin
          case t^.Info.ReservedKey of
            KEY_RESOURCESTRING,
              KEY_CONST:
              Mode := MODE_CONST;
            KEY_FUNCTION,
              KEY_PROCEDURE:
              begin
                GetLastComment(d);
                if (not ParseCDFP(m, '', t^.Data^.GetString, t^.Info.ReservedKey, d)) then
                  begin
                    Exit;
                  end;
                m^.InsertMethod(m, U^.FuncsProcs);
                Mode := MODE_UNDEFINED;
              end;
            KEY_IMPLEMENTATION:
              Finished := True;
            KEY_TYPE:
              Mode := MODE_TYPE;
            KEY_USES:
              if (not ParseUses(U)) then Exit;
            KEY_THREADVAR,
              KEY_VAR:
              Mode := MODE_VAR;
          else
            begin
              ErrorMessage := Scanner^.GetStreamInfo + ': Error, unexpected keyword ' + t^.Data^.GetString;
              Exit;
            end;
          end;
        end;
    end;
    if Assigned(t) then Dispose(t, Done);
  until Finished;
  Result := True;
end;

function TParser.ParseProperty(var p: PProperty): Boolean;
var
  Finished: Boolean;
  t: PToken;
begin
  ParseProperty := False;
  if (not GetNextNonWCToken(t)) then Exit;
  if (t^.MyType <> TOK_IDENTIFIER) then
    begin
      Dispose(t, Done);
      ErrorMessage := Scanner^.GetStreamInfo + ': expected identifier as property name.';
      Exit;
    end;
  p := New(PProperty, Init);
  if (not Assigned(p)) then
    begin
      ErrorMessage := 'Could not create property item.';
      Exit;
    end;
  p^.Name := t^.Data^.GetString;
  PrintLn(5, 'Parsing property ' + p^.Name);
  p^.IndexDecl := '';
  p^.Proptype := '';
  Dispose(t, Done);
  GetLastComment(p^.Description);
  if (not GetNextNonWCToken(t)) then Exit;
  { get index }
  if (t^.IsSymbol(SYM_LEFT_BRACKET)) then
    begin
      Dispose(t, Done);
      p^.IndexDecl := '[';
      repeat
        if (not Scanner^.GetToken(t)) then
          begin
            PrintLn(2, Scanner^.GetStreamInfo + ': Error, could not parse property.');
            Exit;
          end;
        if (t^.MyType <> TOK_COMMENT) and (t^.MyType <> TOK_DIRECTIVE) then
          p^.IndexDecl := p^.IndexDecl + t^.Data^.GetString;
        Finished := t^.IsSymbol(SYM_RIGHT_BRACKET);
        Dispose(t, Done);
      until Finished;
      { next nonwc token should be the colon }
      if (not GetNextNonWCToken(t)) then
        Exit;
    end
  else
    begin
      if (t^.IsSymbol(SYM_SEMICOLON)) then
        begin
          p^.FullDeclaration := p^.Name + ';';
          // p^.FullDeclaration := p^.Name + p^.IndexDecl + ': ' + p^.Proptype;
          Dispose(t, Done);
          ParseProperty := True;
          Exit;
        end;
    end;

  { now if there is a colon, it is followed by the type }
  if t^.IsSymbol(SYM_COLON) then
    begin
      { get property type }
      if (not GetNextNonWCToken(t)) then Exit;
      if (t^.MyType <> TOK_IDENTIFIER) and (t^.MyType <> TOK_RESERVED) then
        begin
          PrintLn(2, 'Property error, identifier expected, found ' + TokenTypeNames[t^.MyType]);
          ErrorMessage := Scanner^.GetStreamInfo + ' identifier expected';
          Exit;
        end;
      p^.Proptype := t^.Data^.GetString;
      Dispose(t, Done);
      p^.FullDeclaration := p^.Name + p^.IndexDecl + ': ' + p^.Proptype + ';';
    end
  else
    p^.FullDeclaration := p^.Name + ';';

  { simply skipping the rest of declaration }
  if (not SkipDeclaration) then
    begin
      PrintLn(2, 'Error: could not skip rest of declaratino');
      Exit;
    end;
  Result := True;

  (*

{ now there should be a colon followed by the type }

if (not t^.IsSymbol(SYM_COLON)) then
  begin
    ErrorMessage := Scanner^.GetStreamInfo + ': expected ":" character.';
    Dispose(p, Done);
    p := nil;
    Dispose(t, Done);
    Exit;
  end;
Dispose(t, Done);

{ get property type }

if (not GetNextNonWCToken(t)) then Exit;
if (t^.MyType <> TOK_IDENTIFIER) and (t^.MyType <> TOK_RESERVED) then
  begin
    PrintLn(2, 'Property error, identifier expected, found ' + TokenTypeNames[t^.MyType]);
    ErrorMessage := Scanner^.GetStreamInfo + ' identifier expected';
    Exit;
  end;
p^.Proptype := t^.Data^.GetString;
Dispose(t, Done);

{ simply skipping the rest of declaration }
if (not SkipDeclaration) then
  begin
    PrintLn(2, 'Error: could not skip rest of declaratino');
    Exit;
  end;
p^.FullDeclaration := p^.Name + p^.IndexDecl + ': ' + p^.Proptype + ';';
ParseProperty := True; *)
end;

{
  TYPENAME =
    class of ... ;               => "normal" type
    class ( ANCESTOR<S> )        => class
          ANYTHING               => class
    object ( ) end ;                 => object
          ANYTHING
    interface end ;              => interface
}
function TParser.ParseType(var U: PUnit; var t: PToken): Boolean;
var
  d: PText;
  i: PItem;
  n: string;
begin
  ParseType := False;
  n := t^.Data^.GetString;
  PrintLn(5, 'Parsing type "' + n + '"');
  Dispose(t, Done);
  GetLastComment(d);
  if (not GetNextNonWCToken(t)) then
    Exit;

  if (not t^.IsSymbol(SYM_EQUAL)) then
    begin
      if (t^.IsSymbol(SYM_SEMICOLON)) then
        begin
          Dispose(t, Done);
          t := nil;
          ParseType := True;
          Exit;
        end;
      ErrorMessage := Scanner^.GetStreamInfo + ': Error, "=" expected.';
      Dispose(t, Done);
      Exit;
    end;

  Dispose(t, Done);
  if (not GetNextNonWCToken(t)) then
    Exit;

  if (t^.MyType = TOK_RESERVED) then
    case t^.Info.ReservedKey of
      KEY_CLASS:
        begin
          Dispose(t, Done);
          if (not GetNextNonWCToken(t)) then Exit;
          if (t^.MyType = TOK_RESERVED) and (t^.Info.ReservedKey = KEY_OF) then
            begin
              { include "identifier = class of something;" as standard type }
            end
          else
            begin
              Scanner^.UnGetToken(t);
              if (not ParseCIO(U, n, CIO_CLASS, d)) then Exit;
              t := nil;
              ParseType := True;
              Exit;
            end;
        end;
      KEY_DISPINTERFACE:
        begin
          if (not ParseCIO(U, n, CIO_DISPINTERFACE, d)) then Exit;
          Dispose(t, Done);
          t := nil;
          ParseType := True;
          Exit;
        end;
      KEY_INTERFACE:
        begin
          if (not ParseCIO(U, n, CIO_INTERFACE, d)) then Exit;
          Dispose(t, Done);
          t := nil;
          ParseType := True;
          Exit;
        end;
      KEY_OBJECT:
        begin
          if (not ParseCIO(U, n, CIO_OBJECT, d)) then Exit;
          Dispose(t, Done);
          t := nil;
          ParseType := True;
          Exit;
        end;
    end;

  Scanner^.UnGetToken(t);

  t := nil; { so that calling function will not try to dispose of it }
  if (not SkipDeclaration) then
    Exit;

  i := New(PItem, Init);
  if (not Assigned(i)) then
    begin
      ErrorMessage := 'Error, unable to create new item.';
      Exit;
    end;
  i^.Name := n;
  i^.Description := d;
  U^.AddType(i);
  ParseType := True;
end;

function TParser.ParseUnit(var U: PUnit): Boolean;
var
  t: PToken;
begin
  ParseUnit := False;
  { get 'unit' keyword }
  if (not GetNextNonWCToken(t)) then Exit;
  if (t^.MyType <> TOK_RESERVED) or (t^.Info.ReservedKey <> KEY_UNIT) then
    begin
      ErrorMessage := Scanner^.GetStreamInfo + ': keyword "unit" expected.';
      Exit;
    end;
  Dispose(t, Done);
  U := New(PUnit, Init);
  GetLastComment(U^.Description);
  if (not GetNextNonWCToken(t)) then Exit;
  { get unit name identifier }
  if (t^.MyType <> TOK_IDENTIFIER) then
    begin
      ErrorMessage := Scanner^.GetStreamInfo + ': identifier (unit name) expected.';
      Exit;
    end;
  U^.Name := t^.Data^.GetString;
  { skip semicolon }
  if (not GetNextNonWCToken(t)) then Exit;
  if (not t^.IsSymbol(SYM_SEMICOLON)) then
    begin
      ErrorMessage := Scanner^.GetStreamInfo + ': semicolon expected.';
      Exit;
    end;
  if not GetNextNonWCToken(t) then Exit;
  { get 'interface' keyword }
  if (t^.MyType <> TOK_RESERVED) or (t^.Info.ReservedKey <> KEY_INTERFACE) then
    begin
      ErrorMessage := Scanner^.GetStreamInfo + ': keyword "INTERFACE" expected.';
      Exit;
    end;
  { now parse the interface section of that unit }
  ParseUnit := ParseInterfaceSection(U);
end;

function TParser.ParseUses(var U: PUnit): Boolean;
var
  Finished: Boolean;
  t: PToken;
begin
  ParseUses := False;
  if (not Assigned(U^.UsesUnits)) then U^.UsesUnits := New(PStringCollection, Init(1, 1));
  if (not Assigned(U^.UsesUnits)) then
    begin
      PrintLn(2, 'Error in TParser.ParseUses: Could not create list of unit names.');
      Exit;
    end;
  repeat
    if (not GetNextNonWCToken(t)) then Exit;
    if (t^.MyType <> TOK_IDENTIFIER) then
      begin
        ErrorMessage := Scanner^.GetStreamInfo + ': Error, unit name expected (found ' +
          t^.GetTypeName + ', ' + t^.Data^.GetString + ')';
        Exit;
      end;
    U^.UsesUnits^.Insert(NewStr(t^.Data^.GetString));
    Dispose(t, Done);
    if (not GetNextNonWCToken(t)) then Exit;
    if (t^.MyType <> TOK_SYMBOL) and
      (t^.Info.SymbolType <> SYM_COMMA) and
      (t^.Info.SymbolType <> SYM_SEMICOLON) then
      begin
        ErrorMessage := Scanner^.GetStreamInfo + ': Error, comma or semicolon expected.';
        Exit;
      end;
    Finished := (t^.Info.SymbolType = SYM_SEMICOLON);
    Dispose(t, Done);
  until Finished;
  ParseUses := True;
end;

function TParser.ParseVariables(var U: PUnit; var t: PToken): Boolean;
var
  Finished: Boolean;
  FirstLoop: Boolean;
  i: PItem;
begin
  ParseVariables := False;
  FirstLoop := True;
  repeat
    i := New(PItem, Init);
    if (not Assigned(i)) then
      begin
        ErrorMessage := 'Error: Not enough memory for new item.';
        Exit;
      end;
    if FirstLoop then
      begin
        i^.Name := t^.Data^.GetString;
        FirstLoop := False;
      end
    else
      begin
        if (not GetNextNonWCToken(t)) then Exit;
        if (t^.MyType <> TOK_IDENTIFIER) then
          begin
            ErrorMessage := Scanner^.GetStreamInfo +
              ': Error, identifier expected.';
            Exit;
          end;
        i^.Name := t^.Data^.GetString;
      end;
    GetLastComment(i^.Description);
    U^.AddVariable(i);
    Dispose(t, Done);
    if (not GetNextNonWCToken(t)) then Exit;
    if (t^.MyType <> TOK_SYMBOL) or
      ((t^.Info.SymbolType <> SYM_COMMA) and
      (t^.Info.SymbolType <> SYM_COLON)) then
      begin
        ErrorMessage := Scanner^.GetStreamInfo +
          ': Error, expected comma or colon in var declaration.';
        Exit;
      end;
    Finished := (t^.Info.SymbolType = SYM_COLON);
    Dispose(t, Done);
    t := nil;
  until Finished;
  if (not SkipDeclaration) then Exit;
  ParseVariables := True;
end;

function TParser.SkipDeclaration: Boolean;
var
  EndLevel: Integer;
  IsSemicolon: Boolean;
  PLevel: Integer;
  t: PToken;
begin
  SkipDeclaration := False;
  EndLevel := 0;
  PLevel := 0;
  repeat
    if (not GetNextNonWCToken(t)) then Exit;

    IsSemicolon := (t^.MyType = TOK_SYMBOL) and (t^.Info.SymbolType = SYM_SEMICOLON);

    if (t^.MyType = TOK_SYMBOL) then
      begin
        if (t^.Info.SymbolType = SYM_LEFT_PARENTHESIS) then
          Inc(PLevel);
        if (t^.Info.SymbolType = SYM_RIGHT_PARENTHESIS) then
          Dec(PLevel);
      end;
    if (t^.MyType = TOK_RESERVED) then
      begin
        if (t^.Info.ReservedKey = KEY_END) then
          Dec(EndLevel)
        else if (t^.Info.ReservedKey = KEY_RECORD) then
          Inc(EndLevel);
      end;
    Dispose(t, Done);
  until IsSemicolon and (EndLevel = 0) and (PLevel = 0);
  SkipDeclaration := True;
end;

function TParser.SkipWhitespaceAndComments: Boolean;
var
  t: PToken;
begin
  repeat
    if (not Scanner^.PeekToken(t)) then
      begin
        SkipWhitespaceAndComments := False;
        ErrorMessage := Scanner^.ErrorMessage;
        Exit;
      end;
    if (t^.MyType = TOK_WHITESPACE) then
      begin
        Scanner^.ConsumeToken;
        Dispose(t, Done);
      end
    else if (t^.MyType = TOK_COMMENT) then
      begin
        Scanner^.ConsumeToken;
        if Assigned(LastCommentToken) then Dispose(LastCommentToken, Done);
        LastCommentToken := t;
      end
    else
      begin
        SkipWhitespaceAndComments := True;
        Exit;
      end;
  until False;
end;

begin
end.

