{
@abstract(provides basic integer types and operations on them)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@created(7 May 1998)
@lastmod(10 Apr 2000)

This unit defines several types / procedures to deal with integer numbers.

The GetXXXX procedures return a value of type XXXX, getting it from an untyped
argument Source, the SetXXXX procedures do the opposite.

A byte order must be specified whenever type XXXX variables are at least two
bytes large.
}

unit Numbers;

{$I platform.inc}

interface

type
  { define several basic integer types;
    naming conventions are as follows
    - first T for type
    - S for signed or U for unsigned
    - Int
    - number of bits (8, 16, 32, 64) }
  { signed 16 bit integer type }
  TSInt16 = Integer;
  { signed 32 bit integer type }
  TSInt32 = LongInt;
  { signed 64 bit integer type }
  TSInt64 =
    {$IFDEF PPC_DELPHI4}
  Int64;
  {$ELSE}
  Comp;
  {$ENDIF}
  { signed 8 bit integer type }
  TSInt8 = ShortInt;
  { unsigned 16 bit integer type }
  TUInt16 = Word;
  { unsigned 32 bit integer type }
  TUInt32 =
    {$IFDEF PPC_DELPHI4}
  LongWord; { it took about 10 years to accomplish this :-) }
  {$ELSE}
  LongInt;
  {$ENDIF}
  { unsigned 8 bit integer type }
  TUInt8 = Byte;
  {
  Simple integer type to be used as argument throughout this
  unit.
  Holds dimension values, so should be at least 16 bits large.
  Also holds sample values, so must be at least the size of the
  largest possible sample.
  Must be signed.
  }
  TNativeInt = LongInt;
  { supported byte order types }
  TByteOrder = (BYTE_ORDER_INTEL, BYTE_ORDER_MOTOROLA);
  { array of Byte (unsigned 8 bit values). }
  TByteArray = array[0..MaxInt div SizeOf(Byte) - 1] of Byte;
  { pointer to @link(TByteArray) }
  PByteArray = ^TByteArray;
  { array of @link(TNativeInt); will be overindexed }
  TIntArray = array[0..3] of TNativeInt;
  { pointer to @link(TIntArray) }
  PIntArray = ^TIntArray;

const
  { maximum value of a TNativeInt variable }
  MAX_NATIVE_INT: TNativeInt = 2147483647;
  { minimum value of a TNativeInt variable }
  MIN_NATIVE_INT: TNativeInt = -2147483647;

  { *** ATTENTION *** }
  { You MUST define exactly one of the two following directives! }

  {$DEFINE HOME_BYTE_ORDER_INTEL}
  { $DEFINE HOME_BYTE_ORDER_MOTOROLA}

  { Compares two unsigned 8 bit values.
    Returns -1 (B1 < B2), 0 (B1 = B2) or 1 (B1 > B2). }
function CompareUInt8(b1, b2: TUInt8): Integer;
{ compares two unsigned 16 bit values; returns -1 (B1 < B2), 0 (B1 = B2) or 1 (B1 > B2) }
function CompareUInt16(V1, v2: TUInt16): Integer;
{ gets a signed 16 bit integer value in a specific byte order from address Source, result in Value }
procedure GetSInt16(ByteOrder: TByteOrder; var Source; var Value: TSInt16);
{ gets a signed 32 bit integer value in a specific byte order from address Source, result in Value }
procedure GetSInt32(ByteOrder: TByteOrder; var Source; var Value: TSInt32);
{ gets a signed 8 bit integer value from address Source, result in Value }
procedure GetSInt8(var Source; var Value: TSInt8);
{ gets an unsigned 16 bit integer value in a specific byte order from address Source, result in Value }
procedure GetUInt16(ByteOrder: TByteOrder; var Source; var Value: TUInt16);
{ gets an unsigned 32 bit integer value in a specific byte order from address Source, result in Value }
procedure GetUInt32(ByteOrder: TByteOrder; var Source; var Value: TUInt32);
{ gets an unsigned 8 bit integer value from address Source, result in Value }
procedure GetUInt8(var Source; var Value: TUInt8);
{ returns string representation of Value in decimal format }
// function IntToString (Value: TSInt32): string;
{ returns string representation of Value in decimal format, filling up to length L with '0' characters }
function IntToStringLZ(l: TUInt16; Value: TSInt32): string;
{ returns string representation of Value }
function IntToStringRadix(Value: TSInt32; Radix: TUInt8): string;
{ writes Value in a given byte order to address Dest}
procedure SetSInt16(ByteOrder: TByteOrder; const Value: TSInt16; var Dest);
{ writes Value in a given byte order to address Dest}
procedure SetSInt32(ByteOrder: TByteOrder; const Value: TSInt32; var Dest);
{ writes Value to address Dest }
procedure SetSInt8(const Value: TSInt8; var Dest);
{ writes Value in a given byte order to address Dest}
procedure SetUInt16(ByteOrder: TByteOrder; const Value: TUInt16; var Dest);
{ writes Value in a given byte order to address Dest}
procedure SetUInt32(ByteOrder: TByteOrder; const Value: TUInt32; var Dest);
{ writes Value to address Dest }
procedure SetUInt8(const Value: TUInt8; var Dest);
{ swaps two 16 bit values (signed or unsigned), i.e. sets A to B and vice versa }
{procedure Swap16(var A, B);}
{ swaps two 32 bit values (signed or unsigned), i.e. sets A to B and vice versa }
{procedure Swap32(var A, B);}

implementation

uses
  SysUtils;

function CompareUInt8(b1, b2: TUInt8): Integer;
begin
  if (b1 = b2) then
    CompareUInt8 := 0
  else
    begin
      if (b1 < b2) then
        CompareUInt8 := -1
      else
        CompareUInt8 := 1;
    end;
end;

function CompareUInt16(V1, v2: TUInt16): TSInt16;
begin
  if (V1 < v2) then
    CompareUInt16 := -1
  else
    if (V1 > v2) then
      CompareUInt16 := 1
    else
      CompareUInt16 := 0;
end;

procedure GetSInt8(var Source; var Value: TSInt8);
begin
  Value := TSInt8(Source);
end;

procedure GetSInt16(ByteOrder: TByteOrder; var Source; var Value: TSInt16);
begin
  if (ByteOrder = BYTE_ORDER_INTEL) then
    Value := (TByteArray(Source)[0] shl 8) + TByteArray(Source)[1]
  else
    Value := (TByteArray(Source)[1] shl 8) + TByteArray(Source)[0];
end;

procedure GetSInt32(ByteOrder: TByteOrder; var Source; var Value: TSInt32);
begin
  {$IFDEF HOME_BYTE_ORDER_INTEL}
  case ByteOrder of
    BYTE_ORDER_INTEL: Value := TSInt32(Source);
    BYTE_ORDER_MOTOROLA:
      Value :=
        TByteArray(Source)[3] or
        (TByteArray(Source)[2] shl 8) or
        (TByteArray(Source)[1] * 65536) or
        (TByteArray(Source)[0] * 16777216);
    {        TByteArray(Source)[3] or
            (TByteArray(Source)[2] shl 8) or
            (TByteArray(Source)[1] shl 16) or
            (TByteArray(Source)[0] shl 24);}
  end;
  {$ENDIF}
  {$IFDEF HOME_BYTE_ORDER_MOTOROLA}
  {$ENDIF}
end;

procedure GetUInt8(var Source; var Value: TUInt8);
begin
  Value := TUInt8(Source);
end;

procedure GetUInt16(ByteOrder: TByteOrder; var Source; var Value: TUInt16);
begin
  {$IFDEF HOME_BYTE_ORDER_INTEL}
  case ByteOrder of
    BYTE_ORDER_INTEL: Value := TUInt16(Source);
  end;
  {$ENDIF}
  {$IFDEF HOME_BYTE_ORDER_MOTOROLA}
  {$ENDIF}
end;

procedure GetUInt32(ByteOrder: TByteOrder; var Source; var Value: TUInt32);
begin
  {$IFDEF HOME_BYTE_ORDER_INTEL}
  case ByteOrder of
    BYTE_ORDER_INTEL: Value := TUInt32(Source);
  end;
  {$ENDIF}
  {$IFDEF HOME_BYTE_ORDER_MOTOROLA}
  {$ENDIF}
end;

function IntToStringLZ(l: TUInt16; Value: TSInt32): string;
var
  s: string;
begin
  s := IntToStr(Value);
  while (Length(s) < l) do
    s := '0' + s;
  IntToStringLZ := s;
end;

procedure SetSInt16(ByteOrder: TByteOrder; const Value: TSInt16; var Dest);
begin
end;

procedure SetSInt32(ByteOrder: TByteOrder; const Value: TSInt32; var Dest);
begin
end;

procedure SetSInt8(const Value: TSInt8; var Dest);
begin
  TSInt8(Dest) := Value;
end;

procedure SetUInt16(ByteOrder: TByteOrder; const Value: TUInt16; var Dest);
begin
  {$IFDEF HOME_BYTE_ORDER_INTEL}
  if (ByteOrder = BYTE_ORDER_INTEL) then
    TUInt16(Dest) := Value
  else
    ;
  {$ENDIF}
end;

procedure SetUInt32(ByteOrder: TByteOrder; const Value: TUInt32; var Dest);
begin
  {$IFDEF HOME_BYTE_ORDER_INTEL}
  if (ByteOrder = BYTE_ORDER_INTEL) then
    TUInt32(Dest) := Value
  else
    ;
  {$ENDIF}
end;

procedure SetUInt8(const Value: TUInt8; var Dest);
begin
  TUInt8(Dest) := Value;
end;

{
function IntToString (Value: TSInt32): string;
begin
  IntToString := IntToStringRadix (Value, 10);
end;
}

function IntToStringRadix(Value: TSInt32; Radix: TUInt8): string;
const
  HexChars: array[0..15] of Char = '0123456789abcdef';
var
  i: LongInt;
  s: string;
  Sign: string[1];
begin
  Result := '';
  if (Radix < 2) or (Radix > 16) then Exit;
  if (Value < 0) then
    begin
      Sign := '-';
      Value := Abs(Value);
    end
  else
    Sign := '';
  s := '';
  repeat
    i := Value mod Radix;
    s := HexChars[i] + s;
    Value := Value div Radix;
  until (Value = 0);
  Result := Sign + s;
end;

end.

