{
@abstract(contains the application object TPasdoc)
@author(Ralf Junker <delphi@zeitungsjunge.de>)
@author(Erwin Scheuch-Heilig (ScheuchHeilig@t-online.de))
@author(Marco Schmidt (marcoschmidt@geocities.com))
@author(Michael van Canneyt (michael@tfdec1.fys.kuleuven.ac.be))
@created(24 Sep 1999)
@lastmod(20 Apr 2000)
}
unit Main;

{$I platform.inc}

interface

{ Wildcard expansion is currently supported for any DOS compiler (FPC/DOS,
  BP) and Delphi.
  Linux doesn't need this, as the shell does the expansion. }

{$IFDEF PPC_DELPHI}
{ tell Delphi that this is a console application }
{$APPTYPE CONSOLE}
{ file search using SysUtils }
{$DEFINE SUPPORT_WILDCARD_EXPANSION}
{$ENDIF}

{$IFDEF OS_DOS}
{ file search using DOS unit }
{$DEFINE SUPPORT_WILDCARD_EXPANSION}
{$DEFINE USE_DOS}
{$ENDIF}

{$IFDEF PPC_FPC}
{ file search using DOS unit }
{$DEFINE SUPPORT_WILDCARD_EXPANSION}
{$DEFINE USE_DOS}
{$ENDIF}

uses
  {$IFDEF USE_DOS}
  DOS,
  {$ENDIF}
  {$IFDEF PPC_DELPHI}
  SysUtils,
  {$ENDIF}
  AppInfo,
  Arrays,
  FileStre,
  GenDoc,
  Html,
  Items,
  Languages,
  Msg,
  Numbers,
  Objects,
  Parsing,
  Streams,
  Texts,
  Time;

type
  { all supported output formats }
  TOutputFormat = (OUTPUT_NONE, OUTPUT_HTML, OUTPUT_HTML_HELP);
  { pointer to @link(TPasDoc) }
  PPasDoc = ^TPasDoc;
  { The main object in the pasdoc application; first scans parameters, then
    parses files.
    All parsed units are then given to documentation generator,
    which creates one or more documentation output files. }
  TPasDoc = object(TObject)
    { all description file names }
    DescriptionFiles: PStringCollection;
    { directory where documentation will be written to }
    DestDir: string;
    { all file names which have been collected from the program
      parameters }
    FileNames: PStringCollection;
    ProjectName: AnsiString;
    Header: AnsiString;
    Footer: AnsiString;
    ContentsFile: AnsiString;
    { collection of @link(PString) values of all directories where pasdoc is
      supposed to search for include files }
    IncludeFilePaths: PCollection;
    { defines if the private parts (fields and methods) of objects are to be
      included in the documentation (default is false) }
    IncludePrivate: Boolean;
    { language used to write documentation }
    Language: TLanguageID;
    { by default, all HTML documents contain a link on the bottom stating
      pasdoc version, time this page was generated and a link to the pasdoc
      homepage; set NoGeneratorInfo to true to prevent this }
    NoGeneratorInfo: Boolean;
    { if true: leave out statements at the beginning and end of TeX document
      to enable users to simply \include the pasdoc output; default is false }
    // NoTexHeadersAndFooters: Boolean;
    { output format to be used (default is HTML) }
    OutputFormat: TOutputFormat;

    {$IFDEF WINDOWS}
    { Registry and HCC only exists on Windows }
    { True if not to call HCC.exe if creating HtmlHelp output.
      Otherwise, rjPasDoc will look for HCC.exe in the registry and
      compile the project.  }
    NoHHC: Boolean;
    {$ENDIF}

    { list of standard compiler directives }
    StdDirectives: PStringCollection;
    { Title of documentation. }
    Title: string;
    { all PUnit objects which have been created from the list of file names }
    Units: PItemCollection;
    { Creates object and sets fields to default values. }
    constructor Init;
    { }
    destructor Done; virtual;
    { Adds S to the list of description files that will be parsed.
      Will compare S to strings in list and will not include it if already
      present.
      However, this will not avoid duplicates on file systems where case is
      not regarded (VFAT etc.). }
    function AddDescriptionFile(s: string): Boolean;
    { Adds parameter S to the internal list of standard directives, which
      are considered present whenever parsing of a unit starts. }
    function AddStandardDirective(s: string): Boolean;
    { Creates a @link(TUnit) object from the stream and adds it to
   @link(Units). }
    procedure HandleStream(InputStream: PInputStream; const SourceFileName: PString);
    { Scans the program parameters, creating a list of file names and
      regarding switches. }
    procedure HandleParams;
    { Loads names of Pascal unit source code files from a text file.
      Adds all file names to @link(FileNames). }
    procedure LoadFileNames(FileName: string);
    { Loads a list of standard directives (one per row) from text file
      FileName.
      These directives are added to the standard directives list
      @link(StdDirectives). }
    procedure LoadStandardDirectives(FileName: string);
    { Calls @link(HandleStream) for each file name in @link(FileNames). }
    procedure LoadAnsiStringFromFile(FileName: string; var s: AnsiString);
    procedure ParseFiles;
    { Prints a detailed list of switches to standard output. }
    procedure PrintHelp;
    { Prints a usage clause to standard output. }
    procedure PrintUsage;
    { Searches the description of each PUnit item in the collection for an
      excluded tag.
      If one is found, the item is removed from the collection.
      If not, the fields, methods and properties collections are called
      with RemoveExcludedItems
      If the collection is empty after removal of all items, it is disposed
      of and the variable is set to nil. }
    procedure RemoveExcludedItems(var c: PItemCollection);
    { Does all the work. }
    procedure Run;
    { Searches for descr tags in the comments of all TItem objects in C. }
    procedure SearchDescrFileTags(var c: PItemCollection);
    { Sets language to NewLanguage. }
    procedure SetLanguage(const NewLanguage: TLanguageID);
  end;

implementation

const
  { the default output language (type @link(TLanguage) }
  // DEFAULT_LANGUAGE: TLanguage = lgEnglish;
  { the output format to be used when none is specified on command line }
  DEFAULT_OUTPUT_FORMAT: TOutputFormat = OUTPUT_HTML;
  { names of all output formats }
  OUTPUT_FORMAT_NAMES: array[TOutputFormat] of string =
  ('', 'HTML', 'HTML Help');
  { the string used to separate paths, a backslash on Windows and a forward
    slash on Unix systems }
  {$IFDEF OS_LINUX}
  PATH_SEPARATOR = '/';
  {$ELSE}
  PATH_SEPARATOR = '\';
  {$ENDIF}

  { find the right-most character 'CharToFind' within string 'Str'
    returns 0 if character not found at all }
function RScan(CharToFind: Char; Str: string): Integer;
var
  CurrPos: Integer;
begin
  CurrPos := Length(Str);
  while (CurrPos > 0) and (Str[CurrPos] <> CharToFind) do
    Dec(CurrPos);
  RScan := CurrPos;
end;

constructor TPasDoc.Init;
begin
  inherited Init;
  { check for Unicode }
  if (SizeOf(Char) <> 1) then
    begin
      PrintLn(1, 'Error: Size of Char is larger than one byte, cannot handle this case.');
      Halt(1);
    end;
  DestDir := '';
  DescriptionFiles := nil;
  FileNames := nil;
  IncludeFilePaths := nil;
  IncludePrivate := False;
  Language := DEFAULT_LANGUAGE;
  OutputFormat := DEFAULT_OUTPUT_FORMAT;
  NoGeneratorInfo := False;
  // NoTexHeadersAndFooters := False;
  StdDirectives := nil;
  Units := nil;
end;

destructor TPasDoc.Done;
begin
  if Assigned(DescriptionFiles) then Dispose(DescriptionFiles, Done);
  if Assigned(FileNames) then Dispose(FileNames, Done);
  if Assigned(Units) then Dispose(Units, Done);
  inherited Done;
end;

function TPasDoc.AddDescriptionFile(s: string): Boolean;
var
  PS: PString;
begin
  Result := False;
  if (not Assigned(DescriptionFiles)) then
    begin
      DescriptionFiles := New(PStringCollection, Init(16, 16));
      if (not Assigned(DescriptionFiles)) then
        begin
          PrintLn(2, 'Errorr: not enough memory to create description file list');
          Exit;
        end;
    end;
  if (DescriptionFiles^.IndexOf(@s) <> -1) then
    begin
      Result := True;
      Exit;
    end;
  PS := NewStr(s);
  if Assigned(PS) then
    DescriptionFiles^.Insert(PS)
  else
    PrintLn(2, 'Error - could not create description file name string');
  Result := Assigned(PS);
end;

function TPasDoc.AddStandardDirective(s: string): Boolean;
var
  PS: PString;
begin
  { convert S to upper case for convenient comparison }
  StringToUpper(s, s);
  { check if S is already included - if so, leave }
  if (Assigned(StdDirectives) and (StdDirectives^.IndexOf(@s) <> -1)) then
    begin
      AddStandardDirective := True;
      Exit;
    end;
  { create StdDirectives collection if necessary }
  if (not Assigned(StdDirectives)) then
    begin
      StdDirectives := New(PStringCollection, Init(16, 16));
      if (not Assigned(StdDirectives)) then
        begin
          PrintLn(1, 'Error: Could not create list of standard directives.');
          Halt(1);
        end;
    end;
  { create PString version of S and insert it into collection }
  PS := NewStr(s);
  if (not Assigned(PS)) then
    begin
      PrintLn(1, 'Error: Could not create new string for directive list');
      Halt(1);
    end;
  StdDirectives^.Insert(PS);
  AddStandardDirective := True;
end;

procedure TPasDoc.HandleParams;
var
  ErrorCode: Integer;
  i: LongInt;
  j: LongInt;
  l: TLanguageID;
  p: PString;
  Path: string;
  s: AnsiString;
  VerbosityLevel: Integer;
  {$IFDEF USE_DOS}
  SR: DOS.SearchRec;
  {$ENDIF}
  {$IFDEF PPC_DELPHI}
  SearchResult: Integer;
  SR: SysUtils.TSearchRec;
  {$ENDIF}
begin
  FileNames := New(PStringCollection, Init(16, 16));
  OutputFormat := OUTPUT_NONE;
  VerbosityLevel := DEFAULT_VERBOSITY_LEVEL;
  i := 1;
  s := ParamStr(i);
  while s <> '' do
    begin
      if (s[1] = '-') or (s[1] = '/') then
        begin
          System.Delete(s, 1, 1);
          if s = '' then
            begin
              PrintLn(1, 'Error: Empty switch (' + IntToStr(i) + ').');
              Continue;
            end;
          case s[1] of
            '?': // Help
              begin
                PrintUsage;
                PrintHelp;
                Halt(0);
              end;
            'C', 'c': // Contents for HtmlHelp
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No file name specified after -C switch.');
                    Continue;
                  end;
                ContentsFile := s;
              end;
            'D': // Compiler Directive
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No directive specified after -D switch.');
                    Continue;
                  end;
                if not AddStandardDirective(s) then
                  PrintLn(1, 'Error: Could not add standard directive (' + s + ').');
              end;
            'd': // Compiler Directive File
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No file name after -d switch, skipping.');
                    Continue;
                  end;
                LoadStandardDirectives(s);
              end;
            'E', 'e': // Output directory
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No path specified after -E switch, skipping.');
                    Continue;
                  end;
                DestDir := s;
                if DestDir[Length(DestDir)] <> PATH_SEPARATOR then
                  DestDir := DestDir + PATH_SEPARATOR;
                if not DirectoryExists(DestDir) then
                  begin
                    PrintLn(1, 'Error: Output path does not exist.');
                    Halt(1);
                  end;
              end;
            'F', 'f': // Footer File
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No file name specified after -H switch, skipping.');
                    Continue;
                  end;
                LoadAnsiStringFromFile(s, Footer);
              end;
            'H', 'h': // Header File
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No file name specified after -H switch, skipping.');
                    Continue;
                  end;
                LoadAnsiStringFromFile(s, Header);
              end;
            'I', 'i': // Include Path
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No directory specified after -U switch, skipping.');
                    Continue;
                  end;
                if (s[Length(s)] <> PATH_SEPARATOR) then
                  s := s + PATH_SEPARATOR;
                if (not Assigned(IncludeFilePaths)) then
                  begin
                    IncludeFilePaths := New(PCollection, Init(16, 16));
                    if (not Assigned(IncludeFilePaths)) then
                      begin
                        PrintLn(1, 'Error: Could not create list of include file paths.');
                        Halt(1);
                      end;
                  end;
                p := NewStr(s);
                if (not Assigned(p)) then
                  begin
                    PrintLn(1, 'Error: Could not create string for include file path.');
                    Halt(1);
                  end;
                IncludeFilePaths^.Insert(p);
              end;
            'L', 'l': // Language
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No language specified after -L switch, skipping.');
                    Continue;
                  end;
                l := Low(LANGUAGE_ARRAY);
                while l <= High(LANGUAGE_ARRAY) do
                  begin
                    if CompareText(LANGUAGE_ARRAY[l].Syntax, s) = 0 then
                      begin
                        SetLanguage(l);
                        Break;
                      end;
                    Inc(l);
                  end;
                if l > High(LANGUAGE_ARRAY) then
                  PrintLn(1, 'Error: Unknown language specifier (' + s + '), skipping.');
              end;
            'N', 'n': // Name of project
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No name specified after -N switch, skipping.');
                    Continue;
                  end;
                ProjectName := s;
              end;
            'O', 'o': // Output Format
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No output format specified after -O switch.');
                    Continue;
                  end;
                if CompareText('Html', s) = 0 then
                  OutputFormat := OUTPUT_HTML
                else
                  if CompareText('HtmlHelp', s) = 0 then
                    OutputFormat := OUTPUT_HTML_HELP
                  else
                    PrintLn(1, 'Error: Unknown output format (' + s + '), skipping.');
              end;
            'P', 'p': // Include Private
              begin
                IncludePrivate := True;
              end;
            'R', 'r': // DescRiption File
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No file name specified after -R switch.');
                    Continue;
                  end;
                if not AddDescriptionFile(s) then
                  PrintLn(1, 'Error: Could not add description file (' + s + ').');
              end;
            'S', 's': // SourceFile
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No filename specified after -S switch.');
                    Continue;
                  end;
                LoadFileNames(s);
              end;
            'T', 't': // Title
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No title specified after -T switch.');
                    Continue;
                  end;
                Title := s;
              end;
            'V': // Version
              begin
                PrintLn(1, APP_NAME_AND_VERSION);
                Halt(0);
              end;
            'v': // Verbosity
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No verbosity level (integer >= 0) specified after -v switch.');
                    Continue;
                  end;
                Val(s, VerbosityLevel, ErrorCode);
                if (ErrorCode <> 0) or (VerbosityLevel < 0) then
                  begin
                    PrintLn(1, 'Error - verbosity level value must be an integer >= 0');
                    VerbosityLevel := DEFAULT_VERBOSITY_LEVEL;
                  end;
              end;
            'X', 'x': // Exclude generator info
              begin
                Delete(s, 1, 1);
                if s = '' then
                  begin
                    PrintLn(1, 'Error: No exclude option specified after -X switch.');
                    Continue;
                  end;
                if CompareText('HomePage', s) = 0 then
                  NoGeneratorInfo := True

                  {$IFDEF WINDOWS}
                  { Registry and HCC only exists on Windows }
                else
                  if CompareText('HHC', s) = 0 then
                    NoHHC := True
                  else
                    PrintLn(1, 'Error: Unknown exclude option (' + s + '), skipping.')
                      {$ENDIF}
                    ;

              end;
          else
            begin
              PrintLn(1, 'Error: Unknown switch ("-' + s + '"), skipping.');
            end;
          end;
        end
      else
        begin
          { it's not a switch, so it must be a unit source code file name or a wildcard }
          { check for wildcards }
          if (System.Pos('*', s) > 0) or (System.Pos('?', s) > 0) then
            begin
              {$IFDEF SUPPORT_WILDCARD_EXPANSION}
              Path := ExtractFilePath(s);
              {$IFDEF USE_DOS}
              DOS.FindFirst(s, 63, SR);
              while (DOS.DosError = 0) do
                begin
                  if ((SR.Attr and 24) = 0) then FileNames^.Insert(NewStr(Path + SR.Name));
                  DOS.FindNext(SR);
                end;
              {$ENDIF}
              {$IFDEF PPC_DELPHI}
              SearchResult := SysUtils.FindFirst(s, 63, SR);
              while (SearchResult = 0) do
                begin
                  if ((SR.Attr and 24) = 0) then FileNames^.Insert(NewStr(Path + SR.Name));
                  SearchResult := FindNext(SR);
                end;
              SysUtils.FindClose(SR);
              {$ENDIF}
              {$ELSE}
              PrintLn(1, 'Warning: Wildcard expansion not supported, skipping ', s);
              {$ENDIF}
            end
          else
            begin
              p := NewStr(s);
              j := FileNames^.IndexOf(p);
              if (j < 0) then
                FileNames^.Insert(p)
              else
                DisposeStr(p);
            end;
        end;
      Inc(i);
      s := ParamStr(i);
    end;

  if i = 1 then
    begin
      PrintUsage;
      PrintLn(1, 'Type ''' + ParamStr(0) + ' /?'' for help.');
      Halt(0);
    end;

  SetVerbosityLevel(VerbosityLevel);
  if (FileNames^.Count <= 0) then
    begin
      PrintLn(1, 'Error: No file has been specified.');
      Halt(1);
    end;
  PrintLn(2, APP_FIRST_LINE);
  PrintLn(2, 'Latest version: ' + APP_HOMEPAGE);
  { if no output format has been defined on the command line, pick
    default output format, show warning }
  if (OutputFormat = OUTPUT_NONE) then
    begin
      PrintLn(2, 'Warning: No output format specified, using default ("' +
        OUTPUT_FORMAT_NAMES[DEFAULT_OUTPUT_FORMAT] + '").');
      OutputFormat := DEFAULT_OUTPUT_FORMAT;
    end;

  // Set default language in case a language does not provide a translation.
  LANGUAGE_ARRAY[DEFAULT_LANGUAGE].proc;
  if Language <> DEFAULT_LANGUAGE then
    LANGUAGE_ARRAY[Language].proc
  else
    PrintLn(2, 'Warning: No output language specified, using default ("English").');

  {  if Assigned(IncludeFilePaths) and (IncludeFilePaths^.Count > 0) then
    begin
      for I := 0 to IncludeFilePaths^.Count - 1 do
      begin
        P := IncludeFilePaths^.At(I);
        if Assigned(P)
        then WriteLn(I, ' ', P^);
      end;
    end;}
end;

procedure TPasDoc.HandleStream(InputStream: PInputStream; const SourceFileName: PString);
var
  p: PParser;
  U: PUnit;
  Index: Integer;
begin
  p := New(PParser, Init(InputStream, StdDirectives, IncludeFilePaths));
  if (not Assigned(p)) then
    begin
      PrintLn(1, InputStream^.GetName + ' Error: Could not create parser.');
      Exit;
    end;
  if p^.ParseUnit(U) then
    begin
      if (not IncludePrivate) then
        U^.RemovePrivateItems;

      if (not Assigned(Units)) then
        Units := New(PItemCollection, Init(16, 16));

      if Units^.Search(U, Index) then
        begin
          PrintLn(2, 'Warning: Duplicate unit name (' + U^.Name + ') in files:');
          PrintLn(2, '    ' + PUnit(Units^.At(Index))^.SourceFileName);
          PrintLn(2, '    ' + SourceFileName^ + ' (dicarded)');
          Dispose(U, Done);
        end
      else
        begin
          U^.SourceFileName := SourceFileName^;
          Units^.Insert(U);
        end;
    end
  else
    begin
      PrintLn(1, 'Error parsing: ' + p^.ErrorMessage);
    end;
  Dispose(p, Done);
end;

procedure TPasDoc.LoadFileNames(FileName: string);
var
  nf: PFileInputStream;
  l: string;
  s: PString;

  Path: AnsiString;
  SearchResult: Integer;
  SR: SysUtils.TSearchRec;
begin
  nf := New(PFileInputStream, Init(FileName));
  while (nf^.HasData) do
    begin
      nf^.ReadLine(l);
      Path := ExtractFilePath(l);

      SearchResult := SysUtils.FindFirst(l, 63, SR);
      while (SearchResult = 0) do
        begin
          if (SR.Attr and 24) = 0 then
            begin
              s := NewStr(Path + SR.Name);
              if FileNames^.IndexOf(@s) = -1 then
                FileNames^.Insert(s)
              else
                DisposeStr(s);
            end;
          SearchResult := FindNext(SR);
        end;
      SysUtils.FindClose(SR);
    end;
  Dispose(nf, Done);
end;

procedure TPasDoc.LoadStandardDirectives(FileName: string);
var
  DF: PFileInputStream;
  l: string;
begin
  DF := New(PFileInputStream, Init(FileName));
  if (not Assigned(DF)) then
    begin
      PrintLn(1, 'Error: Could not open file "' + FileName + '" to read standard directives.');
      Halt(1);
    end;
  while (DF^.HasData) do
    begin
      DF^.ReadLine(l);
      if (not AddStandardDirective(l)) then Break;
    end;
  Dispose(DF, Done);
end;

procedure TPasDoc.LoadAnsiStringFromFile(FileName: string; var s: AnsiString);
var
  nf: PFileInputStream;
  l: LongInt;
begin
  nf := New(PFileInputStream, Init(FileName));
  if (not Assigned(nf)) then
    begin
      PrintLn(2, 'Error: Could not open file "' + FileName + '" for reading.');
      Halt(1);
    end;
  l := nf^.GetSize;
  SetLength(s, l);
  nf^.ReadRawBytes(l, Pointer(s)^);
  Dispose(nf, Done);
end;

procedure TPasDoc.ParseFiles;
var
  i: LongInt;
  p: PString;
  InputStream: PFileInputStream;
begin
  if (not Assigned(FileNames)) or (FileNames^.Count < 1) then Exit;
  i := 0;
  repeat
    p := FileNames^.At(i);
    PrintLn(3, 'Loading "' + p^ + '"...');
    InputStream := New(PFileInputStream, Init(p^));
    if (not Assigned(InputStream)) then
      begin
        PrintLn(2, 'Warning: Could not open "' + p^ + '" for reading, skipping.');
      end
    else
      HandleStream(InputStream, p);
    Inc(i);
  until (i = FileNames^.Count);
end;

procedure TPasDoc.PrintHelp;
var
  l: TLanguageID;
begin
  PrintLn(1, '');
  PrintLn(1, 'General:');
  PrintLn(1, ' -?                           Show this screen and terminate');
  PrintLn(1, ' -V                           Show program version and terminate');
  PrintLn(1, ' -v<LEVEL>                    Set verbosity to LEVEL (default 2)');
  PrintLn(1, 'Parser:');
  PrintLn(1, ' -R<FILE>                     Read descRiptions from FILE');
  PrintLn(1, ' -D<DIRECTIVE>                Define conditional');
  PrintLn(1, ' -d<FILE>                     Read conditional from FILE');
  // PrintLn(1, ' -f, --directivefile FILE     Add directives from FILE');
  PrintLn(1, ' -I<PATH>                     Search for include files in PATH');
  PrintLn(1, ' -P                           Include private fields, methods & properties');
  PrintLn(1, ' -S<FILE>                     Read Source file names from FILE');
  PrintLn(1, 'Documentation:');
  PrintLn(1, '  -C<FILE>                    Read Contents for HtmlHelp from FILE');
  PrintLn(1, '  -F<FILE>                    Include FILE as footer in output');
  PrintLn(1, '  -H<FILE>                    Include FILE as header in output');
  PrintLn(1, '  -N<NAME>                    Name for documentation or HtmlHelp project');
  PrintLn(1, ' -O<OUTPUTFORMAT>             Specify Output format');
  PrintLn(1, '    Html                        Output plain Html');
  PrintLn(1, '    HtmlHelp                    Output Html & HtmlHelp project');
  PrintLn(1, '  -T<TITLE>                   Title for documentation or HtmlHelp');
  PrintLn(1, ' Language:');
  PrintLn(1, '  -L<LANGUAGE>                Specify output language');
  for l := Low(LANGUAGE_ARRAY) to High(LANGUAGE_ARRAY) do
    PrintLn(1, '    ' + LANGUAGE_ARRAY[l].Syntax + StringOfChar(#32, 28 - Length(LANGUAGE_ARRAY[l].Syntax)) + LANGUAGE_ARRAY[l].Name);
  PrintLn(1, ' Other:');
  PrintLn(1, '  -E<PATH>                    Output doc file(s) to PATH');
  PrintLn(1, '  -X<OPTION>                  Exclude <OPTION> while running');
  PrintLn(1, '     HomePage                   Suppress ' + APP_NAME + ' homepage link in output');

  {$IFDEF WINDOWS}
  { Registry and HCC only exists on Windows }
  PrintLn(1, '     HHC                        Do not invoke HHC.exe when generating HtmlHelp');
  {$ENDIF}

  PrintLn(1, '');
  PrintLn(1, 'Visit the ' + APP_NAME + ' homepage: ' + APP_HOMEPAGE);
end;

procedure TPasDoc.PrintUsage;
begin
  PrintLn(1, APP_FIRST_LINE);
  PrintLn(1, 'Documentation generator for Pascal unit source code files.');
  PrintLn(1, 'Usage: ' + ParamStr(0) + ' [<OPTIONs>] <FILEs>');
end;

procedure TPasDoc.RemoveExcludedItems(var c: PItemCollection);
var
  i: LongInt;
  p: PItem;
begin
  if (not Assigned(c)) then Exit;
  i := 0;
  while (i < c^.Count) do
    begin
      p := c^.At(i);
      if Assigned(p) and Assigned(p^.Description) and
        //(p^.Description^.FindTag ('EXCLUDE', Offs1, Offs2, Offs3)) then
      (p^.Description^.PosCI('@EXCLUDE') >= 0) then
        begin
          PrintLn(3, 'Hint: Excluding item ' + p^.Name);
          c^.AtFree(i);
        end
      else
        begin
          { P has no excluded tag; but if it is a class, interface, object or
            unit, one of its parts may be excluded }
          if (TypeOf(p^) = TypeOf(TCIO)) then
            begin
              RemoveExcludedItems(PCIO(p)^.Fields);
              RemoveExcludedItems(PItemCollection(PCIO(p)^.Properties));
              RemoveExcludedItems(PItemCollection(PCIO(p)^.Methods));
            end
          else
            begin
              if (TypeOf(p^) = TypeOf(TUnit)) then
                begin
                  RemoveExcludedItems(PUnit(p)^.CIO);
                  RemoveExcludedItems(PUnit(p)^.Constants);
                  RemoveExcludedItems(PItemCollection(PUnit(p)^.FuncsProcs));
                  RemoveExcludedItems(PUnit(p)^.Types);
                  RemoveExcludedItems(PUnit(p)^.Variables);
                end;
            end;
          Inc(i);
        end;
    end;
  if (c^.Count = 0) then
    begin
      Dispose(c, Done);
      c := nil;
    end;
end;

procedure TPasDoc.Run;
var
  Generator: PDocGenerator;
  t1, t2: TDateTime;
begin
  t1 := Now;
  HandleParams;
  ParseFiles;
  {P^.SearchDescrFileTags(P^.Units);}
  RemoveExcludedItems(Units);
  { check if parsing was successful }
  if (Units = nil) or (Units^.Count <= 0) then
    begin
      PrintLn(1, 'Error: At least one unit must have been successfully parsed to write docs.');
      Halt(1);
    end;
  PrintLn(3, 'Creating ' + OUTPUT_FORMAT_NAMES[OutputFormat] + ' documentation file(s)...');
  { create desired output generator }
  case OutputFormat of
    OUTPUT_HTML, OUTPUT_HTML_HELP:
      begin
        Generator := New(PHTMLDocGenerator, Init);
        // Additional settings for Html Help
        if OutputFormat = OUTPUT_HTML_HELP then
          with PHTMLDocGenerator(Generator)^ do
            begin
              ContentsFile := Self.ContentsFile;
              HtmlHelp := True;
              {$IFDEF WINDOWS}
              NoHHC := Self.NoHHC;
              {$ENDIF}
            end;
      end;
  else
    begin
      Generator := nil; { to prevent 'uninitialized' warnings }
      PrintLn(1, 'Error: Output format unknown.');
      Halt(1);
    end;
  end;
  { copy some fields from P to Generator }
  Generator^.Header := Header;
  Generator^.Footer := Footer;
  Generator^.Language := Language;
  Generator^.DestDir := DestDir;
  Generator^.NoGeneratorInfo := NoGeneratorInfo;

  if ProjectName <> '' then
    Generator^.ProjectName := ProjectName
  else
    Generator^.ProjectName := Translation[trHelp];

  Generator^.Title := Title;
  Generator^.Units := Units;

  Generator^.BuildLinks;

  Generator^.ExpandDescriptions;
  Generator^.LoadDescriptionFiles(DescriptionFiles);

  // Write Binary Files first, ...
  Generator^.WriteBinaryFiles;
  // ... because WriteDocumentation may need them (i.e. when calling HHC.exe).
  Generator^.WriteDocumentation;

  Dispose(Generator, Done);

  t2 := Now;
  PrintLn(2, 'Done, worked ' + FloatToStrF((t2 - t1) * (24 * 60 * 60), ffFixed, 0, 2) + ' second(s).');
end;

procedure TPasDoc.SearchDescrFileTags(var c: PItemCollection);
var
  Found: Boolean;
  i: LongInt;
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  p: PItem;
  s: AnsiString;
begin
  if (not Assigned(c)) then Exit;
  i := 0;
  while (i < c^.Count) do
    begin
      p := c^.At(i);
      Inc(i);
      if (not Assigned(p)) then Continue;
      if Assigned(p^.Description) then
        begin
          Offs1 := 0;
          repeat
            Found := p^.Description^.FindTag('DESCRFILE', Offs1, Offs2, Offs3);
            if Found then
              begin
                p^.Description^.ExtractTag(Offs1, Offs2, Offs3, s);
                PrintLn(3, 'Adding description file "' + s + '"');
                AddDescriptionFile(s);
                Offs1 := Offs3 + 1;
              end;
          until (not Found);
        end;
      if (TypeOf(p^) = TypeOf(TCIO)) then
        begin
          SearchDescrFileTags(PCIO(p)^.Fields);
          SearchDescrFileTags(PItemCollection(PCIO(p)^.Methods));
          SearchDescrFileTags(PItemCollection(PCIO(p)^.Properties));
        end;
      if (TypeOf(p^) = TypeOf(TUnit)) then
        begin
          SearchDescrFileTags(PUnit(p)^.CIO);
          SearchDescrFileTags(PUnit(p)^.Constants);
          SearchDescrFileTags(PItemCollection(PUnit(p)^.FuncsProcs));
          SearchDescrFileTags(PUnit(p)^.Types);
          SearchDescrFileTags(PUnit(p)^.Variables);
        end;
    end;
end;

procedure TPasDoc.SetLanguage(const NewLanguage: TLanguageID);
begin
  if (Language <> DEFAULT_LANGUAGE) and (Language <> NewLanguage) then
    begin
      PrintLn(1, 'Error - You can only define one language.');
      Halt(1);
    end;
  Language := NewLanguage;
end;

end.

