{
@abstract(defines all items that can appear within a Pascal unit's interface)
@created(11 Mar 1999)
@lastmod(10 Feb 2000)
@author(Marco Schmidt (marcoschmidt@geocities.com))

For each type, variable, class etc. (short: item) that may appear in a Pascal
source code file and can thus be taken into the documentation, this unit
provides an object type which will store name, unit, description and more
on this item.
}
unit Items;

{$I platform.inc}

interface

uses
  Objects,
  Texts;

const
  { indicates field or method is public (default state) }
  STATE_PUBLIC = 0;
  { indicates field or method is protected }
  STATE_PROTECTED = 1;
  { indicates field or method is private }
  STATE_PRIVATE = 2;
  { indicates field or method is published }
  STATE_PUBLISHED = 3;

type
  { pointer to @link(TItemCollection) }
  PItemCollection = ^TItemCollection;
  { pointer to @link(TMethodCollection) }
  PMethodCollection = ^TMethodCollection;
  { pointer to @link(TPropertyCollection) }
  PPropertyCollection = ^TPropertyCollection;
  { pointer to @link(TMethod) }
  PMethod = ^TMethod;
  { pointer to @link(TProperty) }
  PProperty = ^TProperty;
  { pointer to @link(TUnit) }
  PUnit = ^TUnit;
  { pointer to @link(TObjectInfo) }
  PCIO = ^TCIO;
  { pointer to @link(TItem) }
  PItem = ^TItem;
  { basic linkable item in pasdoc hierarchy }
  TItem = object(TObject)
    { unique number of this item }
    AnchorNumber: LongInt;
    { list of strings, each representing one author of this item }
    Authors: PStringCollection;
    { if assigned, contains string with date of creation }
    Created: PString;
    { description of this item, a single sentence }
    Description: PText;
    { more detailed description of this item, mostly more than one
      sentence }
    DetailedDescription: PText;
    { a full link that should be enough to link this item from anywhere else }
    FullLink: string[127];
    { if assigned, contains string with date of last modification }
    LastMod: PString;
    { if this item is part of an object or class, the corresponding
      info object is stored here, nil otherwise }
    MyObject: PCIO;
    { pointer to unit this item belongs to }
    MyUnit: PUnit;
    { name of the item }
    Name: string;
    { One of the STATE_xxx constants, determines access rights
      (public, private, etc.). }
    State: Byte;
    { }
    destructor Done; virtual;
    function FindItem(ItemName: string): PItem; virtual;
    { }
    function FindName(s1, s2, S3: string; n: Integer): PItem; virtual;
    { Returns DetailedDescription if available, otherwise Description,
      otherwise nil. }
    function GetDescription: PText;
    { Searches for an abstract tag within the Description field of
      this item. If one is found, Description is copied to DetailedDescription
      and the abstract tag becomes the new Description. This procedure
      should be called after the dates (created and lastmod) and the
      author tags have been handled, as they are searched in Description. }
    { }
    procedure HandleAbstractTag;
    { }
    procedure HandleAuthorTags;
    { }
    procedure HandleCreatedTag;
    { }
    procedure HandleLastModTag;
    { Returns true if there is a detailled or a normal description available. }
    function HasDescription: Boolean;
    { Inserts an item into a collection.
   Creates collection if it does not exist already. }
    procedure InsertItem(Item: PItem; var c: PItemCollection);
    { inserts a method into a collection; creates collection if necessary }
    procedure InsertMethod(Method: PMethod; var c: PMethodCollection);
    { inserts a property into a collection; creates collection if necessary }
    procedure InsertProperty(Prop: PProperty; var c: PPropertyCollection);
  end;
  { extends @link(TItem) to store method and function-/procedure-specific
    information }
  TMethod = object(TItem)
    { full declaration, including parameter list and procedural directives }
    FullDeclaration: string;
    { 0=constructor 1=destructor 2=function or procedure }
    What: Byte;
  end;

  TProperty = object(TItem)
    { full declaration, including read/write and storage specifiers }
    FullDeclaration: string;
    { contains the optional index declaration, including brackets }
    IndexDecl: string;
    { contains the type of the property }
    Proptype: string;
    { read specifier }
    Reader: string;
    { write specifier }
    Writer: string;
    { true if the property is the default property }
    Default: Boolean;
    { keeps default value specifier }
    DefaultID: string;
    { true if Nodefault property }
    NoDefault: Boolean;
    { keeps Stored specifier }
    StoredId: string;
    { Get a string that gives a short declaration, just giving the name and
      the type, optionally the index. }
  {  function GetDeclaration : String;}
    { Get a string that describes the specifiers such as read-only,
      write-only etc. If Detailed is true, then the values are
      specified too. }
  {  function GetSpecifiers(Detailed : Boolean) : String;}
  end;
  { enumeration type to determine type of TObjectInfo item: class,
    interface or object }
  TCIOType = (CIO_CLASS, CIO_DISPINTERFACE, CIO_INTERFACE, CIO_OBJECT);
  { Extends @link(TItem) to store all items in a class / an object, e.g.
    fields. }
  TCIO = object(TItem)
    { name of the ancestor class / object }
    Ancestors: PStringCollection;
    { list of all fields }
    Fields: PItemCollection;
    { list of all methods }
    Methods: PMethodCollection;
    { determines if this is a class, an interface or an object }
    MyType: TCIOType;
    { name of documentation output file (if each class / object gets
      its own file, that's the case for HTML, but not for TeX) }
    OutputFileName: string;
    { list of properties }
    Properties: PPropertyCollection;
    { Simply returns the result of a call to @link(FindFieldMethodProperty). }
    function FindItem(ItemName: string): PItem; virtual;
    { If this class (or interface or object) contains a field, method or
      property with the name of ItemName, the corresponding item pointer is
      returned. }
    function FindFieldMethodProperty(ItemName: string): PItem;
    {function GetType: Char;}
  end;
  { extends @link(TItem) to store anything about a unit, its constants,
    types etc.; also provides methods for parsing a complete unit }
  TUnit = object(TItem)
    { list of classes and objects defined in this unit }
    CIO: PItemCollection;
    { list of constants defined in this unit }
    Constants: PItemCollection;
    { list of functions and procedures defined in this unit }
    FuncsProcs: PMethodCollection;
    { name of documentation output file
      THIS SHOULD NOT BE HERE! }
    OutputFileName: string;
    { the names of all units mentioned in a uses clause in the interface
      section of this unit }
    UsesUnits: PStringCollection;
    { list of classes and objects defined in this unit }
    SourceFileName: string;
    { list of types defined in this unit }
    Types: PItemCollection;
    { list of variables defined in this unit }
    Variables: PItemCollection;
    { create this object with given input stream and file name }
    constructor Init;
    { dispose of all dynamically allocated memory in this object }
    destructor Done; virtual;
    procedure AddCIO(i: PCIO);
    procedure AddConstant(i: PItem);
    procedure AddType(i: PItem);
    procedure AddVariable(i: PItem);
    function FindFieldMethodProperty(s1, s2: string): PItem;
    function FindItem(ItemName: string): PItem; virtual;
    procedure RemovePrivateItems;
  end;
  { extends @link(TSortedCollection) to store a list of items }
  TItemCollection = object(TSortedCollection)
    { Moves all elements from C to this object, but does not
      dispose of the - empty - C }
    procedure AddItems(var c: PItemCollection);
    { Compares two @link(TItem) objects by their @link(TItem.Name), @link(TItem.MyObject) and
      @link(TItem.MyUnit) fields. }
    function Compare(Key1, Key2: Pointer): Integer; virtual;
    { Copies all Items from c to this object, not changing c at all. }
    procedure CopyItems(const c: PItemCollection);
    { Counts classes, interfaces and objects within this collection. }
    procedure CountCIO(var c, i, o: LongInt);
    { Compares each element's name field with Name and returns the item on
      success, nil otherwise.
      Name's case is not regarded. }
    function FindName(Name: string): PItem;
    { Inserts all items of C into this collection.
      Disposes C and sets it to nil. }
    procedure InsertItems(var c: PItemCollection);
    { Checks each element's State field and removes all elements with a value
      of STATE_PRIVATE. }
    procedure RemovePrivateItems;
  end;
  { extends @link(TItemCollection) by defining a different compare function: two
    methods are first compared by their @link(TMethod.What) field, which determines
    whether they are constructor, destructor or function / procedure }
  TMethodCollection = object(TItemCollection)
    { compares two method objects (it is assumed Key1 and
      Key2 are of type PMethod) }
    function Compare(Key1, Key2: Pointer): Integer; virtual;
  end;
  { a TPropertyCollection holds a collection of properties. It introduces no
    new methods when compared to @link(TITemCollection), but this may be
    implemented in a later stage }
  TPropertyCollection = object(TItemCollection)
  end;

  { If C is assigned and C^.Count is zero, the collection's destructor is
    called and C is set to nil.
    If C is nil, nothing happens. }
procedure FreeEmptyCollection(var c: PItemCollection);

implementation

uses
  SysUtils,

  Arrays,
  Msg,
  Numbers;

procedure FreeEmptyCollection(var c: PItemCollection);
begin
  if (not Assigned(c)) then Exit;
  if (c^.Count < 1) then
    begin
      Dispose(c, Done);
      c := nil;
    end;
end;

{ TItem }

destructor TItem.Done;
begin
  if Assigned(Authors) then Dispose(Authors, Done);
  if Assigned(Created) then Objects.DisposeStr(Created);
  if Assigned(Description) then Dispose(Description, Done);
  if Assigned(DetailedDescription) then Dispose(DetailedDescription, Done);
  if Assigned(LastMod) then Objects.DisposeStr(LastMod);
  inherited Done;
end;

function TItem.FindItem(ItemName: string): PItem;
begin
  FindItem := nil;
end;

function TItem.FindName(s1, s2, S3: string; n: Integer): PItem;
var
  p: PItem;
begin
  Result := nil;

  case n of
    0:
      begin
        if Assigned(MyObject) then
          begin { this item is a method or field }
            p := MyObject^.FindFieldMethodProperty(s1);
            if Assigned(p) then
              begin
                Result := p;
                Exit;
              end;
          end;

        if Assigned(MyUnit) then
          begin
            p := MyUnit^.FindItem(s1);
            if Assigned(p) then
              begin
                Result := p;
                Exit;
              end;
          end;

        if Assigned(MyUnit) and (CompareStringsCI(s1, MyUnit^.Name) = 0) then
          begin
            Result := MyUnit;
            Exit;
          end;

      end;

    1:
      begin

        if Assigned(MyObject) then
          begin
            if (CompareStringsCI(MyObject^.Name, s1) = 0) then
              begin
                p := MyObject^.FindFieldMethodProperty(s2);
                if Assigned(p) then
                  begin
                    Result := p;
                    Exit;
                  end;
              end;
          end;

        // RJ: To find links in Unit's objects!
        if Assigned(MyUnit) then
          begin
            p := MyUnit^.FindFieldMethodProperty(s1, s2);
            if Assigned(p) then
              begin
                Result := p;
                Exit;
              end;
          end;

      end;

  end;

end;

function TItem.GetDescription: PText;
begin
  if Assigned(DetailedDescription) then
    Result := DetailedDescription
  else
    begin
      if Assigned(Description) then
        Result := Description
      else
        Result := nil;
    end;
end;

{ ---------- }

procedure TItem.HandleAbstractTag;
var
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  s: AnsiString;
  p: PText;
begin
  if (not Assigned(Description)) then Exit;
  Offs1 := 0;

  while Offs1 < Description^.Content do
    begin
      if (Description^.Data[Offs1] = '@') and
        Description^.FindTag('ABSTRACT', Offs1, Offs2, Offs3) then
        begin
          Description^.ExtractTag(Offs1, Offs2, Offs3, s);
          if (Length(s) <= 0) then Continue;
          p := New(PText, Init);
          p^.AppendString(s);
          DetailedDescription := Description;
          Description := p;
          Exit;
        end;
      Inc(Offs1);
    end;
end;

{ ---------- }

procedure TItem.HandleAuthorTags;
var
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  p: PString;
  s: AnsiString;
begin
  if not Assigned(Description) then Exit;
  Offs1 := 0;
  { we could have more than one author, so repeat until we have all }
  while Offs1 < Description^.Content do
    begin
      if (Description^.Data[Offs1] = '@') and
        Description^.FindTag('AUTHOR', Offs1, Offs2, Offs3) then
        begin
          { we found one, remove it from the description and add it to the author list }
          Description^.ExtractTag(Offs1, Offs2, Offs3, s);
          if (Length(s) <= 0) then Continue;
          if (not Assigned(Authors)) then
            Authors := New(PUnSortedStringCollection, Init(16, 16));
          p := Objects.NewStr(s);
          Authors^.Insert(p)
        end;
      Inc(Offs1);
    end;
end;

procedure TItem.HandleCreatedTag;
var
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  s: ansistring;
begin
  if not Assigned(Description) then Exit;
  Offs1 := 0;
  while Offs1 < Description^.Content do
    begin
      if (Description^.Data[Offs1] = '@') and
        Description^.FindTag('CREATED', Offs1, Offs2, Offs3) then
        begin
          Description^.ExtractTag(Offs1, Offs2, Offs3, s);
          Created := Objects.NewStr(s);
          Exit;
        end;
      Inc(Offs1);
    end;
end;

procedure TItem.HandleLastModTag;
var
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  s: ansistring;
begin
  if (not Assigned(Description)) then Exit;
  Offs1 := 0;
  while Offs1 < Description^.Content do
    begin
      if (Description^.Data[Offs1] = '@') and
        Description^.FindTag('LASTMOD', Offs1, Offs2, Offs3) then
        begin
          Description^.ExtractTag(Offs1, Offs2, Offs3, s);
          LastMod := Objects.NewStr(s);
          Exit;
        end;
      Inc(Offs1);
    end;
end;

function TItem.HasDescription: Boolean;
begin
  HasDescription := (Assigned(Description) or Assigned(DetailedDescription));
end;

procedure TItem.InsertItem(Item: PItem; var c: PItemCollection);
begin
  if (not Assigned(Item)) then Exit;
  if (not Assigned(c)) then c := New(PItemCollection, Init(16, 16));
  c^.Insert(Item);
end;

procedure TItem.InsertMethod(Method: PMethod; var c: PMethodCollection);
begin
  if (not Assigned(Method)) then Exit;
  if (not Assigned(c)) then
    begin
      c := New(PMethodCollection, Init(16, 16));
      c^.Duplicates := True;
    end;
  c^.Insert(Method);
end;

procedure TItem.InsertProperty(Prop: PProperty; var c: PPropertyCollection);

begin
  if (not Assigned(Prop)) then Exit;
  if (not Assigned(c)) then
    c := New(PPropertyCollection, Init(16, 16));
  c^.Insert(Prop);
end;

{ TItemCollection }

procedure TItemCollection.AddItems(var c: PItemCollection);
var
  i: LongInt;
  p: Pointer;
begin
  if (not Assigned(c)) or (c^.Count < 1) then Exit;
  for i := 0 to c^.Count - 1 do
    begin
      p := c^.At(i);
      Insert(p);
    end;
  c^.DeleteAll;
end;

function TItemCollection.Compare(Key1, Key2: Pointer): Integer;
begin
  Result := CompareText(PItem(Key1)^.Name, PItem(Key2)^.Name);
  if Result = 0 then
    if Assigned(PItem(Key1)^.MyObject) and Assigned(PItem(Key2)^.MyObject) then
      Result := CompareText(PItem(Key1)^.MyObject^.Name, PItem(Key2)^.MyObject^.Name);
  if Result = 0 then
    if Assigned(PItem(Key1)^.MyUnit) and Assigned(PItem(Key2)^.MyUnit) then
      Result := CompareText(PItem(Key1)^.MyUnit^.Name, PItem(Key2)^.MyUnit^.Name)
end;

procedure TItemCollection.CopyItems(const c: PItemCollection);
var
  i: LongInt;
begin
  if c = nil then Exit;
  for i := 0 to c^.Count - 1 do
    Insert(c^.At(i));
end;

procedure TItemCollection.CountCIO(var c, i, o: LongInt);
var
  j: LongInt;
  p: PCIO;
begin
  c := 0;
  i := 0;
  o := 0;
  if (Count < 1) then Exit;
  j := 0;
  repeat
    p := At(j);
    case p^.MyType of
      CIO_CLASS: Inc(c);
      CIO_INTERFACE: Inc(i);
      CIO_OBJECT: Inc(o);
    end;
    Inc(j);
  until (j = Count);
end;

function TItemCollection.FindName(Name: string): PItem;
var
  i: Integer;
  Item: PItem;
begin
  if (Count < 1) then
    begin
      Result := nil;
      Exit;
    end;

  for i := 0 to Count - 1 do
    begin
      Item := At(i);
      if Assigned(Item) then
        begin
          if (Arrays.CompareStringsCI(Name, Item^.Name) = 0) then
            begin
              Result := Item;
              Exit;
            end;
        end;
    end;
  Result := nil;
end;

procedure TItemCollection.InsertItems(var c: PItemCollection);
var
  i: LongInt;
begin
  if (not Assigned(c)) or (c^.Count < 1) then Exit;
  i := c^.Count - 1;
  repeat
    Insert(c^.At(i));
    c^.atDelete(i);
    Dec(i);
  until (i = -1);
end;

procedure TItemCollection.RemovePrivateItems;
var
  i: LongInt;
  Item: PItem;
begin
  i := 0;
  while (i < Count) do
    begin
      Item := At(i);
      if Assigned(Item) and (Item^.State = STATE_PRIVATE) then
        begin
          atFree(i);
        end
      else
        Inc(i);
    end;
end;

{
function TItemCollection.SplitLink(S: String; var S1, S2, S3: String; var N: LongInt): Boolean;
var
  I: LongInt;
begin
  SplitLink := false;
  S1 := '';
  S2 := '';
  S3 := '';
  N := 0;
  I := 1;
  while (I <= Length(S)) and (not (S[I] in IdentifierStart)) do
    Inc(I);
  if (I > Length(S))
  then exit;
  if (I > 1)
  then System.Delete(S, 1, I - 1);
  I := 2;
  while (I <= Length(S)) and (S[I] in IdentifierOther) do
    Inc(I);
  System.Delete(S, I - 1, Length(S) - I + 1);
  I := System.Pos('.', S);
  SplitLink := true;
  if (I = 0)
  then
    begin
      S1 := S;
      N := 1;
      exit;
    end;
  S1 := System.Copy(S, 1, I - 1);
  System.Delete(S, 1, I);
  I := System.Pos('.', S);
end;
}

function TCIO.FindItem(ItemName: string): PItem;
begin
  FindItem := FindFieldMethodProperty(ItemName);
end;

function TCIO.FindFieldMethodProperty(ItemName: string): PItem;
var
  c: PItemCollection;
  i: Integer;
  Item: PItem;
begin
  for i := 0 to 2 do
    begin
      case i of
        0: c := Fields;
        1: c := Methods;
        2: c := Properties;
      else
        c := nil;
      end;
      if Assigned(c) then
        begin
          Item := c^.FindName(ItemName);
          if Assigned(Item) then
            begin
              FindFieldMethodProperty := Item;
              Exit;
            end;
        end;
    end;
  FindFieldMethodProperty := nil;
end;
{
function TCIO.GetType: Char;
begin
  case MyType of
    CIO_CLASS: GetType := 'C';
    CIO_INTERFACE: GetType := 'I';
    CIO_OBJECT: GetType := 'O';
    else GetType := '?';
  end;
end;
}
{ TMethodCollection }

function TMethodCollection.Compare(Key1, Key2: Pointer): Integer;
var
  p1: PMethod;
  p2: PMethod;
  r: Integer;
begin
  p1 := PMethod(Key1);
  p2 := PMethod(Key2);
  { compare 'method type', order is constructor > destructor > function, procedure }
  r := Numbers.CompareUInt8(p1^.What, p2^.What);
  if (r <> 0) then
    begin
      Compare := r;
      Exit;
    end;
  { if 'method type' is equal, compare names }
  Compare := Arrays.CompareStringsCI(p1^.Name, p2^.Name);
end;

{ TUnit }

constructor TUnit.Init;
begin
  inherited Init;
  {  Stream := New(PUnitStream, Init(InputStream));}
  {  SourceFileName := FileName;}
  Types := nil;
  Constants := nil;
  Variables := nil;
  FuncsProcs := nil;
  CIO := nil;
end;

destructor TUnit.Done;
begin
  {  if Assigned(Stream)
    then Dispose(Stream, Done);}
  if Assigned(Types) then Dispose(Types, Done);
  inherited Done;
end;

procedure TUnit.AddCIO(i: PCIO);
begin
  if (not Assigned(CIO)) then
    begin
      CIO := New(PItemCollection, Init(16, 16));
      if (not Assigned(CIO)) then Exit;
    end;
  CIO^.Insert(i);
end;

procedure TUnit.AddConstant(i: PItem);
begin
  if (not Assigned(Constants)) then
    begin
      Constants := New(PItemCollection, Init(16, 16));
      if (not Assigned(Constants)) then
        begin
          PrintLn(1, 'Error - could not create list of constants.');
          Halt(1);
        end;
    end;
  Constants^.Insert(i);
end;

procedure TUnit.AddType(i: PItem);
begin
  if (not Assigned(Types)) then
    begin
      Types := New(PItemCollection, Init(16, 16));
      if (not Assigned(Types)) then
        begin
          PrintLn(1, 'Error - could not create list of types.');
          Halt(1);
        end;
    end;
  Types^.Insert(i);
end;

procedure TUnit.AddVariable(i: PItem);
begin
  if (not Assigned(Variables)) then
    begin
      Variables := New(PItemCollection, Init(16, 16));
      if (not Assigned(Variables)) then Exit;
    end;
  Variables^.Insert(i);
end;

function TUnit.FindFieldMethodProperty(s1, s2: string): PItem;
var
  PI: PItem;
  po: PCIO;
begin
  FindFieldMethodProperty := nil;
  if (not Assigned(CIO)) then Exit;
  po := PCIO(CIO^.FindName(s1));
  if Assigned(po) then
    begin
      PI := po^.FindFieldMethodProperty(s2);
      if Assigned(PI) then FindFieldMethodProperty := PI;
    end;
end;

function TUnit.FindItem(ItemName: string): PItem;
var
  i: Integer;
  Item: PItem;
  CioItem: PCIO;
begin
  Result := nil;
  Item := nil;
  if (CompareStringsCI(Name, ItemName) = 0) then
    begin
      Result := @Self;
      Exit;
    end;
  if Assigned(Constants) then Item := Constants^.FindName(ItemName);
  if Assigned(Item) then
    begin
      Result := Item;
      Exit;
    end;

  if Assigned(Types) then
    Item := Types^.FindName(ItemName);
  if Assigned(Item) then
    begin
      Result := Item;
      Exit;
    end;

  if Assigned(Variables) then
    Item := Variables^.FindName(ItemName);
  if Assigned(Item) then
    begin
      Result := Item;
      Exit;
    end;

  if Assigned(FuncsProcs) then
    Item := FuncsProcs^.FindName(ItemName);
  if Assigned(Item) then
    begin
      Result := Item;
      Exit;
    end;

  if Assigned(CIO) then
    Item := CIO^.FindName(ItemName);
  if Assigned(Item) then
    begin
      Result := Item;
      Exit;
    end;

  if Assigned(CIO) then
    for i := 0 to CIO^.Count - 1 do
      begin
        CioItem := CIO^.At(i);
        if Assigned(CioItem) then
          Result := CioItem^.FindFieldMethodProperty(ItemName);
        if Result <> nil then Exit;
      end;

end;

procedure TUnit.RemovePrivateItems;
var
  i: LongInt;
  p: PCIO;
begin
  {  WriteLn('Removing private items of unit ', Name);}
  if (not Assigned(CIO)) then Exit;
  i := 0;
  while (i < CIO^.Count) do
    begin
      p := PCIO(CIO^.At(i));

      if Assigned(p^.Fields) then p^.Fields^.RemovePrivateItems;
      FreeEmptyCollection(p^.Fields);

      if Assigned(p^.Methods) then p^.Methods^.RemovePrivateItems;
      FreeEmptyCollection(PItemCollection(p^.Methods));

      if Assigned(p^.Properties) then p^.Properties^.RemovePrivateItems;
      FreeEmptyCollection(PItemCollection(p^.Properties));

      Inc(i);
    end;
end;

{ TProperty }

{Function TProperty.GetDeclaration : String;

Var T : String;

begin
  T:='Property '+Name;
  If Length(IndexName)<>0 Then
  T:=T+'['+IndexName+':'+IndexType+']';
  T:=T+ ' : '+Proptype+';';
  GetDeclaration:=T;
end;

Function TProperty.GetSpecifiers (Detailed : Boolean) : String;

Var S : String;

Procedure addto (Const T : String);

begin
  If Length(S)>0 then
    S:=S+', '+T
  else
    S:=T;
end;

begin
  S:='';
  If Length(Reader)=0 Then
    begin
    If Not detailed then
     AddTo('Write Only')
    end
  else
    If Detailed then
       AddTo('Read '+Reader);
  if Length(Writer)=0 Then
    begin
    If not detailed then
      AddTo('Read Only')
    end
  else
    If Detailed then
      AddTo('Write '+Writer);
  If Default then
    AddTo('Default property');
  If NoDefault then
    AddTo('NoDefault');
  If Length(DefaultID)<>0 then
  If Not Detailed then
    AddTo('Has default')
  else
    AddTo('default '+DefaultId);
  If Length (StoredID)<>0 then
    If Detailed then
      addTo('Stored '+StoredId)
    else
      AddTo ('Is stored');
  If S<>'' then
    S:='('+S+')';
  GetSpecifiers:=S;
end;
}

end.

