{
@abstract(provides HTML doc generator object)
@author(Ralf Junker (delphi@zeitungsjunge.de))
@author(Wim van der Vegt (wvd_vegt@knoware.nl))
@author(Erwin Scheuch-Heilig (ScheuchHeilig@t-online.de))
@author(Marco Schmidt (marcoschmidt@geocities.com))
@author(Alexander Lisnevsky (alisnevsky@yandex.ru))
@created(1 Sep 1998)
@lastmod(07 May 2002)

Implements an object to generate HTML documentation, overriding many of
@link(TDocGenerator)'s virtual methods.
}

unit Html;

{$I platform.inc}

interface

uses
  AppInfo,
  Arrays,
  FileStre,
  GenDoc,
  Items,
  Languages,
  Numbers,
  Objects,
  Parsing,
  Streams,
  Texts,
  Time;

type
  THTMLColorIndex = (
    HTML_BACKGROUND,
    HTML_TEXT,
    HTML_LINK,
    HTML_VLINK,
    HTML_ALINK,
    HTML_TABLE_BACKGROUND,
    HTML_TABLE_TEXT);

type
  { pointer to @link(THTMLDocGenerator) }
  PHTMLDocGenerator = ^THTMLDocGenerator;
  { @abstract(generates HTML documentation)
    Extends @link(TDocGenerator) and overwrites many of its methods to generate
    output in HTML (HyperText Markup Language) format.
    This type of output is well suited to be read with a web browser at the
    computer, as a reference manual that does not have to be printed.
    For printed output, use @link(Tex.TTexDocGenerator). }
  THTMLDocGenerator = object(TDocGenerator)
    { Contains Name of a file to read HtmlHelp Contents from.
      If empty, create default contents file. }
    ContentsFile: AnsiString;
    { If True, generate Html Help project files. }
    HtmlHelp: Boolean;
    { True if not to call HCC.exe if creating HtmlHelp output.
      Otherwise, rjPasDoc will look for HCC.exe in the registry and
      compile the project.  }
    NoHHC: Boolean;
    { Makes a string look like a coded string, i.e. <CODE>TheString</CODE>
      in Html. }
    function CodeString(const s: AnsiString): AnsiString; virtual;
    { Converts C to HTML, simply copying normal characters and converting
      special characters like the ampersand to escape sequences. }
    function ConvertChar(c: Char): string; virtual;
    { Returns a link to an anchor within a document. HTML simply concatenates
      the strings with a "#" character between them. }
    function CreateLink(Item: PItem): string; virtual;
    { Creates a valid HTML link, starting with an anchor that points to Link,
      encapsulating the text ItemName in it. }
    function CreateReferencedLink(ItemName, Link: string): string; virtual;
    function ExistsFullPath(s: string): Boolean;
    { Returns HTML file extension ".htm". }
    function GetFileExtension: string; virtual;
    { Writes information on doc generator to current output stream,
      including link to pasdoc homepage. }
    procedure WriteAppInfo;
    { Writes authors to output, at heading level HL. Will not write anything
      if collection of authors is not assigned or empty. }
    procedure WriteAuthors(HL: Byte; Authors: PStringCollection);
    { Writes a single class, interface or object CIO to output, at heading
      level HL. }
    procedure WriteCIO(HL: Byte; const CIO: PCIO); virtual;
    { Calls @link(WriteCIO) with each element in the argument collection C,
      using heading level HL. }
    procedure WriteCIOs(HL: Byte; c: PItemCollection); virtual;
    procedure WriteCIOSummary(HL: Byte; c: PItemCollection); virtual;
    procedure WriteCodeWithLinks(const p: PItem; const Code: string; const ItemLink: AnsiString);
    { Writes a comment S to output file. }
    procedure WriteComment(const s: string); virtual;
    { Writes dates Created and LastMod at heading level HL to output
      (if at least one the two has a value assigned). }
    procedure WriteDates(HL: Byte; Created, LastMod: PString); virtual;
    { The method that does everything - writes documentation for all units
      and creates overview files. }
    procedure WriteDocumentation; virtual;
    { Writes the beginning of the HTML document, including opening HTML element,
      a complete HEAD element and an opening BODY element.
      See @link(WriteEndOfDocument). }
    procedure WriteDocumentHeadline; virtual;
    { Writes an empty table cell, '&nbsp;'. }
    procedure WriteEmptyCell;

    procedure WriteEndOfCode; virtual;
    { Writes the end of an HTML anchor, '</A>'. }
    procedure WriteEndOfAnchor;
    { See @link(WriteDocumentHeadline). }
    procedure WriteEndOfDocument;
    procedure WriteEndOfLink;
    { Finishes an HTML paragraph element by writing a closing P tag. }
    procedure WriteEndOfParagraph;
    { Finishes an HTML table cell by writing a closing TD tag. }
    procedure WriteEndOfTableCell;
    { Finishes an HTML table by writing a closing TABLE tag. }
    procedure WriteEndOfTable;
    { Finishes an HTML table row by writing a closing TR tag. }
    procedure WriteEndOfTableRow;
    procedure WriteFields(const Order: Byte; const Fields: PItemCollection);
    procedure WriteFooter;
    { Writes information on functions and procedures or methods of a unit or
      class, interface or object to output.
      If argument Methods is true, they will be considered methods of a class,
      interface or object, otherwise they're considered functions or procedures
      of a unit.
      The functions are stored in the FuncsProcs argument. }
    procedure WriteFuncsProcs(HL: Byte; Methods: Boolean; FuncsProcs: PMethodCollection); virtual;
    { Writes heading S to output, at heading level I.
      For HTML, only levels 1 to 6 are valid, so that values smaller
      than 1 will be set to 1 and arguments larger than 6 are set to 6.
      The string S will then be enclosed in an element from H1 to H6,
      according to the level. }
    procedure WriteHeading(i: Byte; const s: string); virtual;
    { Reads the default HTML Images from the rjPasDoc executable and writes
      them to the Output Directory. Existing files will not be overwritten. }
    procedure WriteBinaryFiles; virtual;
    procedure WriteItemDescription(Item: PItem);
    { Writes the Item's DetailedDescription. If the Item also has Discription
      (extracted from @@abstract), this is written to a separate paragraph
      in front of the DetailedDescription. }
    procedure WriteItemDetailedDescription(Item: PItem);
    procedure WriteItems(Order: Byte; Heading: string; const Anchor: string; i: PItemCollection); virtual;
    procedure WriteOverviewFiles;
    procedure WriteParagraph(HL: Byte; s: string; t: PText);
    procedure WriteProperties(HL: Byte; p: PPropertyCollection); virtual;
    procedure WritePropertiesSummary(HL: Byte; p: PPropertyCollection); virtual;
    { Writes an opening A element, including a name attribute given by the
      argument. }
    procedure WriteStartOfAnchor(const Name: string);
    procedure WriteStartOfCode; virtual;
    procedure WriteStartOfDocument(Name: string); virtual;
    procedure WriteStartOfLink(const Name: string);
    { Starts an HTML paragraph element by writing an opening P tag. }
    procedure WriteStartOfParagraph;
    procedure WriteStartOfTableCell;
    procedure WriteStartOfTable1Column(t: string);
    procedure WriteStartOfTable2Columns(t1, t2: string);
    procedure WriteStartOfTable3Columns(t1, t2, T3: string);
    procedure WriteStartOfTableRow;
    { Writes the topic files for Html Help Generation }
    procedure WriteHtmlHelpProject;

    procedure WriteUnit(HL: Byte; U: PUnit); virtual;
    procedure WriteUnitDescription(HL: Byte; U: PUnit); virtual;
    { Creates an output stream that lists up all units and short descriptions. }
    procedure WriteUnitOverviewFile;
    { Writes a cell into a table row with the Item's visibility image. }
    procedure WriteVisibilityCell(const Item: PItem);

  end;

const
  { background color of a table header row; a light gray slightly darker
    than the light gray of @link(HTML_HEADER_BACKGROUND_COLOR)  }
  HTML_ROW_BACKGROUND_COLOR: string[6] = 'efefef';
  { background color of a normal table row; a light gray slightly lighter
    than the light gray of @link(HTML_ROW_BACKGROUND_COLOR) }
  HTML_HEADER_BACKGROUND_COLOR: string[6] = 'e0e0e0';

  HTML_TABLE_CELLPADDING = '4';
  HTML_TABLE_CELLSPACING = '2';

implementation

uses
  {$IFNDEF PPC_KYLIX}
  Windows,
  ShellApi,
  Registry,
  {$ELSE}
  Types,
  {$ENDIF}
  SysUtils,
  Classes,

  Msg;
{
html things to be customized
- standard background color (white)
- background color for table headings and overview list at the top of each file (light gray)
- background color for normal table cells (light gray, slightly lighter than the above)
- standard foreground color (black)
- unused link color (blue)
- used link color (purple)
- link color while being clicked on (red)
- normal font (Times Roman)
- heading font (Helvetica)
- code font (Courier)
}

function THTMLDocGenerator.CodeString(const s: AnsiString): AnsiString;
begin
  Result := '<CODE>' + s + '</CODE>';
end;

function THTMLDocGenerator.ConvertChar(c: Char): string;
begin
  { We do not convert standard Html characters <>&" to allow Html code in comments.
    To insert Html characters literally, use @<, @>, @&, @" instead. }
  // '<': Result := '&lt;';              { less than }
  // '>': Result := '&gt;';              { greater than }
  // '&': Result := '&amp;';             { ampersand }
  // '"': Result := '&quot;';            { quote }

  case Language of
    lgRussian_1251, lgRussian_866, lgRussian_koi8:
      begin
        // No conversions for Russion character sets.
        Result := c;
      end;
  else
    // Character translations for all other languages not covered above.
    // Currently covers German Umlauts.
    case c of
      #$C4: Result := '&Auml;'; {  (large Umlaut A) }
      #$D6: Result := '&Ouml;'; {  (large Umlaut O) }
      #$DC: Result := '&Uuml;'; { U (large Umlaut U) }
      #$DF: Result := '&szlig;'; {  (sz) }
      #$E4: Result := '&auml;'; {  (small Umlaut a) }
      #$F6: Result := '&ouml;'; {  (small Umlaut o) }
      #$FC: Result := '&uuml;'; {  (small Umlaut u) }
      { more special characters to be added here }
    else
      { otherwise, simply return the argument character }
      Result := c;
    end;
  end;
end;

function THTMLDocGenerator.CreateLink(Item: PItem): string;
var
  i: Integer;
begin
  Result := '';
  if (not Assigned(Item)) then exit;
  if Assigned(Item^.MyUnit) then
    begin
      if Assigned(Item^.MyObject) then
        begin
          { it's a method, a field or a property - only those have MyObject initialized }
          // Result := Item^.MyObject^.Name + GetFileExtension + '#' + AnchorToString(Item^.AnchorNumber);
          Result := Item^.MyObject^.FullLink + '#' + AnchorToString(Item^.AnchorNumber);
        end
      else
        begin
          if (TypeOf(TCIO) = TypeOf(Item^)) then
            begin
              { it's an object / a class }
              // Result := Item^.Name + GetFileExtension;
              Result := Item^.Name + GetFileExtension;
              i := 0;
              while ExistsFullPath(Result) do
                begin
                  Inc(i);
                  Result := Item^.Name + IntToStr(i) + GetFileExtension;
                end;
            end
          else
            begin
              { it's a constant, a variable, a type or a function / procedure }
              Result := Item^.MyUnit^.FullLink + '#' + AnchorToString(Item^.AnchorNumber);
            end;
        end;
    end
  else
    begin
      { it's a unit - only units don't have a MyUnit pointer }
      Result := Item^.Name + GetFileExtension;
    end;
end;

function THTMLDocGenerator.CreateReferencedLink(ItemName, Link: string): string;
begin
  Result := CodeString('<A href="' + Link + '">' + ItemName + '</A>');
end;

function THTMLDocGenerator.ExistsFullPath(s: string): Boolean;
var
  i, j: LongInt;
  CO: PCIO;
  U: PUnit;
begin
  Result := False;

  if (not Assigned(Units)) or (Units^.Count < 1) then
    exit;

  for i := 0 to Units^.Count - 1 do
    begin
      U := Units^.At(i);
      Result := CompareText(U^.FullLink, s) = 0;
      if Result then exit;

      if Assigned(U^.CIO) and (U^.CIO^.Count > 0) then
        begin
          for j := 0 to U^.CIO^.Count - 1 do
            begin
              CO := U^.CIO^.At(j);
              Result := CompareText(CO^.FullLink, s) = 0;
              if Result then exit;
            end;
        end;
    end;
end;

function THTMLDocGenerator.GetFileExtension: string;
begin
  { '.html' makes DOS version unhappy - welcome to the past ;-) }
  GetFileExtension :=
    {$IFDEF OS_DOS}
  '.htm';
  {$ELSE}
  '.htm';
  {$ENDIF}
end;

{
function THTMLDocGenerator.GetLink(ItemReference, ItemName: String): String;
begin
  GetLink := '<A href="' + ItemReference + '">' + ItemName + '</A>';
end;
}

procedure THTMLDocGenerator.WriteAppInfo;
var
  t: TTime;
begin
  { check if user does not want a link to the pasdoc homepage }
  if NoGeneratorInfo then exit;
  { get current date/time }
  t.Init;
  { write a horizontal line, pasdoc version and a link to the pasdoc homepage }
  Stream^.WriteLine('<HR noshade size=1><EM>Generated by <A href="' + AppInfo.APP_HOMEPAGE +
    '">' + AppInfo.APP_NAME_AND_VERSION + '</A> on ' +
    t.GetDefaultString + '</EM>');
end;

procedure THTMLDocGenerator.WriteAuthors(HL: Byte; Authors: PStringCollection);
var
  i: LongInt;
  p: PString;
  s1: string;
  s2: string;
  EmailAddress: string;
begin
  if (not Assigned(Authors)) or (Authors^.Count < 1) then exit;
  if (Authors^.Count = 1) then
    WriteHeading(HL, Translation[trAuthor])
  else
    WriteHeading(HL, Translation[trAuthors]);
  for i := 0 to Authors^.Count - 1 do
    begin
      p := Authors^.At(i);
      WriteStartOfParagraph;
      if Assigned(p) then
        begin
          if ExtractEmailAddress(p^, s1, s2, EmailAddress) then
            begin
              WriteString(s1);
              Stream^.WriteString('<A href="mailto:' + EmailAddress + '">');
              WriteString(EmailAddress);
              Stream^.WriteString('</A>');
              WriteString(s2);
            end
          else
            begin
              WriteString(p^);
            end;
        end;
      WriteEndOfParagraph;
    end;
end;

procedure THTMLDocGenerator.WriteCIO(HL: Byte; const CIO: PCIO);
var
  i: LongInt;
  p: PString;
  s: string;
  Item: PCIO;
begin
  if not Assigned(CIO) then exit;

  case CIO^.MyType of
    CIO_CLASS: s := Translation[trClass];
    CIO_DISPINTERFACE: s := Translation[trDispInterface];
    CIO_INTERFACE: s := Translation[trInterface];
    CIO_OBJECT: s := Translation[trObject];
  else
    s := '';
  end;

  s := s + ' ' + CIO^.Name;

  WriteStartOfDocument(CIO^.MyUnit^.Name + ': ' + s);
  if not HtmlHelp then
    WriteDocumentHeadline;

  WriteStartOfAnchor(AnchorToString(CIO^.AnchorNumber));
  WriteEndOfAnchor;
  WriteHeading(HL, s);
  { write unit link }
  if Assigned(CIO^.MyUnit) then
    begin
      WriteHeading(HL + 1, Translation[trUnit]);
      Stream^.WriteString('<A href="' + CIO^.MyUnit^.FullLink + '">' +
        CIO^.MyUnit^.Name + '</A><BR>');
    end;

  { write declaration link }
  WriteHeading(HL + 1, Translation[trDeclaration]);
  Stream^.WriteString('<P>');
  WriteStartOfCode;
  Stream^.WriteString('type ' + CIO^.Name + ' = ');
  case CIO^.MyType of
    CIO_CLASS: Stream^.WriteString('class');
    CIO_DISPINTERFACE: Stream^.WriteString('dispinterface');
    CIO_INTERFACE: Stream^.WriteString('interface');
  else
    Stream^.WriteString('object');
  end;
  if Assigned(CIO^.Ancestors) and (CIO^.Ancestors^.Count > 0) then
    begin
      Stream^.WriteString('(');
      for i := 0 to CIO^.Ancestors^.Count - 1 do
        begin
          p := PString(CIO^.Ancestors^.At(i));
          // RJ s := SearchLink(p^, nil);
          s := SearchLink(p^, CIO);
          Stream^.WriteString(s);
          if (i <> CIO^.Ancestors^.Count - 1) then Stream^.WriteString(', ');
        end;
      Stream^.WriteString(')');
    end;
  WriteEndOfCode;
  Stream^.WriteString('</P>');

  { Write Description }
  // WriteDescription(HL + 1, Translation[trDescription], CIO);
  WriteHeading(HL + 1, Translation[trDescription]);
  WriteItemDetailedDescription(CIO);

  { Write Hierarchy }
  if Assigned(CIO^.Ancestors) and (CIO^.Ancestors^.Count > 0) then
    begin
      p := PString(CIO^.Ancestors^.At(0));
      Item := PCIO(SearchItem(p^, CIO));
      if Assigned(Item) then
        begin
          WriteHeading(HL + 1, Translation[trHierarchy]);
          repeat
            s := CreateReferencedLink(Item^.Name, Item^.FullLink);
            Stream^.WriteString(s);

            if Assigned(Item^.Ancestors) and (Item^.Ancestors^.Count > 0) then
              begin
                p := PString(Item^.Ancestors^.At(0));
                Item := PCIO(SearchItem(p^, Item));
                if Assigned(Item) then Stream^.WriteString('&nbsp;&gt; ');
              end
            else
              Break;
          until Item = nil;
        end;
    end;

  WriteFields(HL + 1, CIO^.Fields);

  WriteFuncsProcs(HL + 1, True, CIO^.Methods);

  if (CIO^.MyType <> CIO_OBJECT) then
    begin
      WritePropertiesSummary(HL + 1, CIO^.Properties);
      WriteProperties(HL + 1, CIO^.Properties);
    end;

  WriteAuthors(HL + 1, CIO^.Authors);
  WriteDates(HL + 1, CIO^.Created, CIO^.LastMod);
  WriteFooter;
  WriteAppInfo;
  WriteEndOfDocument;
end;

procedure THTMLDocGenerator.WriteCIOs(HL: Byte; c: PItemCollection);
var
  i: LongInt;
  p: PCIO;
begin
  if (not Assigned(c)) or (c^.Count < 1) then exit;
  for i := 0 to c^.Count - 1 do
    begin
      p := c^.At(i);
      // if (not CreateStream(p^.Name))
      if (not CreateStream(p^.OutputFileName)) then
        begin
          PrintLn(1, 'Error: Could not create class/interface/object doc file.');
          Continue;
        end;
      PrintLn(3, 'Creating class/interface/object file "' + Stream^.GetName + '"...');
      WriteCIO(HL, p);
    end;
  CloseStream;
end;

procedure THTMLDocGenerator.WriteCIOSummary(HL: Byte; c: PItemCollection);
var
  j: LongInt;
  p: PCIO;
begin
  if (not Assigned(c)) or (c^.Count <= 0) then exit;

  if HtmlHelp then
    WriteString('<A name=Classes></A>');

  WriteHeading(HL, Translation[trCio]);
  WriteStartOfTable2Columns(Translation[trName], Translation[trDescription]);
  for j := 0 to c^.Count - 1 do
    begin
      p := c^.At(j);
      WriteStartOfTableRow;
      { name of class/interface/object and unit }
      WriteStartOfTableCell;
      WriteString(GetCIOTypeName(p^.MyType));
      Stream^.WriteString('&nbsp;');
      WriteStartOfLink(p^.FullLink);
      WriteString(p^.Name);
      WriteEndOfLink;
      WriteEndOfTableCell;

      { Description of class/interface/object }
      if j = 0 then
        Stream^.WriteString('<TD width=100%>')
      else
        WriteStartOfTableCell;
      { Write only the description and do not opt for DetailedDescription,
        like WriteItemDescription does. }
      if Assigned(p^.Description) and (p^.Description^.Content > 0) then
        WriteText(p^.Description)
      else
        Stream^.WriteString('&nbsp;');

      WriteEndOfTableCell;
      WriteEndOfTableRow;
    end;
  WriteEndOfTable;
end;

procedure THTMLDocGenerator.WriteCodeWithLinks(const p: PItem; const Code: string; const ItemLink: AnsiString);
var
  SearchForLink: Boolean;
  FoundItem: PItem;
  i, l: LongInt;
  s: string;

  n: Integer;
  s1: string;
  s2: string;
  S3: string;
begin
  WriteStartOfCode;
  i := 1;
  SearchForLink := False;
  l := length(Code);
  while i <= l do
    begin
      case Code[i] of
        '_', 'A'..'Z', 'a'..'z':
          begin
            { assemble item }
            s := '';
            repeat
              s := s + Code[i];
              Inc(i);
            until (i > l) or (not (Code[i] in ['.', '_', '0'..'9', 'A'..'Z', 'a'..'z']));

            if s = p^.Name then
              if ItemLink <> '' then
                begin
                  WriteStartOfLink(ItemLink);
                  WriteString('<B>' + s + '</B>');
                  WriteEndOfLink;
                end
              else
                WriteString('<B>' + s + '</B>')
            else
              begin
                { search for item of name  L }
                if SearchForLink and (SplitLink(s, s1, s2, S3, n)) then
                  begin
                    FoundItem := p^.FindName(s1, s2, S3, n);
                    if not Assigned(FoundItem) then
                      FoundItem := FindGlobal(s1, s2, S3, n);
                  end
                else
                  FoundItem := nil;

                if Assigned(FoundItem) then
                  begin
                    WriteStartOfLink(FoundItem^.FullLink);
                    Stream^.WriteString(s);
                    WriteEndOfLink;
                  end
                else
                  WriteString(s);

                Continue; // We don't want to miss out on any ':' or ';'
              end;
          end;
        ':':
          SearchForLink := True;
        ';':
          SearchForLink := False;
      end;
      WriteChar(Code[i]);
      Inc(i);
    end;
  WriteEndOfCode;
end;

procedure THTMLDocGenerator.WriteComment(const s: string);
begin
  Stream^.WriteString('<!--');
  WriteString(s);
  Stream^.WriteLine('-->');
end;

procedure THTMLDocGenerator.WriteDates(HL: Byte; Created, LastMod: PString);
begin
  if Assigned(Created) then
    begin
      WriteHeading(HL, Translation[trCreated]);
      WriteStartOfParagraph;
      WriteString(Created^);
      WriteEndOfParagraph;
    end;
  if Assigned(LastMod) then
    begin
      WriteHeading(HL, Translation[trLastModified]);
      WriteStartOfParagraph;
      WriteLine(LastMod^);
      WriteEndOfParagraph;
    end;
end;

procedure THTMLDocGenerator.WriteDocumentation;
{$IFDEF WINDOWS}
var
  HhcPath: AnsiString;
  {$ENDIF}
begin
  WriteUnits(1);
  WriteOverviewFiles;

  {$IFDEF WINDOWS}
  { Registry and HCC only exists on Windows }
  if HtmlHelp and not NoHHC then
    // Try to call HCC.exe
    with TRegistry.Create do
      try
        if OpenKey('\Software\Microsoft\HTML Help Workshop', False) then
          begin
            HhcPath := ReadString('InstallDir');
            if (HhcPath = '') or
              not DirectoryExists(HhcPath) or
              (ShellExecute(GetDeskTopWindow(), 'Open',
              PChar(HhcPath + '\hhc.exe'),
              PChar(DestDir + ProjectName + '.hhp'),
              '', SW_SHOW) <= 32)
              then
              PrintLn(1, 'Error: Could not compile HtmlHelp.');
          end;
      finally
        Free;
      end;
  {$ENDIF}
end;

procedure THTMLDocGenerator.WriteDocumentHeadline;
var
  i: Byte;
begin
  Stream^.WriteLine('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
  Stream^.WriteLine('<TR bgcolor="' + HTML_HEADER_BACKGROUND_COLOR + '">');
  for i := 0 to NUM_OVERVIEW_FILES - 1 do
    begin
      Stream^.WriteString('<TD><A href="' + OverviewFilenames[i] + GetFileExtension + '"><CENTER>');
      case i of
        0: WriteString(Translation[trUnits]);
        1: WriteString(Translation[trCio]);
        2: WriteString(Translation[trTypes]);
        3: WriteString(Translation[trVariables]);
        4: WriteString(Translation[trConstants]);
        5: WriteString(Translation[trFunctionsAndProcedures]);
        6: WriteString(Translation[trIdentifiers]);
      end;
      Stream^.WriteLine('</CENTER></A></TD>');
    end;
  Stream^.WriteLine('</TR>');
  Stream^.WriteLine('</TABLE>');
end;

procedure THTMLDocGenerator.WriteEmptyCell;
begin
  Stream^.WriteString('&nbsp;');
end;

procedure THTMLDocGenerator.WriteEndOfDocument;
begin
  Stream^.WriteLine('</BODY>');
  Stream^.WriteLine('</HTML>');
end;

procedure THTMLDocGenerator.WriteEndOfAnchor;
begin
  Stream^.WriteString('</A>');
end;

procedure THTMLDocGenerator.WriteEndOfCode;
begin
  Stream^.WriteString('</CODE>');
end;

procedure THTMLDocGenerator.WriteEndOfLink;
begin
  Stream^.WriteString('</A>');
end;

procedure THTMLDocGenerator.WriteEndOfParagraph;
begin
  Stream^.WriteLine('</P>');
end;

procedure THTMLDocGenerator.WriteEndOfTableCell;
begin
  Stream^.WriteLine('</TD>');
end;

procedure THTMLDocGenerator.WriteEndOfTable;
begin
  Stream^.WriteLine('</TABLE>');
end;

procedure THTMLDocGenerator.WriteEndOfTableRow;
begin
  Stream^.WriteLine('</TR>');
end;

{ ---------------------------------------------------------------------------- }

procedure THTMLDocGenerator.WriteFields(const Order: Byte; const Fields: PItemCollection);
var
  j: LongInt;
  Item: PItem;
begin
  if not Assigned(Fields) or (Fields^.Count <= 0) then exit;

  WriteString('<A name=Fields></A>');
  WriteHeading(Order, Translation[trFields]);

  Stream^.WriteString('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
  Stream^.WriteString('<TR bgcolor="#' + HTML_HEADER_BACKGROUND_COLOR + '">');
  Stream^.WriteLine('<TH>&nbsp;</TH><TH>' + Translation[trName] + '</TH><TH>' + Translation[trDescription] + '</TH></TR>');

  for j := 0 to Fields^.Count - 1 do
    begin
      Item := Fields^.At(j);
      WriteStartOfTableRow;

      WriteVisibilityCell(Item);

      WriteStartOfTableCell;
      WriteStartOfAnchor(AnchorToString(Item^.AnchorNumber));
      WriteEndOfAnchor;
      WriteString(Item^.Name);
      WriteEndOfTableCell;

      if j = 0 then
        Stream^.WriteString('<TD width=100%>')
      else
        WriteStartOfTableCell;
      WriteItemDetailedDescription(Item);
      WriteEndOfTableCell;

      WriteEndOfTableRow;
    end;
  Stream^.WriteString('</TABLE>');
end;

{ ---------------------------------------------------------------------------- }

procedure THTMLDocGenerator.WriteFooter;
begin
  if Footer <> '' then
    Stream^.WriteBytes(length(Footer), Pointer(Footer)^);
end;

procedure THTMLDocGenerator.WriteFuncsProcs(HL: Byte; Methods: Boolean;
  FuncsProcs: PMethodCollection);
var
  i: LongInt;
  j: LongInt;
  p: PMethod;
  s: string;
begin
  if (not Assigned(FuncsProcs)) or (FuncsProcs^.Count <= 0) then exit;

  if Methods then
    begin
      if HtmlHelp then
        WriteString('<A name=Methods></A>');
      WriteHeading(HL, Translation[trMethods]);
    end
  else
    begin
      if HtmlHelp then
        WriteString('<A name=FuncsProcs></A>');
      WriteHeading(HL, Translation[trFunctionsAndProcedures]);
    end;

  for i := 0 to 1 do
    begin
      if (i = 0) then
        begin
          WriteHeading(HL + 1, Translation[trOverview]);
          WriteStartOfTable1Column('');
        end
      else
        WriteHeading(HL + 1, Translation[trDescription]);

      for j := 0 to FuncsProcs^.Count - 1 do
        begin
          p := FuncsProcs^.At(j);
          if (i = 0) then
            begin
              WriteStartOfTableRow;

              { Only write visibility for functions of classes and objects. }
              if Methods then WriteVisibilityCell(p);

              if j = 0 then
                Stream^.WriteString('<TD width=100%>')
              else
                WriteStartOfTableCell;

              s := p^.FullLink;
              if Assigned(p^.MyUnit) then
                if CompareText(p^.MyUnit^.FullLink, Copy(s, 1, length(p^.MyUnit^.FullLink))) = 0 then
                  Delete(s, 1, length(p^.MyUnit^.FullLink));

              WriteCodeWithLinks(p, p^.FullDeclaration, s);
              (* WriteStartOfLink(s);
               // Stream^.WriteString('<P>');
               WriteStartOfCode;
               // WriteString (p^.Name); // The method's name only

               s := p^.FullDeclaration;
               s := StringReplace(s, p^.Name, '<B>' + p^.Name + '</B>', [rfIgnoreCase]);
               WriteString(s); // The method's full declaration
               WriteEndOfCode;
               // Stream^.WriteString('</P>');
               WriteEndOfLink;  *)

              WriteEndOfTableCell;
              WriteEndOfTableRow;
            end
          else
            begin
              WriteStartOfTable1Column('');
              WriteStartOfTableRow;

              if Methods then WriteVisibilityCell(p);

              Stream^.WriteString('<TD width=100%>');
              WriteStartOfAnchor(AnchorToString(p^.AnchorNumber));
              WriteEndOfAnchor;

              // s := StringReplace(s, p^.Name, '<B>' + p^.Name + '</B>', [rfIgnoreCase]);
              WriteCodeWithLinks(p, p^.FullDeclaration, '');
              {Stream^.WriteString('<code>');
              WriteString(P^.FullDeclaration);
              Stream^.WriteString('</code>');}
              WriteEndOfTableCell;
              WriteEndOfTableRow;
              WriteEndOfTable;
              // RJ WriteEndOfParagraph;

              WriteStartOfParagraph;
              WriteItemDetailedDescription(p);
              WriteEndOfParagraph;
            end;
        end;
      if (i = 0) then WriteEndOfTable;
    end;
end;

procedure THTMLDocGenerator.WriteHeading(i: Byte; const s: string);
var
  c: Char;
begin
  if (i < 1) then i := 1;
  if (i > 6) then
    begin
      PrintLn(2, 'Warning: HTML generator cannot write headlines of level 7 or greater; will use 6 instead!');
      i := 6;
    end;
  c := Chr(i + 48);
  Stream^.WriteString('<H' + c + '>');
  WriteString(s);
  Stream^.WriteLine('</H' + c + '>');
end;

{ ---------- }

procedure THTMLDocGenerator.WriteBinaryFiles;
begin
  WriteResourceToFile('PRIVATE', RT_RCDATA, 'private.gif');
  WriteResourceToFile('PROTECTED', RT_RCDATA, 'protected.gif');
  WriteResourceToFile('PUBLIC', RT_RCDATA, 'public.gif');
  WriteResourceToFile('PUBLISHED', RT_RCDATA, 'published.gif');
  WriteResourceToFile('CSS', RT_RCDATA, ProjectName + '.css')
end;

{ ---------- }

procedure THTMLDocGenerator.WriteItemDescription(Item: PItem);
begin
  if not Assigned(Item) then exit;

  if Assigned(Item^.Description) and (Item^.Description^.Content > 0) then
    WriteText(Item^.Description)
  else
    if Assigned(Item^.DetailedDescription) and (Item^.DetailedDescription^.Content > 0) then
      WriteText(Item^.DetailedDescription)
    else
      Stream^.WriteString('&nbsp;');
end;

procedure THTMLDocGenerator.WriteItemDetailedDescription(Item: PItem);
begin
  if not Assigned(Item) then exit;

  if Assigned(Item^.Description) and (Item^.Description^.Content > 0) then
    begin
      WriteText(Item^.Description);

      if Assigned(Item^.DetailedDescription) and (Item^.DetailedDescription^.Content > 0) then
        begin
          Stream^.WriteString('<P>');
          WriteText(Item^.DetailedDescription);
        end;
    end
  else
    if Assigned(Item^.DetailedDescription) and (Item^.DetailedDescription^.Content > 0) then
      WriteText(Item^.DetailedDescription)
    else
      Stream^.WriteString('&nbsp;');

  {  if not Assigned(Item) then Exit;

    if Assigned(Item^.DetailedDescription) and (Item^.DetailedDescription^.Content > 0) then
      WriteText(Item^.DetailedDescription)
    else
      if Assigned(Item^.Description) and (Item^.Description^.Content > 0) then
        WriteText(Item^.Description)
      else
        Stream^.WriteString('&nbsp;'); }
end;

procedure THTMLDocGenerator.WriteItems(Order: Byte; Heading: string; const Anchor: string; i: PItemCollection);
var
  j: LongInt;
  Item: PItem;
begin
  if (not Assigned(i)) or (i^.Count <= 0) then exit;

  if HtmlHelp and (Anchor <> '') then
    WriteString('<A name=' + Anchor + '></A>');

  WriteHeading(Order, Heading);

  Stream^.WriteString('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
  Stream^.WriteString('<TR bgcolor="#' + HTML_HEADER_BACKGROUND_COLOR + '">');
  Stream^.WriteLine('<TH>' + Translation[trName] + '</TH><TH>' + Translation[trDescription] + '</TH></TR>');

  for j := 0 to i^.Count - 1 do
    begin
      Item := i^.At(j);
      WriteStartOfTableRow;

      WriteStartOfTableCell;
      WriteStartOfAnchor(AnchorToString(Item^.AnchorNumber));
      WriteEndOfAnchor;
      WriteString(Item^.Name);
      WriteEndOfTableCell;

      if j = 0 then
        Stream^.WriteString('<TD width=100%>')
      else
        WriteStartOfTableCell;
      WriteItemDetailedDescription(Item);
      WriteEndOfTableCell;

      WriteEndOfTableRow;
    end;
  Stream^.WriteString('</TABLE>');
end;

{ ---------- }

procedure THTMLDocGenerator.WriteOverviewFiles;
var
  c1: PItemCollection;
  c2: PItemCollection;
  c3: PItemCollection; // Collect all Items for final listing.
  i: LongInt;
  Item: PItem;
  j: LongInt;
  PU: PUnit;
begin
  if HtmlHelp then
    WriteHtmlHelpProject;

  WriteUnitOverviewFile;
  if (not Assigned(Units)) or (Units^.Count < 1) then exit;
  c3 := New(PItemCollection, Init(16, 16));

  for i := 1 to 5 do
    begin
      // if (not CreateStream(OverviewFilenames[i]))
      if (not CreateStream(DestDir + OverviewFilenames[i] + GetFileExtension)) then
        begin
          PrintLn(1, 'Error: Could not create output file "' +
            OverviewFilenames[i] + '".');
          exit;
        end;
      PrintLn(3, 'Writing overview file ' + Stream^.GetName + '...');

      case i of
        1: WriteStartOfDocument(Translation[trHeadlineCio]);
        2: WriteStartOfDocument(Translation[trHeadlineTypes]);
        3: WriteStartOfDocument(Translation[trHeadlineVariables]);
        4: WriteStartOfDocument(Translation[trHeadlineConstants]);
        5: WriteStartOfDocument(Translation[trHeadlineFunctionsAndProcedures]);
        6: WriteStartOfDocument(Translation[trHeadlineIdentifiers]);
      end;

      if not HtmlHelp then WriteDocumentHeadline;

      case i of
        1: WriteHeading(1, Translation[trHeadlineCio]);
        2: WriteHeading(1, Translation[trHeadlineTypes]);
        3: WriteHeading(1, Translation[trHeadlineVariables]);
        4: WriteHeading(1, Translation[trHeadlineConstants]);
        5: WriteHeading(1, Translation[trHeadlineFunctionsAndProcedures]);
        6: WriteHeading(1, Translation[trHeadlineIdentifiers]);
      end;

      c1 := New(PItemCollection, Init(16, 16));

      // Allow for Duplicates in Classes
      if i = 1 then
        c1^.Duplicates := True
      else
        c1^.Duplicates := False;

      for j := 0 to Units^.Count - 1 do
        begin
          PU := Units^.At(j);
          case i of
            1: c2 := PU^.CIO;
            2: c2 := PU^.Types;
            3: c2 := PU^.Variables;
            4: c2 := PU^.Constants;
            5: c2 := PU^.FuncsProcs;
          end;
          c1^.InsertItems(c2);
        end;

      if Assigned(c1) and (c1^.Count > 0) then
        begin
          WriteStartOfTable3Columns(Translation[trName], Translation[trUnit], Translation[trDescription]);
          for j := 0 to c1^.Count - 1 do
            begin
              Item := c1^.At(j);
              WriteStartOfTableRow;

              WriteStartOfTableCell;
              WriteStartOfLink(Item^.FullLink);
              WriteString(Item^.Name);
              WriteEndOfLink;
              WriteEndOfTableCell;

              WriteStartOfTableCell;
              WriteStartOfLink(Item^.MyUnit^.FullLink);
              WriteString(Item^.MyUnit^.Name);
              WriteEndOfLink;
              WriteEndOfTableCell;

              if j = 0 then
                Stream^.WriteString('<TD width=100%>')
              else
                WriteStartOfTableCell;
              WriteItemDescription(Item);
              WriteEndOfTableCell;

              WriteEndOfTableRow;
            end;
          WriteEndOfTable;
        end
      else
        begin
          WriteStartOfParagraph;
          WriteString(Translation[trNone]);
          WriteEndOfParagraph;
        end;

      c3^.InsertItems(c1);
      Dispose(c1, Done);
      WriteFooter;
      WriteAppInfo;
      WriteEndOfDocument;
      CloseStream;
    end;

  if not CreateStream(DestDir + OverviewFilenames[6] + GetFileExtension) then
    begin
      PrintLn(1, 'Error: Could not create overview output file "' +
        OverviewFilenames[6] + '".');
      exit;
    end;
  PrintLn(3, 'Writing overview file ' + Stream^.GetName + '...');
  WriteStartOfDocument(Translation[trHeadlineIdentifiers]);
  if not HtmlHelp then
    WriteDocumentHeadline;
  WriteHeading(1, Translation[trHeadlineIdentifiers]);
  WriteStartOfTable3Columns(Translation[trName], Translation[trUnit], Translation[trDescription]);

  for j := 0 to c3^.Count - 1 do
    begin
      Item := c3^.At(j);
      WriteStartOfTableRow;

      WriteStartOfTableCell;
      WriteStartOfLink(Item^.FullLink);
      WriteString(Item^.Name);
      WriteEndOfLink;
      WriteEndOfTableCell;

      WriteStartOfTableCell;
      WriteStartOfLink(Item^.MyUnit^.FullLink);
      WriteString(Item^.MyUnit^.Name);
      WriteEndOfLink;
      WriteEndOfTableCell;

      if j = 0 then
        Stream^.WriteString('<TD width=100%>')
      else
        WriteStartOfTableCell;
      WriteItemDescription(Item);
      WriteEndOfTableCell;

      WriteEndOfTableRow;
    end;
  WriteEndOfTable;
  WriteFooter;
  WriteAppInfo;
  WriteEndOfDocument;
  CloseStream;
end;

procedure THTMLDocGenerator.WriteParagraph(HL: Byte; s: string; t: PText);
begin
  if (not Assigned(t)) or (t^.Content < 1) then exit;
  WriteHeading(HL, s);
  Stream^.WriteLine('<P>');
  Stream^.WriteBytes(t^.Content, t^.Data^);
  Stream^.WriteLine('</P>');
end;

procedure THTMLDocGenerator.WriteProperties(HL: Byte; p: PPropertyCollection);
var
  j: LongInt;
  Prop: PProperty;
begin
  if (not Assigned(p)) or (p^.Count <= 0) then exit;

  WriteHeading(HL + 1, Translation[trDescription]);
  for j := 0 to p^.Count - 1 do
    begin
      Prop := p^.At(j);

      WriteStartOfTable1Column('');
      WriteStartOfTableRow;

      WriteVisibilityCell(Prop);

      Stream^.WriteString('<TD width=100%>');
      WriteStartOfAnchor(AnchorToString(Prop^.AnchorNumber));
      WriteEndOfAnchor;
      WriteCodeWithLinks(Prop, 'property ' + Prop^.FullDeclaration, '');

      WriteEndOfTableCell;
      WriteEndOfTableRow;
      WriteEndOfTable;

      WriteStartOfParagraph;
      WriteItemDetailedDescription(Prop);
      WriteEndOfParagraph;

    end;
end;

procedure THTMLDocGenerator.WritePropertiesSummary(HL: Byte; p: PPropertyCollection);
var
  j: LongInt;
  Prop: PProperty;
begin
  if (not Assigned(p)) or (p^.Count <= 0) then exit;

  if HtmlHelp then
    WriteString('<A name=Properties></A>');

  WriteHeading(HL, Translation[trProperties]);
  WriteHeading(HL + 1, Translation[trOverview]);

  WriteStartOfTable1Column('');
  for j := 0 to p^.Count - 1 do
    begin
      Prop := p^.At(j);
      WriteStartOfTableRow;

      WriteVisibilityCell(Prop);
      if j = 0 then
        Stream^.WriteString('<TD width=100%>')
      else
        WriteStartOfTableCell;

      WriteCodeWithLinks(Prop, 'property ' + Prop^.FullDeclaration, Prop^.FullLink);
      { s := Prop^.FullDeclaration;
        s := StringReplace(s, Prop^.Name, '<B>' + Prop^.Name + '</B>', [rfIgnoreCase]);
        Stream^.WriteString(CreateReferencedLink(s, Prop^.FullLink)); }

      WriteEndOfTableCell;
      WriteEndOfTableRow;
    end;
  WriteEndOfTable;
end;

procedure THTMLDocGenerator.WriteStartOfAnchor(const Name: string);
begin
  Stream^.WriteString('<A name="' + Name + '">');
end;

procedure THTMLDocGenerator.WriteStartOfCode;
begin
  Stream^.WriteString('<CODE>');
end;

procedure THTMLDocGenerator.WriteStartOfDocument(Name: string);
begin
  Stream^.WriteLine('<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0 Transitional//EN" "http://www.w3.org/TR/REC-html40/loose.dtd">');
  Stream^.WriteLine('<HTML>');
  Stream^.WriteLine('<HEAD>');
  Stream^.WriteLine('<META name="GENERATOR" content="' + APP_NAME_AND_VERSION + '">');
  // Check if we need to specify character sets
  if LANGUAGE_ARRAY[Language].CharSet <> '' then
    Stream^.WriteLine('<META http-equiv="content-type" content="text/html; charset=' + LANGUAGE_ARRAY[Language].CharSet + '">');
  // Title
  Stream^.WriteString('<TITLE>');
  if not HtmlHelp and (Title <> '') then
    WriteString(Title + ': ');
  WriteString(Name);
  Stream^.WriteLine('</TITLE>');
  // StyleSheet
  WriteString('<LINK rel="StyleSheet" href="');
  WriteString(ProjectName);
  Stream^.WriteLine('.css">');

  Stream^.WriteLine('</HEAD>');
  Stream^.WriteLine('<BODY bgcolor="#ffffff" text="#000000" link="#0000ff" vlink="#800080" alink="#FF0000">');

  if Header <> '' then
    Stream^.WriteBytes(length(Header), Pointer(Header)^);
end;

procedure THTMLDocGenerator.WriteStartOfLink(const Name: string);
begin
  Stream^.WriteString('<A href="' + Name + '">');
end;

procedure THTMLDocGenerator.WriteStartOfParagraph;
begin
  Stream^.WriteString('<P>');
end;

procedure THTMLDocGenerator.WriteStartOfTable1Column(t: string);
begin
  Stream^.WriteLine('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
end;

procedure THTMLDocGenerator.WriteStartOfTable2Columns(t1, t2: string);
begin
  Stream^.WriteLine('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
  Stream^.WriteString('<TR bgcolor="#' + HTML_HEADER_BACKGROUND_COLOR + '"><TH>');
  WriteString(t1);
  Stream^.WriteString('</TH><TH>');
  WriteString(t2);
  Stream^.WriteLine('</TH></TR>');
end;

procedure THTMLDocGenerator.WriteStartOfTable3Columns(t1, t2, T3: string);
begin
  Stream^.WriteLine('<TABLE cellspacing=' + HTML_TABLE_CELLSPACING + ' cellpadding=' + HTML_TABLE_CELLPADDING + ' width=100%>');
  Stream^.WriteString('<TR bgcolor="#' + HTML_HEADER_BACKGROUND_COLOR + '"><TH>');
  WriteString(t1);
  Stream^.WriteString('</TH><TH>');
  WriteString(t2);
  Stream^.WriteString('</TH><TH>');
  WriteString(T3);
  Stream^.WriteLine('</TH></TR> ');
end;

procedure THTMLDocGenerator.WriteStartOfTableCell;
begin
  Stream^.WriteString('<TD>');
end;

procedure THTMLDocGenerator.WriteStartOfTableRow;
begin
  Stream^.WriteString('<TR bgcolor=#' + HTML_ROW_BACKGROUND_COLOR + ' valign=top>');
end;

{ ---------------------------------------------------------------------------- }
{ HtmlHelp Content Generation inspired by Wim van der Vegt <wvd_vegt@knoware.nl>
{ ---------------------------------------------------------------------------- }

function BeforeEqualChar(const s: AnsiString): AnsiString;
var
  i: Cardinal;
begin
  Result := s;
  i := Pos('=', Result);
  if i <> 0 then
    SetLength(Result, i - 1);
end;

function AfterEqualChar(const s: AnsiString): AnsiString;
var
  i: Cardinal;
begin
  Result := s;
  i := Pos('=', Result);
  if i <> 0 then
    Delete(Result, 1, i)
  else
    Result := '';
end;

function GetLevel(var s: AnsiString): Integer;
var
  l: Cardinal;
  p: PAnsiChar;
begin
  Result := 0;
  p := Pointer(s);
  l := length(s);
  while (l > 0) and (p^ in [' ', #9]) do
    begin
      Inc(Result);
      Inc(p);
      Dec(l);
    end;
  Delete(s, 1, Result);
end;

procedure THTMLDocGenerator.WriteHtmlHelpProject;
var
  DefaultContentsWritten: Boolean;
  DefaultTopic: AnsiString;

  procedure WriteLiObject(const Name, Local: AnsiString);
  begin
    Stream^.WriteLine('<LI><OBJECT type="text/sitemap">');
    WriteLine('<PARAM name="Name" value="' + Name + '">');
    if Local <> '' then
      begin
        WriteLine('<PARAM name="Local" value="' + Local + '">');
        if DefaultTopic = '' then
          DefaultTopic := Local;
      end;
    Stream^.WriteLine('</OBJECT>');
  end;

  { ---------- }

  procedure WriteItemCollection(const c: PItemCollection);
  var
    i: Integer;
    Item: PItem;
  begin
    if Assigned(c) then
      begin
        Stream^.WriteLine('<UL>');
        for i := 0 to c^.Count - 1 do
          begin
            Item := c^.At(i);
            WriteLiObject(Item^.Name, Item^.FullLink);
          end;
        Stream^.WriteLine('</UL>');
      end;
  end;

  { ---------- }

  procedure WriteItemHeadingCollection(const Title, FullLink: AnsiString; const c: PItemCollection);
  begin
    if Assigned(c) and (c^.Count > 0) then
      begin
        WriteLiObject(Title, FullLink);
        WriteItemCollection(c);
      end;
  end;

  { ---------- }

  procedure InternalWriteCIO(const ClassItem: PCIO);
  begin
    WriteLiObject(ClassItem^.Name, ClassItem^.FullLink);
    Stream^.WriteLine('<UL>');

    WriteItemHeadingCollection('Fields', ClassItem^.FullLink + '#Fields', ClassItem^.Fields);
    WriteItemHeadingCollection('Properties', ClassItem^.FullLink + '#Properties', ClassItem^.Properties);
    WriteItemHeadingCollection('Methods', ClassItem^.FullLink + '#Methods', ClassItem^.Methods);

    Stream^.WriteLine('</UL>');
  end;

  { ---------- }

  procedure ContentWriteUnits(const Text: AnsiString);
  var
    c: PItemCollection;
    j, k: Integer;
    PU: PUnit;
  begin
    if Text <> '' then
      WriteLiObject(Text, OverviewFilenames[0] + GetFileExtension)
    else
      WriteLiObject(Translation[trUnits], OverviewFilenames[0] + GetFileExtension);
    Stream^.WriteLine('<UL>');

    // Iterate all Units
    for j := 0 to Units^.Count - 1 do
      begin
        PU := Units^.At(j);
        WriteLiObject(PU^.Name, PU^.FullLink);
        Stream^.WriteLine('<UL>');

        // For each unit, write classes (if there are any).
        c := PU^.CIO;
        if Assigned(c) then
          begin
            WriteLiObject(Translation[trClasses], PU^.FullLink + '#Classes');
            Stream^.WriteLine('<UL>');

            for k := 0 to c^.Count - 1 do
              InternalWriteCIO(c^.At(k));

            Stream^.WriteLine('</UL>');
          end;

        // For each unit, write Functions & Procedures.
        WriteItemHeadingCollection(Translation[trFunctionsAndProcedures], PU^.FullLink + '#FuncsProcs', PU^.FuncsProcs);
        // For each unit, write Types.
        WriteItemHeadingCollection(Translation[trTypes], PU^.FullLink + '#Types', PU^.Types);
        // For each unit, write Constants.
        WriteItemHeadingCollection(Translation[trConstants], PU^.FullLink + '#Constants', PU^.Constants);

        Stream^.WriteLine('</UL>');
      end;
    Stream^.WriteLine('</UL>');
  end;

  { ---------- }

  procedure ContentWriteClasses(const Text: AnsiString);
  var
    c: PItemCollection;
    j: Integer;
    PU: PUnit;
  begin
    // Write Classes to Contents
    if Text <> '' then
      WriteLiObject(Text, OverviewFilenames[1] + GetFileExtension)
    else
      WriteLiObject(Translation[trClasses], OverviewFilenames[1] + GetFileExtension);
    Stream^.WriteLine('<UL>');
    c := New(PItemCollection, Init(16, 16));
    c^.Duplicates := True;
    // First collect classes in sorted order.
    for j := 0 to Units^.Count - 1 do
      begin
        PU := Units^.At(j);
        c^.CopyItems(PU^.CIO);
      end;
    // Output sorted classes
    for j := 0 to c^.Count - 1 do
      InternalWriteCIO(c.At(j));
    c^.DeleteAll;
    Dispose(c, Done);
    Stream^.WriteLine('</UL>');
  end;

  { ---------- }

  procedure ContentWriteOverview(const Text: AnsiString);
  var
    j: Integer;
  begin
    if Text <> '' then
      WriteLiObject(Text, '')
    else
      WriteLiObject(Translation[trOverview], '');
    Stream^.WriteLine('<UL>');
    for j := 0 to NUM_OVERVIEW_FILES - 1 do
      begin
        Stream^.WriteLine('<LI><OBJECT type="text/sitemap">');
        case j of
          0: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineUnits] + '">');
          1: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineCio] + '">');
          2: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineTypes] + '">');
          3: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineVariables] + '">');
          4: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineConstants] + '">');
          5: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineFunctionsAndProcedures] + '">');
          6: WriteLine('<PARAM name="Name" value="' + Translation[trHeadlineIdentifiers] + '">');
        end;
        WriteLine('<PARAM name="Local" value="' + OverviewFilenames[j] + '.htm">');
        Stream^.WriteLine('</OBJECT>');
      end;
    Stream^.WriteLine('</UL>');
  end;

  { ---------- }

  procedure ContentWriteLegend(const Text: AnsiString);
  begin
    if Text <> '' then
      WriteLiObject(Text, 'Legend.htm')
    else
      WriteLiObject(Translation[trLegend], 'Legend.htm');
  end;

  procedure ContentWriteCustom(const Text, Link: AnsiString);
  begin
    if CompareText('@Classes', Link) = 0 then
      begin
        DefaultContentsWritten := True;
        ContentWriteClasses(Text);
      end
    else
      if CompareText('@Units', Link) = 0 then
        begin
          DefaultContentsWritten := True;
          ContentWriteUnits(Text);
        end
      else
        if CompareText('@Overview', Link) = 0 then
          begin
            DefaultContentsWritten := True;
            ContentWriteOverview(Text);
          end
        else
          if CompareText('@Legend', Link) = 0 then
            begin
              DefaultContentsWritten := True;
              ContentWriteLegend(Text);
            end
          else
            WriteLiObject(Text, Link);
  end;

  procedure IndexWriteItem(const Item, PreviousItem, NextItem: PItem);
    { Item is guaranteed to be assigned, i.e. not to be nil. }
  begin
    if Assigned(Item^.MyObject) then
      begin
        if (Assigned(NextItem) and Assigned(NextItem^.MyObject) and (CompareText(Item^.MyObject^.Name, NextItem^.MyObject^.Name) = 0)) or
          (Assigned(PreviousItem) and Assigned(PreviousItem^.MyObject) and (CompareText(Item^.MyObject^.Name, PreviousItem^.MyObject^.Name) = 0)) then
          WriteLiObject(Item^.MyObject^.Name + ' - ' + Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink)
        else
          WriteLiObject(Item^.MyObject^.Name, Item^.FullLink);
      end
    else
      begin
        WriteLiObject(Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink);
      end;
  end;

  { ---------------------------------------------------------------------------- }

var
  j, k: Integer;
  CurrentLevel, Level: LongInt;
  CIO: PCIO;
  PU: PUnit;
  c: PItemCollection;
  Item, NextItem, PreviousItem: PItem;
  s, Text, Link: AnsiString;
  sl: TStringList;
begin
  // At this point, at least one unit has been parsed:
  // Units is assigned and Units^.Count > 0
  // No need to test this again.

  if (not CreateStream(DestDir + ProjectName + '.hhc')) then
    begin
      PrintLn(1, 'Error: Could not create HtmlHelp Content file "' + ProjectName + '.hhc' + '".');
      exit;
    end;
  PrintLn(3, 'Writing unit overview file "' + Stream^.GetName + '"...');

  // File Header
  Stream^.WriteLine('<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML//EN">');
  Stream^.WriteLine('<HTML>');
  Stream^.WriteLine('<HEAD>');
  Stream^.WriteLine('<META name="GENERATOR" content="' + APP_NAME_AND_VERSION + '">');
  Stream^.WriteLine('</HEAD><BODY>');
  Stream^.WriteLine('<UL>');

  DefaultContentsWritten := False;
  DefaultTopic := '';
  if (ContentsFile <> '') then
    begin
      sl := TStringList.Create;
      try
        sl.LoadFromFile(ContentsFile);
      except
        on e: Exception do
          PrintLn(1, 'Error: ' + e.Message + '. Writing default.');
      end;

      CurrentLevel := 0;
      for j := 0 to sl.Count - 1 do
        begin
          s := sl[j];
          Text := BeforeEqualChar(s);
          Level := GetLevel(Text);
          Link := AfterEqualChar(s);

          if Level = CurrentLevel then
            ContentWriteCustom(Text, Link)
          else
            if CurrentLevel = (Level - 1) then
              begin
                Stream^.WriteLine('<UL>');
                Inc(CurrentLevel);
                ContentWriteCustom(Text, Link)
              end
            else
              if CurrentLevel > Level then
                begin
                  Stream^.WriteLine('</UL>');
                  Dec(CurrentLevel);
                  while CurrentLevel > Level do
                    begin
                      Stream^.WriteLine('</UL>');
                      Dec(CurrentLevel);
                    end;
                  ContentWriteCustom(Text, Link)
                end

              else
                begin
                  PrintLn(1, 'Error: Invalid level ' + IntToStr(Level) + 'in Content file (line ' + IntToStr(j) + ').');
                  exit;
                end;
        end;
      sl.Free;
    end;

  if not DefaultContentsWritten then
    begin
      ContentWriteUnits('');
      ContentWriteClasses('');
      ContentWriteOverview('');
      ContentWriteLegend('');
    end;

  // End of File
  Stream^.WriteLine('</UL>');
  Stream^.WriteLine('</BODY></HTML>');
  CloseStream;

  // Create Keyword Index
  // First collect all Items
  c := New(PItemCollection, Init(16, 16));
  c.Duplicates := True;
  for j := 0 to Units^.Count - 1 do
    begin
      PU := Units^.At(j);

      if Assigned(PU^.CIO) then
        for k := 0 to PU^.CIO^.Count - 1 do
          begin
            CIO := PU^.CIO^.At(k);
            c^.Insert(CIO);
            c^.CopyItems(CIO^.Fields);
            c^.CopyItems(CIO^.Properties);
            c^.CopyItems(CIO^.Methods);
          end;

      c^.CopyItems(PU^.Types);
      c^.CopyItems(PU^.Variables);
      c^.CopyItems(PU^.Constants);
      c^.CopyItems(PU^.FuncsProcs);
    end;

  if (not CreateStream(DestDir + ProjectName + '.hhk')) then
    begin
      PrintLn(1, 'Error: Could not create HtmlHelp Index file "' + ProjectName + '.hhk' + '".');
      exit;
    end;
  PrintLn(3, 'Writing unit overview file "' + Stream^.GetName + '"...');
  Stream^.WriteLine('<!DOCTYPE HTML PUBLIC "-//IETF//DTD HTML//EN">');
  Stream^.WriteLine('<HTML>');
  Stream^.WriteLine('<HEAD>');
  WriteLine('<META name="GENERATOR" content="' + APP_NAME_AND_VERSION + '">');
  Stream^.WriteLine('</HEAD><BODY>');
  Stream^.WriteLine('<UL>');

  { // Write all Items to KeyWord Index
   Item := nil;
   if c^.Count > 0 then
     begin
       NextItem := c^.At(0);

       j := 1;
       while j < c^.Count do
         begin
           Item := NextItem;
           WriteLiObject(Item^.Name, Item^.FullLink);

           // Does the next Item have the same name?
           NextItem := c^.At(j);
           if CompareText(Item^.Name, NextItem^.Name) = 0 then
             begin
               Stream^.WriteLine('<UL>');

               if Assigned(Item^.MyObject) then
                 begin
                   if Assigned(NextItem^.MyObject) and (CompareText(Item^.MyObject^.Name, NextItem^.MyObject^.Name) = 0) then
                     WriteLiObject(Item^.MyObject^.Name + ' - ' + Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink)
                   else
                     WriteLiObject(Item^.MyObject^.Name, Item^.FullLink);
                 end
               else
                 begin
                   WriteLiObject(Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink);
                 end;

               PreviousItem := Item;
               Item := NextItem;
               NextItem := nil;
               Inc(j);

               while (j < c^.Count) do
                 begin
                   NextItem := c^.At(j);

                   if Assigned(Item^.MyObject) then
                     begin
                       if ((CompareText(Item^.Name, NextItem^.Name) = 0) and
                         Assigned(NextItem^.MyObject) and
                         (CompareText(Item^.MyObject^.Name, NextItem^.MyObject^.Name) = 0)) or
                         (Assigned(PreviousItem^.MyObject) and
                         (CompareText(Item^.MyObject^.Name, PreviousItem^.MyObject^.Name) = 0)) then
                         WriteLiObject(Item^.MyObject^.Name + ' - ' + Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink)
                       else
                         WriteLiObject(Item^.MyObject^.Name, Item^.FullLink);
                     end
                   else
                     begin
                       WriteLiObject(Item^.MyUnit^.Name + #32 + Translation[trUnit], Item^.FullLink);
                     end;

                   if CompareText(Item^.Name, NextItem^.Name) <> 0 then Break;

                   PreviousItem := Item;
                   Item := NextItem;
                   Inc(j);
                 end;

               Stream^.WriteLine('</UL>');
             end;

           Inc(j);
         end;

       // Write the last Item
       // if Assigned(Item) and (CompareText(Item^.Name, NextItem^.Name) = 0) then
       if Assigned(NextItem) and (CompareText(Item^.Name, NextItem^.Name) = 0) then
         begin
           Stream^.WriteLine('<UL>');

           if Assigned(Item^.MyObject) then
             WriteLiObject(Item^.MyObject^.Name, Item^.FullLink)
           else
             WriteLiObject(Item^.Name, Item^.FullLink);

           if Assigned(NextItem^.MyObject) then
             WriteLiObject(NextItem^.MyObject^.Name, NextItem^.FullLink)
           else
             WriteLiObject(NextItem^.Name, NextItem^.FullLink)
         end
       else
         // WriteLiObject(NextItem^.Name, NextItem^.FullLink)
         WriteLiObject(Item^.Name, Item^.FullLink);
     end; }

      // Write all Items to KeyWord Index

  if c^.Count > 0 then
    begin
      Item := c^.At(0);
      j := 1;

      while j < c^.Count do
        begin
          NextItem := c^.At(j);

          // Does the next Item have a different name?
          if CompareText(Item^.Name, NextItem^.Name) <> 0 then
            begin
              WriteLiObject(Item^.Name, Item^.FullLink);
              Item := NextItem;
            end
          else
            begin
              // Write the Item. It acts as a header for the subitems to follow.
              WriteLiObject(Item^.Name, Item^.FullLink);
              // Indent by one.
              Stream^.WriteLine('<UL>');

              // No previous Item as we start.
              PreviousItem := nil;

              // Keep on writing Items with the same name as subitems.
              repeat
                IndexWriteItem(Item, PreviousItem, NextItem);

                PreviousItem := Item;
                Item := NextItem;
                Inc(j);

                if j >= c^.Count then Break;
                NextItem := c^.At(j);

                // Break as soon Items' names are different.
              until CompareText(Item^.Name, NextItem^.Name) <> 0;

              // No NextItem as we write the last one of the same Items.
              IndexWriteItem(Item, PreviousItem, nil);

              Item := NextItem;
              Stream^.WriteLine('</UL>');
            end;

          Inc(j);
        end;

      // Don't forget to write the last item. Can it ever by nil?
      WriteLiObject(Item^.Name, Item^.FullLink);
    end;

  c^.DeleteAll;
  Dispose(c, Done);

  Stream^.WriteLine('</UL>');
  Stream^.WriteLine('</BODY></HTML>');
  CloseStream;

  // Create a HTML Help Project File
  if (not CreateStream(DestDir + ProjectName + '.hhp')) then
    begin
      PrintLn(1, 'Error: Could not create HtmlHelp Project file "' + ProjectName + '.hhp' + '".');
      exit;
    end;
  PrintLn(3, 'Writing unit overview file "' + Stream^.GetName + '"...');
  Stream^.WriteLine('[OPTIONS]');
  Stream^.WriteLine('Binary TOC=Yes');
  Stream^.WriteLine('Compatibility=1.1 or later');
  Stream^.WriteLine('Compiled file=' + ProjectName + '.chm');
  Stream^.WriteLine('Contents file=' + ProjectName + '.hhc');
  Stream^.WriteLine('Default Window=Default');
  Stream^.WriteLine('Default topic=' + DefaultTopic);
  Stream^.WriteLine('Display compile progress=Yes');
  Stream^.WriteLine('Error log file=' + ProjectName + '.log');
  Stream^.WriteLine('Full-text search=Yes');
  Stream^.WriteLine('Index file=' + ProjectName + '.hhk');
  if Title <> '' then
    Stream^.WriteLine('Title=' + Title)
  else
    Stream^.WriteLine('Title=' + ProjectName);

  Stream^.WriteLine('');
  Stream^.WriteLine('[WINDOWS]');
  if Title <> '' then
    Stream^.WriteLine('Default="' + Title + '","' + ProjectName + '.hhc","' + ProjectName + '.hhk",,,,,,,0x23520,,0x300e,,,,,,,,0')
  else
    Stream^.WriteLine('Default="' + ProjectName + '","' + ProjectName + '.hhc","' + ProjectName + '.hhk",,,,,,,0x23520,,0x300e,,,,,,,,0');

  Stream^.WriteLine('');
  Stream^.WriteLine('[FILES]');

  { HHC seems to know about the files by reading the Content and Index.
    So there is no need to specify them in the FILES section.

  Stream^.WriteLine('Legend.htm');
  for k := 0 to NUM_OVERVIEW_FILES - 1 do
    Stream^.WriteLine(OverviewFilenames[k] + '.htm');

  if Assigned(Units) then
    for k := 0 to Units^.Count - 1 do
      begin
        Item := Units^.At(k);
        PU := Units^.At(k);
        Stream^.WriteLine(Item^.FullLink);
        c := PU^.CIO;
        if Assigned(c) then
          for l := 0 to c^.Count - 1 do
            begin
              Item2 := c^.At(l);
              Stream^.WriteLine(Item2^.FullLink);
            end;
      end;}

  Stream^.WriteLine('');

  Stream^.WriteLine('[INFOTYPES]');

  Stream^.WriteLine('');

  Stream^.WriteLine('[MERGE FILES]');

  CloseStream;

  // Create a Main Topic
  if (not CreateStream(DestDir + 'Legend.htm')) then
    begin
      PrintLn(1, 'Error: Could not create file "Legend.htm".');
      exit;
    end;
  PrintLn(3, 'Writing Legend.htm...');
  WriteStartOfDocument('Legend');
  WriteHeading(1, 'Legend');
  WriteString('<TABLE cellpadding=5>');
  WriteString('<TR><TD><IMG src="private.gif" alt="' + Translation[trPrivate] + '"></TD><TD>' + Translation[trPrivate] + '</TD></TR>');
  WriteString('<TR><TD><IMG src="protected.gif" alt="' + Translation[trProtected] + '"></TD><TD>' + Translation[trProtected] + '</TD></TR>');
  WriteString('<TR><TD><IMG src="public.gif" alt="' + Translation[trPublic] + '"></TD><TD>' + Translation[trPublic] + '</TD></TR>');
  WriteString('<TR><TD><IMG src="published.gif" alt="' + Translation[trPublished] + '"></TD><TD>' + Translation[trPublished] + '</TD></TR>');
  WriteString('</TABLE>');
  WriteFooter;
  WriteAppInfo;
  WriteEndOfDocument;
  CloseStream;
end;

{ ---------------------------------------------------------------------------- }

procedure THTMLDocGenerator.WriteUnit(HL: Byte; U: PUnit);
begin
  if (not Assigned(U)) then
    begin
      PrintLn(1, 'Error in THTMLDocGenerator.WriteUnit: Unit variable has not been initialized.');
      exit;
    end;
  // if (not CreateStream(u^.Name)) then
  if (not CreateStream(U^.OutputFileName)) then
    begin
      PrintLn(1, 'Error: Could not create HTML unit doc file for unit ' + U^.Name);
      exit;
    end;
  PrintLn(3, 'Writing documentation for unit "' + U^.Name + '"...');
  WriteStartOfDocument(U^.Name);

  if not HtmlHelp then
    WriteDocumentHeadline;
  WriteHeading(HL, Translation[trUnit] + ' ' + U^.Name);

  WriteUnitDescription(HL + 1, U);
  WriteCIOSummary(HL + 1, U^.CIO);
  WriteFuncsProcs(HL + 1, False, U^.FuncsProcs);
  WriteTypes(HL + 1, U^.Types);
  WriteConstants(HL + 1, U^.Constants);
  WriteVariables(HL + 1, U^.Variables);
  WriteAuthors(HL + 1, U^.Authors);
  WriteDates(HL + 1, U^.Created, U^.LastMod);
  WriteFooter;
  WriteAppInfo;
  WriteEndOfDocument;
  CloseStream;
  WriteCIOs(HL, U^.CIO);
end;

procedure THTMLDocGenerator.WriteUnitDescription(HL: Byte; U: PUnit);
begin
  WriteHeading(HL, Translation[trDescription]);
  WriteItemDetailedDescription(U);
end;

procedure THTMLDocGenerator.WriteUnitOverviewFile;
var
  c: PItemCollection;
  Item: PItem;
  j: LongInt;
begin
  c := Units;
  if (not CreateStream(DestDir + OverviewFilenames[0] + GetFileExtension)) then
    begin
      PrintLn(1, 'Error: Could not create overview output file "' + OverviewFilenames[0] + '".');
      exit;
    end;
  PrintLn(3, 'Writing unit overview file "' + Stream^.GetName + '"...');
  WriteStartOfDocument(Translation[trHeadlineUnits]);
  if not HtmlHelp then
    WriteDocumentHeadline;
  WriteHeading(1, Translation[trHeadlineUnits]);
  if Assigned(c) and (c^.Count > 0) then
    begin
      WriteStartOfTable2Columns(Translation[trName], Translation[trDescription]);
      for j := 0 to c^.Count - 1 do
        begin
          Item := c^.At(j);
          WriteStartOfTableRow;
          WriteStartOfTableCell;
          WriteStartOfLink(Item^.FullLink);
          WriteString(Item^.Name);
          WriteEndOfLink;
          WriteEndOfTableCell;

          if j = 0 then
            Stream^.WriteString('<TD width=100%>')
          else
            WriteStartOfTableCell;
          WriteItemDescription(Item);
          WriteEndOfTableCell;
          WriteEndOfTableRow;
        end;
      WriteEndOfTable;
    end;
  WriteFooter;
  WriteAppInfo;
  WriteEndOfDocument;
  CloseStream;
end;

procedure THTMLDocGenerator.WriteVisibilityCell(const Item: PItem);
begin
  WriteStartOfTableCell;
  case Item^.State of
    STATE_PRIVATE:
      Stream^.WriteString('<IMG src="private.gif" alt="' + Translation[trPrivate] + '">');
    STATE_PROTECTED:
      Stream^.WriteString('<IMG src="protected.gif" alt="' + Translation[trProtected] + '">');
    STATE_PUBLIC:
      Stream^.WriteString('<IMG src="public.gif" alt="' + Translation[trPublic] + '">');
    STATE_PUBLISHED:
      Stream^.WriteString('<IMG src="published.gif" alt="' + Translation[trPrivate] + '">');
  end;
  WriteEndOfTableCell;
end;

end.

