{
@abstract(basic doc generator object)
@author(Ralf Junker (delphi@zeitungsjunge.de))
@author(Ivan Montes Velencoso (senbei@teleline.es))
@author(Marco Schmidt (marcoschmidt@geocities.com))
@author(Philippe Jean Dit Bailleul (jdb@abacom.com))
@author(Rodrigo Urubatan Ferreira Jardim (rodrigo@netscape.net))
@created(30 Aug 1998)
@lastmod(20 Apr 2000)

GenDoc contains the basic documentation generator object @link(TDocGenerator).
It is not sufficient by itself but the basis for all generators that produce
documentation in a specific format like HTML or LaTex.
They override @link(TDocGenerator)'s virtual methods.
}
unit GenDoc;

{$I platform.inc}

interface

uses
  Arrays,
  // Chars,
  FileStre,
  Items,
  Languages,
  Numbers,
  Objects,
  Parsing,
  Streams,
  Texts;

const
  { number of overview files that pasdoc generates for
    multiple-document-formats like @link(HTML) }
  NUM_OVERVIEW_FILES = 7;

  { names of all overview files, extensions not included }
  OverviewFilenames: array[0..NUM_OVERVIEW_FILES - 1] of string[8] =
  ('AllUnits',
    'AllClass',
    'AllTypes',
    'AllVaria',
    'AllConst',
    'AllFuncs',
    'AllIdent');

type
  { pointer to @link(TDocGenerator) }
  PDocGenerator = ^TDocGenerator;
  { @abstract(basic documentation generator object)
    @author(Marco Schmidt (marcoschmidt@geocities.com))
    This abstract object will do the complete process of writing
    documentation files.
    It will be given the collection of units that was the result of the
    parsing process and a configuration object that was created from default
    values and program parameters.
    Depending on the output format, one or more files may be created (HTML
    will create several, Tex only one). }
  TDocGenerator = object(TObject)
    { destination directory for documentation; must include terminating
      forward slash or backslash so that valid file names can be created
      by concatenating DestDir and a pathless file name }
    DestDir: string;
    // KeepLineFeeds: Boolean;
    { the (human) output language of the documentation file(s);
      one of the LANG_xxx constants, e.g. @link(LANG_ENGLISH);
      default language is @link(DEFAULT_LANGUAGE) }
    Header: AnsiString;
    Footer: AnsiString;
    Language: TLanguageID;
    { Name of the project to create. }
    ProjectName: AnsiString;
    { if true, no link to pasdoc homepage will be included at the bottom of
      HTML files;
      default is false }
    NoGeneratorInfo: Boolean;
    { the output stream that is currently written to; depending on the
      output format, more than one output stream will be necessary to
      store all documentation }
    Stream: POutputStream;
    { Title of documentation. }
    Title: string;
    { list of all units that were successfully parsed }
    Units: PItemCollection;
    constructor Init;
    { Creates anchors and links for all items in all units. }
    procedure BuildLinks;
    { Checks if D is assigned and empty - if so, disposes of D and sets it to
      nil.
      If there are characters in D, it is checked whether at least one
      non-whitespace character is present - if all characters are whitespace,
      disposes of D. }
    procedure CheckForEmptyDescription(var d: PText);
    { If field @link(Stream) is assigned, it is disposed and set to nil. }
    procedure CloseStream;
    { Makes a string look like a coded string, i.e. <CODE>TheString</CODE>
      in Html. }
    function CodeString(const s: ansistring): ansistring; virtual;
    { Character conversion function - heirs are supposed to override this
      version to return escaped versions for special characters, like a
      backslash in @link(Tex) or an ampersand character in @link(HTML).
      This version returns an empty string for all input characters. }
    function ConvertChar(c: Char): string; virtual;
    { Calls @link(ConvertChar) for each character in S, thus assembling a
      string that is returned and can be written to the documentation file. }
    function ConvertString(s: string): string;
    { Abstract function to be overwritten in descendants.
      This function is supposed to return a reference to an item, that is the
      name combined with some linking information like a hyperlink element in
      HTML or a page number in Tex. }
    function CreateLink(Item: PItem): string; virtual;
    { If field @link(Stream) still exists (@<@> nil), it is closed.
      Then, a new output stream in the destination directory with given
      name and file extension typical for this document format is created and
      assigned to Stream.
      No path or extension should therefore be in Name.
      Typical values for Name would be 'Objects' or 'AllUnits'.
      Returns true if creation was successful, false otherwise. }
    function CreateStream(Name: string): Boolean;
    { Must be overwritten.
      From an item name and its link, this creates a language-specific
      reference to that item. }
    function CreateReferencedLink(ItemName, Link: string): string; virtual;
    { Takes description D of the item Item, expands links (using Item),
      converts output-specific characters.
      Returns true on success, false otherwise (not enough memory?). }
    function ExpandDescription(Item: PItem; var d: PText): Boolean;
    { Calls @link(ExpandDescription) for each item in each unit of
      @link(Units). }
    procedure ExpandDescriptions;
    { Searches for an email address in string S. Searches for first appearance
      of the @@ character}
    function ExtractEmailAddress(s: string; var s1, s2, EmailAddress: string): Boolean;
    { Searches all items in all units (given by field @link(Units)) for item
      S1.S2.S3 (first N  strings not empty).
      Returns a pointer to the item on success, nil otherwise. }
    function FindGlobal(const s1, s2, S3: string; n: Integer): PItem;
    function GetCIOTypeName(MyType: TCIOType): string;
    { Abstract function that provides file extension for documentation format.
      Must be overwritten by descendants. }
    function GetFileExtension: string; virtual;
    function GetSpecialCharacter(IntValue: Integer): string; virtual;
    { Loads descriptions from file N and replaces or fills the corresponding
      comment sections of items. }
    procedure LoadDescriptionFile(n: string);
    { Assumes C contains file names as PString variables.
      Calls @link(LoadDescriptionFile) with each file name. }
    procedure LoadDescriptionFiles(c: PStringCollection);
    function SearchItem(s: string; const Item: PItem): PItem;
    { Searches for an item of name S which was linked in the description
      of Item. Starts search within item, then does a search on all items in all
      units using @link(FindGlobal).
      Returns a link as string on success or an empty string on failure. }
    function SearchLink(s: string; const Item: PItem): string;
    { A link provided in a tag can be made up of up to three parts,
      separated by dots.
      If this link is not a valid identifier or if it has more than
      three parts, false is returned, true otherwise.
      The parts are returned in S1, S2 and S3, with the number of
      parts minus one being returned in N. }
    function SplitLink(s: string; var s1, s2, S3: string; var n: Integer): Boolean;
    procedure StoreDescription(ItemName: string; var t: PText);
    { Writes the names of the authors to standard output, one author per row.
      The collection Authors is expected to store zero or more names,
      one name per string.
      The heading Author(s) (@link(trAUTHOR) or @link(trAUTHORS)) is
      written in the current language at heading level HL. }
    procedure WriteAuthors(HL: Byte; Authors: PStringCollection);
    { Converts C to output format using @link(ConvertChar) and writes
      resulting string to output using @link(Stream)'s
      @link(TOutputStream.WriteString) method. }
    procedure WriteChar(c: Char);
    { Writes all information on a class, object or interface (CIO) to output,
      at heading level HL. }
    procedure WriteCIO(HL: Byte; const CIO: PCIO); virtual;
    { Writes all classes, interfaces and objects in C to output, calling
      @link(WriteCIO) with each, at heading level HL. }
    procedure WriteCIOs(HL: Byte; c: PItemCollection); virtual;
    { Abstract procedure, must be overwritten by descendants.
      Writes a list of all classes, interfaces and objects in C at heading
      level HL to output. }
    procedure WriteCIOSummary(HL: Byte; c: PItemCollection); virtual;
    { Writes collection T, which is supposed to contain constant items only
      to output at heading level HL with heading Translation[trTYPES) calling
      @link(WriteItems).
      Can be overwritten by descendants. }
    procedure WriteConstants(HL: Byte; c: PItemCollection); virtual;
    { If they are assigned, the date values for creation time and time of last
      modification are written to output at heading level HL. }
    procedure WriteDates(HL: Byte; Created, LastMod: PString); virtual;
    { Writes an already-converted description T to output.
      Takes @link(TItem.DetailedDescription) if available,
      @link(TItem.Description) otherwise.
      If none of them is assigned, nothing is written. }
    procedure WriteDescription(HL: Byte; const Heading: string; const Item: TItem);
    { Abstract procedure, must be overwritten.
      Writes all documentation.
      Will create either a single file or one file for each unit and each
      class, interface or object, depending on output format. }
    procedure WriteDocumentation; virtual;
    { Writes a list of functions / procedure or constructors / destructors /
      methods I to output.
      Heading level HL is used.
      If Methods is true, the 'Methods' heading is used, 'Functions and
      procedures' otherwise.
      Usually, a list of all items is written first, followed by detailed
      descriptions of each item.
      However, this is dependent on the output format. }
    procedure WriteFuncsProcs(HL: Byte; Methods: Boolean; FuncsProcs: PMethodCollection); virtual;
    { Abstract procedure that must be overwritten by descendants.
      Writes a heading S at level HL to output.
      In HTML, heading levels are regarded by choosing the appropriate
      element from H1 to H6.
      In TeX, headings will result in chapter, section, subsection or
      subsubsection elements.
      The minimum heading level is 1, the maximum level depends on the
      output format.
      However, it is no good idea to choose a heading level larger than
      five or six.
      Anyway, a descendant should be able to deal with to large HL values,
      e.g. by assigning subsubsection to all Tex headings >= 4. }
    procedure WriteHeading(HL: Byte; const s: string); virtual;

    procedure WriteBinaryFiles; virtual;
    { Writes items in I to output, including a heading of level HL and text
      Heading.
      Each item in I should be written with its short description and a
      reference.
      In HTML, this results in a table with two columns. }
    procedure WriteItems(HL: Byte; Heading: string; const Anchor: string; i: PItemCollection); virtual;
    { Calls @link(WriteString) with S, then writes a line feed. }
    procedure WriteLine(const s: string);
    { Abstract method, must be overwritten by descendants to implement
      functionality.
      Writes a list of properties P to output.
      Heading level HL is used for the heading Translation[trPROPERTIES). }
    procedure WriteProperties(HL: Byte; p: PPropertyCollection); virtual;
    { Writes a resources to DestDir. Existing files will not be overwritten. }
    procedure WriteResourceToFile(const ResourceName, ResourceType: PChar; const FileName: AnsiString);
    { Writes string S to output, converting each character using
      @link(ConvertChar). }
    procedure WriteString(const s: string);
    { Simply copies characters in text T to output. }
    procedure WriteText(t: PText); virtual;
    { Writes collection T, which is supposed to contain type items (TItem) to
      output at heading level HL with heading Translation[trTYPES) calling
      @link(WriteItems).
      Can be overwritten in descendants. }
    procedure WriteTypes(HL: Byte; t: PItemCollection); virtual;
    { Abstract method that writes all documentation for a single unit U to
      output, starting at heading level HL.
      Implementation must be provided by descendant objects and is dependent
      on output format.
      Will call some of the WriteXXX methods like @link(WriteHeading),
      @link(WriteCIOs) or @link(WriteUnitDescription). }
    procedure WriteUnit(HL: Byte; U: PUnit); virtual;
    { Abstract method to be implemented by descendant objects.
      Writes the (detailed, if available) description T of a unit to output,
      including a Translation[trDESCRIPTION) headline at heading level HL. }
    procedure WriteUnitDescription(HL: Byte; U: PUnit); virtual;
    { Writes documentation for all units, calling @link(WriteUnit) for each
      unit. }
    procedure WriteUnits(HL: Byte);
    { Writes collection V, which is supposed to contain variable items (TItem)
      to output at heading level HL with heading Translation[trTYPES) calling
      @link(WriteItems).
      Can be overwritten in descendants. }
    procedure WriteVariables(HL: Byte; V: PItemCollection); virtual;

    procedure WriteStartOfCode; virtual;
    procedure WriteEndOfCode; virtual;
  end;

  { Creates a string from a number, concatenating the string 'link' and a string
    version of Number. }
function AnchorToString(Number: LongInt): string;

{$IFNDEF PPC_DELPHI6}
{ @abstract(DirectoryExists determines whether a specified directory exists.)
  Call @Name to determine whether the directory specified by the Name parameter
  exists. If the directory exists, the function returns True. If the directory
  does not exist, the function returns False.
  <P>If a full path name is entered, DirectoryExists searches for the directory
  along the designated path. Otherwise, the Name parameter is interpreted as a
  relative path name within the current directory. }
function DirectoryExists(const Name: AnsiString): Boolean;
{$ENDIF}

implementation

uses
  {$IFNDEF PPC_KYLIX}
  Windows,
  {$ELSE}
  Classes,
  {$ENDIF}
  SysUtils,

  Msg;

{ ---------------------------------------------------------------------------- }
{ Utilities
{ ---------------------------------------------------------------------------- }

function AnchorToString(Number: LongInt): string;
begin
  AnchorToString := {'link' +} IntToStr(Number);
end;

{$IFNDEF PPC_DELPHI6}
{ For Kylix and Delphi 6, DirectoryExists already exists in SysUtils.pas.
  In earlier Delphi versions it is part of FileCtrl.pas, which blows up this
  small programm's exe size by including the entire Forms.pas code. Since this
  is not what we want, we duplicate it here. }
function DirectoryExists(const Name: AnsiString): Boolean;
var
  Code: Cardinal;
begin
  if Pointer(Name) <> nil then
    begin
      Code := GetFileAttributes(Pointer(Name));
      Result := (Code <> $FFFFFFFF) and (Code and FILE_ATTRIBUTE_DIRECTORY <> 0);
    end
  else
    Result := False;
end;
{$ENDIF}

{ ---------------------------------------------------------------------------- }
{ TDocGenerator
{ ---------------------------------------------------------------------------- }

constructor TDocGenerator.Init;
begin
  inherited Init;
  Stream := nil;
end;

procedure TDocGenerator.BuildLinks;

  procedure AssignLinks(MyUnit: PUnit; MyObject: PCIO;
    const DocName: string; c: PItemCollection; var CurrentNumber: LongInt);
  var
    i: LongInt;
    p: PItem;
  begin
    if (not Assigned(c)) or (c^.Count < 1) then Exit;
    for i := 0 to c^.Count - 1 do
      begin
        p := c^.At(i);
        p^.AnchorNumber := CurrentNumber;
        p^.MyObject := MyObject;
        p^.MyUnit := MyUnit;
        p^.FullLink := CreateLink(p);
        p^.HandleAuthorTags;
        p^.HandleCreatedTag;
        p^.HandleLastModTag;
        p^.HandleAbstractTag;
        Inc(CurrentNumber);
      end;
  end;

var
  {  C: PItemCollection;}
  CO: PCIO;
  i: LongInt;
  {  Item: PItem;}
  j: LongInt;
  {  K: LongInt;}
  n: LongInt;
  U: PUnit;
begin
  PrintLn(3, 'Assigning links...');
  if (not Assigned(Units)) or (Units^.Count < 1) then
    Exit;
  n := 0;
  for i := 0 to Units^.Count - 1 do
    begin
      Inc(n);
      U := Units^.At(i);
      U^.AnchorNumber := n;
      U^.FullLink := CreateLink(U);
      U^.OutputFileName := DestDir + U^.FullLink;
      U^.HandleAuthorTags;
      U^.HandleCreatedTag;
      U^.HandleLastModTag;
      U^.HandleAbstractTag;
      AssignLinks(U, nil, U^.FullLink, U^.Constants, n);
      AssignLinks(U, nil, U^.FullLink, U^.Variables, n);
      AssignLinks(U, nil, U^.FullLink, U^.Types, n);
      AssignLinks(U, nil, U^.FullLink, U^.FuncsProcs, n);
      if Assigned(U^.CIO) and (U^.CIO^.Count > 0) then
        begin
          for j := 0 to U^.CIO^.Count - 1 do
            begin
              Inc(n);
              CO := U^.CIO^.At(j);
              CO^.AnchorNumber := n;
              CO^.MyUnit := U;

              // CO^.OutputFileName := DestDir + CO^.Name + GetFileExtension;
              // CO^.FullLink := CreateLink(CO);

              CO^.FullLink := CreateLink(CO);
              CO^.OutputFileName := DestDir + CO^.FullLink;

              CO^.HandleAuthorTags;
              CO^.HandleCreatedTag;
              CO^.HandleLastModTag;
              CO^.HandleAbstractTag;
              { Do not reset N - gives problems with TeX output}
              Inc(n);
              AssignLinks(U, CO, CO^.FullLink, CO^.Fields, n);
              AssignLinks(U, CO, CO^.FullLink, CO^.Methods, n);
              AssignLinks(U, CO, CO^.FullLink, CO^.Properties, n);
            end;
        end;
    end;
  PrintLn(3, ' ' + IntToStr(n) + ' items linked.');
end;

procedure TDocGenerator.CheckForEmptyDescription(var d: PText);
var
  i: LongInt;
  NonWhitespaceFound: Boolean;
begin
  if (not Assigned(d)) then Exit;

  if (d^.Content < 1) then
    begin
      Dispose(d, Done);
      d := nil;
      Exit;
    end;
  i := 0;
  repeat
    NonWhitespaceFound := not (d^.Data[i] in Whitespace);
    Inc(i);
  until (i = d^.Content) or NonWhitespaceFound;
  if (not NonWhitespaceFound) then
    begin
      Dispose(d, Done);
      d := nil;
      Exit;
    end;
end;

procedure TDocGenerator.CloseStream;
begin
  if Assigned(Stream) then
    begin
      Dispose(Stream, Done);
      Stream := nil;
    end;
end;

function TDocGenerator.ConvertChar(c: Char): string;
begin
  Result := '';
end;

function TDocGenerator.ConvertString(s: string): string;
var
  i: Integer;
  t: string;
begin
  t := '';
  i := 1;
  while (i <= Length(s)) do
    begin
      t := t + ConvertChar(s[i]);
      Inc(i);
    end;
  ConvertString := t;
end;

function TDocGenerator.CreateLink(Item: PItem): string;
begin
  CreateLink := Item^.Name;
end;

function TDocGenerator.CreateStream(Name: string): Boolean;
begin
  CloseStream;
  // Name := DestDir + Name + GetFileExtension;
  PrintLn(4, 'Creating output stream "' + Name + '".');
  Stream := New(PFileOutputStream, Init(Name, True));
  Result := (Stream <> nil);
end;

function TDocGenerator.CreateReferencedLink(ItemName, Link: string): string;
begin
  { abstract }
end;

function TDocGenerator.CodeString(const s: ansistring): ansistring;
begin
  Result := s;
end;

function TDocGenerator.ExpandDescription(Item: PItem; var d: PText): Boolean;
var
  Ancestor: PCIO;
  i, Run: LongInt;
  {  LinkedItem: PItem;}
  Offs1: LongInt;
  Offs2: LongInt;
  Offs3: LongInt;
  TheLink: string;
  s: AnsiString;
  t: PText;
begin
  Result := True;
  { check for cases "no id" and "id is empty" }
  if (not Assigned(d)) then Exit;

  if (d^.Content < 1) then
    begin
      Dispose(d, Done);
      d := nil;
      Exit;
    end;

  { create temporary TText object }
  t := New(PText, Init);
  Run := 0;
  repeat
    if (d^.Data[Run] = '@') then
      begin
        { this is @@ (literal '@')? }
        if (Run < d^.Content - 1) and (d^.Data[Run + 1] = '@') then
          begin
            { literal @ }
            t^.AppendChar('@');
            Inc(Run, 2);
          end
        else
          { Is it @Name?
            * Name must follow directly after @.
            * There are no brackets after @Name. }
          if (Run < d^.Content - 4) and
            ((d^.Data[Run + 1] = 'N') or (d^.Data[Run + 1] = 'n')) and
            ((d^.Data[Run + 2] = 'A') or (d^.Data[Run + 2] = 'a')) and
            ((d^.Data[Run + 3] = 'M') or (d^.Data[Run + 3] = 'm')) and
            ((d^.Data[Run + 4] = 'E') or (d^.Data[Run + 4] = 'e')) then
            begin
              t^.AppendString(CodeString(Item^.Name));
              Inc(Run, 5);
            end
          else
            if (Run < d^.Content - 9) and
              ((d^.Data[Run + 1] = 'C') or (d^.Data[Run + 1] = 'c')) and
              ((d^.Data[Run + 2] = 'L') or (d^.Data[Run + 2] = 'l')) and
              ((d^.Data[Run + 3] = 'A') or (d^.Data[Run + 3] = 'a')) and
              ((d^.Data[Run + 4] = 'S') or (d^.Data[Run + 4] = 's')) and
              ((d^.Data[Run + 5] = 'S') or (d^.Data[Run + 5] = 's')) and
              ((d^.Data[Run + 6] = 'N') or (d^.Data[Run + 6] = 'n')) and
              ((d^.Data[Run + 7] = 'A') or (d^.Data[Run + 7] = 'a')) and
              ((d^.Data[Run + 8] = 'M') or (d^.Data[Run + 8] = 'm')) and
              ((d^.Data[Run + 9] = 'E') or (d^.Data[Run + 9] = 'e')) and
              Assigned(Item^.MyObject) then
              begin
                t^.AppendString(CodeString(Item^.MyObject^.Name));
                Inc(Run, 10);
              end
            else
              { Is it @True? }
              if (Run < d^.Content - 4) and
                ((d^.Data[Run + 1] = 'T') or (d^.Data[Run + 1] = 't')) and
                ((d^.Data[Run + 2] = 'R') or (d^.Data[Run + 2] = 'r')) and
                ((d^.Data[Run + 3] = 'U') or (d^.Data[Run + 3] = 'u')) and
                ((d^.Data[Run + 4] = 'E') or (d^.Data[Run + 4] = 'e')) then
                begin
                  t^.AppendString(CodeString('True'));
                  Inc(Run, 5);
                end
              else
                { Is it @False ? }
                if (Run < d^.Content - 5) and
                  ((d^.Data[Run + 1] = 'F') or (d^.Data[Run + 1] = 'f')) and
                  ((d^.Data[Run + 2] = 'A') or (d^.Data[Run + 2] = 'a')) and
                  ((d^.Data[Run + 3] = 'L') or (d^.Data[Run + 3] = 'l')) and
                  ((d^.Data[Run + 4] = 'S') or (d^.Data[Run + 4] = 's')) and
                  ((d^.Data[Run + 5] = 'E') or (d^.Data[Run + 5] = 'e')) then
                  begin
                    t^.AppendString(CodeString('False'));
                    Inc(Run, 6);
                  end
                else
                  { Is it @nil ? }
                  if (Run < d^.Content - 3) and
                    ((d^.Data[Run + 1] = 'N') or (d^.Data[Run + 1] = 'n')) and
                    ((d^.Data[Run + 2] = 'I') or (d^.Data[Run + 2] = 'i')) and
                    ((d^.Data[Run + 3] = 'L') or (d^.Data[Run + 3] = 'l')) then
                    begin
                      t^.AppendString(CodeString('nil'));
                      Inc(Run, 4);
                    end
                  else
                    if (Run < d^.Content - 9) and
                      ((d^.Data[Run + 1] = 'I') or (d^.Data[Run + 1] = 'i')) and
                      ((d^.Data[Run + 2] = 'N') or (d^.Data[Run + 2] = 'n')) and
                      ((d^.Data[Run + 3] = 'H') or (d^.Data[Run + 3] = 'h')) and
                      ((d^.Data[Run + 4] = 'E') or (d^.Data[Run + 4] = 'e')) and
                      ((d^.Data[Run + 5] = 'R') or (d^.Data[Run + 5] = 'r')) and
                      ((d^.Data[Run + 6] = 'I') or (d^.Data[Run + 6] = 'i')) and
                      ((d^.Data[Run + 7] = 'T') or (d^.Data[Run + 7] = 't')) and
                      ((d^.Data[Run + 8] = 'E') or (d^.Data[Run + 8] = 'e')) and
                      ((d^.Data[Run + 9] = 'D') or (d^.Data[Run + 9] = 'd')) and
                      Assigned(Item^.MyObject) then
                      begin
                        // Try to find inherited property of item.
                        // Updated 25 Feb 2002
                        if Assigned(Item^.MyObject^.Ancestors) and (Item^.MyObject^.Ancestors^.Count > 0) then
                          begin
                            s := PString(Item^.MyObject^.Ancestors^.At(0))^;
                            Ancestor := PCIO(SearchItem(s, Item));
                            if Assigned(Ancestor) then
                              repeat
                                TheLink := SearchLink(s + '.' + Item^.Name, Item);
                                if TheLink <> '' then Break;

                                if Assigned(Ancestor^.Ancestors) and (Ancestor^.Ancestors^.Count > 0) then
                                  begin
                                    s := PString(Ancestor^.Ancestors^.At(0))^;
                                    Ancestor := PCIO(SearchItem(s, Ancestor));
                                  end
                                else
                                  Break;
                              until Ancestor = nil;
                          end;

                        { for i := 0 to Item^.MyObject^.Ancestors^.Count - 1 do
                            begin
                              s := PString(Item^.MyObject^.Ancestors^.At(i))^;
                              TheLink := SearchLink(s + '.' + Item^.Name, Item);
                              if TheLink <> '' then Break;
                            end;}
                        if TheLink <> '' then
                          begin
                            Inc(Run, 10);
                            t^.AppendString(TheLink);
                          end
                        else
                          begin
                            Inc(Run);
                            if Assigned(Item^.MyUnit) then
                              PrintLn(2, 'Warning: Could not expand @Inherited of "' + Item^.Name + '" in unit ' + Item^.MyUnit^.Name)
                            else
                              PrintLn(2, 'Warning: Could not expand @Inherited of "' + Item^.Name + '"');
                            t^.AppendString('WARNING: @');
                          end;
                      end
                    else
                      { Is it @<? }
                      if (Run < d^.Content - 1) and
                        (d^.Data[Run + 1] = '<') then
                        begin
                          t^.AppendString('&lt;');
                          Inc(Run, 2);
                        end
                      else
                        { Is it @>? }
                        if (Run < d^.Content - 1) and
                          (d^.Data[Run + 1] = '>') then
                          begin
                            t^.AppendString('&gt;');
                            Inc(Run, 2);
                          end
                        else
                          { Is it @<? }
                          if (Run < d^.Content - 1) and
                            (d^.Data[Run + 1] = '&') then
                            begin
                              t^.AppendString('&amp;');
                              Inc(Run, 2);
                            end
                          else
                            { Is it @=? }
                            if (Run < d^.Content - 1) and
                              (d^.Data[Run + 1] = '=') then
                              begin
                                t^.AppendString('&quot;');
                                Inc(Run, 2);
                              end
                            else
                              begin
                                Offs1 := Run;
                                if d^.FindTag('LINK', Offs1, Offs2, Offs3) then
                                  begin
                                    d^.GetTag(False, Offs1, Offs2, Offs3, s);

                                    while Run < Offs1 do
                                      begin
                                        t^.AppendString(ConvertChar(d^.Data[Run]));
                                        Inc(Run);
                                      end;

                                    Run := Offs3 + 1;
                                    TheLink := SearchLink(s, Item);

                                    if Assigned(Item^.MyUnit) then
                                      if CompareText(Item^.MyUnit^.FullLink, Copy(TheLink, 1, Length(Item^.MyUnit^.FullLink))) = 0 then
                                        Delete(TheLink, 1, Length(Item^.MyUnit^.FullLink));

                                    t^.AppendString(SearchLink(s, Item));
                                  end
                                else

                                  begin
                                    Offs1 := Run;
                                    if d^.FindTag('CODE', Offs1, Offs2, Offs3) then
                                      begin
                                        d^.GetTag(False, Offs1, Offs2, Offs3, s);
                                        while Run < Offs1 do
                                          begin
                                            t^.AppendString(ConvertChar(d^.Data[Run]));
                                            Inc(Run);
                                          end;
                                        Run := Offs3 + 1;
                                        t^.AppendString(CodeString(s));
                                      end
                                    else
                                      begin
                                        Inc(Run);
                                        // s := d^.GetTagName(Run);
                                        // if //(s <> 'ABSTRACT') and
                                        //  (s <> 'AUTHOR') and
                                        //  (s <> 'EXCLUDE') then
                                        //   begin
                                        if Assigned(Item^.MyUnit) then
                                          PrintLn(2, 'Warning: Found non-link tag when expanding descriptions of "' + Item^.Name + '" in unit ' + Item^.MyUnit^.Name)
                                        else
                                          PrintLn(2, 'Warning: Found non-link tag when expanding descriptions of "' + Item^.Name + '"');
                                        t^.AppendString('WARNING: @');
                                        //  end;
                                        //  Inc(Run);
                                      end;
                                  end;
                              end;
      end
    else
      begin
        if (d^.Data[Run] in [#9, #13, #10]) then d^.Data[Run] := ' ';

        { if (not KeepLineFeeds) and (j > 80) and (d^.Data[Run] in Whitespace) then
           begin
             t^.AppendString (#10);
             j := 0;
           end; }

        t^.AppendString(ConvertChar(d^.Data[Run]));
        Inc(Run);
      end;
  until (Run >= d^.Content);

  Dispose(d, Done);
  d := t;
end;

procedure TDocGenerator.ExpandDescriptions;

{ expands Description and DetailedDescription of Item }
  procedure ExpandItem(Item: PItem);
  begin
    if (not Assigned(Item)) then Exit;

    // RJ
    if Assigned(Item^.Description) then
      Item^.Description^.CompressWhiteSpace;
    if Assigned(Item^.DetailedDescription) then
      Item^.DetailedDescription^.CompressWhiteSpace;

    CheckForEmptyDescription(Item^.Description);
    CheckForEmptyDescription(Item^.DetailedDescription);
    if (not ExpandDescription(Item, Item^.Description)) or
      (not ExpandDescription(Item, Item^.DetailedDescription)) then
      begin
        PrintLn(2, 'Warning: Could not expand description from ' + Item^.Name);
      end;
  end;

  { for all items in collection C, expands descriptions }
  procedure ExpandCollection(c: PItemCollection);
  var
    i: LongInt;
    p: PItem;
    {T: PText;}
  begin
    if (not Assigned(c)) or (c^.Count < 1) then Exit;
    for i := 0 to c^.Count - 1 do
      begin
        p := c^.At(i);
        ExpandItem(p);
      end;
  end;

var
  {C: PItemCollection;}
  CO: PCIO;
  i: LongInt;
  j: LongInt;
  U: PUnit;
begin
  PrintLn(3, 'Expanding descriptions...');
  if (not Assigned(Units)) or
    (Units^.Count < 1) then Exit;
  for i := 0 to Units^.Count - 1 do
    begin
      U := Units^.At(i);
      ExpandItem(U);
      ExpandCollection(U^.Constants);
      ExpandCollection(U^.Variables);
      ExpandCollection(U^.Types);
      ExpandCollection(U^.FuncsProcs);
      if Assigned(U^.CIO) and (U^.CIO^.Count > 0) then
        begin
          for j := 0 to U^.CIO^.Count - 1 do
            begin
              CO := U^.CIO^.At(j);
              ExpandItem(CO);
              ExpandCollection(CO^.Fields);
              ExpandCollection(CO^.Methods);
              ExpandCollection(CO^.Properties);
            end;
        end;
    end;
end;

function TDocGenerator.ExtractEmailAddress(s: string; var s1, s2, EmailAddress: string): Boolean;
const
  ALLOWED_CHARS = ['a'..'z', 'A'..'Z', '-', '.', '_', '0'..'9'];
  Letters = ['a'..'z', 'A'..'Z'];
var
  atPos: Integer;
  i: Integer;
begin
  ExtractEmailAddress := False;
  if (Length(s) < 6) { minimum length of email address: a@b.cd } then Exit;
  atPos := Pos('@', s);
  if (atPos < 2) or (atPos > Length(s) - 3) then Exit;
  { assemble address left of @ }
  i := atPos - 1;
  while (i >= 1) and (s[i] in ALLOWED_CHARS) do
    Dec(i);
  EmailAddress := System.Copy(s, i + 1, atPos - i - 1) + '@';
  s1 := '';
  if (i > 1) then s1 := System.Copy(s, 1, i);
  { assemble address right of @ }
  i := atPos + 1;
  while (i <= Length(s)) and (s[i] in ALLOWED_CHARS) do
    Inc(i);
  EmailAddress := EmailAddress + System.Copy(s, atPos + 1, i - atPos - 1);
  if (Length(EmailAddress) < 6) or
    (not (EmailAddress[Length(EmailAddress)] in Letters)) or
  (not (EmailAddress[Length(EmailAddress) - 1] in Letters)) then Exit;
  s2 := '';
  if (i <= Length(s)) then s2 := System.Copy(s, i, Length(s) - i + 1);
  ExtractEmailAddress := True;
end;

function TDocGenerator.FindGlobal(const s1, s2, S3: string; n: Integer): PItem;
var
  i: LongInt;
  Item: PItem;
  CIO: PCIO;
  U: PUnit;
begin
  Result := nil;
  if (not Assigned(Units)) or (Units^.Count = 0) then Exit;
  case n of
    0: { }
      for i := 0 to Units^.Count - 1 do
        begin
          U := Units^.At(i);
          Item := U^.FindItem(s1);
          if Assigned(Item) then
            begin
              Result := Item;
              Exit;
            end;
        end;
    1:
      begin

        { object.field_method_property }
        for i := 0 to Units^.Count - 1 do
          begin
            U := Units^.At(i);
            if Assigned(U^.CIO) then
              begin
                CIO := Pointer(U^.CIO^.FindName(s1));
                if Assigned(CIO) then
                  begin
                    Item := CIO^.FindFieldMethodProperty(s2);
                    if Assigned(Item) then
                      begin
                        Result := Item;
                        Exit;
                      end;
                  end;
              end;
          end;

        { unit.cio_var_const_type }
        U := PUnit(Units^.FindName(s1));
        if Assigned(U) then
          begin
            Item := U^.FindItem(s2);
            Result := Item;
            Exit;
          end;
      end;
    2: { unit.objectorclassorinterface.fieldormethodorproperty }
      begin
        U := PUnit(Units^.FindName(s1));
        if (not Assigned(U)) then Exit;
        {      WriteLn('DEBUG1: Found unit ' + U^.Name);}
        Item := U^.FindItem(s2);
        if (not Assigned(Item)) then Exit;
        {      WriteLn('DEBUG2: Found item ' + Item^.Name);}
        Item := Item^.FindItem(S3);
        if (not Assigned(Item)) then Exit;
        {      WriteLn('DEBUG3: Found item ' + Item^.Name);}
        Result := Item;
        Exit;
      end;
  end;
  Result := nil;
end;

function TDocGenerator.GetCIOTypeName(MyType: TCIOType): string;
var
  s: string;
begin
  case MyType of
    CIO_CLASS: s := Translation[trClass];
    CIO_DISPINTERFACE: s := Translation[trDispInterface];
    CIO_INTERFACE: s := Translation[trInterface];
    CIO_OBJECT: s := Translation[trObject];
  else
    s := '';
  end;
  GetCIOTypeName := s;
end;

function TDocGenerator.GetFileExtension: string;
begin
  { abstract }
  GetFileExtension := '';
end;

function TDocGenerator.GetSpecialCharacter(IntValue: Integer): string;
begin
  { abstract }
  Result := '(UNDEFINED SPECIAL CHARACTER)';
end;

procedure TDocGenerator.LoadDescriptionFile(n: string);
var
  f: PFileInputStream;
  ItemName: string;
  s: string;
  t: PText;
begin
  if (Length(n) < 1) then Exit;
  PrintLn(3, 'Loading descriptions from file "' + n + '"');
  f := New(PFileInputStream, Init(n));
  if (not Assigned(f)) then
    begin
      PrintLn(2, 'Error: Could not open description file "' + n + '"');
      Exit;
    end;
  t := nil;
  while (f^.HasData) do
    begin
      f^.ReadLine(s);
      {    PrintLn(4, 'DEBUG - descr "' + S + '"');}
      if (Length(s) > 0) then
        begin
          { # means: description of another item begins }
          if (s[1] = '#') then
            begin
              { if there is an old description, deal with it }
              StoreDescription(ItemName, t);
              { delete # char }
              System.Delete(s, 1, 1);
              { skip whitespace }
              while (Length(s) > 0) and (s[1] in [' ', #9]) do
                System.Delete(s, 1, 1);
              { find item }
              ItemName := '';
              while (Length(s) > 0) and (s[1] in ['A'..'Z', 'a'..'z', '_', '.', '0'..'9']) do
                begin
                  ItemName := ItemName + s[1];
                  System.Delete(s, 1, 1);
                end;
              t := New(PText, Init);
              {PrintLn(2, 'DEBUG - loaded description for "' + ItemName + '"');}
              { dispose of item's current description }
              {if Assigned(I) and Assigned(I^.Description) then
              begin
                Dispose(I^.Description, Done);
                T := New(PText, Init);
                if (not Assigned(T)) then
                begin
                end;
                I^.Description := T;
              end;}
            end
          else
            begin
              { check if there is a text }
              if (not Assigned(t)) then
                begin
                  PrintLn(2, 'Error: First line of description file must start with "# item_name"');
                  Break; { leave while loop }
                end;
              t^.AppendString(s + #10);
            end;
        end;
    end;
  StoreDescription(ItemName, t);
  Dispose(f, Done);
end;

procedure TDocGenerator.LoadDescriptionFiles(c: PStringCollection);
var
  i: LongInt;
  s: PString;
begin
  if (not Assigned(c)) or (c^.Count < 1) then Exit;
  PrintLn(3, 'Loading description files...');
  for i := 0 to c^.Count - 1 do
    begin
      s := PString(c^.At(i));
      if Assigned(s) then LoadDescriptionFile(s^);
    end;
end;

function TDocGenerator.SearchItem(s: string; const Item: PItem): PItem;
var
  n: Integer;
  s1: string;
  s2: string;
  S3: string;
begin
  { S is supposed to have 0 to 2 dots in it - S1, S2 and S3 contain
    the parts between the dots, N the number of dots }
  if (not SplitLink(s, s1, s2, S3, n)) then
    begin
      PrintLn(2, 'Warning: the link "' + s + '" is invalid');
      Result := nil;
      Exit;
    end;

  { first try to find link starting at Item }
  if Assigned(Item) then
    begin
      Result := Item^.FindName(s1, s2, S3, n);
    end
  else
    Result := nil;

  if (not Assigned(Result)) then Result := FindGlobal(s1, s2, S3, n);
end;

function TDocGenerator.SearchLink(s: string; const Item: PItem): string;
var
  i, n: Integer;
  s1: string;
  s2: string;
  S3: string;
  UnitName: string;
  FoundItem: PItem;
  U: PUnit;
begin
  { S is supposed to have 0 to 2 dots in it - S1, S2 and S3 contain
    the parts between the dots, N the number of dots }
  if (not SplitLink(s, s1, s2, S3, n)) then
    begin
      PrintLn(2, 'Warning: the link "' + s + '" in unit ' + Item^.MyUnit^.Name + ' is invalid');
      Result := 'UNKNOWN';
      Exit;
    end;

  { first try to find link starting at Item }
  if Assigned(Item) then
    begin
      FoundItem := Item^.FindName(s1, s2, S3, n);
    end
  else
    FoundItem := nil;

  { Next try to find link in items's unit uses units. }
  if FoundItem = nil then
    if Assigned(Item^.MyUnit) then
      if Assigned(Item^.MyUnit^.UsesUnits) then
        begin
          i := Item^.MyUnit^.UsesUnits^.Count;
          while i > 0 do
            begin
              Dec(i);
              UnitName := PString(Item^.MyUnit^.UsesUnits^.At(i))^;
              U := PUnit(Units^.FindName(UnitName));
              if U <> nil then
                // FoundItem := U^.FindName(s1, s2, S3, n);
                FoundItem := U^.FindFieldMethodProperty(s1, s2);
              if FoundItem <> nil then Break;
            end;
        end;

  { Find Global }
  if FoundItem = nil then
    FoundItem := FindGlobal(s1, s2, S3, n);

  if Assigned(FoundItem) then
    Result := CreateReferencedLink(FoundItem^.Name, FoundItem^.FullLink)
  else
    Result := ''; // RJ ConvertString(s);
end;

function TDocGenerator.SplitLink(s: string; var s1, s2, S3: string;
  var n: Integer): Boolean;

  procedure SplitInTwo(s: string; var s1, s2: string);
  var
    i: Integer;
  begin
    i := Pos('.', s);
    if (i = 0) then
      begin
        s1 := s;
        s2 := '';
      end
    else
      begin
        s1 := System.Copy(s, 1, i - 1);
        s2 := System.Copy(s, i + 1, Length(s));
      end;
  end;

var
  i: LongInt;
  t: string;
begin
  Result := False;
  s1 := '';
  s2 := '';
  S3 := '';
  n := 0;
  {  I := 1;}
  s := Trim(s);
  if (Length(s) = 0) then Exit;
  if (not (s[1] in IdentifierStart)) then Exit;
  i := 2;
  while (i <= Length(s)) do
    begin
      if (not (s[i] in IdentifierOther)) then Exit;
      Inc(i);
    end;
  SplitInTwo(s, s1, s2);
  if (Length(s2) = 0) then
    begin
      n := 0;
    end
  else
    begin
      t := s2;
      SplitInTwo(t, s2, S3);
      if (Length(S3) = 0) then
        n := 1
      else
        n := 2;
    end;
  Result := True;
end;

procedure TDocGenerator.StoreDescription(ItemName: string; var t: PText);
var
  Item: PItem;
  n: Integer;
  s1: string;
  s2: string;
  S3: string;
begin
  if (not Assigned(t)) then
    begin
      Exit;
    end;
  PrintLn(5, 'Storing description for ' + ItemName);
  if SplitLink(ItemName, s1, s2, S3, n) then
    begin
      {  	Write('DEBUG Split into ', N + 1, ' pieces: ');
         if (N >= 0) then Write(S1 + ' ');
         if (N >= 1) then Write(S2 + ' ');
         if (N >= 2) then Write(S3 + ' ');
         WriteLn;}
      Item := FindGlobal(s1, s2, S3, n);
      if Assigned(Item) then
        begin
          if Assigned(Item^.Description) then
            begin
              PrintLn(2, 'Warning - more than one description for ' + ItemName);
              Dispose(t, Done);
            end
          else
            begin
              Item^.Description := t;
            end;
        end
      else
        begin
          PrintLn(2, 'Warning - could not find item ' + ItemName);
          Dispose(t, Done);
        end;
    end
  else
    begin
      PrintLn(2, 'Warning: Could not split item "' + ItemName + '"');
    end;
  t := nil;
end;

procedure TDocGenerator.WriteAuthors(HL: Byte; Authors: PStringCollection);
begin
  { abstract }
end;

procedure TDocGenerator.WriteChar(c: Char);
begin
  Stream^.WriteString(ConvertChar(c));
end;

procedure TDocGenerator.WriteCIO(HL: Byte; const CIO: PCIO);
begin
  { abstract }
end;

procedure TDocGenerator.WriteCIOs(HL: Byte; c: PItemCollection);
var
  i: LongInt;
  p: PCIO;
begin
  if (not Assigned(c)) or (c^.Count < 1) then Exit;
  for i := 0 to c^.Count - 1 do
    begin
      p := c^.At(i);
      WriteCIO(HL, p);
    end;
end;

procedure TDocGenerator.WriteCIOSummary(HL: Byte; c: PItemCollection);
begin
  WriteItems(HL, Translation[trSummaryCio], 'Classes', c);
end;

procedure TDocGenerator.WriteConstants(HL: Byte; c: PItemCollection);
begin
  WriteItems(HL, Translation[trConstants], 'Constants', c);
end;

procedure TDocGenerator.WriteDates(HL: Byte; Created, LastMod: PString);
begin
  { abstract }
end;

procedure TDocGenerator.WriteDescription(HL: Byte; const Heading: string; const Item: TItem);
var
  d: PText;
begin
  if (Assigned(Item.DetailedDescription)) then
    d := Item.DetailedDescription
  else
    if (Assigned(Item.Description)) then
      d := Item.Description
    else
      Exit;

  if (Length(Heading) > 0) then
    WriteHeading(HL, Heading);
  WriteText(d);
end;

procedure TDocGenerator.WriteDocumentation;
begin
  { abstract }
end;

procedure TDocGenerator.WriteFuncsProcs(HL: Byte; Methods: Boolean; FuncsProcs: PMethodCollection);
begin
  { abstract }
end;

procedure TDocGenerator.WriteHeading(HL: Byte; const s: string);
begin
  { abstract }
end;

procedure TDocGenerator.WriteBinaryFiles;
begin
  { abstract }
end;

procedure TDocGenerator.WriteItems(HL: Byte; Heading: string; const Anchor: string; i: PItemCollection);
begin
  { abstract }
end;

procedure TDocGenerator.WriteLine(const s: string);
begin
  WriteString(s);
  Stream^.WriteLine('');
end;

procedure TDocGenerator.WriteProperties(HL: Byte; p: PPropertyCollection);
begin
  { abstract }
end;

{ ---------- }

procedure TDocGenerator.WriteResourceToFile(const ResourceName, ResourceType: PChar; const FileName: AnsiString);
var
  HResInfo: HRSRC;
  HGlobal: THandle;
  {$IFNDEF PPC_KYLIX}
  FileHandle: THandle;
  BytesWritten: DWORD;
  {$ENDIF}
begin
  HResInfo := FindResource(MainInstance, ResourceName, ResourceType);
  if HResInfo = 0 then Exit;

  HGlobal := LoadResource(MainInstance, HResInfo);
  if HGlobal = 0 then Exit;

  {$IFNDEF PPC_KYLIX}
  // Could use the same code as for Kylix (below), but keep Streams from Classes.pas out for now.
  FileHandle := CreateFile(PChar(DestDir + FileName), GENERIC_WRITE, 0, nil, CREATE_New, FILE_ATTRIBUTE_NORMAL or FILE_FLAG_SEQUENTIAL_SCAN, 0);
  if FileHandle = INVALID_HANDLE_VALUE then Exit;
  try
    WriteFile(FileHandle, LockResource(HGlobal)^, SizeOfResource(MainInstance, HResInfo), BytesWritten, nil)
  finally
    CloseHandle(FileHandle);
  end;
  {$ELSE}
  with TFileStream.Create(DestDir + FileName, fmOpenWrite) do begin
      try
        Write(LockResource(HGlobal)^, SizeOfResource(MainInstance, HResInfo));
      finally
        Free;
      end;
    end;
  {$ENDIF}
end;

{ ---------- }

procedure TDocGenerator.WriteString(const s: string);
var
  i: LongInt;
  l: LongInt;
  t: string;
begin
  l := Length(s);
  if (l = 0) then Exit;
  t := '';
  for i := 1 to l do
    t := t + ConvertChar(s[i]);
  Stream^.WriteString(t);
end;

procedure TDocGenerator.WriteText(t: PText);
var
  i: Integer;
begin
  if (t = nil) or (t^.Content < 1) then Exit;
  i := 0;
  while (i < t^.Content) do
    begin
      Stream^.WriteChar(t^.Data[i]);
      Inc(i);
    end;
end;

procedure TDocGenerator.WriteTypes(HL: Byte; t: PItemCollection);
begin
  WriteItems(HL, Translation[trTypes], 'Types', t);
end;

procedure TDocGenerator.WriteUnit(HL: Byte; U: PUnit);
begin
  { abstract }
end;

procedure TDocGenerator.WriteUnitDescription(HL: Byte; U: PUnit);
begin
  { abstract }
end;

procedure TDocGenerator.WriteUnits(HL: Byte);
var
  i: LongInt;
  p: PUnit;
begin
  if (not Assigned(Units)) or (Units^.Count < 1) then Exit;
  for i := 0 to Units^.Count - 1 do
    begin
      p := PUnit(Units^.At(i));
      WriteUnit(HL, p);
    end;
end;

procedure TDocGenerator.WriteVariables(HL: Byte; V: PItemCollection);
begin
  WriteItems(HL, Translation[trVariables], 'Variables', V);
end;

procedure TDocGenerator.WriteEndOfCode;
begin
  // abstract
end;

procedure TDocGenerator.WriteStartOfCode;
begin
  // abstract
end;

begin
end.

