{
@abstract(provides streams to read and write files)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@created(7 May 1998)
@lastmod(7 Feb 2000)

Read from or write to a file, having all functionality from unit @link(Streams).
Do not use these streams if you want to write to / read from the same file or
if you want to do random access.
}

unit FileStre;

{$I platform.inc}

interface

uses
  Streams;

type
  { pointer to @link(TFileInputStream) }
  PFileInputStream = ^TFileInputStream;
  { @abstract(file input stream)
    Overwrites @link(TInputStream) to implement an object to read from a file.
    Has a private field of type File which is used in combination with the RTL's
    functions to work on it (Assign, Reset, BlockRead, Close). }
  TFileInputStream = object (TInputStream)
    { call @link(TFileInputStream.Open) }
    constructor Init (FileName: string);
    { close file, keeping all buffers and other settings }
    procedure Close; virtual;
    function GetSize: TStreamOffset; virtual;
    function HasData: Boolean; virtual;
    { open file FileName for sequential (i.e. non-random-access) reading }
    procedure Open (FileName: string);
    { read N bytes from file to Dest }
    function ReadRawBytes (n: TStreamOffset; var Dest): TStreamOffset; virtual;
  private
    { file handle }
    f: file;
  end;
  { pointer to @link(TFileOutputStream) }
  PFileOutputStream = ^TFileOutputStream;
  { @abstract(file output stream)
    Overwrites @link(TOutputStream) to implement an object to write to a file.
    Has a private field of type File which is used in combination with the RTL's
    functions to work on it (Assign, Rewrite, BlockWrite, Close). }
  TFileOutputStream = object (TOutputStream)
    { Calls inherited @link(TOutputStream.Init) and then @link(Open). }
    constructor Init (FileName: string; CreateNewFile: Boolean);
    { Closes file, keeping all buffers and other stream settings }
    procedure Close; virtual;
    { Opens file FileName, creates new file if desired. Returns if we were
            successful. }
    procedure Open (FileName: string; CreateNewFile: Boolean);
    { }
    function WriteRawBytes (n: TStreamOffset; var Source): TStreamOffset; virtual;
  private
    { file handle }
    f: file;
  end;

implementation

{ TFileInputStream }

constructor TFileInputStream.Init (FileName: string);
begin
  inherited Init;
  Open (FileName);
  if (not IsOpen) then Fail;
end;

procedure TFileInputStream.Close;
var
  MustClose         : Boolean;
begin
  MustClose := IsOpen;
  inherited Close;
  if MustClose then System.Close (f);
end;

procedure TFileInputStream.Open (FileName: string);
var
  Success           : Boolean;
begin
  FileMode := 0;
  Assign (f, FileName);
  {$I-}
  Reset (f, 1);
  {$I+}
  Success := (IOResult = 0);
  SetOpenStatus (Success);
  SetName (FileName);
end;

function TFileInputStream.HasData: Boolean;
begin
  HasData := IsOpen and ((inherited HasData) or (not System.EOF (f)));
end;

function TFileInputStream.ReadRawBytes (n: TStreamOffset; var Dest): TStreamOffset;
var
  LastRead          : Integer;
begin
  BlockRead (f, Dest, n, LastRead);
  EndOfStream := (LastRead = 0);
  ReadRawBytes := LastRead;
end;

function TFileInputStream.GetSize: TStreamOffset;
begin
  Result := FileSize (f);
end;

{ TFileOutputStream }

constructor TFileOutputStream.Init (FileName: string; CreateNewFile: Boolean);
begin
  inherited Init;
  Open (FileName, CreateNewFile);
end;

procedure TFileOutputStream.Close;
var
  MustClose         : Boolean;
begin
  MustClose := IsOpen;
  inherited Close;
  if MustClose then System.Close (f);
end;

procedure TFileOutputStream.Open (FileName: string; CreateNewFile: Boolean);
begin
  Assign (f, FileName);
  Rewrite (f, 1);
  SetOpenStatus (True);
  SetName (FileName);
end;

function TFileOutputStream.WriteRawBytes (n: TStreamOffset; var Source): TStreamOffset;
var
  LastWritten       : Integer;
begin
  System.BlockWrite (f, Source, n, LastWritten);
  WriteRawBytes := LastWritten;
end;

end.

