{
@abstract(provides standard operations on arrays and strings)
@author(Marco Schmidt (marcoschmidt@geocities.com))
@created(31 Jul 1998)
@lastmod(10 Apr 2000)

Perform basic operations on arrays of primitive data types: sort, convert
characters to another case, search - should be optimized with assembler
whenever possible.
}

unit Arrays;

{$I platform.inc}

interface

uses
  Numbers;

type
  { array of characters }
  TCharArray = array[0..30000] of Char; { will be overindexed }
  { pointer to @link(TCharArray) }
  PCharArray = ^TCharArray;

  { Compares two arrays of byte, Source and Dest, both being N bytes
    large. Returns relation as -1 (Source < Dest), 0 (equal) or 1
    (Source > Dest). }
function CompareBytes (var Source, Dest; n: TSInt32): Integer;
{ Compares two arrays of characters, Source and Dest, both being N characters
  large. Before comparing two single characters, they're converted to
  uppercase, so 'hallo' and 'HalLo' will be considered equal.
  Returns relation as -1, 0 or 1 (meaning as in @link(CompareBytes)). }
function CompareCharsCI (var Source, Dest; n: TSInt32): Integer;
{ Compares two strings, Source and Dest, not regarding the case ('hallo' and
  'HallO' will be considered equal). Returns relation as -1, 0 or 1 (meaning
  as in @link(CompareBytes)). }
function CompareStringsCI (var Source, Dest: string): Integer;
{ Converts array of bytes Source to array of characters Dest, which will
  store a hexadecimal representation of the input bytes (example: 0 10 255
  will be converted to "000aff". Input array size is N bytes, output array
  must have space for 2 * N characters! }
procedure ConvertBytesToHex (var Source, Dest; n: TSInt32);
{ Copies character array Source to character array Dest, both N characters
  large, converting to lower case. }
procedure ConvertBytesToLowerCase (var Source, Dest; n: TSInt32);
{ Copies character array Source to character array Dest, both N characters
  large, converting to upper case. }
procedure ConvertBytesToUpperCase (var Source, Dest; n: TSInt32);
{ Converts array of characters Source, size 2 * N characters, to array of
  bytes Dest, N bytes large. It is assumed Source's characters are in
  hexadecimal format. Example: "0aff13" (six characters) will be converted to
  10 255 19 (three bytes). Does not check if input data is correct,
  non-hexadecimal input will result in zero output bytes. }
procedure ConvertHexToBytes (var Source, Dest; n: TSInt32);
{procedure FillBytes(var Data; N: TSInt32; B: Byte);}
{function FindByte(var Data; N: TSInt32; B: Byte): TSInt32;}
{ Searches array of bytes Data, size N, for minimum and maximum value.
  Returns these in Min and Max. }
procedure FindMinMaxBytes (var Data; n: TSInt32; var Min, Max: Byte);
{ Copies array of bytes Source to Dest, both N bytes large, taking each
  source value as an index into the array[0..255] of byte NewBytes, writing
  the looked-up value to Dest. }
procedure ReplaceBytes (var Source, Dest; n: TSInt32; var NewBytes);
{$IFNDEF PPC_DELPHI}
procedure SetLength (var s: string; l: Integer);
{$ENDIF}
{procedure SortBytes(var Data; N: TSInt32);}
{ Sorts array of TNativeInt Data, N integers large, in ascending order,
  using a standard Quicksort implementation. }
procedure SortNativeInts (var Data; n: TSInt32);
{ Copies input string S to U, converting all characters to uppercase. }
procedure StringToUpper (var s, U: string);

implementation

const
  ToHexTable        : array[0..15] of Byte =
    (48, 49, 50, 51, 52, 53, 54, 55,
    56, 57, 65, 66, 67, 68, 69, 70);

var
  FromHexTable      : array[0..255] of Byte;
  ToLowerTable      : array[0..255] of Byte;
  ToUpperTable      : array[0..255] of Byte;

function CompareBytes (var Source, Dest; n: TSInt32): Integer;
var
  i                 : TSInt32;
begin
  CompareBytes := 0;
  i := 0;
  while (i < n) do
    begin
      if (TByteArray (Source)[i] < TByteArray (Dest)[i]) then
        begin
          CompareBytes := -1;
          Exit;
        end
      else
        if (TByteArray (Source)[i] > TByteArray (Dest)[i]) then
          begin
            CompareBytes := 1;
            Exit;
          end;
      Inc (i);
    end;
end;

function CompareCharsCI (var Source, Dest; n: TSInt32): Integer;
var
  i                 : LongInt;
  c1                : Integer;
  c2                : Integer;
begin
  i := 0;
  while (i < n) do
    begin
      c1 := ToUpperTable[TByteArray (Source)[i]];
      c2 := ToUpperTable[TByteArray (Dest)[i]];
      if (c1 <> c2) then
        begin
          if (c1 < c2) then
            CompareCharsCI := -1
          else
            CompareCharsCI := 1;
          Exit;
        end;
      Inc (i);
    end;
  CompareCharsCI := 0;
end;

function CompareStringsCI (var Source, Dest: string): Integer;
var
  l1                : Byte;
  l2                : Byte;
  Minimum           : Byte;
  r                 : TSInt32;
  Rel               : Integer;
begin
  l1 := Length (Source);
  l2 := Length (Dest);
  if (l1 < l2) then
    begin
      Rel := -1;
      Minimum := l1
    end
  else
    if (l1 > l2) then
      begin
        Rel := 1;
        Minimum := l2;
      end
    else
      begin
        Rel := 0;
        Minimum := l1;
      end;
  r := CompareCharsCI (Source[1], Dest[1], Minimum);
  if (r = 0) then
    CompareStringsCI := Rel
  else
    CompareStringsCI := r;
end;

{
N = number of bytes to be converted
Source must be N bytes large
Dest must be N * 2 bytes large
Source must not be the same as Dest
}
procedure ConvertBytesToHex (var Source, Dest; n: TSInt32);
var
  b                 : Byte;
  i                 : TSInt32;
  j                 : TSInt32;
begin
  i := 0;
  j := 0;
  while (i <> n) do
    begin
      b := TByteArray (Source)[i];
      Inc (i);
      TByteArray (Dest)[j] := ToHexTable[b shr 4];
      Inc (j);
      TByteArray (Dest)[j] := ToHexTable[b and 15];
      Inc (j);
    end;
end;

procedure ConvertBytesToLowerCase (var Source, Dest; n: TSInt32);
begin
  ReplaceBytes (Source, Dest, n, ToLowerTable);
end;

procedure ConvertBytesToUpperCase (var Source, Dest; n: TSInt32);
begin
  ReplaceBytes (Source, Dest, n, ToUpperTable);
end;

procedure ConvertHexToBytes (var Source, Dest; n: TSInt32);
var
  i                 : TSInt32;
  j                 : TSInt32;
begin
  i := 0;
  j := 0;
  while (j <> n) do
    begin
      TByteArray (Dest)[j] :=
        (FromHexTable[TByteArray (Source)[i]] shr 4) or
      FromHexTable[TByteArray (Source)[i + 1]];
      Inc (j);
      Inc (i, 2);
    end;
end;

procedure FindMinMaxBytes (var Data; n: TSInt32; var Min, Max: Byte);
var
  b                 : Byte;
  i                 : TSInt32;
begin
  Max := TByteArray (Data)[0];
  Min := TByteArray (Data)[0];
  i := 1;
  while (i < n) do
    begin
      b := TByteArray (Data)[i];
      if (b < Min) then Min := b;
      if (b > Max) then Max := b;
      Inc (i);
    end;
end;

procedure InitTables;
var
  i                 : Integer;
begin
  for i := 0 to 255 do
    begin
      FromHexTable[i] := 0;
      if (i >= 65) and (i <= 90)        { A .. Z } then
        begin
          ToLowerTable[i] := i + 32;
          FromHexTable[i] := i - 55;
        end
      else
        ToLowerTable[i] := i;
      if (i >= 97) and (i <= 122)       { a .. z } then
        begin
          ToUpperTable[i] := i - 32;
          FromHexTable[i] := i - 87;
        end
      else
        ToUpperTable[i] := i;
      if (i >= 48) and (i <= 57)        { 0 .. 9 } then FromHexTable[i] := i - 48;
    end;
end;

procedure ReplaceBytes (var Source, Dest; n: TSInt32; var NewBytes);
var
  i                 : TSInt32;
begin
  i := 0;
  while (i <> n) do
    begin
      TByteArray (Dest)[i] := TByteArray (NewBytes)[TByteArray (Source)[i]];
      Inc (i);
    end;
end;

{$IFNDEF PPC_DELPHI}
procedure SetLength (var s: string; l: Integer);
begin
  s[0] := Chr (l);
end;
{$ENDIF}

procedure SortNativeInts (var Data; n: TSInt32);
type
  TNativeIntArray = array[0..0] of TNativeInt;

  procedure SubSort (l, r: TSInt32);
  var
    i, j            : TSInt32;
    x, y            : TNativeInt;
  begin
    i := l;
    j := r;
    x := TNativeIntArray (Data)[(l + r) div 2];
    repeat
      while (TNativeIntArray (Data)[i] < x) do
        i := i + 1;
      while (x < TNativeIntArray (Data)[j]) do
        j := j - 1;
      if (i <= j) then
        begin
          y := TNativeIntArray (Data)[i];
          TNativeIntArray (Data)[i] := TNativeIntArray (Data)[j];
          TNativeIntArray (Data)[j] := y;
          i := i + 1;
          j := j - 1;
        end;
    until (i > j);
    if (l < j) then SubSort (l, j);
    if (i < r) then SubSort (i, r);
  end;

begin
  if (n > 1) then SubSort (0, n - 1);
end;

procedure StringToUpper (var s, U: string);
begin
  ConvertBytesToUpperCase (s[1], U[1], Length (s));
  SetLength (U, Length (s));
end;

begin
  InitTables;
end.

