{*************************************************************}
{            Blind Guardian Component for Delphi 16/32        }
{ Version:   1.2                                              }
{ Author:    Aleksey Kuznetsov                                }
{ E-Mail:    aleksey@utilmind.com                             }
{ Home Page: http://www.utilmind.com                          }
{ Created:   March, 16, 1999                                  }
{ Modified:  September, 4, 1999                               }
{ Thanks for minor changes to:                                }
{      v1.1: Falzetti Edorado (Edoardo.Falzetti@formula.it)   }
{      v1.2: Alexander (iamt@inetus.com)                      }
{                                                             }
{ Legal:     Copyright (c) 1999, UtilMind Solutions           }
{*************************************************************}
{   TBlindGuardian                                            }
{ Protects Shareware programs against illegal copying and     }
{ manages trial period handling.                              }
{*************************************************************}
{ PROPERTIES:                                                 }
{    TrialPeriod: Word (positive value of trial period)       }
{     Registered: Boolean (registered flag)                   }
{ EVENTS:                                                     }
{    OnRegistration: Calls Registration dialog.               }
{ PROTECTED FUNCTIONS:                                        }
{    GetSettings and Modify Settings - Get and Modify the     }
{                                      protection settings    }
{ ADDITIONAL PUBLIC VARIABLES                                 }
{    StartedFirstTime: Boolean;                               }
{    DaysPassed: LongInt (number of passed days from first    }
{                                                    start)   }
{                                                             }
{ NOTE:                                                       }
{   If you are using Application.OnIdleEvent then you must    }
{   show your Registration Dialog manually.                   }
{   OnRegistration event will started only if you not used    }
{   OnIdle event.                                             }
{*************************************************************}
{                     IMPORTANT NOTE:                         }
{ This software is provided 'as-is', without any express or   }
{ implied warranty. In no event will the author be held       }
{ liable for any damages arising from the use of this         }
{ software.                                                   }
{ Permission is granted to anyone to use this software for    }
{ any purpose, including commercial applications, and to      }
{ alter it and redistribute it freely, subject to the         }
{ following restrictions:                                     }
{ 1. The origin of this software must not be misrepresented,  }
{    you must not claim that you wrote the original software. }
{    If you use this software in a product, an acknowledgment }
{    in the product documentation would be appreciated but is }
{    not required.                                            }
{ 2. Altered source versions must be plainly marked as such,  }
{    and must not be misrepresented as being the original     }
{    software.                                                }
{ 3. This notice may not be removed or altered from any       }
{    source distribution.                                     }
{*************************************************************}

unit BlindGuardian;

interface

uses
  {$IFDEF WIN32} Windows, {$ELSE} WinTypes, WinProcs, {$ENDIF}
  Messages, Classes, Controls, SysUtils, Forms;

type
  TOnRegistrationEvent = procedure(Sender: TObject; TrialPeriod, DaysPassed: Integer) of object;
  TBlindGuardian = class(TComponent)
  private
    FTrialPeriod: Word;
    FRegistered: Boolean;
    FOnRegistration: TOnRegistrationEvent;
    FOnCracked: TNotifyEvent;
    FirstTime: Boolean;
    DateTime: TDateTime;
    AlreadyDaysPassed: LongInt;
    OldIdle: TIdleEvent;

    procedure GuardianIdleProc(Sender: TObject; var Done: Boolean);
  protected
    procedure GetSettings; virtual;
    procedure ModifySettings; virtual;
  public
    StartedFirstTime: Boolean;
    DaysPassed: LongInt;

    constructor Create(aOwner: TComponent); override;
    procedure Loaded; override;

    procedure SetRegistered(Value: Boolean);
  published
    property TrialPeriod: Word read FTrialPeriod write FTrialPeriod;
    property Registered: Boolean read FRegistered write SetRegistered;
    property OnRegistration: TOnRegistrationEvent read FOnRegistration write FOnRegistration;
    property OnCracked: TNotifyEvent read FOnCracked write FOnCracked;
  end;

procedure Register;

implementation

function FileDateTime(const FileName: string): System.TDateTime;
var
  Age: Longint;
begin
  Age := FileAge(FileName);
  if Age = -1 then
    Result := 0
  else
    Result := FileDateToDateTime(Age);
end;

function GetSystemDir: string;
{$IFDEF WIN32}
var
  Buffer: array[0..1023] of Char;
begin
  SetString(Result, Buffer, GetSystemDirectory(Buffer, SizeOf(Buffer)));
{$ELSE}
begin
  Result[0] := Char(GetSystemDirectory(@Result[1], 254));
{$ENDIF}
end;

constructor TBlindGuardian.Create(aOwner: TComponent);
begin
  inherited Create(aOwner);
  OldIdle := Application.OnIdle;
  Application.OnIdle := GuardianIdleProc;
  FTrialPeriod := 30;
  FRegistered := False;
  AlreadyDaysPassed := 0;
  FirstTime := True;
  DateTime := Now;
end;

procedure TBlindGuardian.Loaded;
begin
  inherited Loaded;
  GetSettings;
end;

procedure TBlindGuardian.GuardianIdleProc(Sender: TObject; var Done: Boolean);
begin
  if FirstTime then
   begin
    if not (csDesigning in ComponentState) then
     if not FRegistered and Assigned(FOnRegistration) then
      begin
       FirstTime := False;
       FOnRegistration(Self, FTrialPeriod, DaysPassed);
      end;
    Application.OnIdle := OldIdle;
    Done := True;
   end;
end;

procedure TBlindGuardian.SetRegistered(Value: Boolean);
begin
  if not FRegistered then
   if FRegistered <> Value then
    begin
     FRegistered := Value;
     ModifySettings;
    end;
end;

{ It is we make that the name of a file was not
  visible in the text of -exe file }
const
  HiddenFileName: String = ''; {'WINSPOOL.CRC' xor $AA }

function GetFileName: String;
var
  i: Integer;
  St: String;
begin
  St := HiddenFileName;
  for i := 1 to Length(St) do
   St[i] := char(byte(St[i]) xor $AA); { for example, $AA }
  Result := GetSystemDir + '\' + St;
end;

procedure TBlindGuardian.GetSettings;
var
  d2: Boolean;
  fs: TFileStream;
  DT: TDateTime;
begin
{ -------
  I show example only. The protected programs should be more artful
  Each way must be unique and every programmer should have his own secrets.
  BTW, if you want to create professional shareware protection, I recomended
  to use ASProtect (www.aspack.com)
  ------- }
  
{ Get Registry Binding }

{ Get Windows Binding }

   try
    fs := TFileStream.Create(GetFileName, fmOpenRead);
    StartedFirstTime := False;
    d2 := False;
   except
    StartedFirstTime := True;
    d2 := True;
   end;

   if not d2 then
    begin

// if date of our file is not equal to date of system files then someone want to crack us...
     if DateTimeToFileDate(FileDateTime(GetSystemDir + '\Kernel32.dll')) <>
        DateTimeToFileDate(FileDateTime(GetFileName)) then
      if Assigned(FOnCracked) then
       begin
        FOnCracked(Self);
        Exit;
       end;

     try
      fs.Read(FRegistered, SizeOf(FRegistered));
      fs.Read(DateTime, SizeOf(DateTime));
      fs.Read(AlreadyDaysPassed, SizeOf(AlreadyDaysPassed));
      fs.Free;
     except
     end;
    end;

   { Looking for passed time. }              
   if Now < DateTime then
    DT := DateTime - Now
   else
    DT := Now - DateTime;

   DaysPassed := Trunc((MSecsPerday * DT) / 1000) div 86400 + AlreadyDaysPassed;
   if Now >= DateTime then
    begin
     AlreadyDaysPassed := DaysPassed;
     DateTime := Now;
    end;

   ModifySettings;
end;

procedure TBlindGuardian.ModifySettings;
var
  fs: TFileStream;
  FileHandle: THandle;
begin
{ -------
  I show an example only. The protected programs should be more artful.
  Each way must be unique and each programmer should have own secrets.
  ------- }

{ Bind to Registry }

{ Bind to Windows }

   try
    fs := TFileStream.Create(GetFileName, fmCreate);
    fs.Write(FRegistered, SizeOf(FRegistered));
    fs.Write(DateTime, SizeOf(DateTime));
    fs.Write(AlreadyDaysPassed, SizeOf(AlreadyDaysPassed));
   except
   end;
   try
    fs.Free;
   except
   end;

   FileHandle := FileOpen(GetFileName, fmOpenWrite or fmShareDenyNone); // get the file descriptor
   if FileHandle > 0 then
    if FileSetDate(FileHandle, DateTimeToFileDate(FileDateTime(GetSystemDir + '\Kernel32.dll'))) = 0 then //    .
     FileClose(FileHandle);
end;

procedure Register;
begin
  RegisterComponents('UtilMind', [TBlindGuardian]);
end;

end.
