/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

/*
 * dll_glue.cpp --- glue code for the dll to communicate with the caller
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#if qDebug
#	define USE_DUP 0
#else /* !qDebug */
#	define USE_DUP 0
#endif /* !qDebug */

#ifdef macintosh
#	if __profile__
#		include <profiler.h>
#	endif /* __profile__ */

#	include <GUSI.h>
#	include <console.h>
//#	include <pool_alloc.h>
//	extern mem_pool_obj __malloc_pool;

	int dllglue_dispatchevent(EventRecord *userevent); /* should go in dll_glue.h */
	int dllglue_handlerevent(spin_msg msg, long param);

	static GUSIEvtHandler dllglue_Events[]	=	{
		(GUSIEvtHandler)dllglue_dispatchevent,		// nullEvent
		
		(GUSIEvtHandler)dllglue_dispatchevent,		// mouseDown
		(GUSIEvtHandler)dllglue_dispatchevent,		// mouseUp
		0L,							// keyDown
		0L,
		
		0L,							// autoKey
		(GUSIEvtHandler)dllglue_dispatchevent,		// updateEvt
		(GUSIEvtHandler)dllglue_dispatchevent,		// diskEvt
		(GUSIEvtHandler)dllglue_dispatchevent,		// activateEvt
		
		0L,
		0L,
		0L,
		0L,
		
		0L,
		0L,
		(GUSIEvtHandler)dllglue_dispatchevent,		// osEvt
		0L,
		
		0L,
		0L,
		0L,
		0L,
		
		0L,
		0L,
		0L,
	};

	static int  msl_write_stdout(__file_handle file, unsigned char * buff,
		size_t * count, __idle_proc idle_proc);
	static int  msl_write_stderr(__file_handle file, unsigned char * buff,
		size_t * count, __idle_proc idle_proc);
#endif /* macintosh */

#ifdef WIN32
#	include <Windows.h>
#	include <Direct.h>
#	include <IO.h>
#	include <FCntl.h>
#	include <Process.h>
#	include <crtdbg.h>
#endif /* WIN32 */

#include <stdio.h>
#include <setjmp.h>
#include <errno.h>

#include "dll_glue.h"
#include "dll_garbage.h"

#ifdef macintosh
#	include "cvs_hqx.h"
#endif /* macintosh */

#ifdef macintosh
	static UniversalProcPtr __glue_getenv = 0L;
	static UniversalProcPtr __glue_consoleout = 0L;
	static UniversalProcPtr __glue_consoleerr = 0L;
	static UniversalProcPtr __glue_consolein = 0L;
	static UniversalProcPtr __glue_eventdispatch = 0L;
	static UniversalProcPtr __glue_eventhandler = 0L;
#else /* !macintosh */
	static dllglue_getenv_func __glue_getenv = 0L;
	static dllglue_consoleout_func __glue_consoleout = 0L;
	static dllglue_consoleout_func __glue_consoleerr = 0L;
	static dllglue_consolein_func __glue_consolein = 0L;
#endif /* !macintosh */

#ifndef macintosh
#	define true 1
#	define false 0
#endif /* !macintosh */

static int __glue_exit_code = 0;
static jmp_buf _glue_buf;

#if defined(WIN32)
	static long outseek;
	static long errseek;
	static char *outname;
	static char *errname;
#	if !USE_DUP
	static FILE *outlog;
	static FILE *errlog;
#	else /* USE_DUP */
	static int outlog = -1;
	static int errlog = -1;
	static int oldout = -1;
	static int olderr = -1;
#	endif /* USE_DUP */
#endif /* WIN32 */
#if defined(WIN32) || defined(UNIX_SHL)
	static int _hasAlreadyThrownExit = 0;
#endif /* WIN32 || UNIX_SHL */

extern int main(int argc, char **argv);

#if defined(WIN32)
#	if !USE_DUP
static void
myflush(FILE *log, long *oldpos)
{
	long newpos, pos;
#	define BUF_SIZE 8000
	char buf[BUF_SIZE];
	size_t len;

	pos = fseek(log, *oldpos, SEEK_SET);
	if(pos != 0)
		goto fail;

	while(!feof(log) && !ferror(log))
	{
		len = fread(buf, sizeof(char), BUF_SIZE, log);
		if(len > 0)
		{
			if(log == outlog && __glue_consoleout != 0L)
				__glue_consoleout(buf, len);
			if(log == errlog && __glue_consoleerr != 0L)
				__glue_consoleerr(buf, len);
		}
	}

	if(ferror(log))
		goto fail;

	newpos = ftell(log);
	if(newpos < 0)
		goto fail;

	*oldpos = newpos;
	return;
fail:
	fprintf(stderr, "Unable to redirect stdout/stderr !\n");
}
#	else /* USE_DUP */
static void
myflush(int log, long *oldpos)
{
	long newpos, pos;
#	define BUF_SIZE 8000
	char buf[BUF_SIZE];
	size_t len;
	int eofval;

	pos = lseek(log, *oldpos, SEEK_SET);
	if(pos < 0)
		goto fail;

	while(!(eofval = eof(log)))
	{
		len = read(log, buf, sizeof(char) * BUF_SIZE);
		if(len > 0)
		{
			if(log == outlog && __glue_consoleout != 0L)
				__glue_consoleout(buf, len);
			if(log == errlog && __glue_consoleerr != 0L)
				__glue_consoleerr(buf, len);
		}
	}

	if(eofval == -1)
		goto fail;

	newpos = tell(log);
	if(newpos < 0)
		goto fail;

	*oldpos = newpos;
	return;
fail:
	fprintf(stderr, "Unable to redirect stdout/stderr !\n");
}
#	endif /* USE_DUP */
#endif /* WIN32 */

void
dllglue_flushconsole(void)
{
#if defined(WIN32) || defined(UNIX_SHL)
	/* on a strategic place, this will
	stop the dll if the application requests it... */
	if(!_hasAlreadyThrownExit && getenv("DLLLIBSTOP") != 0L)
	{
		/*
		 * This is ugly because it depends on cvs.
		 * But we need it in order to clean the locks.
		 */
		extern error_exit();
		_hasAlreadyThrownExit = 1;
		error_exit();
	}
#endif /* WIN32 || UNIX_SHL */

	fflush(stdout);
	fflush(stderr);
#if defined(WIN32)
	if(outlog != 0L)
		myflush(outlog, &outseek);
	if(errlog != 0L)
		myflush(errlog, &errseek);
#endif /* WIN32 */
}

void dllglue_saveconsole(void)
{
#if defined(WIN32)
#	if USE_DUP
	dup2(oldout, 1);
	dup2(olderr, 2);
#	endif /* USE_DUP */
#endif /* WIN32 */
}

void dllglue_restoreconsole(void)
{
#if defined(WIN32)
#	if USE_DUP
	dup2(outlog, 1);
	dup2(errlog, 2);
#	endif /* USE_DUP */
#endif /* WIN32 */
}

static void initialize_all(void)
{
	garbyNew();
#ifdef macintosh
#	if __profile__
	ProfilerInit(collectDetailed, bestTimeBase, 1200, 200);
	ProfilerSetStatus(1);
#	endif /* __profile__ */
	GUSISetup(GUSIwithInternetSockets);
	GUSISetEvents(dllglue_Events);
	stdout->write_proc = msl_write_stdout;
	stderr->write_proc = msl_write_stderr;
#endif /* macintosh */
#ifdef WIN32
#	ifndef tempnam
#		define tempnam _tempnam
#	endif /* !tempnam */
#endif /* WIN32 */
#if defined(WIN32)
    outname = tempnam(0L, 0L);
	if(outname == 0L)
	{
		fprintf(stderr, "Unable to open tmp file !\n");
		exit(1);
	}
    errname = tempnam(0L, 0L);
	if(errname == 0L)
	{
		fprintf(stderr, "Unable to open tmp file !\n");
		exit(1);
	}
#	if !USE_DUP
	outlog = freopen(outname, "w+", stdout);
	if(outlog == 0L)
	{
		fprintf(stderr, "Unable to reopen stdout !\n");
		exit(1);
	}
#	else /* USE_DUP */
	oldout = dup(1);
	if(oldout == -1)
	{
		fprintf(stderr, "Unable to reopen stdout !\n");
		exit(1);
	}
	outlog = open(outname, O_RDWR | O_CREAT | O_TRUNC/* | O_RANDOM*/, 0666);
	if(outlog == -1 || dup2(outlog, 1) == -1)
	{
		fprintf(stderr, "Unable to reopen stdout !\n");
		exit(1);
	}
#	endif /* USE_DUP */
#	if !USE_DUP
	errlog = freopen(errname, "w+", stderr);
	if(errlog == 0L)
	{
		fprintf(stderr, "Unable to reopen stderr !\n");
		exit(1);
	}
#	else /* USE_DUP */
	olderr = dup(2);
	if(olderr == -1)
	{
		fprintf(stderr, "Unable to reopen stderr !\n");
		exit(1);
	}
	errlog = open(errname, O_RDWR | O_CREAT | O_TRUNC/* | O_RANDOM*/, 0666);
	if(errlog == -1 && dup2(errlog, 2) == -1)
	{
		fprintf(stderr, "Unable to reopen stderr !\n");
		exit(1);
	}
#	endif /* USE_DUP */
	outseek = 0;
	errseek = 0;
#endif /* WIN32 */
}

static void flush_all(void)
{
	garbyRemove(true, false);
	dllglue_flushconsole();
	dllglue_setgetenv(0L);
	dllglue_setconsoleout(0L);
	dllglue_setconsoleerr(0L);
	dllglue_setconsolein(0L);
#ifdef macintosh
	dllglue_seteventdispatch(0L);
	dllglue_seteventhandler(0L);
	/*
	 * Can't do that, cause at this time, some
	 * static destructors in GUSI still remain :
	 * so we do that after the segment terminates.
	 */
//	__pool_free_all(&__malloc_pool);
#	if __profile__
	ProfilerSetStatus(0);
	ProfilerDump("\pcvs.prof");
	ProfilerTerm();
#	endif /* __profile__ */
#endif /* macintosh */
#if defined(WIN32)
#	if !USE_DUP
	if(outlog != 0L)
		fclose(outlog);
	if(errlog != 0L)
		fclose(errlog);
#	else /* USE_DUP */
	if(oldout != -1)
	{
		dup2(oldout, 1);
		close(oldout);
	}
	if(olderr != -1)
	{
		dup2(olderr, 2);
		close(olderr);
	}
	if(outlog != -1)
		close(outlog);
	if(errlog != -1)
		close(errlog);
#	endif /* USE_DUP */
	if(outname != 0L)
	{
		unlink(outname);
		free(outname);
	}
	if(errname != 0L)
	{
		unlink(errname);
		free(errname);
	}
#endif /* WIN32 */
}

void
dllglue_exit(int code)
{
	__glue_exit_code = code;
	longjmp(_glue_buf, 1);
}

char *
dllglue_getenv(const char * name)
{
	if(__glue_getenv != 0L)
	{
#ifdef macintosh
		return (char *)CallUniversalProc(__glue_getenv, uppGetEnvProcInfo, name);
#else /* !macintosh */
		return __glue_getenv(name);
#endif /* !macintosh */
	}
	return 0L;
}

#ifdef UNIX_SHL
char *dllglue_getpass (const char *prompt)
{
	/* let the application get the password */
	return getenv("CVS_GETPASS");
}
#endif /* UNIX_SHL */

#ifdef macintosh
int dllglue_dispatchevent(EventRecord *userevent)
{
	if(__glue_eventdispatch != 0L)
	{
		return (int)CallUniversalProc(__glue_eventdispatch, uppEventDispatchProcInfo, userevent);
	}
	return 0;
}

int dllglue_handlerevent(spin_msg msg, long param)
{
	if(__glue_eventhandler != 0L)
	{
		return (int)CallUniversalProc(__glue_eventhandler, uppEventHandlerProcInfo, msg, param);
	}
	return 0;
}
#endif /* macintosh */

int
dllglue_main(const char *path, int argc, char **argv)
{
#if qDebug
	//Debugger();
#endif
	if(setjmp(_glue_buf) == 0)
	{
		initialize_all();

		if(path != 0L && chdir(path) != 0)
		{
			fprintf(stderr, "Error: Unable to chdir %s (error %d)\n", path, errno);
			exit(1);
		}
	
		main(argc, argv);
	}
	else
	{
	}
	
	flush_all();
	
	return __glue_exit_code;
}

int
#ifdef macintosh
dllglue_setgetenv(UniversalProcPtr agetenv)
#else /* !macintosh */
dllglue_setgetenv(dllglue_getenv_func agetenv)
#endif /* !macintosh */
{
	if(__glue_getenv != 0L)
	{
#ifdef macintosh
		DisposeRoutineDescriptor(__glue_getenv);
#endif /* macintosh */
		__glue_getenv = 0L;
	}
	__glue_getenv = agetenv;
	return 0;
}

#ifdef macintosh
int
dllglue_seteventdispatch(UniversalProcPtr aeventdispatch)
{
	if(__glue_eventdispatch != 0L)
	{
		DisposeRoutineDescriptor(__glue_eventdispatch);
		__glue_eventdispatch = 0L;
	}
	__glue_eventdispatch = aeventdispatch;
	return 0;
}

int
dllglue_seteventhandler(UniversalProcPtr aeventhandler)
{
	if(__glue_eventhandler != 0L)
	{
		DisposeRoutineDescriptor(__glue_eventhandler);
		__glue_eventhandler = 0L;
	}
	__glue_eventhandler = aeventhandler;
	GUSISetSpin(dllglue_handlerevent);
	return 0;
}
#endif /* macintosh */

int
#ifdef macintosh
dllglue_setconsoleout(UniversalProcPtr aconsoleout)
#else /* !macintosh */
dllglue_setconsoleout(dllglue_consoleout_func aconsoleout)
#endif /* !macintosh */
{
	if(__glue_consoleout != 0L)
	{
#ifdef macintosh
		DisposeRoutineDescriptor(__glue_consoleout);
#endif /* macintosh */
		__glue_consoleout = 0L;
	}
	__glue_consoleout = aconsoleout;
	return 0;
}

int
#ifdef macintosh
dllglue_setconsoleerr(UniversalProcPtr aconsoleerr)
#else /* !macintosh */
dllglue_setconsoleerr(dllglue_consoleout_func aconsoleerr)
#endif /* !macintosh */
{
	if(__glue_consoleerr != 0L)
	{
#ifdef macintosh
		DisposeRoutineDescriptor(__glue_consoleerr);
#endif /* macintosh */
		__glue_consoleerr = 0L;
	}
	__glue_consoleerr = aconsoleerr;
	return 0;
}

int
#ifdef macintosh
dllglue_setconsolein(UniversalProcPtr aconsolein)
#else /* !macintosh */
dllglue_setconsolein(dllglue_consolein_func aconsolein)
#endif /* !macintosh */
{
	if(__glue_consolein != 0L)
	{
#ifdef macintosh
		DisposeRoutineDescriptor(__glue_consolein);
#endif /* macintosh */
		__glue_consolein = 0L;
	}
	__glue_consolein = aconsolein;
	return 0;
}

#ifdef macintosh
	static int gOutErrSemaphore = 0;

	static int  msl_write_stdout(__file_handle file, unsigned char * buff,
		size_t * count, __idle_proc idle_proc)
	{
		if(gOutErrSemaphore++ == 0)
			fflush(stderr);
		
		if(__glue_consoleout != 0L)
			*count = CallUniversalProc(__glue_consoleout, uppConsoleOutProcInfo, buff, *count);

		--gOutErrSemaphore;
		return (*count == -1) ? __io_error : __no_io_error;
	}
	
	static int  msl_write_stderr(__file_handle file, unsigned char * buff,
		size_t * count, __idle_proc idle_proc)
	{
		if(gOutErrSemaphore++ == 0)
			fflush(stdout);
		
		if(__glue_consoleerr != 0L)
			*count = CallUniversalProc(__glue_consoleerr, uppConsoleOutProcInfo, buff, *count);

		--gOutErrSemaphore;
		return (*count == -1) ? __io_error : __no_io_error;
	}

	short InstallConsole(short fd)
	{
		return 0;
	}

	void RemoveConsole(void)
	{
	}

	long WriteCharsToConsole(char *buffer, long n)
	{
		if(__glue_consoleout != 0L)
			return CallUniversalProc(__glue_consoleout, uppConsoleOutProcInfo, buffer, n);
		
		return n;
	}

	long ReadCharsFromConsole(char *buffer, long n)
	{
		if(__glue_consolein != 0L)
			return CallUniversalProc(__glue_consolein, uppConsoleInProcInfo, buffer, n);

		return 0;
	}
#endif /* macintosh */

#ifdef _MSC_VER
	/*
	 * This crap comes from the fact "pipe" doesn't work when called from
	 * a MFC application or from a dll. There are two reasons :
	 *
	 * 1) The standard C library provided with VisualC++ allows only
	 * dos commands to call "pipe"
	 * 2) The anonymous pipes won't work as well. Because the system to
	 * give the console IO using "SetStdHandle" won't work if the call is
	 * issued from anything but a dos command.
	 *
	 * So the alternative was to emulate "pipe(..., "w")" by capturing
	 * the input to give to the process in a file, then call "dospipe"
	 * which will do the work for us and map correctly the console IOs
	 * for the process we want to call.
	 */

	static char *pipeInName = 0L;
	static char *pipeOutName = 0L;
	static FILE *pipeIn = 0L;
	static FILE *pipeOut = 0L;
	static char *pipeCmd = 0L;
	static char pipePath[_MAX_PATH] = {'\0'};
	static int pipeRead = 0;
	static DWORD pipeRes = -1;

	FILE *mypopen(const char *cmd, const char *mode);
	int mypclose(FILE *file);

	BOOL WINAPI DllMain(  HINSTANCE hinstDLL,  // handle to DLL module
		DWORD fdwReason,     // reason for calling function
		LPVOID lpvReserved   // reserved
		);

	BOOL WINAPI DllMain(  HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved)
	{
		if ( fdwReason == DLL_PROCESS_ATTACH )
		{
#if qDebug
		int tmpDbgFlag = _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG);
		tmpDbgFlag |= _CRTDBG_LEAK_CHECK_DF;
		_CrtSetDbgFlag(tmpDbgFlag);
#endif
			if(GetModuleFileName(hinstDLL, pipePath, _MAX_PATH))
			{
				char *tmp = pipePath;
				char *last = 0L;
				while((tmp = strchr(tmp, '\\')) != 0L)
					last = ++tmp;
				if(last != 0L)
					*last = '\0';
			}
		}
		return TRUE;
	}

	/* call dospipe.exe */
	static HANDLE
	ForkChildProcess(char *cmd)
	{
		SECURITY_ATTRIBUTES lsa;
		STARTUPINFO         si;
		PROCESS_INFORMATION pi;

		lsa.nLength=sizeof(SECURITY_ATTRIBUTES);
		lsa.lpSecurityDescriptor=NULL;
		lsa.bInheritHandle=TRUE;

		memset(&si,0,sizeof(STARTUPINFO));
		si.cb=sizeof(STARTUPINFO);
		si.wShowWindow = SW_SHOWMINNOACTIVE;
		si.dwFlags = STARTF_USESHOWWINDOW;

		if (!CreateProcess
			 (
				NULL,
				cmd,
				NULL,
				NULL,
				TRUE,
				NORMAL_PRIORITY_CLASS,
				NULL,
				NULL,
				&si,
				&pi)
			 )
		{
			if (GetLastError()==2)
				fprintf(stderr, "Executable %s not found\n", cmd);
			fprintf(stderr, "Could Not Create Child Process (error %d)\n", GetLastError());
		}

		CloseHandle(pi.hThread);

		return(pi.hProcess);
	}

	static char *appendstr(char *txt, const char *what, int addspace)
	{
		size_t len = strlen(what);
		if(txt == 0L)
		{
			txt = malloc((len + 1) * sizeof(char));
			if(txt != 0L)
				strcpy(txt, what);
		}
		else
		{
			txt = realloc(txt, (strlen(txt) + len + 1 + (addspace ? 1 : 0)) * sizeof(char));
			if(txt != 0L)
			{
				if(addspace)
					strcat(txt, " ");
				strcat(txt, what);
			}
		}
		return txt;
	}

	FILE *mypopen(const char *cmd, const char *mode)
	{
		char *pipeErrName = 0L;
		FILE *pipeErr = 0L;
		HANDLE hChild = 0L;
		char buf[1024];
		int nread;

		if(cmd == 0L || mode == 0L)
			return 0L;

		if(*mode != 'w' && *mode != 'r')
		{
			fprintf(stderr, "mypopen Internal error : '%c' not implemented\n", *mode);
			return 0L;
		}

		if(*mode == 'w')
		{
			pipeRead = 0;

			pipeInName = _tempnam(0L, 0L);
			if(pipeInName == 0L)
			{
				fprintf(stderr, "mypopen : Unable to open tmp file !\n");
				return 0L;
			}

			pipeCmd = strdup(cmd);
			if(pipeCmd == 0L)
			{
				fprintf(stderr, "mypopen : Impossible to allocate %d bytes !\n", strlen(cmd));
				return 0L;
			}

			pipeIn = fopen(pipeInName, "w");
			return pipeIn;
		}

		pipeRead = 1;

		pipeInName = _tempnam(0L, 0L);
		if(pipeInName == 0L)
		{
			fprintf(stderr, "mypopen : Unable to open tmp file !\n");
			goto error;
		}

		pipeIn = fopen(pipeInName, "w");
		if(pipeIn == 0L)
		{
			fprintf(stderr, "mypopen : Unable to open tmp file !\n");
			goto error;
		}

		pipeRes = -1;
		fclose(pipeIn);
		pipeIn = 0L;

		pipeOutName = _tempnam(0L, 0L);
		if(pipeOutName == 0L)
		{
			fprintf(stderr, "mypclose : Unable to open tmp file !\n");
			goto error;
		}

		pipeErrName = _tempnam(0L, 0L);
		if(pipeErrName == 0L)
		{
			fprintf(stderr, "mypclose : Unable to open tmp file !\n");
			goto error;
		}

		/* invoke dospipe */
		pipeCmd = appendstr(pipeCmd, """", 0);
		pipeCmd = appendstr(pipeCmd, pipePath, 0);
		pipeCmd = appendstr(pipeCmd, "dospipe.exe", 0);
		pipeCmd = appendstr(pipeCmd, """", 0);
		pipeCmd = appendstr(pipeCmd, pipeInName, 1);
		pipeCmd = appendstr(pipeCmd, pipeOutName, 1);
		pipeCmd = appendstr(pipeCmd, pipeErrName, 1);
		pipeCmd = appendstr(pipeCmd, cmd, 1);

		hChild = ForkChildProcess(pipeCmd);
		if(hChild == 0L)
			goto error;

		WaitForSingleObject(hChild, (DWORD)(-1L));
		GetExitCodeProcess(hChild, &pipeRes);
		CloseHandle(hChild);
		
		/* load & redirect stdout, stderr */
		pipeOut = fopen(pipeOutName, "r");
		if(pipeOut == 0L)
		{
			fprintf(stderr, "mypclose : Unable to open stdout file !\n");
			goto error;
		}
		pipeErr = fopen(pipeErrName, "r");
		if(pipeErr == 0L)
		{
			fprintf(stderr, "mypclose : Unable to open stdout file !\n");
			goto error;
		}

		while(!feof(pipeErr) && !ferror(pipeErr))
		{
			nread = fread(buf, sizeof(char), 1024, pipeErr);
			if(nread <= 0)
				break;
			fwrite(buf, sizeof(char), nread, stderr);
		}

		dllglue_flushconsole();
	error:
		if(pipeIn != 0L)
			fclose(pipeIn);
		if(pipeErr != 0L)
			fclose(pipeErr);
		if(pipeInName != 0L)
			unlink(pipeInName);
		if(pipeErrName != 0L)
			unlink(pipeErrName);
		if(pipeCmd != 0L)
			free(pipeCmd);

		pipeCmd = 0L;
		pipeInName = 0L;
		pipeIn = 0L;
		return pipeOut;
	}

	int mypclose(FILE *file)
	{
		char *pipeErrName = 0L;
		FILE *pipeErr = 0L;
		HANDLE hChild = 0L;
		char *cmd = 0L;
		char buf[1024];
		int nread;

		if(!pipeRead)
		{
			if(pipeIn == 0L || pipeIn != file || pipeInName == 0L || pipeCmd == 0L)
				goto error;

			pipeRes = -1;
			fclose(pipeIn);
			pipeIn = 0L;

			pipeOutName = _tempnam(0L, 0L);
			if(pipeOutName == 0L)
			{
				fprintf(stderr, "mypclose : Unable to open tmp file !\n");
				goto error;
			}

			pipeErrName = _tempnam(0L, 0L);
			if(pipeErrName == 0L)
			{
				fprintf(stderr, "mypclose : Unable to open tmp file !\n");
				goto error;
			}

			/* invoke dospipe */
			cmd = appendstr(cmd, """", 0);
			cmd = appendstr(cmd, pipePath, 0);
			cmd = appendstr(cmd, "dospipe.exe", 0);
			cmd = appendstr(cmd, """", 0);
			cmd = appendstr(cmd, pipeInName, 1);
			cmd = appendstr(cmd, pipeOutName, 1);
			cmd = appendstr(cmd, pipeErrName, 1);
			cmd = appendstr(cmd, pipeCmd, 1);

			hChild = ForkChildProcess(cmd);
			if(hChild == 0L)
				goto error;

			WaitForSingleObject(hChild, (DWORD)(-1L));
			GetExitCodeProcess(hChild, &pipeRes);
			CloseHandle(hChild);
			
			/* load & redirect stdout, stderr */
			pipeOut = fopen(pipeOutName, "r");
			if(pipeOut == 0L)
			{
				fprintf(stderr, "mypclose : Unable to open stdout file !\n");
				goto error;
			}
			pipeErr = fopen(pipeErrName, "r");
			if(pipeErr == 0L)
			{
				fprintf(stderr, "mypclose : Unable to open stdout file !\n");
				goto error;
			}

			while(!feof(pipeOut) && !ferror(pipeOut))
			{
				nread = fread(buf, sizeof(char), 1024, pipeOut);
				if(nread <= 0)
					break;
				fwrite(buf, sizeof(char), nread, stdout);
			}

			while(!feof(pipeErr) && !ferror(pipeErr))
			{
				nread = fread(buf, sizeof(char), 1024, pipeErr);
				if(nread <= 0)
					break;
				fwrite(buf, sizeof(char), nread, stderr);
			}

			dllglue_flushconsole();
error:
			if(pipeIn != 0L)
				fclose(pipeIn);
			if(pipeOut != 0L)
				fclose(pipeOut);
			if(pipeErr != 0L)
				fclose(pipeErr);
			if(pipeInName != 0L)
				unlink(pipeInName);
			if(pipeOutName != 0L)
				unlink(pipeOutName);
			if(pipeErrName != 0L)
				unlink(pipeErrName);
			if(pipeCmd != 0L)
				free(pipeCmd);
			if(cmd != 0L)
				free(cmd);

			pipeCmd = 0L;
			pipeInName = 0L;
			pipeOutName = 0L;
			pipeIn = 0L;
			pipeOut = 0L;
		}
		else
		{
			if(pipeOut == 0L || pipeOut != file || pipeOutName == 0L)
				goto error2;

error2:
			if(pipeOut != 0L)
				fclose(pipeOut);
			if(pipeOutName != 0L)
				unlink(pipeOutName);

			pipeOutName = 0L;
			pipeOut = 0L;
		}
		return pipeRes;
	}
#endif /* _MSC_VER */
