/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

/*
 * TextBinary.cpp --- utilities to ckeck if files are binary or text
 */

#include "stdafx.h"

#ifdef WIN32
#	include <sys/stat.h>
#	include <direct.h>
#endif /* WIN32 */

#ifdef macintosh
#	ifndef qMacCvsPP
#		include <Unix.h>
#	else /* qMacCvsPP */
#		include <Resources.h>
#		include <unistd.h>
#		include <sys/types.h>
#		include <stat.h>
#	endif /* qMacCvsPP */
#	include "MacMisc.h"
#endif /* macintosh */

#ifdef qQT
#	include <unistd.h>
#endif /* qQT */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include "TextBinary.h"
#include "AppConsole.h"
#include "CPStr.h"

#if !defined(S_ISLNK) && defined(S_IFLNK)
#	if defined(S_IFMT)
#		define	S_ISLNK(m) (((m) & S_IFMT) == S_IFLNK)
#	else
#		define S_ISLNK(m) ((m) & S_IFLNK)
#	endif
#endif

#define MAX_TEST_SIZE 100000

#ifdef WIN32
#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

static char *sBuf = 0L;
static int sBufSize;

static kTextBinTYPE FillBuffer(const char *arg, const char *dir, const FSSpec * spec, bool bin)
{
	if(sBuf == NULL)
	{
		sBuf = (char *)malloc(MAX_TEST_SIZE * sizeof(char));
		if(sBuf == 0L)
		{
			cvs_err("Unable to allocate %d bytes\n", MAX_TEST_SIZE * sizeof(char));
			return kFileMissing;
		}
	}
	
	if(chdir(dir) != 0)
	{
		cvs_err("Unable to chdir to ""%s"" (error %d)\n", dir, errno);
		return kFileMissing;
	}
	
	FILE *samp = fopen(arg, bin ? "rb" : "r");
	if(samp == 0L)
	{
		cvs_err("Unable to open ""%s"" (error %d)\n", arg, errno);
		return kFileMissing;
	}
	
	sBufSize = fread(sBuf, sizeof(char), MAX_TEST_SIZE, samp);
	if(sBufSize < 0)
	{
		cvs_err("Unable to read ""%s"" (error %d)\n", arg, errno);
		return kFileMissing;
	}
	
	fclose(samp);
	return kFileIsOK;
}

static kTextBinTYPE TestBuffer(const char *arg, const char *dir, const FSSpec * spec, bool bin)
{
	if(sBufSize == 0)
		return kFileIsOK;
	
	// Windows : fread skips the \r, so we don't care.
	// Mac : MSL exchanges when reading the \r and \n,
	// so we look for \n
	// Others : default to \n
	
	const unsigned char nativeNL = '\n';
	const unsigned char nonnativeNL = '\r';
	
	// figure it it is binary (the test is more than 10% binary
	// is binary)
	int numBinChars = 0;
	int numTextChars = 0;
	int i;
	unsigned char c;
	char *tmp;
	
	for(tmp = sBuf, i = 0; i < sBufSize; i++)
	{
		c = (unsigned char)*tmp++;
		if((c < 0x20 && c != nativeNL && c != nonnativeNL && c != '\t') || (c >= 0x80
#ifdef macintosh
		&& c != (unsigned char)'' && c != (unsigned char)'' /* powerplant */
#endif /* macintosh */		
		))
			numBinChars++;
		else
			numTextChars++;
	}
	
	float percent = (float)numBinChars / (float)sBufSize;
	bool isbin = percent >= 0.05;
	if(isbin && !bin)
		return kTextIsBinary;
	if(!isbin && bin)
		return kBinIsText;
	
	
	// for the text files, check if the new lines are good
	if(!bin)
	{
		int numNonNative = 0, numNative = 0;
		for(tmp = sBuf, i = 0; i < sBufSize; i++)
		{
			c = (unsigned char)*tmp++;
			if(c == nonnativeNL)
				numNonNative++;
			else if(c == nativeNL)
				numNative++;
		}
		if(numNative == 0)
		{
			if(numNonNative > 0)
				return kTextWrongLF;
		}
		else
		{
			percent = (float)numNonNative / (float)numNative;
			if(percent >= 0.1)
				return kTextWrongLF;
		}
		
		if(numBinChars > 0)
			return kTextEscapeChar;
	}
	
	return kFileIsOK;
}

#ifdef macintosh
static kTextBinTYPE GetMacSig(const char *arg, const char *dir, const FSSpec * spec, bool bin)
{
	CInfoPBRec info;

	info.dirInfo.ioVRefNum 		= spec->vRefNum;
	info.dirInfo.ioDrDirID 		= spec->parID;
	info.dirInfo.ioNamePtr 		= (StringPtr)spec->name;
	info.dirInfo.ioFDirIndex 	= 0;
	info.dirInfo.ioACUser 		= 0;
		
	OSErr err = PBGetCatInfoSync(&info);
	if(err != noErr || (info.dirInfo.ioFlAttrib & 0x10)/*folder*/)
	{
		cvs_err("Unable to get info on ""%s"" (error %d)\n", arg, err);
		return kFileMissing;
	}
	
	if(info.hFileInfo.ioFlFndrInfo.fdFlags & (1 << 15))
	{
		return kFileIsAlias;
	}
	
	if(bin)
	{
		if(info.hFileInfo.ioFlFndrInfo.fdType == 'TEXT')
			return kBinWrongSig;
	}
	else
	{
		if(info.hFileInfo.ioFlRLgLen > 0 && info.hFileInfo.ioFlLgLen == 0)
			return kTextIsBinary;
			
		
		if(info.hFileInfo.ioFlFndrInfo.fdType != 'TEXT')
			return kTextWrongSig;
	}
	
	return kFileIsOK;
}
#endif /* macintosh */


static kTextBinTYPE TestValidName(const char *filename)
{
	kTextBinTYPE result = kFileIsOK;

	if(strchr(filename, '/') != 0L)
		result = kFileInvalidName;

#ifdef macintosh
	if(strcmp(filename, ".") == 0 || strcmp(filename, "..") == 0)
		result = kFileInvalidName;
	if(strcmp(filename, "CVS") != 0 && stricmp(filename, "cvs") == 0)
		result = kFileInvalidName;
#endif /* !macintosh */

	return result;
}

static kTextBinTYPE TestFileIsAlias(const char *arg, const char *dir, const FSSpec * spec)
{
	kTextBinTYPE result = kFileIsOK;

#ifdef S_ISLNK
	CPStr fullname;
	struct stat sb;

	fullname = dir;
	if(!fullname.endsWith(kPathDelimiter))
		fullname << kPathDelimiter;
	fullname << arg;

#	ifdef qMacCvsPP
	if (lstat(fullname, &sb) == -1)
#	else /* !qMacCvsPP */
	if (stat(fullname, &sb) == -1)
#	endif /* !qMacCvsPP */
		result = kFileMissing;
	else if(S_ISLNK(sb.st_mode))
		result = kFileIsAlias;
#endif /* S_ISLNK */

	return result;
}

kTextBinTYPE FileIsText(const char *arg, const char *dir, const FSSpec * spec)
{
	kTextBinTYPE state;

	state = FillBuffer(arg, dir, spec, false);
	if(state != kFileIsOK)
		return state;
	
	if(sBufSize == 0)
		return kFileIsOK;

	state = TestBuffer(arg, dir, spec, false);
	if(state != kFileIsOK)
		return state;

#ifdef macintosh
	state = GetMacSig(arg, dir, spec, false);
	if(state != kFileIsOK)
		return state;
#endif /* macintosh */
	
	state = TestValidName(arg);
	if(state != kFileIsOK)
		return state;
	
	state = TestFileIsAlias(arg, dir, spec);
	if(state != kFileIsOK)
		return state;

	return kFileIsOK;
}

kTextBinTYPE FileIsBinary(const char *arg, const char *dir, const FSSpec * spec)
{
	kTextBinTYPE state;

	state = FillBuffer(arg, dir, spec, true);
	if(state != kFileIsOK)
		return state;
	
	state = TestBuffer(arg, dir, spec, true);
	if(state != kFileIsOK)
		return state;
	
#ifdef macintosh
	state = GetMacSig(arg, dir, spec, true);
	if(state != kFileIsOK)
		return state;
#endif /* macintosh */
	
	state = TestValidName(arg);
	if(state != kFileIsOK)
		return state;
	
	state = TestFileIsAlias(arg, dir, spec);
	if(state != kFileIsOK)
		return state;

	return kFileIsOK;
}

bool SplitPath(const char *dir, CStr & uppath, CStr & folder)
{
	char newdir[512];
	char olddir[512];
	char newfile[256];
	int lendir = strlen(dir);
	strcpy(olddir, dir);
	if(dir[lendir - 1] == kPathDelimiter)
	{
		olddir[lendir - 1] = '\0';
		--lendir;
	}
	const char *tmp = olddir, *prev = olddir;
	
	while((tmp = strchr(tmp, kPathDelimiter)) != 0L)
		prev = ++tmp;
	
	strcpy(newdir, olddir);
	newdir[prev - olddir] = '\0';
	strcpy(newfile, prev);

	uppath = newdir;
	folder = newfile;

	return true;
}
