/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- December 1997
 */

/*
 * Persistent.h --- utilities to store persitent values
 */

#ifndef PERSISTENT_H
#define PERSISTENT_H

#include <string.h>

#ifdef macintosh
#	include <Files.h>
#endif /* macintosh */

// virtual class to access the date to load/save
class CPersistent
{
public:
	virtual unsigned int SizeOf(void) const = 0L;
	virtual const void *GetData(void) const = 0L;
	virtual void SetData(const void *ptr, unsigned int size) = 0L;
		// virtual access
	
	static void Register(CPersistent *value, const char *uniqueName);
	static void UnRegister(CPersistent *value);
		// register/unregister a value to store/load
	
	static void SaveAll(void);
	static bool LoadAll(void);
		// save/load the registred values

protected:
	struct CPersistentEntry
	{
		const char *fUniqueName;
		CPersistent *fValue;
	};
	static CPersistentEntry *fAllEntries;
	static int fNumEntries;

	static CPersistentEntry * Find(const char *uniqueName);
};

// this is OK for most of the persistant values

template <class T>
class CPersistentT : public CPersistent
{
public:
	CPersistentT(const char *uniqueName, T defValue)
	{
		fValue = fDefValue = defValue;
		CPersistent::Register(this, uniqueName);
	}
	virtual ~CPersistentT()
	{
		CPersistent::UnRegister(this);
	}

	inline operator const T &() const {return fValue;}
	inline operator T &() {return fValue;}
	inline const T * operator ->() const {return &fValue;}
	inline T * operator ->() {return &fValue;}
	
	inline T & operator=(const T & val) {return fValue = val;}
#ifdef _MSC_VER
	inline bool operator==(const T & val) const {return fValue == val;}
#endif

	virtual unsigned int SizeOf(void) const {return sizeof(T);}
	virtual const void *GetData(void) const {return &fValue;}
	virtual void SetData(const void *ptr, unsigned int size)
	{
		if(size == sizeof(T))
			memcpy(&fValue, ptr, size);
	}
	
protected:
	T fDefValue, fValue;
};

#ifdef macintosh
	OSErr MacGetPrefsFolder(FSSpec & theFolder, Str255 & thePath);
#endif /* macintosh */

#ifdef WIN32
#	define PROFILE_NAME "CVS settings"
#endif /* WIN32 */

#endif /* PERSISTENT_H */
