/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- February 1998
 */

/*
 * Persistent.cpp --- utilities to store persitent values
 */

#include "stdafx.h"
#include <stdlib.h>
#include <string.h>

#include "Persistent.h"
#include "AppConsole.h"
#include "CPStr.h"

#ifdef macintosh
#	ifndef Assert_
#		define Assert_(a) a
#	endif /* !Assert_ */
#	include <Resources.h>
#	include <TextUtils.h>
#	include <Folders.h>
#	include "VolsPaths.h"
#	include "MacMisc.h"
#	define kCVS_prefID 'cprf'
#	define kCVS_prefName "\pMacCvs Prefs"
#endif /* macintosh */

#ifdef qQT
#	include "qcvsapp.h"
#	include "CvsPrefs.h"
#	include "TextBinary.h"
#	ifdef HAVE_ERRNO_H
#	include <errno.h>
#	endif
#endif /* qQT */

#ifdef WIN32
#	ifdef _DEBUG
#	define new DEBUG_NEW
#	undef THIS_FILE
	static char THIS_FILE[] = __FILE__;
#	endif
#endif /* WIN32 */

int CPersistent::fNumEntries = 0;
CPersistent::CPersistentEntry *CPersistent::fAllEntries = 0L;

void CPersistent::Register(CPersistent *value, const char *uniqueName)
{
	for(int i = 0; i < fNumEntries; i++)
	{
		if(strcmp(uniqueName, fAllEntries[i].fUniqueName) == 0)
		{
#ifdef macintosh
			Assert_(0);
#endif
#if defined(WIN32) || defined(qQT)
			ASSERT(0);
#endif
		}
	}
	if(fNumEntries == 0)
	{
		//garbyNew(); /*RANGECHECK*/
		fAllEntries = (CPersistentEntry *)malloc(sizeof(CPersistentEntry));
	}
	else
	{
		fAllEntries = (CPersistentEntry *)realloc(fAllEntries,
			(fNumEntries + 1) * sizeof(CPersistentEntry));
	}

#ifdef macintosh
	Assert_(fAllEntries != 0L);
#endif
#if defined(WIN32) || defined(qQT)
	ASSERT(fAllEntries != 0L);
#endif
	fAllEntries[fNumEntries].fUniqueName = uniqueName;
	fAllEntries[fNumEntries++].fValue = value;
}

void CPersistent::UnRegister(CPersistent *value)
{
	int i;
	for(i = 0; i < fNumEntries; i++)
	{
		if(value == fAllEntries[i].fValue)
			break;
	}
#ifdef macintosh
	Assert_(i != fNumEntries);
#endif
#if defined(WIN32) || defined(qQT)
	ASSERT(i != fNumEntries);
#endif
	if(i == fNumEntries)
		return;
	
	memmove(&fAllEntries[i], &fAllEntries[i + 1],
		(fNumEntries - i - 1) * sizeof(CPersistentEntry));
	
	if(--fNumEntries == 0)
	{
		free(fAllEntries);
		fAllEntries = 0L;
		//garbyRemove(0, 0); /*RANGECHECK*/
	}
	else
	{
		fAllEntries = (CPersistentEntry *)realloc(fAllEntries,
			fNumEntries * sizeof(CPersistentEntry));
	}
}

#ifdef macintosh
	static OSErr GetPreferencesFile(FSSpec & mMacFileSpec)
	{
		short	theVRef;
		long	theDirID;
		OSErr	theErr;

		theErr = ::FindFolder(kOnSystemDisk, kPreferencesFolderType,
						kDontCreateFolder, &theVRef, &theDirID);
						
		if (theErr == noErr)
		{
			::FSMakeFSSpec(theVRef, theDirID, kCVS_prefName, &mMacFileSpec);
		}
		
		return theErr;

	}

	OSErr
	MacGetPrefsFolder(FSSpec & theFolder, Str255 & thePath)
	{
		short foundVRefNum;
		long foundDirID;
		OSErr err;

		err = FindFolder(kOnSystemDisk, kPreferencesFolderType,
			kDontCreateFolder, &foundVRefNum, &foundDirID);
		if (err != noErr)
		{
			cvs_err("MacCvs: Unable to find the preferences folder (error %d)\n", err);
			return err;
		}
			
		if(PathNameFromDirID(foundDirID, foundVRefNum, (char *)thePath) != noErr)
		{
			cvs_err("MacCvs: Unable to convert path name\n");
			return fnfErr;
		}
		c2pstr((char *)thePath);

		err = FSMakeFSSpecCompat(foundVRefNum, foundDirID, "\p", theFolder);
		return err;
	}
#endif /* macintosh */

void CPersistent::SaveAll(void)
{
#ifdef macintosh
	FSSpec theFolder;
	short FRef, oldRef;
	Handle hdl;
	OSErr err;
	
	err = GetPreferencesFile(theFolder);
	if(err != noErr)
	{
		cvs_err("Error while searching preference folder (error %d)!\n", err);
		return;
	}

	oldRef = CurResFile();
	err = HSetVol(0L, theFolder.vRefNum, theFolder.parID);
	Assert_(err == noErr);
	err = FSDelete(kCVS_prefName, 0/*theFolder.vRefNum*/);
	Assert_(err == noErr || err == fnfErr);
	CreateResFile (kCVS_prefName);
	FRef = OpenResFile(kCVS_prefName);
	if(FRef < 0)
	{
		cvs_err("Error while saving preferences (error %d)!\n", ResError());
		return;
	}

	UseResFile (FRef);
	CPStr name;

	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		int size = entry.fValue->SizeOf();
		if(size <= 0)
			continue;
		
		//short id = Unique1ID(kCVS_prefID);
		name = entry.fUniqueName;

		Handle hdl = NewHandle(size);
		Assert_(hdl != 0L);
		HLock(hdl);
		BlockMove(entry.fValue->GetData(), *hdl, size);
		HUnlock(hdl);
		AddResource(hdl, kCVS_prefID, i, name);
		Assert_(ResError() == noErr);
		UpdateResFile(FRef);
		Assert_(ResError() == noErr);
		ReleaseResource(hdl);
		Assert_(ResError() == noErr);
	}

	UseResFile(oldRef);
	Assert_(ResError() == noErr);
	CloseResFile (FRef);
	Assert_(ResError() == noErr);
#endif /* macintosh */
#ifdef WIN32
	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		unsigned int size = entry.fValue->SizeOf();
		if(size == 0)
			continue;
		
		if(!AfxGetApp()->WriteProfileBinary(PROFILE_NAME, entry.fUniqueName,
			(BYTE *)entry.fValue->GetData(), size))
		{
			cvs_err("Error while saving preferences !\n");
			break;
		}
	}
#endif /* WIN32 */
#ifdef qQT
	CStr home;
	home = gCvsPrefs.Home();
	if(!home.endsWith(kPathDelimiter))
		home << kPathDelimiter;
	home << ".qcvsrc";
	FILE *out = fopen(home, "wb");
	if(out == 0L)
	{
		cvs_err("Unable to open '%s' for writing (error %d)", (char *)home, errno);
		return;
	}
	int version = 1;
	fwrite(&version, sizeof(int), 1, out);
	int numEntries = fNumEntries;
	fwrite(&numEntries, sizeof(int), 1, out);
	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		unsigned int size = entry.fValue->SizeOf();
		
		fwrite(entry.fUniqueName, sizeof(char), strlen(entry.fUniqueName) + 1, out);
		fwrite(&size, sizeof(unsigned int), 1, out);
		fwrite(entry.fValue->GetData(), 1, size, out);
	}

	if(ferror(out))
		cvs_err("Error while writing '%s' (error %d)", (char *)home, errno);
	fclose(out);
#endif /* qQT */
}

bool CPersistent::LoadAll(void)
{
#ifdef macintosh
	FSSpec theFolder;
	short FRef, oldRef;
	Handle hdl;
	OSErr err;
	
	err = GetPreferencesFile(theFolder);
	if(err != noErr)
		return false;

	oldRef = CurResFile();
	err = HSetVol(0L, theFolder.vRefNum, theFolder.parID);
	FRef = OpenResFile(kCVS_prefName);
	if(FRef < 0)
		return false;

	CPStr name;
	UseResFile(FRef);
	
	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		name = entry.fUniqueName;

		Handle hdl = Get1NamedResource(kCVS_prefID, name);
		if(hdl == 0L)
			continue;
		
		HLock(hdl);
		entry.fValue->SetData(*hdl, GetHandleSize(hdl));
		HUnlock(hdl);
	}
	
	CloseResFile (FRef);
	UseResFile(oldRef);
	return true;
#endif /* macintosh */
#ifdef WIN32
	bool found_one = false;
	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		BYTE* ppData;
		UINT pBytes;

		if(!AfxGetApp()->GetProfileBinary(PROFILE_NAME, entry.fUniqueName,
			&ppData, &pBytes))
		{
			continue;
		}
		found_one = true;
		entry.fValue->SetData(ppData, pBytes);
		delete ppData;
	}
	return found_one;
#endif /* WIN32 */
#ifdef qQT
	CStr home;
	home = gCvsPrefs.Home();
	if(!home.endsWith(kPathDelimiter))
		home << kPathDelimiter;
	home << ".qcvsrc";
	FILE *out = fopen(home, "rb");
	if(out == 0L)
		return false;

	int version;
	fread(&version, sizeof(int), 1, out);
	if(version != 1)
	{
		cvs_err("Unexpected version %d in the file '%s'\n", version, (char *)home);
		goto close;
	}
	int numEntries;
	fread(&numEntries, sizeof(int), 1, out);
	CStaticAllocT<char> alloc;
	if(numEntries < 0)
	{
		cvs_err("Unexpected # %d in the file '%s'\n", numEntries, (char *)home);
		goto close;
	}
	for(int i = 0; i < numEntries; i++)
	{
		CStr name;
		while(!feof(out) && !ferror(out))
		{
			int c = getc(out);
			name << (char)c;
			if(c == '\0')
				break;
		}
		if(feof(out))
		{
			cvs_err("Unexpected end of file in '%s'\n", (char *)home);
			goto close;
		}
		if(ferror(out))
		{
			cvs_err("Unexpected error %d in '%s'\n", errno, (char *)home);
			goto close;
		}
		CPersistentEntry * entry = CPersistent::Find(name);
		if(entry == 0L)
		{
			cvs_err("Unexpected token '%s' in '%s'\n", (char *)name, (char *)home);
			goto close;
		}

		unsigned int size;
		fread(&size, sizeof(unsigned int), 1, out);
		
		alloc.AdjustSize(size);
		fread((char *)alloc, 1, size, out);
		entry->fValue->SetData((char *)alloc, size);
	}

	if(ferror(out))
		cvs_err("Error while reading '%s' (error %d)", (char *)home, errno);
	fclose(out);
	return true;
close:
	fclose(out);
#endif /* qQT */
	return false;
}

CPersistent::CPersistentEntry * CPersistent::Find(const char *uniqueName)
{
	for(int i = 0; i < fNumEntries; i++)
	{
		CPersistentEntry & entry = fAllEntries[i];
		if(!strcmp(uniqueName, entry.fUniqueName))
			return &fAllEntries[i];
	}
	return 0L;
}
