/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@geocities.com> --- April 1998
 */

/*
 * CvsEntries.h --- adaptation from cvs/src/entries.c
 */

#ifndef CVSENTRIES_H
#define CVSENTRIES_H

#include "SortList.h"
#include "CvsIgnore.h"

typedef enum
{
	ENT_FILE,
	ENT_SUBDIR
} ent_type;

class EntnodeData
{
public:
	EntnodeData(ent_type atype) : ref(1), type(atype), user(0L), desc(0L),
		missing(false), visited(false), unknown(false),
		unmodified(true), ignored(false), locked(false)
	{
#ifdef macintosh
		macspec.vRefNum = 0;
		macspec.parID = 0;
		macspec.name[0] = '\0';
#endif /* macintosh */
	}
	virtual ~EntnodeData();

	enum
	{
		kName = 0,
		kStatus = 3
	};

	inline ent_type GetType(void) const {return type;}
	inline static int Compare(const EntnodeData & node1, const EntnodeData & node2)
	{
#ifdef WIN32
		return stricmp(node1.user, node2.user);
#else /* !WIN32 */
		return strcmp(node1.user, node2.user);
#endif /* !WIN32 */
	}
	inline EntnodeData *Ref(void) {++ref; return this;}
	inline EntnodeData *UnRef(void)
	{
		if(--ref == 0)
		{
			delete this;
			return 0L;
		}
		return this;
	}

	virtual const char *operator[](int index) const
	{
		if(index == kName)
			return user;
		else if(index == kStatus)
			return desc;
		return 0L;
	}

	inline void SetMissing(bool state) {missing = state;}
	inline bool IsMissing(void) const {return missing;}
	inline void SetVisited(bool state) {visited = state;}
	inline bool IsVisited(void) const {return visited;}
	inline void SetUnknown(bool state) {unknown = state;}
	inline bool IsUnknown(void) const {return unknown;}
	inline void SetIgnored(bool state) {ignored = state;}
	inline bool IsIgnored(void) const {return ignored;}
	inline void SetLocked(bool state) {locked = state;}
	inline bool IsLocked(void) const {return locked;}
	void SetDesc(const char *newdesc);
	inline const char *GetDesc(void) const {return desc;}
	inline void SetUnmodified(bool state) {unmodified = state;}
	inline bool IsUnmodified(void) const {return unmodified;}

#ifdef macintosh
	FSSpec & MacSpec(void) {return macspec;}
	const FSSpec & GetMacSpec(void) const {return macspec;}
#endif /* macintosh */
protected:
	int ref;
	ent_type type;
	char *user;
	char *desc;
	bool missing;
	bool visited;
	bool unknown;
	bool unmodified;
	bool ignored;
	bool locked;
#ifdef macintosh
	FSSpec macspec;
#endif /* macintosh */
};

class EntnodeFile : public EntnodeData
{
public:
	EntnodeFile(const char *newuser, const char *newvn = 0L,
		const char *newts = 0L, const char *newoptions = 0L,
		const char *newtag = 0L, const char *newdate = 0L,
		const char *newts_conflict = 0L);

	virtual ~EntnodeFile();

	enum
	{
		kVN = 1,
		kTS = 5,
		kOption = 2,
		kTag = 4,
		kConflict = 6
	};

	// access fields
	virtual const char *operator[](int index) const
	{
		if(index == kVN)
			return vn;
		else if(index == kTS)
			return ts;
		else if(index == kOption)
			return option;
		else if(index == kTag)
			return tag != 0L ? tag : date;
		else if(index == kConflict)
			return ts_conflict;

		return EntnodeData::operator[](index);
	}

protected:
	char *vn;
	char *ts;
	char *option;
	char *tag; // can be nil
	char *date; // can be nil
	char *ts_conflict; // can be nil
};

class EntnodeFolder : public EntnodeData
{
public:
	EntnodeFolder(const char *newuser, const char *newvn = 0L,
		const char *newts = 0L, const char *newoptions = 0L, const char *newtag = 0L,
		const char *newdate = 0L, const char *newts_conflict = 0L);

	virtual ~EntnodeFolder();

	// access fields
	virtual const char *operator[](int index) const
	{
		return EntnodeData::operator[](index);
	}

protected:
};

// ENTNODE gets a node file or a node folder
// and reference it.
class ENTNODE
{
public:
	ENTNODE() : shareData(0L) {}
	ENTNODE(EntnodeData *data) : shareData(data->Ref()) {}
	ENTNODE(EntnodeData & data) : shareData(data.Ref()) {}
	ENTNODE(const ENTNODE & anode) : shareData(0L)
	{
		*this = anode;
	}
	~ENTNODE()
	{
		if(shareData != 0L)
			shareData->UnRef();
	}

	inline static int Compare(const ENTNODE & node1, const ENTNODE & node2)
	{
		return EntnodeData::Compare(*node1.shareData, *node2.shareData);
	}

	inline ENTNODE & operator=(const ENTNODE & anode)
	{
		if(shareData != 0L)
		{
			shareData->UnRef();
			shareData = 0L;
		}
		if(anode.shareData != 0L)
			shareData = ((ENTNODE *)&anode)->shareData->Ref();
		return *this;
	}

	inline EntnodeData *Data(void) {return shareData;}
protected:
	EntnodeData *shareData;
};

bool Entries_Open (CSortList<ENTNODE> & entries, const char *fullpath);
	// return false if no CVS/Entries

EntnodeData *Entries_SetVisited(CSortList<ENTNODE> & entries, const char *name,
	const struct stat & finfo, bool isFolder, const vector<CStr> * ignlist = 0L);
	// fill an ENTNODE when the file appears on the disk : it set
	// some flags like "visited", "unknown", "ignored"... and return
	// a reference to the node.

void Entries_SetMissing(CSortList<ENTNODE> & entries);
	// After the traversal and for the entries which didn't have a
	// "Entries_SetVisited" call associated with, mark them to be "missing".

bool Tag_Open(CStr & tag, const char *fullpath);
	// return false if no CVS/Tag

#endif /* CVSENTRIES_H */
