/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
* Author : Jerzy Kaczorowski <kaczoroj@hotmail.com> --- July 2002
*/

// BrowseViewHandlers.cpp : implementation file
//

#include "stdafx.h"

#ifdef WIN32
#include "wincvs.h"
#endif /* !WIN32 */

#include "BrowseViewHandlers.h"
#include "PromptFiles.h"
#include "MultiFiles.h"

/*!
	Constructor
	\param strCommand Command name
	\param needSelection Need selection indicator
*/
KiSelectionHandler::KiSelectionHandler(const char* strCommand, bool needSelection /*= true*/)
	: m_strCommand(strCommand), m_pmf(NULL), m_selType(kSelNone), 
	  m_autoDelete(false), m_needSelection(needSelection)
{
}

KiSelectionHandler::~KiSelectionHandler()
{
	if( m_autoDelete && m_pmf )
	{
		delete m_pmf;
	}
}

/*!
	Get the selection type
	\return The selection type
*/
kSelType KiSelectionHandler::GetSelType() const
{
	return m_selType;
}

/*!
	Get the need selection flag
	\return	The need selection flag
*/
bool KiSelectionHandler::GetNeedSelection() const
{
	return m_needSelection;
}

/*!
	Called for selected folder
	\param fullpath Directory path to add to selection
	\note Tyically you should call the KiSelectionHandler::OnFolder in your override first
*/
void KiSelectionHandler::OnFolder(const char* fullpath)
{
	m_selType = kSelDir;
	if( fullpath )
	{
		m_pmf = new MultiFiles;
		if( m_pmf )
		{
			m_autoDelete = true;
			m_pmf->newdir(fullpath);
		}
	}
}

/*!
	Called for selected files
	\param pmf Files selection as MultiFiles
	\note Tyically you should call the KiSelectionHandler::OnFiles in your override first
*/
void KiSelectionHandler::OnFiles(MultiFiles* pmf)
{
	m_selType = kSelFiles;
	m_pmf = pmf;
}

/*!
	Get the selection
	\return The selection as a pointer to MultiFiles object
	\note Prompts the user if neccessary (controlled by KiSelectionHandler::m_needSelection flag)
*/
MultiFiles* KiSelectionHandler::GetSelection()
{
	if( !m_pmf && m_needSelection )
	{
		switch( m_selType )
		{
		case kSelFiles:
			m_pmf = new MultiFiles;
			if( m_pmf )
			{
				m_autoDelete = true;
				
				CStr strPrompt;
				strPrompt << "Select files for ";
				strPrompt << m_strCommand;
				strPrompt << ":";
				
				if( !BrowserGetMultiFiles(strPrompt, *m_pmf) )
				{
					delete m_pmf;
					m_pmf = NULL;
				}
			}
			break;
		case kSelDir:
			m_pmf = new MultiFiles;
			if( m_pmf )
			{
				CStr strPrompt;
				strPrompt << "Select directory to ";
				strPrompt << m_strCommand;
				strPrompt << ":";
				
				const char* dir = BrowserGetDirectory(strPrompt);
				if( !dir )
				{
					delete m_pmf;
					m_pmf = NULL;
				}
				
				m_pmf->newdir(dir);
			}
			break;
		default:
#ifdef WIN32
			ASSERT(FALSE);	// type must be set to something if m_needSelection is true
#endif
			break;
		}
	}
	
	return m_pmf;
}
