/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Author : Alexandre Parenteau <aubonbeurre@hotmail.com> --- December 1997
 */

/*
 * dll_loader.cpp --- code to load dynamic code
 */

#ifdef HAVE_CONFIG_H
#	include "config.h"
#endif /* HAVE_CONFIG_H */

#if !defined(WIN32) && !defined(TARGET_OS_MAC)
#	include "dynl.h"
#endif /* !WIN32 && !TARGET_OS_MAC */

#ifdef WIN32
#	include <windows.h>
#endif /* WIN32 */

#include <string.h>

#include "dll_loader.h"

void* sCompatProc;

#ifdef TARGET_OS_MAC
#	include <TextUtils.h>
	static ProcPtr getSymFunction(CompatConnectID connID, const Str255 name)
	{
		long symCount;
		Ptr symAddr;
		int i;
		OSErr err = CountSymbols(connID, &symCount);
		if(err != noErr)
			return 0L;
		
		for(i = 0; i < symCount; i++)
		{
			CFragSymbolClass symClass;
			Str255 symName;
			err = GetIndSymbol(connID, i, symName, &symAddr, &symClass);
			if(err != noErr)
				return 0L;
			if(memcmp(symName, name, name[0]) == 0)
				break;
		}
		if(i == symCount)
			return 0L;
		return (ProcPtr)symAddr;
	}
#endif /* TARGET_OS_MAC */

/*!
	Load the shared library
	\param connID Return the connection ID
	\param name_dll	Filename of the library (on Mac the fragment name of the shared library)
	\return true on success, false otherwise
*/
int CompatLoadLibrary(CompatConnectID* connID, const char* name_dll)
{
	if( connID == 0L )
		return 0;

#ifdef TARGET_OS_MAC
	{
		OSErr err;
		Ptr mainAddr;
		Str255 errMessage;
		Str255 pname;
		
		c2pstrcpy(pname, name_dll);
		err = GetSharedLibrary(pname, kPowerPCCFragArch, kPrivateCFragCopy, connID, &mainAddr, errMessage);
		return err == noErr;
	}
#elif defined(WIN32)
	*connID = LoadLibrary(name_dll);
	return (*connID) != 0L;
#else
	*connID = dynl_link(name_dll);
	return (*connID) != 0L;
#endif
}

/*!
	Load the shared library
	\param connID The connection ID, should not be used anymore after this call 
	\return true on success, false otherwise
*/
int CompatCloseLibrary(CompatConnectID* connID)
{
	if( connID == 0L )
		return 0;

#ifdef TARGET_OS_MAC
	CloseConnection(connID);
#elif defined(WIN32)
	FreeLibrary(*connID);
	connID = 0L;
#else
	dynl_unlink(*connID);
	connID = 0L;
#endif

	return 1;
}

/*!
	Call a function in the shared library
	\param connID The connection ID
	\param name_func Function name be called
	\return The function address
*/
void* _CompatCallLibrary(CompatConnectID connID, const char* name_func)
{
#ifdef TARGET_OS_MAC
	ProcPtr symAddr;
	Str255 pname;

	c2pstrcpy(pname, name_func);

	symAddr = getSymFunction(connID, pname);
	return symAddr;
#elif defined(WIN32)
	return GetProcAddress(connID, name_func);
#else
	return dynl_func(name_func, connID);
#endif
}
