/*
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 1, or (at your option)
** any later version.

** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.

** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef __BROWSEVIEW_COLUMN_H__
#define __BROWSEVIEW_COLUMN_H__

class EntnodeData;

#include "ustr.h"

/// Context in which a column is accessed
class KoColumnContext
{
// Attributes
public:
	UStr m_path;	/*!< Current path */

public:
	KoColumnContext(const char* path)
	: m_path(path)
	{}	
};

#ifndef WIN32
	struct LV_COLUMN
	{
		int width;
		const char* name;
	};
#endif

/// Column descriptor interface
class KiColumn 
{
// Interfaces
public:
	/// Retrieve setup data
	virtual void GetSetupData(LV_COLUMN* lvc) const = 0;

	/// Tells if the column shall be sorted ascending initially
	virtual bool IsDefaultAscending() const = 0;

	/// Retrieve formatted text as pertained to a node
	virtual void GetText(KoColumnContext* context, EntnodeData* data, char* buffer, int buffersize) const = 0;

	/// Compare two nodes (-1,0,+1)
	virtual int Compare(KoColumnContext* context, EntnodeData* d1, EntnodeData* d2) const = 0;
};

/// Columnar model of the data		
class KiColumnModel
{
// Interfaces
public:
	/// Gets column descriptor by its index
	virtual const KiColumn* GetAt(int pos) const = 0;

	/// Gets column type by its index
	virtual int GetType(int pos) const = 0;

	/// Returns count of the columns
	virtual int GetCount() const = 0;

	/// Gets column index by its type
	virtual int GetColumn(int type) const;

// Factory methods
public:
	static KiColumnModel* GetRecursiveModel();
	static KiColumnModel* GetRegularModel();
};

/// Sort parameters
struct CSortParam
{
	CSortParam (bool fAccendant, bool fGroupSubDir, const KiColumn* column, const KiColumn* columnAlt, KoColumnContext* context) 
	: m_column(column)
	, m_columnAlt(columnAlt)
	, m_context(context)
	, m_fAccendant(fAccendant)
	, m_fGroupSubDir(fGroupSubDir)
	{};

	const KiColumn* m_column;	  // descriptor of the column being sorted
	const KiColumn* m_columnAlt;// descriptor of the alternate sorting column
	KoColumnContext* m_context;
	bool m_fAccendant;
	bool m_fGroupSubDir;	
};

/// Compare callback
#ifdef WIN32
int CALLBACK _Compare(LPARAM data1, LPARAM data2, LPARAM data);
#else
int _Compare(void* data1, void* data2, void* data);
#endif

#endif
// __BROWSEVIEW_COLUMN_H__
