/*
 * GUSITTY.cp
 * (c) 2000 Jean-Pierre Stierlin.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "GUSIInternal.h"
#include "GUSIDevice.h"
#include "GUSIDescriptor.h"
#include "GUSIBasics.h"
#include "GUSIDiag.h"

#include "GUSIlsh.h"

#include <LowMem.h>

#include <fcntl.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#define LSH_VERSION_H_INCLUDED
#include "GUSIMSLSetup.h"
#include "dll_glue.h"

extern "C" {
	void error (int, int, const char *, ...);
}

static gusilsh_ctx *gContext;

#if PRAGMA_STRUCT_ALIGN
#pragma options align=native
#endif

class GUSILSHDevice : public GUSIDevice {
public:
	static GUSILSHDevice *	Instance();

	// [[GUSILSHDevice]] is prepared to handle an [[open]] on a limited set of names.
 //                                                                         
 // <Overridden member functions for class [[GUSILSHDevice]]>=            
 virtual bool Want(GUSIFileToken & file);
 // [[open]] returns instance of [[GUSILSHSocket]].              
 //                                                                         
 // <Overridden member functions for class [[GUSILSHDevice]]>=            
 virtual GUSISocket * open(GUSIFileToken &, int flags);
private:
	GUSILSHDevice() {}

	static GUSILSHDevice *	sInstance;
};

class GUSILSHSocket : public GUSISocket {
public:
	~GUSILSHSocket();
	
	// Some member functions are fairly trivial wrappers.                      
 //                                                                         
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 ssize_t	read(const GUSIScatterer & buffer);
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 ssize_t write(const GUSIGatherer & buffer);
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 virtual int	ioctl(unsigned int request, va_list arg);
 // Since we know we're running on a pseudodevice, we can pass on that fact.
 //                                                                         
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 virtual int	fstat(struct stat * buf);
 // And we also know we're a TTY.                                           
 //                                                                         
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 virtual int	isatty();
 // [[select]] requires a walk of the event queue.                          
 //                                                                         
 // <Overridden member functions for class [[GUSILSHSocket]]>=            
 bool select(bool * canRead, bool * canWrite, bool *);

protected:
	GUSILSHSocket(gusilsh_ctx *context);

	gusilsh_ctx *mContext;
	
	friend class GUSILSHDevice;
};


#if PRAGMA_STRUCT_ALIGN
#pragma options align=reset
#endif



/*
 * GUSIwithTTYSockets
 */
void GUSIwithLSHSockets()
{
	if(lsh_init != 0L)
	{
		GusiLSHSetWriteConsole(dllglue_consoleout);
		GusiLSHSetErrConsole(dllglue_consoleerr);
		GusiLSHSetInConsole(dllglue_consolein);
		GusiLSHSetHandleEvent(dllglue_dispatchevent);
		lsh_init();
		GUSIDeviceRegistry::Instance()->AddDevice(GUSILSHDevice::Instance());
	}
}

void GUSIwithoutLSHSockets()
{
	if(lsh_init != 0L)
	{
		// it's important to remove it since the cvs code
		// is dynamically loaded, but GUSI is not
		GusiLSHSetWriteConsole(0L);
		GusiLSHSetErrConsole(0L);
		GusiLSHSetInConsole(0L);
		GusiLSHSetHandleEvent(0L);
		GUSIDeviceRegistry::Instance()->RemoveDevice(GUSILSHDevice::Instance());
		lsh_terminate();
	}
}



/*
 * GUSILSHDevice::Instance
 */

GUSILSHDevice * GUSILSHDevice::sInstance;

GUSILSHDevice * GUSILSHDevice::Instance()
{
	if (!sInstance)
		sInstance = new GUSILSHDevice();
	return sInstance;
}

/*
 * GUSILSHDevice::Want
 */
bool GUSILSHDevice::Want(GUSIFileToken &file)
{
	const char *path = file.Path();

	switch (file.WhichRequest()) {
	case GUSIFileToken::kWillOpen:
		return file.IsDevice() && file.StrFragEqual(path+4, "lsh");
	default:
		return false;
	}
}

/*
 * GUSILSHDevice::open
 */
GUSISocket * GUSILSHDevice::open(GUSIFileToken &file, int flags)
{
	GUSISocket *sock;

	if ( !(sock = new GUSILSHSocket(gContext) )) {
		GUSISetPosixError(ENOMEM);
	}
	return sock;
}


/*
 * GUSILSHSocket::GUSILSHSocket
 */
GUSILSHSocket::GUSILSHSocket(gusilsh_ctx *context) : mContext(context)
{
}


/*
 * GUSILSHSocket::~GUSILSHSocket
 */
GUSILSHSocket::~GUSILSHSocket()
{
	// we rely on lsh to close properly. We prefer
	// it to die first so we make sure no code remains executed
	// when the code fragment is unloaded later on.
	long howLong = ::TickCount();
	int times = 10;
	while(lsh_running(mContext->lsh_ctx))
	{
		if((::TickCount() - howLong) > 120)
		{
			howLong = ::TickCount();
			if(--times == 0)
			{
				lsh_delete(mContext->lsh_ctx);
				mContext->doAbort = 1;
				break;
			}
			error(0, 0, "Waiting for ssh to die (%d)...", times);
		}

		lsh_yield();		
	}
}

/*
 * GUSILSHSocket::read
 */
ssize_t GUSILSHSocket::read(const GUSIScatterer & buffer)
{
	// flush stdout first
	while(lsh_running(mContext->lsh_ctx) && lsh_canwrite(mContext->lsh_ctx) > 0)
	{
		lsh_yield();
	}
	
	// get some stdin
	while(lsh_running(mContext->lsh_ctx) && lsh_canread(mContext->lsh_ctx) == 0)
	{
		lsh_yield();
	}
	
	return buffer.SetLength(
		lsh_read(mContext->lsh_ctx, buffer.Buffer(), buffer.Length()));
}

/*
 * GUSILSHSocket::write
 */
ssize_t GUSILSHSocket::write(const GUSIGatherer & buffer)
{
	return lsh_write(mContext->lsh_ctx, buffer.Buffer(), buffer.Length());
}

/*
 * GUSILSHSocket::ioctl
 */
int GUSILSHSocket::ioctl(unsigned int request, va_list)
{
	switch (request)	{
	case FIOINTERACTIVE:
		return 0;
	default:
		return GUSISetPosixError(EOPNOTSUPP);
	}
}

/*
 * GUSILSHSocket::fstat
 */
int	GUSILSHSocket::fstat(struct stat * buf)
{
	GUSISocket::fstat(buf);
	buf->st_mode =	S_IFCHR | 0666;
	
	return 0;
}

/*
 * GUSILSHSocket::isatty
 */
int GUSILSHSocket::isatty()
{ 
	return 1;
}

/*
 * GUSILSHSocket::select
 */
bool GUSILSHSocket::select(bool * canRead, bool * canWrite, bool *)
{
	bool cond = false;
	if (canRead)
		if (*canRead = lsh_canread(mContext->lsh_ctx))
			cond = true;
	if (canWrite)
		cond = *canWrite = true;
		
	return cond;
}

void GUSILSH_set_context(gusilsh_ctx *ctx)
{
	gContext = ctx;
}

void GUSILSH_cleanup (int sig)
{
	/* we get here when we gt a SIGINT signal (probably
	a user which hits Cmd-.).*/
	
	if(gContext)
	{
		gContext->doAbort = 1;
		
		/* we are on the lsh thread */
		if(lsh_running(gContext->lsh_ctx))
		{
			lsh_delete(gContext->lsh_ctx);
		}
		if(gContext->oldHandler.sa_handler != 0L)
			gContext->oldHandler.sa_handler(sig);
	}
}

void GUSILSH_hdlevt(long userData, long sleepTime)
{
	if(gContext && gContext->doAbort && lsh_current() != 0L)
	{
		
	}
}
