<?php
//============================================================+
// File name   : tce_db_dal_postgresql.php
// Begin       : 2003-10-12
// Last Update : 2006-11-24
// 
// Description : PostgreSQL driver for TCExam Database
//               Abstraction Layer (DAL).
//               This abstraction use the same SQL syntax
//               of MySQL.
//
// Author: Nicola Asuni
//
// (c) Copyright:
//               Tecnick.com S.r.l.
//               Via Ugo Foscolo n.19
//               09045 Quartu Sant'Elena (CA)
//               ITALY
//               www.tecnick.com
//               info@tecnick.com
//
// License: GNU GENERAL PUBLIC LICENSE v.2
//          http://www.gnu.org/copyleft/gpl.html
//============================================================+

/**
 * PostgreSQL driver for TCExam Database Abstraction Layer (DAL).
 * This abstraction layer uses the same SQL syntax of MySQL.
 * @package com.tecnick.tcexam.shared.postgresql
 * @author Nicola Asuni
 * @copyright Copyright &copy; 2004-2006, Tecnick.com S.r.l. - Via Ugo Foscolo n.19 - 09045 Quartu Sant'Elena (CA) - ITALY - www.tecnick.com - info@tecnick.com
 * @link www.tecnick.com
 * @since 2004-12-21
 */

/**
 * Open a connection to a PostgreSQL Server and select a database.
 * If a second call is made to this function with the same arguments, no new link will be established, but instead, the link identifier of the already opened link will be returned.
 * @param string $server Database server path. It can also include a port number. e.g. "hostname:port" or a path to a local socket e.g. ":/path/to/socket" for the localhost.
 * @param string $username Name of the user that owns the server process.
 * @param string $password Password of the user that owns the server process.
 * @param string $database Database name.
 * @return PostgreSQL link identifier on success, or FALSE on failure. 
 */
function F_db_connect($host = 'localhost', $port = '5432', $username = 'postgres', $password = '', $database = 'template1') {
	$connection_string = "host='".$host."' port='".$port."' dbname='".$database."' user='".$username."' password='".$password."'";
	if(!$db = @pg_connect($connection_string)) {
		return FALSE;
	}
	return $db;
}

/**
 * Closes the non-persistent connection to a database associated with the given connection resource.
 * @param resource $link_identifier database link identifier.
 * @return bool TRUE on success or FALSE on failure
 */
function F_db_close($link_identifier) {
	return pg_close($link_identifier);
}

/**
 * Returns the text of the error message from previous database operation
 * @return string error message.
 */
function F_db_error() {
	return pg_last_error();
}

/**
 * Sends a query to the currently active database on the server that's associated with the specified link identifier.<br>
 * NOTE: Convert MySQL RAND() function to PostgreSQL RANDOM() on ORDER BY clause of selection queries.
 * @param string $query The query tosend. The query string should not end with a semicolon. 
 * @param resource $link_identifier database link identifier.
 * @return FALSE in case of error, TRUE or resource-identifier in case of success.
 */
function F_db_query($query, $link_identifier) {
	// convert MySQL RAND() function to PostgreSQL RANDOM()
	$query = preg_replace("/ORDER BY RAND\(\)/si", "ORDER BY RANDOM()", $query);
	return pg_query($link_identifier, $query);
}

/**
 * Fetch a result row as an associative and numeric array.
 * Note: This function sets NULL fields to PHP NULL value.
 * @param resource $result result resource to the query result.
 * @return Returns an array that corresponds to the fetched row, or FALSE if there are no more rows.
*/
function F_db_fetch_array($result) {
	return pg_fetch_array($result);
}

/**
 * Returns number of rows (tuples) affected by the last INSERT, UPDATE or DELETE query associated with link_identifier.
 * @param resource $link_identifier database link identifier [UNUSED].
 * @param resource $result result resource to the query result.
 * @return Number of rows.
 */
function F_db_affected_rows($link_identifier, $result) {
	return pg_affected_rows($result);
}

/**
 * Get number of rows in result.
 * @param resource $result result resource to the query result.
 * @return Number of affected rows.
 */
function F_db_num_rows($result) {
	return pg_num_rows($result);
}

/**
 * Get the ID generated from the previous INSERT operation
 * @param resource $link_identifier database link identifier.
 * @param string Table name.
 * @param string Field name (column name).
 * @return int ID generated from the last INSERT operation.
 */
function F_db_insert_id($link_identifier, $tablename = '', $fieldname = '') {
	if(!$result = @pg_query($link_identifier, "SELECT CURRVAL('".$tablename."_".$fieldname."_seq')")) {
		return 0;
	}
	if(!$seq_array = pg_fetch_row($result, 0)) {
		return 0;
	}
	return $seq_array[0]; 
}

//============================================================+
// END OF FILE                                                 
//============================================================+
?>