/* Module:          SQLCancel.c
 *
 * Description:     Cancels the processing on a statement.
 *
 * Classes:         
 *
 * API functions:   SQLCancel
 *
 * Comments:        See "notice.txt" for copyright and license information.
 *
 */

#include "driver.h"

SQLRETURN SQL_API SQLCancel(SQLHSTMT	hDrvStmt)  /* Statement to cancel. */
{
	static char *func="SQLCancel";
	StatementClass *stmt = (StatementClass *) hDrvStmt;
	SQLRETURN result;
	#ifdef WIN32
	HMODULE hmodule;
	FARPROC addr;
	#endif

	mylog( "%s: entering...\n", func);

	/*	Check if this can handle canceling in the middle of a SQLPutData? */
	if ( ! stmt)
	{
		SC_log_error(func, "", NULL);
		return SQL_INVALID_HANDLE;
	}

	/*	Not in the middle of SQLParamData/SQLPutData so cancel like a close. */
	if (stmt->data_at_exec < 0)
	{
		/*	MAJOR HACK for Windows to reset the driver manager's cursor state:
			Because of what seems like a bug in the Odbc driver manager,
			SQLCancel does not act like a SQLFreeStmt(CLOSE), as many
			applications depend on this behavior.  So, this 
			brute force method calls the driver manager's function on
			behalf of the application.  
		*/
		#ifdef WIN32
		if (globals.cancel_as_freestmt)
		{
			hmodule = GetModuleHandle("ODBC32");
			addr = GetProcAddress(hmodule, "SQLFreeStmt");
			result = addr( (char *) (stmt->phstmt) - 96, SQL_CLOSE);
		}
		else
		{
			result = SQLFreeStmt( hDrvStmt, SQL_CLOSE);
		}
		#else
		result = SQLFreeStmt( hDrvStmt, SQL_CLOSE);
		#endif

		mylog("SQLCancel:  SQLFreeStmt returned %d\n", result);

		SC_clear_error(hDrvStmt);
		return SQL_SUCCESS;
	}

	/*	In the middle of SQLParamData/SQLPutData, so cancel that. */
	/*	Note, any previous data-at-exec buffers will be freed in the recycle */
	/*	if they call SQLExecDirect or SQLExecute again. */

	stmt->data_at_exec = -1;
	stmt->current_exec_param = -1;
	stmt->put_data = FALSE;

	return SQL_SUCCESS;

}
