/*
 * Created on Feb 26, 2005
 */
package dguitar.players.sound;

import dguitar.adaptors.song.SongMeasure;
import dguitar.adaptors.song.SongTrack;
import dguitar.adaptors.song.Tempo;
import dguitar.adaptors.song.TimeSignature;



/**
 * A Performance is an interface to a prepared version of a Song. Implementations
 * of this interface must be prepared to deliver PerformanceEvent objects to listeners
 * when the music is playing. Implementations will also typically have to generate
 * specific note data (such as MIDI messages or wave files).
 * @author crnash
 **/
public interface Performance
{
    /**
     * Set the frequency of timer events, in clicks per beat.
     * @param frequency
     */
    void setTimerFrequency(int frequency);
    
    /**
     * Generate initialization events (such as controller initialization and
     * patch selection) for the given SongTrack. The implementor is responsible
     * for creating the actual performed data as well as a PerformanceTrackInitEvent
     * on virtual track 0 of the given SongTrack.
     * @param track the SongTrack to add. 
     */
    void initializeTrack(SongTrack track);
    
    /**
     * Generate timer events for the given measure, which will be at the specified
     * location. It is up to the implementor to ensure that at least one event
     * (corresponding to the measure start) is generated by this call.
     * @param measure the measure for which to add information
     * @param location the time offset, measured in timing pulses
     */
    void addTimerEvents(SongMeasure measure,int location);
    
    /**
     * Adds the given list of musical events to the performance. The particular
     * performance implementation is responsible for adding these events, as-is,
     * to the event stream for the particular index of track and virtual track.
     * In addition, the performance implementation must generate any necessary
     * audio that will be played by the given player (for example, MIDI events
     * or wave files).
     * <p>
     * Typically the PerformanceEvent is little more than a wrapped SongEvent
     * with added time information. Note this means that a PerformanceEvent
     * applies strictly to the <b>notation</b> of a Song, not necessarily
     * the actual notes. For example, a tied note may be delivered as multiple
     * events, which a typical implementation must be ready to handle as a
     * single played note (but keep the multiple events in the event stream).
     * <p>
     * The implementor should not necessarily assume the incoming list of
     * events is in precise performance order (rather, the events will arrive
     * in the order they appear on a score). Events on each call will
     * originate from the same virtual track. Note that it is the responsibility
     * of the implementation to expect and handle calls from multiple virtual
     * tracks into the same physical track.
     * @param events an event stream of PerformanceEvents to add 
     */
    void addMusicalEvents(EventStream events);

    /**
     * @return a Tempo object for his Performance
     */
    Tempo getTempo();

    /**
     * @param location
     * @param currentTimeSignature
     */
    void setTimeSignature(int location, TimeSignature currentTimeSignature);
}
