/*
** makedata.c - Database creator for Oblique Strategies
**
** Copyright 1998, Lonnon Foster <lonnief@pobox.com>
** All rights reserved.
**
** The original Oblique Strategies are Copyright 1975, 78, 79
** by Brian Eno and Peter Schmidt.
**
** This program is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** This program is distributed in the hope that it will be useful, but
** WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program (see LICENSE.TXT); if not, write to the
** Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139,
** USA.
**
** Created 30 June, 1998.  Last edited 7 July, 1998.
*/

#include <System/SysAll.h>
#include <UI/UIAll.h>

#include "makedata.h"
#include "resource.h"


/* Global variables */
static short        CurrNo = 0;
DmOpenRef           obstratDB;
char                obstratDBName[] = "ObStratDB";


/*
** AboutViewHandleEvent
**   Event handler for the About form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean AboutViewHandleEvent(EventPtr event)
{
    Boolean handled = false;
    FormType frm;
    
    switch (event->eType)
    {
        case frmOpenEvent:
            FrmDrawForm(FrmGetActiveForm());
            handled = true;
            break;
        case ctlEnterEvent:
            if (event->data.popSelect.controlID == btnID_AboutOK)
            {
                FrmReturnToForm(frmID_Main);
                handled = true;
                break;
            }
    }
    return(handled);
}


/*
** MainViewHandleEvent
**   Event handler for the Main form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean MainViewHandleEvent(EventPtr event)
{
    Boolean handled = false;
    FormType frm;
    RectangleType rect;
    
    switch (event->eType)
    {
        case frmOpenEvent:
            FrmDrawForm(FrmGetActiveForm());
            break;
        case menuEvent:
            MenuEraseStatus(0);
            switch (event->data.menu.itemID)
            {
                case mniID_MainAbout:
                    FrmPopupForm(frmID_About);
                    handled = true;
                    break;
            }
            handled = true;
            break;
    }
    return(handled);
}


/*
** ApplicationHandleEvent
**   Loads a form and sets the event handler for the form.
** Inputs
**   event    Pointer to an event type structure.
** Returns
**   True if the event has been dealt with.
*/
static Boolean ApplicationHandleEvent(EventPtr event)
{
    FormPtr frm;
    Int formId;
    Boolean handled = false;
    
    if (event->eType == frmLoadEvent) {
        formId = event->data.frmLoad.formID;
        frm = FrmInitForm(formId);
        FrmSetActiveForm(frm);
        
        switch (formId) {
            case frmID_Main:
                FrmSetEventHandler(frm, MainViewHandleEvent);
                break;
            case frmID_About:
                FrmSetEventHandler(frm, AboutViewHandleEvent);
                break;
        }
        handled = true;
    }
    return handled;
}


/*
** CreateDatabase
**   Creates the application database from the structure in memory
*/
Boolean CreateDatabase(void)
{
    UInt         index = 0;
    VoidHand     recHandle;
    Ptr          recPointer;
    int          i;
    UInt         cardNo;
    LocalID      dbID;
    Word         attributes;
    
    obstratDB = DmOpenDatabaseByTypeCreator(obstratDBType, obstratAppID,
                                            dmModeReadWrite);
    if (!obstratDB) {
        /* Create a new database */
        DmCreateDatabase(0, obstratDBName, obstratAppID, obstratDBType,
                         false);

        /* Open the new database */
        obstratDB = DmOpenDatabaseByTypeCreator(obstratDBType, obstratAppID,
                                                dmModeReadWrite);
        
        /* Fill the database with records from the ObStrat list */
        for (i = 0; i < MAXNO; i++) {
            index = DmNumRecords(obstratDB);
            recHandle = DmNewRecord(obstratDB, &index,
                                    StrLen(ObStrat[i]) + 1);
            recPointer = MemHandleLock(recHandle);
            DmWrite(recPointer, 0, ObStrat[i], StrLen(ObStrat[i]) + 1);
            MemPtrUnlock(recPointer);
            DmReleaseRecord(obstratDB, index, true);
        }
    }
    /* Set the backup bit for the database */
    DmOpenDatabaseInfo(obstratDB, &dbID, NULL, NULL, &cardNo, NULL);
    DmDatabaseInfo(cardNo, dbID, NULL, &attributes, NULL, NULL,
                   NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    attributes |= dmHdrAttrBackup;
    DmSetDatabaseInfo(cardNo, dbID, NULL, &attributes, NULL, NULL,
                      NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    return 1;
}


/*
** StartApplication
**   Takes care of things which need to be done when opening the app.
*/
int StartApplication(void)
{
    int error;
    
    /* Create application database */
    error = CreateDatabase();

    FrmGotoForm(frmID_Main);
}


/*
** StopApplication
**   Takes care of cleanup when switching to another app.
*/
void StopApplication(void)
{
    DmCloseDatabase(obstratDB);
    FrmCloseAllForms();
}


/*
** EventLoop
**   Hands out events from the Event Manager to the system,
**   menu, and application event handlers before passing them
**   to the application for processing.
*/
static void EventLoop(void)
{
    EventType event;
    Word error;
    do
    {
        EvtGetEvent(&event, evtWaitForever);
        if (! SysHandleEvent(&event))
            if (! MenuHandleEvent(NULL, &event, &error))
                if (! ApplicationHandleEvent(&event))
                    FrmDispatchEvent(&event);
    }
    while (event.eType != appStopEvent);
}

/*
** PilotMain
**   Entry point for the app.
** Inputs
**   cmd          command specifying how to launch the app
**   cmdPBP       parameter block for the command
**   launchFlags  flags used to configure the launch
** Returns
**   Error code
*/
DWord PilotMain (Word cmd, Ptr cmdPBP, Word launchFlags)
{
    int error;

    if (cmd == sysAppLaunchCmdNormalLaunch) {
        error = StartApplication();
        if (error) return error;
        EventLoop();
        StopApplication();
    }
    return 0;
}
