/*
 *      FileXFer.c - based on SimpleTerm 0.5, a simple Pilot terminal
 *      
 *	This program is used for file transfers between a pilot
 *	and a computer running a terminal program.
 *  
 */

#include <Pilot.h>
#include <SerialMgr.h>
#include <CallBack.h>
#ifdef PAMLIII
#include <FontSelect.h>
#endif
#define   NON_PORTABLE
#include <SystemPrv.h>

#include "FileXFer.h"

/*
 * Structeres
 *
 */
typedef struct {
	UInt baud;
	UInt bits;
	UInt parity;
	UInt xon;
	UInt rts;
	UInt version;
        FontID editFont;
        FontID mainFont;
} FileXFerPreferenceType;

typedef struct {
        CharPtr name;
        CharPtr contents;
} FileXFerUnpackedRecType;

typedef FileXFerUnpackedRecType * FileXFerUnpackedRecPtr;

/*
 * Defines
 *
 */
#define versionRequired 0x02000000

#define idAppFileXFer 'SPFX'
#define dbTypeFileXFer 'Data'
#define dbNameFileXFer "FileXFer"

#define noRecordSelected -1
#define noFormLoaded -1

#define replaceString "\\0"

/*
 * Prototypes for internal functions
 *
 */
static Err StartApplication(void);
static void StopApplication(void);
static Err RomVersionCompatible(DWord requiredVersion, Word launchFlags);

static Boolean OpenDatabase(void);
static Err OpenDatabaseInfo(DmOpenRef db);
static Boolean CreateRecord(void);
static void GetUnpackedRecord(VoidHand recordHandle, FileXFerUnpackedRecPtr record);
static Boolean DeleteRecord(Int recordNum);
static void GotoRecord(Int record);

static void UpdateCategory(FormPtr frm);
static void SetCategoryTrigger(FormPtr frm, Word categoryNum);
static void SelectCategory(FormPtr frm);
static Word SelectCategoryOnly(FormPtr frm, Word categoryNum);
static void SetCategory(void);
static FieldPtr GetFocusObjectPtr(void);
static FontID nextFont(FontID font);
static Boolean FormHandleMenuEvent(EventPtr event);

static void MainFormInit(FormPtr frmFileXFer);
static void MainFormTableScroll(FormPtr frmFileXFer, Int linesToScroll, Boolean relative);
static void DrawFileNames(VoidPtr tablePtr, Word row, Word column, RectanglePtr bounds);
static Boolean MainFormHandleEvent(EventPtr event);

static void EditFormInit(FormPtr frmFileEdit);
static void EditFormUpdateScrollBar(FormPtr frmEdit);
static void EditFormScroll(FormPtr frmEdit, Short linesToScroll);
static Boolean EditFormHandleEvent(EventPtr event);

static void InitPreferences(FormPtr frm);
static Boolean PrefFormHandleEvent(EventPtr event);

static Boolean DetailsFormHandleEvent(EventPtr event);

static Boolean QuickSendFormHandleEvent(EventPtr event);

static void EventLoop(void);
static Boolean RxData(void);
static void SetSerialSettings(void);
static Boolean OpenPort(void);
static void SendData(FieldPtr field);

static void FrmError(CharPtr s);
static void DebugDialog(CharPtr v1, Long v2I, Long v3I);

/*
 * Global variables
 *
 */
static FileXFerPreferenceType prefs;
static UInt serialRef;

static DmOpenRef dbFileXFer;
static Int currentRecordNum = noRecordSelected;
static Int currentForm = noFormLoaded;
static Word currentCategoryNum;
Char categoryName[dmCategoryLength];

static Boolean portClosed;

/*
 * Opens the serial library and initializes the database
 *
 */
static Err StartApplication(void) {
	Err error;

	error = SysLibFind("Serial Library", &serialRef);
	ErrNonFatalDisplayIf(error, errOpeningSerial);

	portClosed = true;

	error = OpenDatabase();
	ErrNonFatalDisplayIf(error, errOpeningDB);

	FrmGotoForm(idFormFileXFer);

	return 0;
}

/*
 * Closes the database
 *
 */
static void StopApplication (void) {
        FrmCloseAllForms();

	PrefSetAppPreferences(idAppFileXFer, 0, 1, &prefs, sizeof(FileXFerPreferenceType), true);
	DmCloseDatabase(dbFileXFer);

	if (!portClosed) SerClose(serialRef);
}

/*
 * Checks for ROM version compatibility
 *
 */
static Err RomVersionCompatible(DWord requiredVersion, Word launchFlags) {
        DWord romVersion;
	
	FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
	if (romVersion < requiredVersion) {
	        if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) == (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) {
		        FrmAlert(idAlertIncompatibleROM);

			if (romVersion < 0x02000000) {
			        Err err;

                                AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
                        }
                        return sysErrRomIncompatible;
		}
        }

        return NULL;
}

/*
 * Opens the database or creates one
 *
 */
static Boolean OpenDatabase(void) {
	Word prefSize;

	currentRecordNum = 0;

        dbFileXFer = DmOpenDatabaseByTypeCreator(dbTypeFileXFer, idAppFileXFer, dmModeReadWrite);
	if (! dbFileXFer) {
	        if (DmCreateDatabase(0, dbNameFileXFer, idAppFileXFer, dbTypeFileXFer, false)) return true;
		dbFileXFer = DmOpenDatabaseByTypeCreator(dbTypeFileXFer, idAppFileXFer, dmModeReadWrite);
		OpenDatabaseInfo(dbFileXFer);
	}

	prefSize = sizeof(FileXFerPreferenceType);
	if (PrefGetAppPreferences(idAppFileXFer, 0, &prefs, &prefSize, true) == noPreferenceFound) {
	        prefs.bits = 8;
		prefs.parity = 0;
		prefs.baud = 9600;
		prefs.xon = 1;
		prefs.rts = 1;
		prefs.version = 1;

		prefs.editFont = stdFont;
		prefs.mainFont = stdFont;

		SetSerialSettings();
	}

	currentCategoryNum = dmAllCategories;

	return false;
}

/*
 * Sets the categories
 *
 */
static Err OpenDatabaseInfo(DmOpenRef db) {
	UInt cardNo;
	VoidHand h;
	LocalID dbID;
	LocalID appInfoID;
	AppInfoPtr appInfoPtr;
	AppInfoPtr nilPtr = 0;

	if (DmOpenDatabaseInfo(db, &dbID, NULL, NULL, &cardNo, NULL)) return dmErrInvalidParam;
	if (DmDatabaseInfo(cardNo, dbID, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &appInfoID, NULL, NULL, NULL)) return dmErrInvalidParam;

	/* If no appInfoID exists create a new one. */
	if (appInfoID == NULL) {
		h = DmNewHandle (db, sizeof(AppInfoType));
		if (! h) return dmErrMemError;
		
		appInfoID = MemHandleToLocalID(h);
		DmSetDatabaseInfo(cardNo, dbID, NULL, NULL, NULL, NULL, NULL, NULL, NULL, &appInfoID, NULL, NULL, NULL);
	}

	appInfoPtr = MemLocalIDToLockedPtr(appInfoID, cardNo);

	/* Clear the app info block */
	DmSet(appInfoPtr, 0, sizeof(AppInfoType), 0);

	/* Set the default categories */
	DmStrCopy(appInfoPtr, (ULong)nilPtr->categoryLabels[0], "Unfiled");
	DmStrCopy(appInfoPtr, (ULong)nilPtr->categoryLabels[1], "Personal");

	/* Initialize the categories */
	CategoryInitialize(appInfoPtr, idListCategories);

	MemPtrUnlock(appInfoPtr);

	return 0;
}

/*
 * Creates a new record and sets the current record to 0
 *
 */
static Boolean CreateRecord(void) {
        VoidHand RecHandle;
	Ptr RecPointer;
	Char recordContents[25];
	UInt size;
	Err error;

	currentRecordNum = 0;

	/* Name */
	StrCopy(recordContents, strUnnamed);
	size = StrLen(recordContents) + 1;
	/* Contents */
	StrCopy(&recordContents[size], "");
	size += 1;

     	RecHandle = DmNewRecord(dbFileXFer, &currentRecordNum, size);
	RecPointer = MemHandleLock(RecHandle);

	error = DmWrite(RecPointer, currentRecordNum, &recordContents, size);
	ErrFatalDisplayIf(error, errCouldNotCreateRecord);

	MemPtrUnlock(RecPointer);
	DmReleaseRecord(dbFileXFer, currentRecordNum, true);

	/* Set Category */
	SetCategory();

	return true;
}

/*
 * Get a record
 *
 */
static void GetUnpackedRecord(VoidHand recordHandle, FileXFerUnpackedRecPtr record) {
        VoidPtr recordPtr;

	recordPtr = MemHandleLock(recordHandle);
	/* Name */
	record->name = recordPtr;
	/* Contents */
	record->contents = recordPtr + StrLen(record->name) + 1;
}

/*
 * Shows a confirmationm alert and deletes a record
 *
 */
static Boolean DeleteRecord(Int recordNum) {
 	if (FrmAlert(idAlertDelete) == 0) {
	        FormPtr form;
		FieldPtr field;

		form = FrmGetFormPtr(idFormFileEdit);
		if (form) {
		        field = (FieldPtr)(FrmGetObjectPtr(form, (FrmGetObjectIndex(form, idFieldFile))));
			if (field) FldSetTextHandle(field, 0);	        
		}
	
		form = FrmGetFormPtr(idFormDetails);
		if (form) {
		        field = (FieldPtr)(FrmGetObjectPtr(form, (FrmGetObjectIndex(form, idFieldName))));
			if (field) FldSetTextHandle(field, 0);
		}

	        DmRemoveRecord(dbFileXFer, recordNum);
		currentRecordNum = noRecordSelected;

		return true;
	}

	return false;
}

/*
 * Shows a record
 *
 */
static void GotoRecord(Int record) {
        OpenPort();

        currentRecordNum = record;
	FrmGotoForm(idFormFileEdit);
}

/*
 * Recieves data and writes it to the current record
 *
 */
static Boolean RxData(void) {
  	FormPtr frmFileEdit;
	FieldPtr fldFile;	
	ULong numBytes = 0;			
	CharPtr recieveBuffer;
	Err error;
					
	if (portClosed)	return false;
								
	error = SerReceiveCheck(serialRef, &numBytes);
	if (error) {
		SerClearErr(serialRef);					
		FrmError(errRecieving); 
		return false;
	} else if (numBytes <= 0) return false;

	recieveBuffer = MemPtrNew(numBytes + 1);
	numBytes = SerReceive(serialRef, recieveBuffer, numBytes, -1, &error);

	if (error) {
 	        FrmError(errRecieving);
		SerClearErr(serialRef);
	} else {
	        frmFileEdit = FrmGetFormPtr(idFormFileEdit);
		if (frmFileEdit) {
		         Boolean allReplaced;
			 Word strLength;
			 CharPtr posString;

	                 fldFile = (FieldPtr)(FrmGetObjectPtr(frmFileEdit, (FrmGetObjectIndex(frmFileEdit, idFieldFile))));
			 
			 recieveBuffer[numBytes] = '\0';
			 strLength = numBytes + 1;
			 while (strLength > 0) {
			         allReplaced = false;

			         while (! allReplaced) {
				         posString = StrStr(recieveBuffer, replaceString);

					 /* Insert another String to be able to convert it back */
					 if (posString != NULL) {
					         ULong diff = (posString - recieveBuffer) + StrLen(replaceString);

						 FldInsert(fldFile, recieveBuffer, diff);
						 FldInsert(fldFile, replaceString, StrLen(replaceString));
						 strLength -= diff;
						 recieveBuffer += diff;
					 } else allReplaced = true;
				 }
				 FldInsert(fldFile, recieveBuffer, StrLen(recieveBuffer));				 

				 strLength -= StrLen(recieveBuffer) + 1;
				 recieveBuffer += StrLen(recieveBuffer) + 1;
				 if (strLength > 0) FldInsert(fldFile, replaceString, StrLen(replaceString));
			 }
			 /* Reset to original address */
			 recieveBuffer -= numBytes + 1;
		}
	}
	MemPtrFree(recieveBuffer);

	return true;
}

/*
 * Sends the text in the field
 *
 */
void SendData(FieldPtr field) {
        Err error;
	CharPtr contents;
	ULong numBytes;
			
        OpenPort();
	if ((contents = FldGetTextPtr(field)) != NULL) {
	        CharPtr sendBuffer;
		Word strLength = 0;
		CharPtr posString;
		Boolean allReplaced = false;
		
	        numBytes = StrLen(contents);
		sendBuffer = MemPtrNew(numBytes + 1);

		StrCopy(sendBuffer, contents);

		while (! allReplaced) {
			 posString = StrStr(sendBuffer, replaceString);

			 if (posString) {
                                 SerSend(serialRef, sendBuffer, posString - sendBuffer, &error);
				 if (error) {
				         FrmError(errSending);
					 SerClearErr(serialRef);	
				 }
				 posString += StrLen(replaceString);
                                 if (StrStr(posString, replaceString) == posString) {
				          /* Two in a row */
				          posString += StrLen(replaceString);
					  SerSend(serialRef, replaceString, StrLen(replaceString), &error);
				 } else {
				          /* Send zero char */
				          SerSend(serialRef, "", 1, &error);
				 }
				 if (error) {
				         FrmError(errSending);
					 SerClearErr(serialRef);	
				 }
				 sendBuffer += posString - sendBuffer;
			 } else {
			         allReplaced = true;
			         strLength = StrLen(sendBuffer);
			         SerSend(serialRef, sendBuffer, strLength, &error);
				 if (error) {
				         FrmError(errSending);
					 SerClearErr(serialRef);	
				 }
			 }
		}

		/* Reset to original address */
		sendBuffer -= numBytes - strLength;
		MemPtrFree(sendBuffer);
	
		// Pause until the send buffer clears...
		SerSendWait(serialRef, -1);
	}
}

/*
 * Set serial settings and save app-prefs
 *
 */
static void SetSerialSettings(void) {
        SerSettingsType serialSettings;
	Err error;
	
	SerGetSettings(serialRef, &serialSettings);

	serialSettings.baudRate = prefs.baud;
	serialSettings.flags = 0;
	serialSettings.ctsTimeout = serDefaultCTSTimeout;

	if (prefs.xon == 1) serialSettings.flags = serSettingsFlagXonXoffM;

	if (prefs.bits == 7) serialSettings.flags |= serSettingsFlagStopBits1 | serSettingsFlagBitsPerChar7;
	else serialSettings.flags |= serSettingsFlagStopBits1 | serSettingsFlagBitsPerChar8;

	if (prefs.rts == 1) serialSettings.flags |= serSettingsFlagCTSAutoM | serSettingsFlagRTSAutoM;
 
	if (prefs.parity == 1) serialSettings.flags |= serSettingsFlagParityOnM;
	else if (prefs.parity == 2) serialSettings.flags |= serSettingsFlagParityOnM | serSettingsFlagParityEvenM;

	if (! portClosed) {
	        error = SerSetSettings(serialRef, &serialSettings);
		if (error) {
		        SerClearErr(serialRef);
			ErrDisplay(errSettingSerial);
		}
	}
		
	PrefSetAppPreferences(idAppFileXFer, 0, 1, &prefs, sizeof(FileXFerPreferenceType), true);
}

/*
 * Opens the serial port
 *
 */
Boolean OpenPort(void) {
	Err error;

        if (!portClosed) return false; // Port is already open!
							
	error = SerOpen(serialRef, 0, prefs.baud);

	if (error) {
		ErrNonFatalDisplayIf( (error == serErrAlreadyOpen), "Another application is using the serial port");
		ErrNonFatalDisplayIf( (error == memErrNotEnoughSpace), "Not enough memory to open the serial port");
		ErrNonFatalDisplayIf( (error == serErrBadParam), "The serial port could not be opened");

		return true;
	}

	portClosed = false;							
	SetSerialSettings();

	return true;
}

/*
 * Updates the global variables
 *
 */
static void UpdateCategory(FormPtr frm) {
	UInt attr;

	DmRecordInfo(dbFileXFer, currentRecordNum, &attr, NULL, NULL);
	currentCategoryNum = attr & dmRecAttrCategoryMask;

	if (frm) SetCategoryTrigger(frm, currentCategoryNum);
	else CategoryGetName(dbFileXFer, currentCategoryNum, categoryName);
}

/* 
 * Sets the category popup trigger
 *
 */
static void SetCategoryTrigger(FormPtr frm, Word categoryNum) {
	ControlPtr ctl;

	/* Set the popup trigger's label that contains the category */
        ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idPopupCategories))));
	if (ctl) {
	        CategoryGetName(dbFileXFer, categoryNum, categoryName);
		CategorySetTriggerLabel (ctl, categoryName);
	}
}

/*
 * Selects the Category
 *
 */
static void SelectCategory(FormPtr frm) {
        /* Update the current category */
        UpdateCategory(NULL);

	CategorySelect(dbFileXFer, frm, idPopupCategories, idListCategories, false, &currentCategoryNum, categoryName, 1, 0);

	SetCategory();
        /* Update the current category and the popup trigger */
	UpdateCategory(frm);
}

/*
 * Selects the Category
 *
 */
static Word SelectCategoryOnly(FormPtr frm, Word categoryNum) {
	CategoryGetName(dbFileXFer, categoryNum, categoryName);
	CategorySelect(dbFileXFer, frm, idPopupCategories, idListCategories, false, &categoryNum, categoryName, 1, 0);

	SetCategoryTrigger(frm, categoryNum);
	return categoryNum;
}

/*
 * Sets the Category in the record
 *
 */
static void SetCategory(void) {
	UInt attr;

        DmRecordInfo(dbFileXFer, currentRecordNum, &attr, NULL, NULL);	
	attr &= ~dmRecAttrCategoryMask;
	if (currentCategoryNum == dmAllCategories) currentCategoryNum = CategoryFind(dbFileXFer, strUnfiled);
	attr |= currentCategoryNum | dmRecAttrDirty;
	DmSetRecordInfo(dbFileXFer, currentRecordNum, &attr, NULL);
}

/*
 * Returns the next font, because FontSelect is not supported by the compiler (gcc 0.5.0)
 *
 */
static FontID nextFont(FontID font) {
        switch(font) {
	case stdFont: return boldFont;
	case boldFont: return largeFont;
	default: return stdFont;
	}
}	

/*
 * Handles the common menu entrys
 *
 */
static Boolean FormHandleMenuEvent(EventPtr event) {
	Boolean	handled = false;

	if (event->eType == menuEvent) {
		MenuEraseStatus(0);
		handled = true;

		switch (event->data.menu.itemID) {
		case idMenuitemQuickSend:
		        FrmPopupForm(idFormQuickSend);
  			break;
		case idMenuitemNew:
		        CreateRecord();
			GotoRecord(currentRecordNum);
			break;
		case idMenuitemKeyboard:
		        SysKeyboardDialog(kbdAlpha);
		        break;
		case idMenuitemGraffiti:
			SysGraffitiReferenceDialog (referenceDefault);
			break;
		case idMenuitemPreferences:
		        FrmPopupForm(idFormPreferences);
  			break;
		case idMenuitemAbout:
		        FrmAlert(idAlertAbout);
			break;
		default:
		        handled = false;
		}
	}

	return handled;
}

/*
 * Initializes the main form
 *
 */
static void MainFormInit(FormPtr frmMain) {
        UInt recordCount;
	TablePtr tablePtr;
	UInt row;
	Word position;

        recordCount = DmNumRecords(dbFileXFer);

	tablePtr = (TablePtr)(FrmGetObjectPtr(frmMain, (FrmGetObjectIndex(frmMain, idTableFiles))));
	for (row = 0; row < TblGetNumberOfRows(tablePtr); row++) {
		/* Name */
		TblSetItemStyle(tablePtr, row, 0, customTableItem);
		/* Size */
		TblSetItemStyle(tablePtr, row, 1, customTableItem);
	}

	TblSetColumnUsable(tablePtr, 0, true);
	TblSetColumnUsable(tablePtr, 1, true);

	TblSetCustomDrawProcedure(tablePtr, 0, DrawFileNames);
	TblSetCustomDrawProcedure(tablePtr, 1, DrawFileNames);

	SetCategoryTrigger(frmMain, currentCategoryNum);

	if (currentRecordNum != noRecordSelected) {
	        if (DmNumRecordsInCategory(dbFileXFer, currentCategoryNum) > 0) position = DmPositionInCategory(dbFileXFer, currentRecordNum, currentCategoryNum);        
		else position = 0;
	} position = 0;

	MainFormTableScroll(frmMain, position, false);
}

/*
 * Scrolls the table with the file names
 *
 */
static void MainFormTableScroll(FormPtr frmFileXFer, Int linesToScroll, Boolean relative) {
	TablePtr tablePtr;
	Err error = 0;
	Word recordCount;
	Word firstRecord;
	UInt maxRows;
	Int row;
	ControlPtr ctlUp;
	ControlPtr ctlDown;
	UInt offset = 0;
	Word recordNum = 0;
	Int position;

	/* Scrollers */
	ctlUp = (ControlPtr)(FrmGetObjectPtr(frmFileXFer, (FrmGetObjectIndex(frmFileXFer, idRepeatFilesUp))));
	ctlDown = (ControlPtr)(FrmGetObjectPtr(frmFileXFer, (FrmGetObjectIndex(frmFileXFer, idRepeatFilesDown))));

	CtlSetUsable(ctlUp, false);
	CtlSetUsable(ctlDown, false);

	/* Table */
        tablePtr = (TablePtr)(FrmGetObjectPtr(frmFileXFer, (FrmGetObjectIndex(frmFileXFer, idTableFiles))));
	recordCount = DmNumRecordsInCategory(dbFileXFer, currentCategoryNum); 
	maxRows = TblGetNumberOfRows(tablePtr);

	/* Get the first record */
	if (relative) offset = TblGetRowID(tablePtr, 0);
	else { 
	        error = DmSeekRecordInCategory(dbFileXFer, &offset, 0, dmSeekForward, currentCategoryNum);
		relative = true;
	}

	/* Calculate first row */
	if (! error) {
	        position = DmPositionInCategory(dbFileXFer, offset, currentCategoryNum);

		if (position + linesToScroll > recordCount - maxRows) linesToScroll = (recordCount - maxRows) - position;
		if (position - linesToScroll < 0) linesToScroll = -position;

		for(row = 0; row < Abs(linesToScroll); row++) {
		        recordNum = 0;
			if (linesToScroll > 0) {
			        offset++;
				DmSeekRecordInCategory(dbFileXFer, &recordNum, offset, dmSeekForward, currentCategoryNum);
			} else if (linesToScroll < 0) {
			        offset--;
				DmSeekRecordInCategory(dbFileXFer, &recordNum, offset, dmSeekBackward, currentCategoryNum);
			}
			offset = recordNum;
		}
	}
	firstRecord = offset;

	/* Set rows */
	for (row = 0; row < maxRows; row++) {
	        if (row < recordCount) {
		        recordNum = 0;
		        DmSeekRecordInCategory(dbFileXFer, &recordNum, offset, dmSeekForward, currentCategoryNum);
			offset = recordNum + 1;
			TblSetRowID(tablePtr, row, recordNum);
			TblSetRowUsable(tablePtr, row, true);
			TblMarkRowInvalid(tablePtr, row);
		} else {
		        TblSetRowUsable(tablePtr, row, false);
		}
	}

	if (recordCount > maxRows) {
	        CtlSetUsable(ctlUp, true);
		CtlSetUsable(ctlDown, true);
		if (DmPositionInCategory(dbFileXFer, firstRecord, currentCategoryNum) > 0) {
		        CtlSetLabel(ctlUp, "\001");
			CtlSetEnabled(ctlUp, true);
		} else {
		        CtlSetLabel(ctlUp, "\003");
			CtlSetEnabled(ctlUp, false);
		}
		/* Check last visible record (pos zero-based)*/
		if (recordCount > DmPositionInCategory(dbFileXFer, recordNum, currentCategoryNum) + 1) {
		        CtlSetLabel(ctlDown, "\002");
			CtlSetEnabled(ctlDown, true);
		} else {
		        CtlSetLabel(ctlDown, "\004");
			CtlSetEnabled(ctlDown, false);
		}
	}

	TblRedrawTable(tablePtr);
}

/*
 * Draw the file names in the list
 *
 */
static void DrawFileNames(VoidPtr tablePtr, Word row, Word column, RectanglePtr bounds) {
	VoidHand recordHandle;
	FileXFerUnpackedRecType record;

        CALLBACK_PROLOGUE
	
	recordHandle = (VoidHand)DmQueryRecord(dbFileXFer, TblGetRowID(tablePtr, row));
	GetUnpackedRecord(recordHandle, &record);
	MemHandleUnlock(recordHandle);

	FntSetFont(prefs.mainFont);

	switch (column) {
	case 0: {
	        /* File Name */
	        Int textLen;
		Int width;
		Boolean fits;
		
		width = bounds->extent.x - 2;
		textLen = StrLen(record.name);
		FntCharsInWidth(record.name, &width, &textLen, &fits);
	
	        WinDrawChars(record.name, textLen, bounds->topLeft.x, bounds->topLeft.y);

		break; }
	case 1: {
	        Char sizeLabel[10];
		
		StrIToA(sizeLabel, StrLen(record.contents) / 1024);
	        /* File Size */
	        StrCat(sizeLabel, " KB");
	        WinDrawChars(sizeLabel, StrLen(sizeLabel), bounds->topLeft.x, bounds->topLeft.y);	  

		break; }
	}

        CALLBACK_EPILOGUE
}

/*
 * Handles events for the "main" form.
 *
 */
static Boolean MainFormHandleEvent(EventPtr event) {
	Boolean	handled = false;

	CALLBACK_PROLOGUE

	if (event->eType == frmOpenEvent) {
	       FormPtr frmFileXFer;

	       frmFileXFer = FrmGetActiveForm();
	       MainFormInit(frmFileXFer);
	       FrmDrawForm(frmFileXFer);

	       handled = true;
	} else if (event->eType == ctlSelectEvent) {
	       FormPtr frmFileXFer;

	       frmFileXFer = FrmGetActiveForm();

	       handled = true;

   	       switch (event->data.ctlEnter.controlID) {
	       case idButtonNew:
		       CreateRecord();
		       GotoRecord(currentRecordNum);

		       break;
	       case idPopupCategories:
		       CategorySelect(dbFileXFer, frmFileXFer, idPopupCategories, idListCategories, true, &currentCategoryNum, categoryName, 1, 0);
		       SetCategoryTrigger(frmFileXFer, currentCategoryNum);
		       MainFormTableScroll(frmFileXFer, 0, false);

		       break;
	       default:
		       handled = false;
	       }
	} else if (event->eType == ctlRepeatEvent) {
	       handled = false;

   	       switch (event->data.ctlEnter.controlID) {
	       case idRepeatFilesUp:
		       MainFormTableScroll(FrmGetActiveForm(), -1, true);

		       break;
	       case idRepeatFilesDown:
		       MainFormTableScroll(FrmGetActiveForm(), 1, true);

		       break;
	       }
	} else if (event->eType == tblSelectEvent) {
  	       if (event->data.tblSelect.tableID == idTableFiles) {
		        GotoRecord(TblGetRowID(event->data.tblSelect.pTable, event->data.tblSelect.row));
	       }
	} else if (event->eType == menuEvent) {
		FontID fontID;

		handled = FormHandleMenuEvent(event);

		if (! handled) {
			switch (event->data.menu.itemID) {
			case idMenuitemSendCategory:
		        	DebugDialog("This function has not been implemented, yet", idMenuitemSendCategory, 0);
				break;
			case idMenuitemFont:
		        	#ifdef PAMLIII
		        	fontID = FontSelect(prefs.mainFont);
				#endif
				fontID = nextFont(prefs.mainFont);
				if (fontID != prefs.mainFont) {
			        	FormPtr frmFileXFer;
					TablePtr tablePtr;

					frmFileXFer = FrmGetActiveForm();
					tablePtr = (TablePtr)(FrmGetObjectPtr(frmFileXFer, (FrmGetObjectIndex(frmFileXFer, idTableFiles))));
					prefs.mainFont = fontID;
					TblEraseTable(tablePtr);
					TblDrawTable(tablePtr);
				}
				break;
			default:
				handled = false;
			}
		}
	}

	CALLBACK_EPILOGUE

	return handled;
}

/*
 * Returns a pointer to the current control
 *
 */
static FieldPtr GetFocusObjectPtr(void) {
        FormPtr form;
	UInt focus;

	form = FrmGetActiveForm();
	focus = FrmGetFocus(form);

	if (focus == noFocus) return NULL;
	return FrmGetObjectPtr(form, focus);
}

/*
 * Initializes the edit form
 *
 */
static void EditFormInit(FormPtr frmFileEdit) {
        FieldPtr fldFile;
	VoidHand recordHandle;
	FileXFerUnpackedRecType record;
	CharPtr recordPtr;
	UInt offset;
	UInt attr;
	FieldAttrType fieldAttr;

	/* Set the database handle */
	fldFile = (FieldPtr)(FrmGetObjectPtr(frmFileEdit, (FrmGetObjectIndex(frmFileEdit, idFieldFile))));
	
	recordHandle = (VoidHand)DmGetRecord(dbFileXFer, currentRecordNum);
	GetUnpackedRecord(recordHandle, &record);
	
	recordPtr = MemDeref(recordHandle);
	offset = record.contents - recordPtr;

	FrmSetTitle(frmFileEdit, record.name);
	FldSetText(fldFile, recordHandle, offset, StrLen(record.contents) + 1);
	       
	MemHandleUnlock(recordHandle);

	/* Set the category trigger without changing "currentCategoryNum" */
	DmRecordInfo(dbFileXFer, currentRecordNum, &attr, NULL, NULL);
	SetCategoryTrigger(frmFileEdit, attr & dmRecAttrCategoryMask);

	/* Set the font */
	FldSetFont(fldFile, prefs.editFont);

	FldGetAttributes(fldFile, &fieldAttr);
	fieldAttr.hasScrollBar = true;
	FldSetAttributes(fldFile, &fieldAttr);

	EditFormUpdateScrollBar(frmFileEdit);
}

/*
 * Updates the scroll bar
 *
 */
static void EditFormUpdateScrollBar(FormPtr frmEdit) {
	Word scrollPos;
	Word textHeight;
	Word fieldHeight;
	Short maxValue;
	FieldPtr fldFile;
	ScrollBarPtr sbarFile;

	fldFile = (FieldPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idFieldFile))));
	sbarFile = (ScrollBarPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idSBarFile))));
	
	FldGetScrollValues(fldFile, &scrollPos, &textHeight,  &fieldHeight);

	if (textHeight > fieldHeight) maxValue = textHeight - fieldHeight;
	else if (scrollPos) maxValue = scrollPos;
	else maxValue = 0;

	SclSetScrollBar(sbarFile, scrollPos, 0, maxValue, fieldHeight - 1);
}

/*
 * Scrolls the file
 *
 */
static void EditFormScroll(FormPtr frmEdit, Short linesToScroll) {
	Int blankLines;
	Short min;
	Short max;
	Short value;
	Short pageSize;
	FieldPtr fldFile;
	ScrollBarPtr sbarFile;
	
	fldFile = (FieldPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idFieldFile))));

	if (linesToScroll < 0) {
		blankLines = FldGetNumberOfBlankLines(fldFile);
		FldScrollField (fldFile, -linesToScroll, up);
		
		if (blankLines) {
		        sbarFile = (ScrollBarPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idSBarFile))));
			SclGetScrollBar(sbarFile, &value, &min, &max, &pageSize);
			if (blankLines > -linesToScroll) max += linesToScroll;
			else max -= blankLines;
			SclSetScrollBar (sbarFile, value, min, max, pageSize);
		}
	} else if (linesToScroll > 0) FldScrollField(fldFile, linesToScroll, down);
}

/*
 * Handles events for the edit form
 *
 */
static Boolean EditFormHandleEvent(EventPtr event) {
	Boolean	handled = false;

	CALLBACK_PROLOGUE

	if (event->eType == frmOpenEvent) {
	        FormPtr frmEdit;
	       
		frmEdit = FrmGetActiveForm();
		EditFormInit(frmEdit);
		FrmDrawForm(frmEdit);

		handled = true;
	} else if (event->eType == frmCloseEvent) {
	        FormPtr frmFileEdit;
		FieldPtr fldFile;

		frmFileEdit = FrmGetActiveForm();
		fldFile = (FieldPtr)(FrmGetObjectPtr(frmFileEdit, (FrmGetObjectIndex(frmFileEdit, idFieldFile))));

		FldSetTextHandle(fldFile, 0);
		DmReleaseRecord(dbFileXFer, currentRecordNum, true);

		/* Make sure the form is disposed by the default handler */
		handled = false;
	} else if (event->eType == frmUpdateEvent) {
	        EditFormInit(FrmGetActiveForm());

		handled = true;
	} else if (event->eType == fldChangedEvent) {
	        EditFormUpdateScrollBar(FrmGetActiveForm());
		
		handled = true;
	} else if (event->eType == sclRepeatEvent) {
	        EditFormScroll(FrmGetActiveForm(), event->data.sclRepeat.newValue - event->data.sclRepeat.value);

		handled = false;
	} else if (event->eType == ctlSelectEvent) {
	        FormPtr frmFileEdit;
		FieldPtr fldFile;

		frmFileEdit = FrmGetActiveForm();
		fldFile = (FieldPtr)(FrmGetObjectPtr(frmFileEdit, (FrmGetObjectIndex(frmFileEdit, idFieldFile))));

		switch (event->data.ctlEnter.controlID) {
		case idPopupCategories:
		        SelectCategory(frmFileEdit);
			break;
		case idButtonDone:
			FrmGotoForm(idFormFileXFer);
			break;
		case idButtonSend:
			SendData(fldFile);
			break;
		case idButtonDetails:
		        FldSetTextHandle(fldFile, 0);
			DmReleaseRecord(dbFileXFer, currentRecordNum, true);
		        FrmPopupForm(idFormDetails);
			break;
		}

		handled = true;
	} else if (event->eType == nilEvent) {		
		while (RxData());
		
		handled = true;
	} else if (event->eType == keyDownEvent) {
		if (event->data.keyDown.chr == pageUpChr) {		// Scroll up key presed?
			FormPtr frmEdit;
			FieldPtr fldFile;	

			frmEdit = FrmGetActiveForm();
			fldFile = (FieldPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idFieldFile))));
			FldScrollField(fldFile, 1, up);

			handled = true;
		} else if (event->data.keyDown.chr == pageDownChr) {	// Scroll down key presed?
			FormPtr frmEdit;
			FieldPtr fldFile;	

			frmEdit = FrmGetActiveForm();
			fldFile = (FieldPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idFieldFile))));
			FldScrollField(fldFile, 1, down);

			handled = true;
		} else {
		        FormPtr frmEdit;
		  
			frmEdit = FrmGetActiveForm();
			FrmHandleEvent(frmEdit, event);
			EditFormUpdateScrollBar(frmEdit);

			handled = true;
		}
	} else if (event->eType == menuEvent) {
	        FieldPtr fldFile;
		FormPtr frmEdit;
		FontID	fontID;

		MenuEraseStatus(0);
		handled = FormHandleMenuEvent(event);

		if (! handled) {
		        frmEdit = FrmGetActiveForm();
			fldFile = (FieldPtr)(FrmGetObjectPtr(frmEdit, (FrmGetObjectIndex(frmEdit, idFieldFile))));

			switch (event->data.menu.itemID) {
				case idMenuitemSend:
				  if (fldFile) FldUndo(fldFile);
				  SendData(fldFile);
				  break;
			case idMenuitemUndo:
				if (fldFile) FldUndo(fldFile);
				break;
			case idMenuitemCut:
				if (fldFile) FldCut(fldFile);
				break;
			case idMenuitemCopy:
				if (fldFile) FldCopy(fldFile);
				break;
			case idMenuitemPaste:
				if (fldFile) FldPaste(fldFile);
				break;
			case idMenuitemFont:
		        	#ifdef PAMLIII
			        fontID = FontSelect(prefs.editFont);
                                #endif
				fontID = nextFont(prefs.editFont);
				if (fontID != prefs.editFont) {
			        	prefs.editFont = fontID;
					FldSetFont(fldFile, fontID);
					EditFormUpdateScrollBar(frmEdit);
				}
				break;
			case idMenuitemGotoTop:
				fldFile = GetFocusObjectPtr();
				if (fldFile) FldSetScrollPosition(fldFile, 0);
				break;
			case idMenuitemGotoBottom:
		        	fldFile = GetFocusObjectPtr();
				if (fldFile) FldSetScrollPosition(fldFile, FldGetTextLength(fldFile));
				break;
			default:
		        	handled = false;
			}
		}
	}

	CALLBACK_EPILOGUE

	return handled;
}

static void InitPreferences(FormPtr frm) {
	ListPtr list;
	ControlPtr ctl;
				
	list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListBaud))));
	if (prefs.baud == 1200) LstSetSelection(list, 0);
	else if (prefs.baud == 2400) LstSetSelection(list, 1);
	else if (prefs.baud == 4800) LstSetSelection(list, 2);
	else if (prefs.baud == 9600) LstSetSelection(list, 3);
	else if (prefs.baud == 19200) LstSetSelection(list, 4);
	else if (prefs.baud == 38400) LstSetSelection(list, 5);
	else LstSetSelection(list, 6);
	ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idPopupBaud))));
	CtlSetLabel(ctl, LstGetSelectionText(list, LstGetSelection(list)));

	list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListBits))));
	if (prefs.bits == 7) LstSetSelection(list,0);
	else LstSetSelection(list, 1);
	ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idPopupBits))));
	CtlSetLabel(ctl, LstGetSelectionText(list, LstGetSelection(list)));

	list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListParity))));
	if (prefs.parity == 0) LstSetSelection(list, 0);
	else if (prefs.parity == 1) LstSetSelection(list, 1);
	else  LstSetSelection(list, 2);
	ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idPopupParity))));
	CtlSetLabel(ctl, LstGetSelectionText(list, LstGetSelection(list)));
	

	ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idCheckXonXoff))));
	if (prefs.xon == 1) CtlSetValue (ctl, 1);
	else CtlSetValue (ctl, 0);

	ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idCheckRTSCTS))));
	if (prefs.rts == 1) CtlSetValue (ctl, 1);
	else CtlSetValue (ctl, 0);
	if (prefs.baud > 19200) CtlSetEnabled(ctl, false);
}

/*
 * Handles the events for the preferences dialog
 *
 */			
static Boolean PrefFormHandleEvent(EventPtr event) {
	Boolean	handled = false;

	CALLBACK_PROLOGUE

	if (event->eType == frmOpenEvent) {
	        FormPtr formPref;

		formPref = FrmGetActiveForm();
		InitPreferences(formPref);
		FrmDrawForm(formPref);
		
		handled = true;
	} else if (event->eType == popSelectEvent) {
	        if (event->data.popSelect.listID == idListBaud) {
		        ControlPtr ctl;
			FormPtr frm;

			frm = FrmGetActiveForm();
			ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idCheckRTSCTS))));

		        /* RTS/CTS needs to be turned on with speeds greater than 19.200 */
		        if (event->data.popSelect.selection > 4) {
				CtlSetValue(ctl, true);
				CtlSetEnabled(ctl, false);
			} else CtlSetEnabled(ctl, true);
		}

		handled = false;
	} else if (event->eType == ctlSelectEvent) {
	        Int choice;
		ListPtr list;
		ControlPtr ctl;
		FormPtr frm;
				
		switch (event->data.ctlSelect.controlID) {
		case idButtonOkay:
			frm = FrmGetActiveForm();
							
			list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListBaud))));
			choice = LstGetSelection(list);
			if (choice == 0) prefs.baud = 1200;
			else if (choice == 1) prefs.baud = 2400;
			else if (choice == 2) prefs.baud = 4800;
			else if (choice == 3) prefs.baud = 9600;
			else if (choice == 4) prefs.baud = 19200;
			else if (choice == 5) prefs.baud = 38400;
			else prefs.baud = 57600;
			
			list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListBits))));
			choice = LstGetSelection(list);
			if (choice == 0) prefs.bits = 7;
			else prefs.bits = 8;

			list = (ListPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idListParity))));
			choice = LstGetSelection(list);
			if (choice == 0) prefs.parity = 0; /* None */
			else if (choice == 1) prefs.parity = 1; /* Odd */
			else prefs.parity = 2; /* Even */

			ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idCheckXonXoff))));
			if (CtlGetValue(ctl)) prefs.xon = 1;
			else prefs.xon = 0;
					
			ctl = (ControlPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idCheckRTSCTS))));
			if (CtlGetValue(ctl)) prefs.rts = 1;
			else prefs.rts = 0;

			SetSerialSettings();
			FrmReturnToForm(currentForm);

			handled = true;
			break;
		case idButtonCancel:
			FrmReturnToForm(currentForm);

			handled = true;
			break;				
		}
	}

	CALLBACK_EPILOGUE

	return handled;
}

/* 
 * Handles events for the details dialog
 *
 */
static Boolean DetailsFormHandleEvent(EventPtr event) {
	Boolean	handled = false;
	static Char orgName[30];
	static Word categoryNum;

	CALLBACK_PROLOGUE

	if (event->eType == frmOpenEvent) {
	        FormPtr formDetails;
		FieldPtr field;
		VoidHand recordHandle;
		FileXFerUnpackedRecType record;
		CharPtr recordPtr;
		UInt offset;
		UInt attr;

		formDetails = FrmGetActiveForm();
		field = (FieldPtr)(FrmGetObjectPtr(formDetails, (FrmGetObjectIndex(formDetails, idFieldName))));

	        recordHandle = DmQueryRecord(dbFileXFer, currentRecordNum);
	        GetUnpackedRecord(recordHandle, &record);

		recordPtr = MemDeref(recordHandle);
		offset = record.name - recordPtr;

		StrCopy(orgName, record.name);
		FldSetText(field, recordHandle, offset, StrLen(record.name) + 1);
	       
		DmRecordInfo(dbFileXFer, currentRecordNum, &attr, NULL, NULL);
		categoryNum = attr & dmRecAttrCategoryMask;
		SetCategoryTrigger(formDetails, categoryNum);

	        MemHandleUnlock(recordHandle);

		FrmDrawForm(formDetails);
		
		handled = true;
	} else if (event->eType == frmCloseEvent) {
	        FormPtr frm;
	        FieldPtr field;
		
		/* In case the form is not closed with on of the buttons */
		frm = FrmGetActiveForm();
		field = (FieldPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idFieldName))));

		FldSetTextHandle(field, 0);

	        /* Make sure the form is disposed by the default handler */
	        handled = false;
	} else if (event->eType == ctlSelectEvent) {
		FieldPtr field;
		FormPtr frm;
			
		frm = FrmGetActiveForm();
		field = (FieldPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idFieldName))));

		switch (event->data.ctlSelect.controlID) {
		case idPopupCategories:
		        categoryNum = SelectCategoryOnly(frm, categoryNum);

			handled = true;
			break;
		case idButtonCancel:
		        FldDelete(field, 0, FldGetMaxChars(field));
			FldInsert(field, orgName, StrLen(orgName));

			FldSetTextHandle(field, 0);

			FrmUpdateForm(currentForm, 0);
			FrmReturnToForm(currentForm);

			handled = true;
			break;
		case idButtonOkay:
			FldSetTextHandle(field, 0);
			currentCategoryNum = categoryNum;			

			FrmUpdateForm(currentForm, 0);
			FrmReturnToForm(currentForm);

			handled = true;
			break;
		case idButtonDelete:
 		        if (DeleteRecord(currentRecordNum)) {
			        FrmCloseAllForms(); // FrmGetFormPtr(currentForm));
				FrmGotoForm(idFormFileXFer);
			}

		        handled = true;
			break;
		}
	}

	CALLBACK_EPILOGUE

	return handled;
}

/* 
 * Handles events for the quick send dialog
 *
 */
static Boolean QuickSendFormHandleEvent(EventPtr event) {
	Boolean	handled = false;

	CALLBACK_PROLOGUE

	if (event->eType == frmOpenEvent) {
		FrmDrawForm(FrmGetActiveForm());
		
		handled = true;
	} else if (event->eType == ctlSelectEvent) {
		FieldPtr fldCmd;
		FormPtr frm;
			
		frm = FrmGetActiveForm();
		fldCmd = (FieldPtr)(FrmGetObjectPtr(frm, (FrmGetObjectIndex(frm, idFieldCommand))));

		switch (event->data.ctlSelect.controlID) {
		case idButtonDone:
			FrmReturnToForm(currentForm);

			handled = true;
			break;
		case idButtonSend:
		        FldInsert(fldCmd, "\n", 1);
			SendData(fldCmd);
		        FldDelete(fldCmd, 0, FldGetTextLength(fldCmd));

			handled = true;
			break;
		}
	}

	CALLBACK_EPILOGUE

	return handled;
}


static Boolean ApplicationHandleEvent(EventPtr event) {
        Boolean handled = false;

        if (event->eType == frmLoadEvent) {				
	        FormPtr frm;
		int formID;

	        formID = event->data.frmLoad.formID;
		frm = FrmInitForm(formID);
		FrmSetActiveForm(frm);

		switch (formID) {
		case idFormFileXFer:
		       currentForm = formID;
		       FrmSetEventHandler(frm, MainFormHandleEvent);
		       break;
		case idFormFileEdit:
		       currentForm = formID;
		       FrmSetEventHandler(frm, EditFormHandleEvent);
		       break;
		case idFormPreferences:
		       FrmSetEventHandler(frm, PrefFormHandleEvent);
		       break;
		case idFormDetails:
		       FrmSetEventHandler(frm, DetailsFormHandleEvent);
		       break;
		case idFormQuickSend:
		       FrmSetEventHandler(frm, QuickSendFormHandleEvent);
		       break;
		}
		handled = true;
	}
	return handled;
}

/*
 * Handles all incoming events
 *
 */
static void EventLoop(void) {
	EventType event;
	Word error;
	
	do {
		// Get the next available event.
		if (portClosed) // not listening for serial events
			EvtGetEvent(&event, evtWaitForever);
                else            // we are, so need to react quickly...
			EvtGetEvent(&event, 1); 
					
		// Give the system a chance to handle the event.
		if (! SysHandleEvent (&event))

			// P2. Give the menu bar a chance to update and handle the event.	
			  if (! MenuHandleEvent(0, &event, &error))

				// Give the application a chance to handle the event.
				if (! ApplicationHandleEvent(&event))
					// Let the form object provide default handling of the event.
					FrmDispatchEvent(&event);
		} 
	while (event.eType != appStopEvent);
}

/*
 * Displays an error message
 *
 */
static void FrmError(CharPtr s) {
        FrmCustomAlert(idAlertError, s, "", "");
}

/*
 * Displays 3 variables in an alert form
 *
 */
static void DebugDialog(CharPtr v1, Long v2I, Long v3I) {
        Char v2[10];
        Char v3[10];

	StrIToA(v3, v3I);
	StrIToA(v2, v2I);
	FrmCustomAlert(idAlertDebug, v1, v2, v3);
}

/*
 * Starts the event loop
 *
 */
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags) {
	Err error;

	error = RomVersionCompatible(versionRequired, launchFlags);
        if (error) return error;

	if (cmd == sysAppLaunchCmdNormalLaunch) {
		error = StartApplication();
		if (error) return error;
		EventLoop();
		StopApplication ();
	}
	return(0);
}
