/****************************************************************************
*                   atari.c
*
*  This module implements Atari ST/TT/Falcon specific routines.
*
*  from Persistence of Vision Raytracer
*  Copyright 1993 Persistence of Vision Team
*---------------------------------------------------------------------------
*  NOTICE: This source code file is provided so that users may experiment
*  with enhancements to POV-Ray and to port the software to platforms other 
*  than those supported by the POV-Ray Team.  There are strict rules under
*  which you are permitted to use this file.  The rules are in the file
*  named POVLEGAL.DOC which should be distributed with this file. If 
*  POVLEGAL.DOC is not available or for more info please contact the POV-Ray
*  Team Coordinator by leaving a message in CompuServe's Graphics Developer's
*  Forum.  The latest version of POV-Ray may be found there as well.
*
* This program is based on the popular DKB raytracer version 2.12.
* DKBTrace was originally written by David K. Buck.
* DKBTrace Ver 2.0-2.12 were written by David K. Buck & Aaron A. Collins.
*
*****************************************************************************/

/* 
 * this is the display code
 * for the Atari Computer Series.
 * Hacked out by Kay Rmer $17/12/92
 *
 * slightly improved by Dirk Klemmt 07/10/93
 */  

#include <math.h>
#include <time.h>
#include "config.h"
#define OBJECT object  
#define NORMAL normal /* to avoid doubly defined structure in aes.h */
#include "frame.h"
#undef OBJECT
#undef NORMAL

#ifdef __MINTLIB__ || __GNUC__
	#define __OLD_WAY__
	#include <mintbind.h>
	#include <aesbind.h>
	#include <vdibind.h>
#elif __TURBOC__
	#include <portab.h>
	#include <tos.h>
	#include <aes.h>
	#include <vdi.h>
#endif

#include "pov.rsh"  /* gem resources */
#define TREECOUNT 2 /* number of trees in pov.rsh */

#undef max
#undef min
#define NEW_COLORS 0 /* request to switch to our colors */
#define OLD_COLORS 1 /* ditto but to restore old colors */
#define C_INDEX 255  /* used color index in high color mode */
#define SET_SLID 100  
#define max(x,y) ((x)>(y) ? (x) : (y))
#define min(x,y) ((x)<(y) ? (x) : (y))
#define imagesize( x, y, planes ) (((long)(x)*(y)*(planes)+7)/8)
#define WinElem NAME|CLOSER|MOVER|SIZER|UPARROW|DNARROW|\
                LFARROW|RTARROW|VSLIDE|HSLIDE|FULLER|INFO

/* definitions for DragDrop-protokoll */
#ifndef AP_DRAGDROP
#define AP_DRAGDROP 63     /* event-number */
#define DD_NAK      1      /* negative acknowledge */
#endif

/* forward declarations */
static void redraw_window PARAMS(( void ));
static void slid_pos PARAMS(( int ));
static void slid_size PARAMS(( void ));
static int  rc_intersect PARAMS(( GRECT*, GRECT* ));
static int  mouse_clip PARAMS(( int[4] ));
static void send_redraw PARAMS(( void ));
static char FSdither PARAMS(( int, int, int, UBYTE, int (*)[][2] ));
static void PlotMethodSelect PARAMS(( void ));

static MFDB  thePixmap;
static FILE *theOutFh;
static char  theInfoStr[160];
static int   theApp,
			 theWin,
			 theVdi,
			 thePlanes,
			 theOfsX = 0,
			 theOfsY = 0,
			 (*errbufR)[][2],
			 (*errbufG)[][2],
			 (*errbufB)[][2],
			 (*old_pal)[][3],
			 (*new_pal)[][3];

static struct { int r, g, b; } col_bits;

static struct pixel
{
	MFDB  mfdb;
	UWORD pix;
} thePixel = { { &thePixel.pix,1,1,1,0,1,0,0,0 }, 0 };

static struct plot_methods
{
	void (*plot_init) PARAMS(( void ));
	void (*plot_exit) PARAMS(( void ));
	int  (*put_pixel) PARAMS(( int, int, UBYTE, UBYTE, UBYTE ));
} thePlot;

extern FRAME Frame;
extern volatile int Stop_Flag;
extern char Input_File_Name[],
            Output_File_Name[];

/*
 * This is for redirecting output from stdout/stderr 
 * to the file POVRAY.OUT, so we do not 
 * destroy the dektop.
 */
void atari_init_povray PARAMS(( void ))
{
	if( (theOutFh = fopen( "povray.out", "w" ))==NULL )
	{
		fprintf( stderr, "DISPLAY: cannot open povray.out" );
		exit(0);
	}
	memcpy( stdout, theOutFh, sizeof(FILE) );
	memcpy( stderr, theOutFh, sizeof(FILE) );
}

void print_other_credits( void )
{
	fprintf (stderr,"  Hi, this is my compilation of POVRay. I compiled it using the\n");
#ifdef __PUREC__
	fprintf (stderr,"  PureC-Compiler on "__DATE__"\n");
#elif __GNUC__
	fprintf (stderr,"  GnuC-Compiler on "__DATE__"\n");
#endif
	fprintf (stderr,"  The original port to atari was written by Kay Roemer.\n\n"); 
	fprintf (stderr,"  POVShell works fine with this version. Take a look at my shell. :-)\n\n");
	fprintf (stderr,"  Corrections, improvements, suggestions, etc. to:\n");
	fprintf (stderr,"     klemmt@informatik.uni-frankfurt.de\n");
	fprintf (stderr,"  bye and happy tracing -- Dirch\n\n");
	fflush( stderr );
}

int matherr ( struct exception *x ) 
{
	switch( x->type )
	{
		case DOMAIN:
		case OVERFLOW:
			x->retval = 1.0e17;
			break;

		case SING:
		case UNDERFLOW:
			x->retval = 0.0;
			break;

		default:
			break;
	}
	return(1);
}

void display_finished PARAMS(( void ))
{}

void display_init PARAMS(( void ))
{
	long mapsize;
	int  du, wx, wy, ww, wh, wwx, wwy, www, wwh, 
		 w_out[57], 
		 w_in[11] = {1,1,1,1,1,1,1,1,1,1,2};
  
	/* 
	 * First initialize AES and VDI, get Screen Info, etc.
	 */
	theApp = appl_init();
	graf_mouse( ARROW, NULL );
	theVdi = graf_handle( &du, &du, &du, &du );
	v_opnvwk( w_in, &theVdi, w_out );
	vq_extnd( theVdi, 1, w_out );
	thePlanes = w_out[4];

	PlotMethodSelect();
  
	/* 
	 * Now allocate an imagemap, because we have to
	 * remember the whole image in case of an redraw,
	 * window sizing, etc.
	 */
	mapsize = imagesize( Frame.Screen_Height, 
						 Frame.Screen_Width, thePlanes );

	if( (thePixmap.fd_addr = malloc( mapsize )) == NULL )
	{
		fprintf( stderr, "DISPLAY_INIT: failed to alloc pixmap\n" );
		exit(0);
	}
	memset( thePixmap.fd_addr, 0x00, mapsize );

	thePixmap.fd_w = Frame.Screen_Width;
	thePixmap.fd_h = Frame.Screen_Height;
	thePixmap.fd_wdwidth = thePixmap.fd_w/16;
	thePixmap.fd_stand = 0;
	thePixmap.fd_nplanes = thePlanes;
	thePixmap.fd_r1 = 
	thePixmap.fd_r2 =
	thePixmap.fd_r3 = 0;

	(*thePlot.plot_init)();

	/* 
	 * do the object conversion
	 * character set coordinates -> pixel coordinates
	 */
	{
		int t_idx, o_idx = 0;

		for( t_idx = 0; t_idx < TREECOUNT; t_idx++, o_idx = 0 )
			do 
				rsrc_obfix( rs_trindex[t_idx], o_idx );
			while( !(rs_trindex[t_idx][o_idx++].ob_flags & LASTOB) );
	}

	/*
	 * Install the menu bar
	 */
	menu_bar( rs_trindex[MENUE], 1 );
  
	/* 
	 * Now create a window of the appropriate size;
	 * see, if it fits onto screen. If not, make 
	 * it as big as possible and map it onto screen.
	 */
	wind_get( 0, WF_WORKXYWH, &wx, &wy, &ww, &wh );  
	wind_calc( WC_BORDER, WinElem, wx, wy, 
			   Frame.Screen_Width, Frame.Screen_Height, 
			   &wwx, &wwy, &www, &wwh );

	wwx = wx;
	wwy = wy; 
	www = min( www, ww-2 );
	wwh = min( wwh, wh-2 );
	           
	if( (theWin = wind_create( WinElem, wwx, wwy, www, wwh )) < 0 )
	{
		fprintf( stderr, "DISPLAY_INIT: No more Windows\n" );
		exit(0);
	}

	sprintf( theInfoStr, " %s -> %s[%dx%d]", Input_File_Name,
											 Output_File_Name,                                     
											 Frame.Screen_Width,
											 Frame.Screen_Height );
	wind_set( theWin, WF_INFO, theInfoStr );
	wind_set( theWin, WF_NAME, "POVRay 2.0" );
	wind_open( theWin, wwx, wwy, www, wwh );	
	slid_size();
}

void display_close PARAMS(( void ))
{
	(*thePlot.plot_exit)();
	menu_bar( rs_trindex[MENUE], 0 );
	wind_close( theWin );
	v_clsvwk( theVdi );
	appl_exit();
	free( thePixmap.fd_addr );
}

void display_plot (int x, int y, UBYTE Red, UBYTE Green, UBYTE Blue)
{
	static int fulled = TRUE;
	static clock_t lap = 0l;
	int    theMessage[8], col[2], du, wx, wy, ww, wh;

	/*
	 * Plot the pixel first.
	 */
	theMessage[0] = theMessage[1] = 0;
	theMessage[2] = theMessage[3] = 0;
	theMessage[4] = theMessage[6] = x;
	theMessage[5] = theMessage[7] = y;

	col[1] = (*thePlot.put_pixel)(x, y, Red, Green, Blue);

	vrt_cpyfm( theVdi, MD_REPLACE, theMessage, 
			   &thePixel.mfdb, 
			   &thePixmap,
			   col );

	/*
	 * Only redraw every second, so it doesn't
	 * take so much time.
	 */
	if( (clock()-lap) > CLK_TCK/4 )
	{ 
		wind_update( BEG_UPDATE );

		/*
		 * When sending ourselves a message to redraw the window
		 * the AES seem to overwrite the other kept messages.
		 * This causes the slow reaction on sliding, etc.
		 * So explicit redrawing is down.
		 */
		send_redraw();  
		redraw_window(); 
		lap = clock();

		/*
		 * Now lets look for envents, caused by someone playing
		 * around with the window elements.
		 */
		while( MU_MESAG == (MU_MESAG & evnt_multi( MU_MESAG|MU_TIMER, 
                                               0,0,0,0,0,0,0,
                                               0,0,0,0,0,0,
                                               theMessage, 
	                                           0, 0,
	                                           &du,&du,&du,
	                                           &du,&du,&du)))
		switch( theMessage[0] )
		{
			case WM_MOVED:
				fulled = FALSE;
				wind_set( theMessage[3], WF_CURRXYWH, 
						  theMessage[4], theMessage[5],
						  theMessage[6], theMessage[7] );
				break;
	
			case WM_TOPPED:
			case WM_NEWTOP:
				wind_set( theMessage[3], WF_TOP );
				break;
					
			case WM_CLOSED: 
				Stop_Flag = TRUE;
				break;

			case WM_REDRAW:
				redraw_window();
				break;

			case WM_VSLID:
				wind_set( theWin, WF_VSLIDE, theMessage[4] );
				slid_pos( SET_SLID );
				break;

			case WM_HSLID:
				wind_set( theWin, WF_HSLIDE, theMessage[4] );
				slid_pos( SET_SLID );
				break;

			case WM_ARROWED:
				slid_pos( theMessage[4] );
				break;

			case MN_SELECTED:
			{
				int fx, fy, fw, fh, t_idx, leaver;

				switch( theMessage[4] )
				{
					case 6:            /* Programminfo */
						t_idx = PINFO;
						break;
					default:
						continue;      /* the outer while loop */
				}

				form_center( rs_trindex[t_idx], &fx, &fy, &fw, &fh );
				fx -= 2; fy -= 2; fw += 4; fh += 4;
				form_dial( FMD_START,  fx, fy, fw, fh, fx, fy, fw, fh );
				objc_draw( rs_trindex[t_idx], 0, 5, fx, fy, fw, fh );
				leaver = form_do( rs_trindex[t_idx], 0 );
				rs_trindex[t_idx][leaver].ob_state ^= SELECTED;
				form_dial( FMD_FINISH, fx, fy, fw, fh, fx, fy, fw, fh );
				menu_tnormal( rs_trindex[MENUE], theMessage[3], 1 );   
				break;
			}

			case WM_FULLED:
				wind_get( theWin, fulled ? WF_PREVXYWH : WF_FULLXYWH, 
						  &wx, &wy, &ww, &wh );
				wind_set( theWin, WF_CURRXYWH, wx, wy, ww, wh );
				fulled = !fulled;
				slid_size();
				break;

			case WM_SIZED:
				fulled = FALSE;
				wind_get( theWin, WF_FULLXYWH, &wx, &wy, &ww, &wh );
				ww = min( ww, theMessage[6] );
				wh = min( wh, theMessage[7] );
				wind_set( theMessage[3], WF_CURRXYWH,
						  theMessage[4], theMessage[5], ww, wh );
				slid_size();
				break;
			case AP_DRAGDROP:
			/* we needn't DragDropping, so we return DD_NAK. The sending
			 * program needn't wait for a time out.
			 */
			{
				LONG fd;
				BYTE pipename[] = "U:\\PIPE\\DRAGDROP.AA",
					 c = DD_NAK;

				pipename[18] = theMessage[7] & 0x00ff;
				pipename[17] = (theMessage[7] & 0xff00) >> 8;

				fd = Fopen( pipename, 2 );
				if( fd >= 0 )
				{
					Fwrite( (WORD)fd, 1, &c );
					Fclose( (WORD)fd );
				}
				break;
			}
		}
		wind_update( END_UPDATE );
	}
}

/*
 * Redraw the window contents.
 */
static void redraw_window PARAMS(( void ))
{
	static MFDB theScreen = { NULL,0,0,0,0,0,0,0,0 };
	GRECT  box, work;
	int    clip[4], pxy[8], m_clip = FALSE;

	wind_get( theWin, WF_WORKXYWH, &work.g_x, &work.g_y,
								   &work.g_w, &work.g_h );
	wind_get( theWin, WF_FIRSTXYWH, &box.g_x, &box.g_y, 
									&box.g_w, &box.g_h );
	pxy[2] = (pxy[0] = theOfsX)  + work.g_w - 1;
	pxy[3] = (pxy[1] = theOfsY)  + work.g_h - 1;
	pxy[6] = (pxy[4] = work.g_x) + work.g_w - 1;
	pxy[7] = (pxy[5] = work.g_y) + work.g_h - 1;

	while ( box.g_w > 0 && box.g_h > 0 )
	{
		if( rc_intersect( &work, &box ) )
		{
			clip[0] = box.g_x;
			clip[1] = box.g_y;
			clip[2] = box.g_x + box.g_w - 1;
			clip[3] = box.g_y + box.g_h - 1;

			if( !m_clip && (m_clip = mouse_clip( clip )) == TRUE ) 
				graf_mouse( M_OFF, NULL );

			vs_clip( theVdi, 1, clip );
			vro_cpyfm( theVdi, S_ONLY, pxy, &thePixmap, &theScreen );
		}
		wind_get( theWin, WF_NEXTXYWH, &box.g_x, &box.g_y, 
									   &box.g_w, &box.g_h );
	}
	if( m_clip )
		graf_mouse( M_ON, NULL );
}

/* 
 * Someone is wanting to scroll the picture
 * up, down, left or right. Move the scroll
 * bars and the picture according to it.
 */
static void slid_pos( int wanted )
{
	int du, vsize, hsize,
		hspos = theOfsX,
		vspos = theOfsY;

	wind_get( theWin, WF_WORKXYWH, &du, &du, &hsize, &vsize );

	switch( wanted )
	{
		case WA_UPPAGE: vspos -= vsize;  break;
		case WA_DNPAGE: vspos += vsize;  break;
		case WA_UPLINE: vspos -= 20;     break;
		case WA_DNLINE: vspos += 20;     break;
		case WA_LFPAGE: hspos -= hsize;  break;
		case WA_RTPAGE: hspos += hsize;  break;
		case WA_LFLINE: hspos -= 20;     break;
		case WA_RTLINE: hspos += 20;     break;
		case SET_SLID:
#ifdef __GNUC__
		wind_get( theWin, WF_HSLIDE, &hspos, &du, &du, &du );
		wind_get( theWin, WF_VSLIDE, &vspos, &du, &du, &du );
#else
		wind_get( theWin, WF_HSLIDE, &hspos );
		wind_get( theWin, WF_VSLIDE, &vspos );
#endif
		theOfsX = (int)((long)hspos*(Frame.Screen_Width-hsize)/1000);
		theOfsY = (int)((long)vspos*(Frame.Screen_Height-vsize)/1000);
		send_redraw();
		return;
	}  

	hspos = min( hspos, Frame.Screen_Width-hsize );
	hspos = max( hspos, 0 );
	vspos = min( vspos, Frame.Screen_Height-vsize );
	vspos = max( vspos, 0 ); 

	if( theOfsX != hspos )
	{
		theOfsX = hspos;
		hspos = (int)((long)hspos*1000/(Frame.Screen_Width-hsize+1));
		wind_set( theWin, WF_HSLIDE, hspos );
		send_redraw();
	}
	if( theOfsY != vspos )
	{
		theOfsY = vspos;
		vspos = (int)((long)vspos*1000/(Frame.Screen_Height-vsize+1));
		wind_set( theWin, WF_VSLIDE, vspos );
		send_redraw();
	}
}

/*
 * Someone sized the window. May be the whole
 * picture doesn't fit into work area any
 * longer. Set the scroll bar size according
 * to the change in size.
 */
static void slid_size PARAMS(( void ))
{
	int vsize, hsize, du;

	wind_get( theWin, WF_WORKXYWH, &du, &du, &hsize, &vsize );
	hsize = (int)((long)hsize*1000/Frame.Screen_Width);
	vsize = (int)((long)vsize*1000/Frame.Screen_Height);

	wind_set( theWin, WF_HSLSIZE, max( 1, hsize ));
	wind_set( theWin, WF_VSLSIZE, max( 1, vsize ));

	slid_pos( SET_SLID );
}

/*
 * See if R1 and R2 intersect, return intersection in R2.
 */
static int rc_intersect( GRECT *r1, GRECT *r2 )
{
	int x, y, w, h;

	x = max( r2->g_x, r1->g_x );
	y = max( r2->g_y, r1->g_y );
	w = min( r2->g_x + r2->g_w, r1->g_x + r1->g_w );
	h = min( r2->g_y + r2->g_h, r1->g_y + r1->g_h );

	r2->g_x = x;
	r2->g_y = y;
	r2->g_w = w - x;
	r2->g_h = h - y;

	return ( ((w > x) && (h > y) ) );
}

/* 
 * Is the mouse arrow in the workarea of the window?
 * If so we have to switch it off before redrawing.
 */
static int mouse_clip ( int clip[4] )
{
	int mkx, mky, du;

	graf_mkstate( &mkx, &mky, &du, &du );
	return ( clip[0]-16 <= mkx && mkx <= clip[2]+16 &&
			 clip[1]-16 <= mky && mky <= clip[3]+16 );
}

/* 
 * Give the AES a hint to redraw the window.
 * Redrawing isn't actually done here.
 */
static void send_redraw PARAMS(( void ))
{
	int theMessage[8];

	theMessage[0] = WM_REDRAW;
	theMessage[1] = theApp;
	theMessage[2] = 0;
	theMessage[3] = theWin;

	wind_get( theWin, WF_WORKXYWH, 
			  &theMessage[4], &theMessage[5], 
			  &theMessage[6], &theMessage[7] );

	appl_write( theApp, 16, theMessage );
}

/*
 * A nice Floyd Steinberg dithering routine.
 */
static char FSdither( int x, int y, int steps, UBYTE col, int (*buf)[][2] )
{
	int err, val, lum;

	y &= 1;
	steps = 255/(steps-1);
	lum = col + ((*buf)[x+1][y]>>4);
	val = lum < 0 ? 0 : (2*lum+steps)/(2*steps);
	err = lum - val*steps;

	(*buf)[x+1][y  ]  = 0;
	(*buf)[x+2][y  ] += 7*err;
	(*buf)[x+2][y^1] += 1*err;
	(*buf)[x+1][y^1] += 5*err;
	(*buf)[x  ][y^1] += 3*err;

	return( val );
}

/* 
 * Switch: desktop colors / POVRay colors
 */ 
static void color_switch PARAMS(( int mode ))
{
	static int aktual = OLD_COLORS;
	int i;

	switch( mode )
	{
		case NEW_COLORS:
			if( aktual == NEW_COLORS )
				break;
			aktual = NEW_COLORS;
			for( i=0; i < (1<<thePlanes); i++ )
			{
				vq_color( theVdi, i, 0, (*old_pal)[i] );
				vs_color( theVdi, i, (*new_pal)[i] );
			}
			break;
		case OLD_COLORS:
			if( aktual == OLD_COLORS )
				break;
			aktual = OLD_COLORS;
			for( i=0; i < (1<<thePlanes); i++ )
				vs_color( theVdi, i, (*old_pal)[i] );
			break;
	}
}

/* 
 * the black&white / color / truecolor
 * plot, init and exit methods
 */
static void BWinit PARAMS(( void ))
{
	/*
	 * Now allocate and prepare the error buffers for
	 * Floyd Steinberg Error Diffusion
	 */

	if( (errbufR=malloc(2*sizeof(int)*(Frame.Screen_Width+2)))==NULL )
	{
		fprintf( stderr, "DISPLAY_INIT: failed to alloc FS buffers\n" );
		exit(0);
	}
	memset( errbufR, 0x00, sizeof(int)*(Frame.Screen_Width+2)*2 );
}

static void COinit PARAMS(( void ))
{
#define PalIndex(re,gr,bl) (((bl)<<(col_bits.g+col_bits.r))|\
                            ((gr)<<col_bits.r)|(re))

	int    rcnt, gcnt, bcnt, index;
	size_t bufsize;
  
	/* 
	 * Get space for the palette entries and the
	 * error buffers for Floyd Steinberg error
	 * diffusion
	 */

	if( (old_pal = malloc( sizeof(**old_pal)*(1<<thePlanes) ))==NULL ||
		(new_pal = malloc( sizeof(**new_pal)*(1<<thePlanes) ))==NULL )
	{
		fprintf( stderr, "DISPLAY_INIT: cannot alloc palette buffers\n" );
		exit(0);
	}

	bufsize = 2*sizeof(int)*(Frame.Screen_Width+2);

	if( (errbufR=malloc( bufsize ))==NULL ||
		(errbufG=malloc( bufsize ))==NULL ||
		(errbufB=malloc( bufsize ))==NULL )
	{
		fprintf( stderr, "DISPLAY_INIT: cannot alloc FS buffers\n" );
		exit(0);
	}
	memset( errbufB, 0x00, bufsize );
	memset( errbufG, 0x00, bufsize );
	memset( errbufR, 0x00, bufsize );

	/* 
	 * split the availeable bits per pixel into 
	 * red/green/blue parts
	 */

	col_bits.r = 
	col_bits.g = (thePlanes+1)/3;
	col_bits.b = thePlanes - 2*col_bits.r;

	/*
	 * initialize the new palette, so that 
	 * the index into palette bbits|gbits|rbits 
	 * maps to the color red   = k*rbits
	 *                   green = k*gbits
	 *                   blue  = k*bbits
	 * with sertain k
	 */

	for( bcnt=0; bcnt < (1<<col_bits.b); bcnt++ )
		for( gcnt=0; gcnt < (1<<col_bits.g); gcnt++ )
			for( rcnt=0; rcnt < (1<<col_bits.r); rcnt++ )
			{
				index = PalIndex( rcnt, gcnt, bcnt );
				(*new_pal)[index][0] = (int)((long)rcnt*1000/((1<<col_bits.r)-1));
				(*new_pal)[index][1] = (int)((long)gcnt*1000/((1<<col_bits.g)-1));
				(*new_pal)[index][2] = (int)((long)bcnt*1000/((1<<col_bits.b)-1));
			}
}

static void HIinit PARAMS(( void ))
{}

static void BWexit PARAMS(( void ))
{
	free( errbufR );
}

static void COexit PARAMS(( void ))
{
	color_switch( OLD_COLORS );
	free( new_pal );
	free( old_pal );
	free( errbufR );
	free( errbufG );
	free( errbufB );
}

static void HIexit PARAMS(( void ))
{}

static int BWPutPixel ( int x, int y, UBYTE red, UBYTE green, UBYTE blue )
{
	UBYTE col = (UBYTE)((307*(ULONG)red+
						599*(ULONG)green+
						118*(ULONG)blue)/1024);

	return ( FSdither( x, y, 2, col, errbufR ) ? WHITE : BLACK );
}

static int COPutPixel ( int x, int y, UBYTE red, UBYTE green, UBYTE blue )
{
	int    theTopWin;
	static clock_t lap = 0;

#ifdef __GNUC__
	int du;
#endif

	if( clock()-lap > CLK_TCK/2 )
	{
		lap = clock();
#ifdef __GNUC__
		wind_get( theWin, WF_TOP, &theTopWin, &du, &du, &du );
#else
		wind_get( theWin, WF_TOP, &theTopWin );
#endif
		color_switch( theTopWin == theWin ? NEW_COLORS : OLD_COLORS );
	}

	return( PalIndex( FSdither( x, y, 1<<col_bits.r, red, errbufR ), 
					  FSdither( x, y, 1<<col_bits.g, green, errbufG ), 
					  FSdither( x, y, 1<<col_bits.b, blue, errbufB ))); 
}

static int HIPutPixel ( int x, int y, UBYTE red, UBYTE green, UBYTE blue )
{
	int rgb[3];

	rgb[0] = (int)((ULONG)red*1000/255);
	rgb[1] = (int)((ULONG)green*1000/255);
	rgb[2] = (int)((ULONG)blue*1000/255);
	vs_color( theVdi, C_INDEX, rgb );
	return( C_INDEX );
}

/*
 * select the appropriate init/exit/plot methods
 * for the color resolution
 */
static void PlotMethodSelect PARAMS(( void ))
{
	if( thePlanes > 8 )
	{
		/* 'true' color */
		thePlot.plot_init = HIinit;
		thePlot.plot_exit = HIexit;
		thePlot.put_pixel = HIPutPixel;
	}
	else if( thePlanes < 3 )
	{
		/* black & white */
		thePlot.plot_init = BWinit;
		thePlot.plot_exit = BWexit;
		thePlot.put_pixel = BWPutPixel;
	}
	else
	{
		/* color */
		thePlot.plot_init = COinit;
		thePlot.plot_exit = COexit;
		thePlot.put_pixel = COPutPixel;
	}
}