/* 
 * Memory allocation and property list functions for
 * the Windows property list demo
 *
 * Written by Bill Hall
 * 3665 Benton Street, #66
 * Santa Clara, CA 95051
 *
 */

#define NOCOMM
#define NOKANJI
#define NOSOUND
#include <windows.h>
#include "walloc.h"

/* local function declaration */
static LPSTR NEAR WirePointer(DWORD size);

/* property list strings */
static char *szCurBlockNum = "CURBLOCKNUM";
static char *szCurSlotNum = "CURSLOTNUM";
static char *szCurSelector = "CURSELECTOR";

/* locally shared variables */
static int slotsize;

/* definitions */
#define SLOTS_PER_BLOCK	5

/* Allocate a memory chunk and store the information
 * on the window property list.  Memory is allocated in
 * a block whose size depends on the number of slots
 * desired and the size of an individual chunk.  It
 * is assumed that a chunk of memory always has the
 * same fixed size.
 */
LPSTR FAR AllocMem(HWND hWnd, int size)
{

    WORD CurBlockNum;	/* current block and slot numbers */
    WORD CurSlotNum;
    LPSTR memptr;	/* memory pointer */
    WORD sel;		/* selector (or segment) value */
    WORD offset;	/* pointer offset */
    char buf[30];	/* space for property strings */
    WORD handle;

    /* get the current block and slot numbers */
    CurBlockNum = GetProp(hWnd, szCurBlockNum);
    CurSlotNum = GetProp(hWnd, szCurSlotNum);

    /* we are either starting off the allocation or we have to
       create a new block
     */
    if ((CurBlockNum == 0) ||
        ((CurBlockNum > 0) && (++CurSlotNum == SLOTS_PER_BLOCK))) {
	slotsize = size;
	/* keep the pointer for the life of the program */
	memptr = WirePointer(size * SLOTS_PER_BLOCK);
	if (memptr) {
	    sel = HIWORD(memptr);	/* break the pointer into components */
	    offset = LOWORD(memptr);
	    CurBlockNum += 1;		/* increment block number */
            CurSlotNum = 0;		/* new slot number is 0 */
	/* remove old props before changing values
	 * this step is not necessary but improves
	 * the display during EnumProps
	 */
	    RemoveProp(hWnd, szCurBlockNum);
	    RemoveProp(hWnd, szCurSlotNum);
            SetProp(hWnd, szCurBlockNum, CurBlockNum); /* set the new ones */
	    SetProp(hWnd, szCurSlotNum, CurSlotNum);
	    /* make a unique string identifying the block */
	    wsprintf(buf, "%s%x", (LPSTR)szCurSelector, CurBlockNum);
	    if (!SetProp(hWnd, buf, sel)) {	/* save to property list */
		handle = LOWORD(GlobalHandle(sel));/* failed, free allocation */
		GlobalUnWire(handle);
		GlobalFree(handle);
		memptr = NULL;
	    }
	}
    }
    /* we are simply adding a new slot to the current block */
    else {
	/* build string identifying the current block */
        wsprintf(buf, "%s%x", (LPSTR)szCurSelector, CurBlockNum);
	sel = GetProp(hWnd, buf);	/* get the selector */
	offset = CurSlotNum * size;	/* create the offset */
	memptr = (LPSTR)MAKELONG(offset, sel);	/* make the pointer */

    /* this next step is not really necessary but
     * it improves the display during EnumProps
     */
	RemoveProp(hWnd, szCurSlotNum);	/* update the current information */
	SetProp(hWnd, szCurSlotNum, CurSlotNum);
    }
    return memptr;
}

/* free all allocated memory and remove the property list items */
void FAR FreeMem(HWND hWnd)
{
    register WORD i;
    WORD CurBlockNum = GetProp(hWnd, szCurBlockNum);
    WORD cursel;
    char buf[30];
    WORD handle;
    BOOL result;

    for (i = 1; i <= CurBlockNum; i++) {
	/* create the property list identifier */
	wsprintf(buf, "%s%x", (LPSTR)szCurSelector, i);
	cursel = GetProp(hWnd, buf);	/* get the selector */
	handle = LOWORD(GlobalHandle(cursel));	/* convert to handle */
	result = GlobalUnWire(handle);	/* release the allocation */
	result = GlobalFree(handle);
	RemoveProp(hWnd, buf);		/* remove the property */
    }
    RemoveProp(hWnd, szCurBlockNum);	/* remove the remaining properties */
    RemoveProp(hWnd, szCurSlotNum);
}

/* add the most recent allocation in the allocation list box */
void FAR UpdateList(HWND hWnd)
{
    WORD CurBlockNum, CurSlotNum;
    char buf[80];
    int pos;
    WORD sel, offset;
    MYDATA FAR *ptr;

    /* get the current block and slot numbers */
    CurBlockNum = GetProp(hWnd, szCurBlockNum);
    CurSlotNum = GetProp(hWnd, szCurSlotNum);

    /* build a pointer to the data in a slot */
    wsprintf(buf, "%s%x", (LPSTR)szCurSelector, CurBlockNum);
    sel = GetProp(hWnd, buf);
    offset = CurSlotNum * slotsize;
    ptr = (MYDATA FAR *)MAKELONG(offset, sel);

    /* build a string to be inserted into the list box */
    if (CurSlotNum == 0)
        wsprintf(buf, "Block: %2u\tSlot: %2u\tPointer: %x:%x\tData: %s", 
		 CurBlockNum, CurSlotNum, sel, offset, (LPSTR)ptr->data);
    else
	wsprintf(buf, "\tSlot: %2u\tPointer: %x:%x\tData: %s", 
		 CurSlotNum, sel, offset, (LPSTR)ptr->data);

    /* insert into list box */
    pos = (int)SendDlgItemMessage(hWnd, IDD_ALLOC,
				  LB_ADDSTRING, 0, (LONG)(LPSTR)buf);

    /* highlight the new entry */
    if (pos != LB_ERR)
        SendDlgItemMessage(hWnd, IDD_ALLOC, LB_SETCURSEL, pos , 0); 
    else
	MessageBeep(0);
}

/* allocate global data and wire the pointer */
static LPSTR NEAR WirePointer(DWORD size)
{

    HANDLE hmem = GlobalAlloc(GMEM_FIXED | GMEM_ZEROINIT, size);
    if (hmem)
	return GlobalWire(hmem);
    else
	return (LPSTR)NULL;

}

/* enumerate the property list and display in list box */
int FAR PASCAL EnumPropCallBack(HWND hWnd, WORD nDummy, PSTR str, HANDLE hData)
{
    char buf[80];

    if (nDummy != 0)
	wsprintf(buf, "%x\t%s\t%x", hWnd, (LPSTR)str, hData);
    else
	wsprintf(buf, "%x\t%x\t%x", hWnd, str, hData);

    SendDlgItemMessage(hWnd,IDD_ENUM,LB_ADDSTRING,0,(LONG)(LPSTR)buf);

    return TRUE;
}
