/*  viewport.h- viewport Definition and Macros */
#ifndef __VIEWPORT_H
#define __VIEWPORT_H

#include <alloc.h>              /* needed for farmalloc prototype */

#ifdef NEAR_HEAP
#       define VMALLOC   malloc /* Allocate memory for buffer.          */
#       define VFREE     free   /* Free memory used for buffer.         */
        typedef char     *cptr; /* pointer to a character buffer        */
        typedef int      *iptr; /* pointer to an int buffer             */
#else
#       define VMALLOC    farmalloc
#       define VFREE      farfree
        typedef char far* cptr;
        typedef int  far* iptr;
#endif

#define VMAXCOLS        80      /* Max. columns in a viewport           */
#define VMAXLINES       50      /* Max. lines (rows) in a viewport      */
#define VMAGIC          0xcdef  /* Magic number used to recognize a     */
                                /* valid viewport.                      */
#define VINIT_CHAR      ' '     /* clearok and v_scroll_region init.    */
                                /* viewport w/ this character           */
typedef struct viewport
{
    unsigned magic;        /* Magic number---identifies a valid viewport */

    unsigned inactive: 1 ; /* viewport is deactivated but still open.        */
    unsigned closed  : 1 ; /* viewport is closed and deactivated.            */
    unsigned clearok : 1 ; /* Clear (fill with blanks) viewport on 1st open. */
    unsigned saveok  : 1 ; /* Save area under viewport before displaying it  */

    unsigned char nrows  ; /* number of rows in viewport                     */
    unsigned char ncols  ; /* number of columns in viewport                  */

    unsigned char row;  /* absolute coordinate of upper-left corner */
    unsigned char col;

    unsigned char cur_row;       /* cursor position     */
    unsigned char cur_col;

    unsigned char fcolor;        /* foreground color    */
    unsigned char bcolor;        /* background color    */

    iptr savebuf;     /* underlying text when window was first opened */
    iptr old_image;   /* old image when window closed or deactivated  */
}
viewport;
 /*----------------------------------------------------------------------*/
#define v_valid(      v       ) ((v)->magic == VMAGIC)
#define v_foreground( v, color) ((v)->fcolor =  (color) )
#define v_background( v, color) ((v)->bcolor =  (color) )
#define v_saveok(     v, on   ) ((v)->saveok =  (on)!=0 )
#define v_clearok(    v, on   ) ((v)->clearok=  (on)!=0 )
#define v_getposn( rp, cp     ) ((*(rp)=(v)->row),   (*(cp)=(v)->col) )
#define v_row(     v ) (v_valid(v)? ( (v)->cur_row                ):-1)
#define v_col(     v ) (v_valid(v)? ( (v)->cur_col                ):-1)
#define v_nrows(   v ) (v_valid(v)? ( (v)->nrows                  ): 0)
#define v_ncols(   v ) (v_valid(v)? ( (v)->ncols                  ): 0)
#define v_ateol(   v ) (v_valid(v)? ( (v)->cur_col== (v)->ncols-1 ): 0)
#define v_atbol(   v ) (v_valid(v)? ( (v)->cur_col== 0            ): 0)
#define v_ateob(   v ) (v_valid(v)? ( (v)->cur_row== (v)->nrows-1 ): 0)
#define v_advance( v ) v_gotorc( (v), (v)->cur_row, (v)->cur_col + 1  )
#define v_backup(  v ) v_gotorc( (v), (v)->cur_row, (v)->cur_col - 1  )
#define v_cr(      v ) v_gotorc( (v), (v)->cur_row    , 0             )
#define v_up(      v ) v_gotorc( (v), (v)->cur_row - 1, (v)->cur_col  )
#define v_down(    v ) v_gotorc( (v), (v)->cur_row + 1, (v)->cur_col  )

#define v_scroll(v,u,l) v_scroll_region(v,u,l,0,0,VMAXCOLS,VMAXLINES)
#define v_move_rel(v, dr, dc) v_move((v), (v)->row-(dr), (v)->col-(dc))

/*----------------------------------------------------------------------*/
int  v_startup    ( int mode    );
void v_shutdown   ( void        );
int  v_construct  ( viewport *v );
int  v_destroy    ( viewport *v );
int  v_open       ( viewport *v );
int  v_close      ( viewport *v );
int  v_deactivate ( viewport *v, int hide );
int  v_size       ( viewport *v, unsigned nrows, unsigned ncols );
int  v_move       ( viewport *v, unsigned row,   unsigned col   );
int  v_gotorc     ( viewport *v, unsigned row,   unsigned col   );
int  v_putc       ( viewport *v, int c, int move_cur );
int  v_puts       ( viewport *v, char *s );
int  v_getc       ( viewport *v );
int  v_gets       ( viewport *v, int n,  char *s  );
int  v_scroll_region( viewport *v, int up, int to_left, int left, int top,
                                                        int right, int bottom );
int  v_clear        ( viewport *v );
int  v_cleartoeol   ( viewport *v );

/* Workhorse functions, not intended to be user callable
*/

void _v_save_scr    (iptr image, int row, int col, int nrows, int ncols);
void _v_restore_scr (iptr image, int row, int col, int nrows, int ncols);
void _v_fill_scr    (int c_att,  int row, int col, int nrows, int ncols);

#endif /* #ifdef __VIEWPORT_H */
