/* textbuf.h -- Listing 1. */

#ifndef __TEXTBUF_H     /* Make sure that textbuf.h isn't included */
#define __TEXTBUF_H     /* twice, the endif is on the last line.   */

#ifdef NEAR_HEAP
#   define BMALLOC   malloc     /* Allocate memory for buffer.  */
#   define BFREE     free       /* Free memory used for buffer. */
#   define MAXBSIZE  4096       /* Maximum row * col size.      */
    typedef char     *bufptr;   /* pointer to the buffer        */
    typedef int      bufsize;   /* holds max buffer size        */

#   define bufset( dst,c,  size) memset (dst,  c,  size)
#   define bufmove(dst,src,size) memmove(dst, src, size)
#else
#   include <alloc.h>           /* needed for farmalloc prototype */
#   define BMALLOC    farmalloc
#   define BFREE      farfree
#   define MAXBSIZE   (0xffffU - 1)  /* 64K - 1 */
    typedef char far* bufptr;
    typedef unsigned  bufsize;

    extern void _bufset  ( bufptr p, const int c, bufsize size  );
    extern void _bufmove ( bufptr dst, bufptr src, bufsize size );
#   define bufset(p,c,s)        _bufset(p,c,s)  /* in textbuf.c */
#   define bufmove(d,s,size)    _bufmove(d,s,size)
#endif

#define PUBLIC  /* for documentation: globally accessible object. */

typedef struct
{
    unsigned magic; /* Magic number for verification    */
    int nrows;      /* Number of rows                   */
    int ncols;      /* Number of columns                */
    int row;        /* Current cursor position, row     */
    int col;        /* Current cursor position, column  */
    bufptr p;       /* Character at that position       */
    bufptr buf;     /* Actual buffer.                   */
}
textbuf;

/* B_LAST_LINE is passed a textbuf pointer and returns a pointer to
 * the beginning of the last line of the buffer. The first line is
 * pointed to by "buf."
 *
 * textbuf.magic is set to B_MAGIC when a textbuf is allocated. Use
 * it to verify that a textbuf pointer is reasonable.
 */
#define B_LAST_LINE(b)  ((b)->buf + (((b)->nrows - 1) * (b)->ncols))
#define B_MAGIC 0xabcdU
/*----------------------------------------------------------------------*/
PUBLIC textbuf *b_new( int nrows, int ncols );
PUBLIC void     b_delete( textbuf *b );

PUBLIC textbuf *b_construct( textbuf *b, int nrows, int ncols );
PUBLIC int      b_destruct( textbuf *b );
/*----------------------------------------------------------------------*/
#define b_current( b )  (  (b)->p       )
#define b_advance( b )  (  (b)->col++,  \
                           (b)->p++     \
                        )
#define b_row(b)        (  (b)->row     )
#define b_col(b)        (  (b)->col     )
#define b_nrows(b)      (  (b)->nrows   )
#define b_ncols(b)      (  (b)->ncols   )
#define b_setrc(b,r,c)  (  ((b)->row = (r)),                            \
                           ((b)->col = (c)),                            \
                           ((b)->p   = (b)->buf + ((r)*(b)->ncols)+(c)) \
                        )
#define b_ateol(b)      ( (b)->col == ((b)->ncols - 1) )
#define b_ateob(b)      ( (b)->row == ((b)->nrows - 1) )
#define b_valid(b,r,c)  ( (b)                                   \
                          && ((b)->magic == B_MAGIC)            \
                          && (0 <= (r) && (r) < (b)->nrows)     \
                          && (0 <= (c) && (c) < (b)->ncols)     \
                        )
#define b_clearbuf(b,c) ( b_setrc(b,0,0),                               \
                          bufset((b)->buf, (c), (b)->nrows * (b)->ncols)\
                        )
#define b_cr(b)         ( ((b)->p -= (b)->col ),        \
                          ((b)->col = 0       )         \
                        )
#define b_lf(b)         (  ( ++(b)->row           ),    \
                           ( (b)->p += (b)->ncols )     \
                        )
#define b_clearline(b,c)  (  b_cr(b),                         \
                             bufset( (b)->p, (c), (b)->ncols )  \
                          )
#define b_available(b)    (  (b)->ncols - (b)->col )
#define b_insert_c(b,n,c) (  bufmove( (b)->p + n, (b)->p,               \
                                                b_available(b) - (n) ), \
                             bufset ( (b)->p    , (c)   , (n) )         \
                          )
#define b_delete_c(b,n,c) (  bufmove( (b)->p, (b)->p+(n), (n) ),        \
                             bufset ( ( (b)->p - (b)->col) +            \
                                      ( (b)->ncols - (n) ), (c), (n))   \
                          )
#define b_closeup(b,c)    ( b_cr(b),                                    \
                            bufmove( (b)->buf + (b)->ncols, (b)->buf,   \
                                                (b)->row * (b)->ncols ),\
                            bufset( (b)->buf, (c), (b)->ncols )         \
                          )
#define b_closedown(b,c)  ( b_cr(b),                                    \
                            bufmove( (b)->p, (b)->p + (b)->ncols,       \
                                  ((b)->nrows-(b)->row-1) * (b)->ncols),\
                            bufset( B_LAST_LINE(b), (c), (b)->ncols )   \
                          )
#define b_openup(b, c)    (  b_cr(b),                                   \
                             bufmove( (b)->buf, (b)->buf + (b)->ncols,  \
                                                (b)->row * (b)->ncols ),\
                             bufset( (b)->p, (c), (b)->ncols )          \
                          )
#define b_opendown(b,c)   ( b_cr(b),                                    \
                            bufmove( (b)->p + (b)->ncols, (b)->p,       \
                                    ((b)->nrows-(b)->row-1)*(b)->ncols),\
                            bufset( (b)->p, (c), (b)->ncols )           \
                          )
#endif /* #ifndef __TEXTBUF_H */

