#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<stdarg.h>
#include	<sys/time.h>

#include	"smbhdr.h"
#include	"smbpd.h"
#include	"netio.h"
#include	"lpt.h"
#include	"cmd.h"
#include	"trans.h"
#include	"util.h"

/*
 * Codes for SMB commands (subset)
 */

#define	SMB_COM_CLOSE			0x04
#define	SMB_COM_WRITE			0x0B
#define	SMB_COM_CHECK_DIRECTORY		0x10
#define	SMB_COM_PROCESS_EXIT		0x11
#define	SMB_COM_TRANSACTION		0x25
#define	SMB_COM_ECHO			0x2B
#define	SMB_COM_OPEN_ANDX		0x2D
#define	SMB_COM_TREE_CONNECT		0x70
#define	SMB_COM_TREE_DISCONNECT		0x71
#define	SMB_COM_NEGOTIATE		0x72
#define	SMB_COM_SESSION_SETUP_ANDX	0x73
#define	SMB_COM_TREE_CONNECT_ANDX	0x75
#define	SMB_COM_OPEN_PRINT_FILE		0xC0
#define	SMB_COM_WRITE_PRINT_FILE	0xC1
#define	SMB_COM_CLOSE_PRINT_FILE	0xC2
#define	SMB_COM_GET_PRINT_QUEUE		0xC3

int do_command(struct smbconn *client, int command, struct smbhdr *sh,
	unsigned char *op, long *ret_smblen);

int		ndialect = 2;
unsigned short	currtid = 0;

/*
 * Utility routine for setting returned word params (varargs)
 * Returns number of bytes for smblen
 * (Later use a format descriptor for non-word params?)
 */

static int set_param_words(unsigned char *p, int nwords, int ndata, ...)
{
	va_list         args;
	int		i;
	short		s;

	va_start(args, ndata);
	*p++ = nwords;
	for (i = 0; i < nwords; ++i)
	{
		s = va_arg(args, int);
		hstouc2(s, p);
		p += 2;
	}
	va_end(args);
	hstouc2(ndata, p);		/* set the number of data bytes too */
	return (nwords * 2 + 3 + ndata);/* +3 for word count and byte count */
}

/*
 * Code for session commands
 */

static void session_cmd(struct smbconn *client, unsigned char *buffer)
{
	int		len;
	char		name[51];

	len = nb_name_len(buffer) & 0xFFFF;
	if (len > sizeof(name)-1)
	{
		printf("%s Invalid name_len (%d) in session request\n", ptime(), len);
		return;
	}
	nbtoasc(buffer, name);
	trim_space(name);
	printf("%s Session start, to:%s from:", ptime(), name);
	nbtoasc(buffer+len, name);
	trim_space(name);
	printf("%s\n", name);
	/* and save a copy of client identity */
	Strncpy(client->cname, name, sizeof(client->cname));
}

static void session_reply(struct smbconn *client)
{
	unsigned char		*p;

	p = (unsigned char *)client->sh;
	/* postive response, length = 0, only set type field */
	memset(p, 0, 4);
	p[0] = 0x82;		/* positive reply */
	(void)write_smb(client, p, 4);
}

static int special_cmd(struct smbconn *client)
{
	unsigned char		*p;

	p = (unsigned char *)client->sh;
#if	0
	fprintf(stderr, "len:%ld type:%x\n", client->smblen, p[0]);
#endif
	switch (p[0])
	{
	case 0x81:		/* start session */
		session_cmd(client, p + 4);
		session_reply(client);
		break;
	case 0x85:		/* keepalive */
		break;
	}
	return (0);
}

/*
 * Code for normal commands
 */

static int do_negotiate(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	short			nbytes, tzmin;
	int			index, dindex;
	char			*p, *dname;
	long			hitime;
	/* should use a struct later on */
	static char		*dialect_name[] =
	{
		"PC NETWORK PROGRAM 1.0",
		"MICROSOFT NETWORKS 3.0",
		"LM1.2X002",
		"LM 0.12",
		0
	};
	extern long		timezone;

	/* find dialect index */
	nbytes = uc2tohs(sh->param1);
	for (dindex = index = 0, dname = p = sh->param2 + 1;
		p < (char *)(sh->param1 + nbytes); ++index)
	{
		if (strstr(p, dialect_name[ndialect]) != 0)
		{
			dindex = index;
			dname = p;
		}
		p += strlen(p) + 2;
	}
	if (verbose >= d->cmddebuglevel)
		printf("dialect:%s", dname);
	switch (ndialect)
	{
	case 0:
		/* basic header + dialectindex: short, bytecount: short */
		/* data bytes = 0 */
		*smblen = set_param_words(op, 1, 0, dindex);
		break;
	case 1:
	case 2:
		tzmin = my_tzsec / 60L;
		*smblen = set_param_words(op, 13, 0,
			dindex, 0 /*share*/, maxbuffer, 1 /*maxmpx*/,
			1 /*maxvcs*/, 0 /*noraw*/, 42, 42 /*sesskey*/,
			make_dos_time(my_ts), make_dos_date(my_ts), tzmin,
			0 /*Ekeylen*/, 0);
		break;
	case 3:
		tzmin = my_tzsec / 60L;
/*
 *	Approximation as I don't want to drag in 64 bit arithmetic or
 *	FP arithmetic. 27111902 is the adjustment to the high time for
 *	a starting year of 1601 (what a stupid starting point).
 */
		hitime = my_currenttime / 430L +
				my_currenttime / 366967L + 27111902;
		*smblen = 37L;
		op[0] = 17;
		op[35] = op[36] = '\0';
/*
 *	We borrow pack_struct from trans.c for this
 *	The first and third ops are really don't cares since we don't
 *	have any descriptors that need a data area.
 */
		(void)pack_struct("WBWWDDDDDDWB", op, op + 1, op,
			dindex, "" /*share*/, 2 /*maxmpx*/, 1 /*maxvcs*/,
			(long)maxbuffer, (long)maxbuffer /*maxraw*/,
			0x4242L /*sesskey*/, 0x301L /* cap: supports RPC */,
			0L, hitime /* time */, tzmin,
			"" /*Ekeylen*/);
		break;
	}
	return (0);
}

static int do_session_setup_andx(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	int		totlen;
	unsigned char	*p;

	p = op + 9;
	strcpy(p, "DOS");
	p += sizeof("DOS");
	totlen = sizeof("DOS");
	strcpy(p, PROGRAM);
	p += sizeof(PROGRAM);
	totlen += sizeof(PROGRAM);
	strcpy(p, my_work_group);
	totlen += strlen(my_work_group) + 1;
	hstouc2(100, sh->uid);
	*smblen = set_param_words(op, 3, totlen, 0, 0, 1 /* guest*/);
	return (0);
}

static int do_tree_connect(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	unsigned char	*p;
	char		*printer;

	if (++currtid == 0)			/* wraparound */
		++currtid;			/* 0 is never valid */
	p = &sh->wordcount + 4;
	if (verbose >= d->cmddebuglevel)
		printf("path:%s ", p);
	printer = get_resource(p);	/* maybe */
	p += strlen(p) + 2;		/* point to password */
	p += strlen(p) + 2;		/* point to service name */
	if (verbose >= d->cmddebuglevel)
		printf("service:%s tid:%u", p, currtid);
	hstouc2(currtid, sh->tid);
	*smblen = set_param_words(op, 2, 0,
		maxbuffer, currtid);	/* TID */
	if (strcmp(p, "LPT1:") == 0)
	{
		if (find_printer(printer) < 0)
			return (ERRSRV | ERRinvnetname);
		/* save a copy */
		Strncpy(client->pname, printer, MAXLABEL);
		hstouc2(currtid, sh->tid);
		return (0);
	}
	if (strcmp(p, "IPC") == 0)
		return (0);
	if (strcmp(p, "A:") == 0)
		return (ERRSRV | ERRinvdevice);
	return (ERRSRV | ERRinvnetname);
}

static int do_tree_connect_andx(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	unsigned char	*p;
	char		*printer;

	if (++currtid == 0)			/* wraparound */
		++currtid;			/* 0 is never valid */
	p = &sh->wordcount + 11;
	p += strlen(p) + 1;		/* point to path */
	if (verbose >= d->cmddebuglevel)
		printf("path:%s ", p);
	printer = get_resource(p);	/* maybe */
	p += strlen(p) + 1;		/* point to service name */
	if (verbose >= d->cmddebuglevel)
		printf("service:%s tid:%u", p, currtid);
	hstouc2(currtid, sh->tid);
	*smblen = set_param_words(op, 2, strlen(p) + 1,
		maxbuffer, currtid);	/* TID */
	strcpy(op + 7, p);
	if (strcmp(p, "LPT1:") == 0)
	{
		if (find_printer(printer) < 0)
			return (ERRSRV | ERRinvnetname);
		/* save a copy */
		Strncpy(client->pname, printer, MAXLABEL);
		hstouc2(currtid, sh->tid);
		return (0);
	}
	if (strcmp(p, "IPC") == 0)
		return (0);
	if (strcmp(p, "A:") == 0)
		return (ERRSRV | ERRinvdevice);
	return (ERRSRV | ERRinvnetname);
}

static int open_printer(struct smbconn *client, unsigned char *jobname,
	unsigned short fid)
{
	int			printer;

	if ((printer = find_printer(client->pname)) < 0)
		return (ERRSRV | ERRinvnetname);
	if (lpt[printer].avail != FREE)
		return (ERRSRV | ERRqfull);
	check_printer_status(printer, &lpt[printer]);
#ifndef	DEBUG
	if (lpt[printer].status & P_NOPAPER)
		return (ERRHRD | ERRnopaper);
#endif
	lpt[printer].avail = BUSY;
	lpt[printer].client = client;
	Strncpy(lpt[printer].jobname, jobname, sizeof(lpt[0].jobname));
	client->fid = fid;			/* save this fid */
	client->printer = printer;
	client->largebuf = lpt[printer].largebuf;
	client->joblen = 0L;
	client->starttime = time(0);
	return (0);
}

static int do_open_andx(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	unsigned char		*p;
	unsigned short		fid, access, ctime, cdate;

	fid = uc2tohs(sh->tid);
	access = uc2tohs(sh->param4);
	ctime = uc2tohs(sh->param7);
	cdate = uc2tohs(sh->param8);
	p = &sh->wordcount + 32;
	if (verbose >= d->cmddebuglevel)
		printf("name:%s access:%04x fid:%u", p, access, fid);
	*smblen = set_param_words(op, 15, 0,
		0 /*andx*/, 0 /*offset*/, fid /*fid*/, 0x20 /*archive*/,
		ctime /*time*/, cdate /*date*/, 0, 0 /*size*/, 1 /*write ok*/,
		0 /*normal file*/, 0 /*state*/, 2 /*created*/, 0, fid /*fid*/,
		0 /*reserved*/);
	return (open_printer(client, "JOB", fid));
}

static int do_write(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	client->datalength = uc2tohs(&sh->wordcount + 14);
	client->dataoffset = 0;
	client->dataptr = &sh->wordcount + 16;
	client->to_read = 0;
	client->state = PRINTING;
	if (verbose >= d->cmddebuglevel)
		printf("datalength:%d", client->datalength);
	*smblen = set_param_words(op, 1, 0, client->datalength);
	return (0);
}

static int do_open_print_file(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	int		setuplength;
	unsigned short	fid;
	unsigned char	*p;

	fid = uc2tohs(sh->tid);
	setuplength = uc2tohs(sh->param1);
	p = &sh->wordcount + 8;
	if (verbose >= d->cmddebuglevel)
		printf("printer:%s name:%s setuplength:%d fid:%u",
			client->pname, p, setuplength, fid);
	*smblen = set_param_words(op, 1, 0, fid);
	return (open_printer(client, p, fid));
}

static int do_write_print_file(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	client->datalength = uc2tohs(&sh->wordcount + 6);
	client->dataoffset = 0;
	client->dataptr = &sh->wordcount + 8;
	client->to_read = 0;
	client->state = PRINTING;
	if (verbose >= d->cmddebuglevel)
		printf("datalength:%d", client->datalength);
	return (0);
}

static int do_close_print_file(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	unsigned short		fid;

	fid = uc2tohs(sh->param1);
	/* check if this tree connection opened the file */
	if (client->printer < 0 || client->fid != fid)
		return (0);
	show_stats(client);
	lpt[client->printer].avail = FREE;
	lpt[client->printer].client = 0;
	lpt[client->printer].jobname[0] = '\0';
	client->fid = 0;
	client->printer = -1;
	client->largebuf = 0;
	if (verbose >= d->cmddebuglevel)
		printf("fid:%u", fid);
	*smblen = set_param_words(op, 0, 0);
	return (0);
}

static int do_get_print_queue(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	static struct queue_info	q;
	int				i, maxcount, startindex, printer;
	struct tm			*ts;

	maxcount = uc2tohs(client->sh->param1);
	startindex = uc2tohs(client->sh->param2);
	if (verbose >= d->cmddebuglevel)
		printf("maxcount:%d startindex:%d", maxcount, startindex);
	if ((printer = find_printer(client->pname)) < 0
		|| lpt[printer].avail == FREE)
		client->queueinfo = 0;
	else
	{
		client->queueinfo = &q;
		ts = localtime(&client->starttime);
		q.jobdate = make_dos_date(ts);
		q.jobtime = make_dos_time(ts);
		q.state = J_PRINTING;
		hstouc2(1, q.spoolnumber);
		hltouc4(client->joblen, q.spoolsize);
		strcpy(q.clientname, client->cname);
	}
	i = (client->queueinfo != 0);
	*smblen = set_param_words(op, 2, 3, i, 1);
	if (i == 0)
	{
		hstouc2(3, op + 5);	/* param3 */
		*(op + 7) = 0x1;
		hstouc2(0, op + 8);
	}
	else
	{
		hstouc2(3 + QUEUE_INFO_SIZE, op + 5);	/* param3 */
		*(op + 7) = 0x1;
		hstouc2(QUEUE_INFO_SIZE, op + 8);
		memcpy(op + 10, client->queueinfo, QUEUE_INFO_SIZE);
		*smblen += QUEUE_INFO_SIZE;
	}
	return (0);
}

static int do_tree_disconnect(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	do_close_print_file(d, client, sh, op, smblen);
	*smblen = set_param_words(op, 0, 0);
	return (0);
}

static void do_reply(struct smbconn *client, struct smbhdr *sh, int status, long smblen)
{
	/* mark it as a reply */
	sh->flags = 0x80;
	sh->flags2[0] = 0x1;		/* support long filenames */
	sh->errorclass = (status & 0xFF00) >> 8;
	sh->error[0] = (status & 0xFF);
	sh->error[1] = 0;
	sh->length[0] = 0;
	smblen += SMB_HDR_SIZE;
	set_smb_length(smblen, (unsigned char *)sh);
	(void)write_smb(client, (unsigned char *)sh, smblen + 4);
}

static int do_echo(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	int			status, i, j, n;

	i = uc2tohs(sh->param1);		/* times */
	n = uc2tohs(sh->param2);		/* bytecount */
	/* can leave bytecount in place */
	if (verbose >= d->cmddebuglevel)
		printf("times:%d bytecount:%d", i, n);
	for (j = 1; j < i; ++j)
	{
		*smblen = set_param_words(op, 1, n, j);
		do_reply(client, sh, status, *smblen);
	}
	*smblen = set_param_words(op, 1, n, j);
	return (0);
}

static int do_nothing(struct cmdentry *d,
	struct smbconn *client, struct smbhdr *sh,
	unsigned char *op, long *smblen)
{
	*smblen = set_param_words(op, 0, 0);
	return (0);
}

struct cmdentry table[] =
{
	{ 0x04,	"Close print file",	1, do_close_print_file },
	{ 0x0B,	"Write", 		2, do_write },
	{ 0x10,	"Check directory", 	1, do_nothing },
	{ 0x11,	"Exit", 	 	1, do_nothing },
	{ 0x25,	"Transaction", 		999, do_transaction },
	{ 0x2B,	"Echo", 		2, do_echo },
	{ 0x2D,	"OpenX", 		1, do_open_andx },
	{ 0x70,	"Tree connect", 	1, do_tree_connect },
	{ 0x71,	"Tree disconnect", 	1, do_tree_disconnect },
	{ 0x72,	"Negotiate", 		1, do_negotiate },
	{ 0x73,	"Session setupX", 	1, do_session_setup_andx },
	{ 0x75,	"Tree connectX", 	1, do_tree_connect_andx },
	{ 0xC0,	"Open print file", 	1, do_open_print_file },
	{ 0xC1,	"Write print file", 	2, do_write_print_file },
	{ 0xC2,	"Close print file", 	1, do_close_print_file },
	{ 0xC3,	"Get print queue", 	1, do_get_print_queue },
};

static int compare(const void *a, const void *b)
{
	return (((struct cmdentry *)a)->cmdcode - ((struct cmdentry *)b)->cmdcode);
}

int do_command(struct smbconn *client, int command, struct smbhdr *sh,
	unsigned char *op, long *ret_smblen)
{
	int			status;
	long			smblen;
	struct cmdentry		entry, *d;

	status = 0;
	entry.cmdcode = command;
	d = bsearch(&entry, table, sizeof(table)/sizeof(table[0]),
		sizeof(table[0]), compare);
	if (d)
	{
		if (verbose >= d->cmddebuglevel)
			printf("%s %s: ", ptime(), d->cmdname);
		(*d->cmdproc)(d, client, sh, op, &smblen);
		if (verbose >= d->cmddebuglevel)
			printf("\n");
	}
	else
	{
		status = ERRSRV | ERRsmbcmd;
		smblen = set_param_words(op, 0, 0);
		printf("%s Unimplemented command 0x%02x\n", ptime(), command);
	}
	*ret_smblen = smblen;
	return (status);
}

static int normal_cmd(struct smbconn *client)
{
	struct smbhdr		*sh, *ish;
	int			status, command;
	long			smblen, thissmblen;
	unsigned char		*p, *ip, *chain;
	short			andxoffset, wordcount;

	sh = client->sh;
	if (strncmp(sh->protocol, "\377SMB", 4) != 0)
	{
		printf("%s Not a SMB command\n", ptime());
		return (-1);
	}
#if	0
	fprintf(stderr, "len:%ld cmd:0x%x wc:%d\n", client->smblen, sh->command,
		sh->wordcount);
#endif
	p = &sh->wordcount;
	command = sh->command;
	if (command == SMB_COM_SESSION_SETUP_ANDX ||
		command == SMB_COM_TREE_CONNECT_ANDX ||
		command == SMB_COM_OPEN_ANDX)
	{
		/* make a copy of incoming SMB so reply doesn't overwrite */
		ish = (struct smbhdr *)insmbreq;
		memcpy(ish->protocol, sh->protocol, (size_t)client->smblen);
		ip = &ish->wordcount;
		smblen = 0L;
		do
		{
			chain = p + 1;
			if ((status = do_command(client, command, ish, p, &thissmblen)) != 0)
			{
				hstouc2(0xFF, chain);	/* end chain */
				hstouc2(0, chain + 2);
				break;
			}
			smblen += thissmblen;
			p += thissmblen;	/* advance by bytes written */
/*
 *	The memcpy is because the command we just did may alter things in the
 *	SMB header and the new values are implicitly used by the following
 *	commands so we need to copy the header from the copy to the output area
 */
			memcpy(sh->protocol, ish->protocol, SMB_HDR_SIZE);
			/* work out offset of next andx reply */
			andxoffset = p - sh->protocol;
			/* and write into reply */
			hstouc2(andxoffset, chain+2);
			/* get next command from current request */
			wordcount = ip[0];
			command = ip[1];
			hstouc2(command, chain);
			andxoffset = uc2tohs(ip + 3);
			/* advance input pointer to next request */
			ip = ish->protocol + andxoffset;
/*
 *	We then copy the SMB header to just in front of the chained command
 *	so that the routines can work on the structure just as if it were a
 *	normal command.
 */
			ish = (struct smbhdr *)(ip - SMB_HDR_SIZE - 4);
			memcpy(ish->protocol, sh->protocol, SMB_HDR_SIZE);
		} while (command != 0xFF);
	}
	else
		status = do_command(client, command, sh, p, &smblen);
	do_reply(client, sh, status, smblen);
	return (command == SMB_COM_PROCESS_EXIT ? -1 : 0);
}

int dispatch(struct smbconn *client)
{
	if (client->sh->length[0] != 0)
		return (special_cmd(client));
	else
		return (normal_cmd(client));
}
