/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWGINIT.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       Init function for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include <stdarg.h>                         /* for (,...) */
#include <string.h>                         /* for str*() */
#include <stdlib.h>                         /* for exit() */
#include "pwinc.h"                          /* main include file */
#include <signal.h>

/* [DOC] Module *************************************************************
** Name:        POWER 3.20
** Descr:       Programmer Oriented Window Environment Routines
** FuncMask:    pw
** Version:     v0.10.
** Author(s):   Ling Thio.
** Date:        1990/06/03.
****************************************************************************/

/* [DOC] Function *********************************************
** Name:        zz_address                              [ADDRESS]
** Descr:       CSE Production information.
**              Write to:    CSE Computer Systems Expertise
**                           G. Gezellelaan 49
**                           2624 KX Delft, The Netherlands
**              Mail to:     LING@HDETUD11.tudelft.nl
**              Donate to:   AMRO: 46.55.47.079
**************************************************************/

static void pwg_openhelp(void);
static FILE *pwg_fopen(char *name, char *type, char *place);
static void CtrlBrkHandler(int signum);
static void pwg_MessageFormat(PWWIN *pWin, int *nrow, int *ncol, char *ptr);

#ifdef DEBUG
int _pw_deblev = 0;
#endif
unsigned int _pwg_status = 0;
FILE *_pw_helpfh;
int _pwcf_mode;                             /* screen access */
char _pwcf_env[ENVSIZE] = "INIT";
char _pw_ltsp[] = "CSE Computer Systems Expertise";
char _pw_date[] = "(C) 1989, 1990";
char _pw_version[] = "v3.20";
int (*_pw_ExitFunction)(void) = NULL;

static int _pwv_oldrow;
static int _pwv_oldcol;

#define STAT_INIT   0x01                    /* Bit 0 */

#define LEN_PATH    128
#define LEN_FNAME   8
#define LEN_EXT     3

static void pwg_read_config(void);

unsigned char _pwcf_backgr[BACKGRSIZE] = {
    '', ''
};
unsigned char _pwcf_border[BORDERSIZE] = {
    '', '', '', '', '', '', '', '', '', '', ''
};
unsigned char _pwcf_attrib[ATTRIBSIZE] = {
        /* Norm  High  Rev   Spec  ActB  Bord */
           0x1b, 0x1f, 0x2e, 0x4f, 0x1b, 0x10,    /* CGA+COL */
           0x1b, 0x1f, 0x4e, 0x4f, 0x1b, 0x10, /* NORMAL */
           0x2e, 0x2f, 0x0f, 0x20, 0x2e, 0x2e, /* HEADER */
           0x2e, 0x2f, 0x0f, 0xce, 0x2e, 0x2e, /* KEYINFO */
           0x4f, 0x4e, 0x2e, 0x40, 0x4e, 0x40, /* MENU */
           0x1b, 0x1f, 0x4e, 0x4f, 0x1b, 0x1b, /* DATA */
           0x4e, 0x4f, 0x2e, 0x2f, 0x4e, 0x40, /* ERROR */
           0x2e, 0x2f, 0x0f, 0x20, 0x2e, 0x20, /* HELP */
           0x1b, 0x1f, 0x4e, 0x4f, 0x1b, 0x10, /* SPECIAL */

        /* Norm  High  Rev   Spec  ActB  Bord */
           0x1b, 0x1f, 0x1f, 0x2f, 0x1b, 0x13,    /* CGA+MONO */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* NORMAL */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* HEADER */
           0x1b, 0x1f, 0x20, 0xcf, 0x1b, 0x13, /* KEYINFO */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* MENU */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x1b, /* DATA */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* ERROR */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* HELP */
           0x1b, 0x1f, 0x20, 0x2f, 0x1b, 0x13, /* SPECIAL */

        /* Norm  High  Rev   Spec  ActB  Bord */
           0x07, 0x0f, 0x70, 0x7f, 0x0f, 0x07,    /* MDA+MONO */
           0x07, 0x0f, 0x70, 0x7f, 0x0f, 0x07, /* NORMAL */
           0x70, 0x7f, 0x07, 0x0f, 0x70, 0x70, /* HEADER */
           0x70, 0x7f, 0x07, 0xff, 0x7f, 0x7f, /* KEYINFO */
           0x07, 0x0f, 0x70, 0x01, 0x0f, 0x07, /* MENU */
           0x07, 0x0f, 0x70, 0x7f, 0x0f, 0x0f, /* DATA */
           0x07, 0x0f, 0x70, 0x7f, 0x0f, 0x07, /* ERROR */
           0x70, 0x7f, 0x07, 0x0f, 0x70, 0x70, /* HELP */
           0x07, 0x0f, 0x70, 0x7f, 0x0f, 0x07  /* SPECIAL */
};

int _pwcf_mon = -1;                         /* assume automatic */

static PWCELL far *_pwv_oldbuf;
static char _pwg_progname[LEN_FNAME+1];
#ifdef EXITHEADER
const PWCELL _pwg_exitheader[] = {
    0x2e00 + ' ',
    0x2e00 + ' ',
    0x2e00 + 'P',
    0x2e00 + 'O',
    0x2e00 + 'W',
    0x2e00 + 'E',
    0x2e00 + 'R',
    0x2e00 + ' ',
    0x2e00 + ' ',
    0x2e00 + 'v',
    0x2e00 + '3',
    0x2e00 + '.',
    0x2e00 + '2',
    0x2e00 + '0',
    0x2e00 + ' ',
    0x2e00 + ' ',
    0x2e00 + ' ',
    0x2f00 + ' ',
    0x2f00 + 'T',
    0x2f00 + 'h',
    0x2f00 + 'e',
    0x2f00 + ' ',
    0x2f00 + 's',
    0x2f00 + 'c',
    0x2f00 + 'r',
    0x2f00 + 'e',
    0x2f00 + 'e',
    0x2f00 + 'n',
    0x2f00 + ' ',
    0x2f00 + 'e',
    0x2f00 + 'n',
    0x2f00 + 'v',
    0x2f00 + 'i',
    0x2f00 + 'r',
    0x2f00 + 'o',
    0x2f00 + 'n',
    0x2f00 + 'm',
    0x2f00 + 'e',
    0x2f00 + 'n',
    0x2f00 + 't',
    0x2f00 + ' ',
    0x2f00 + 'b',
    0x2f00 + 'y',
    0x2f00 + ' ',
    0x2f00 + ' ',
    0x2f00 + ' ',
    0x2f00 + ' ',
    0x2f00 + ' ',
    0x2f00 + 'C',
    0x2f00 + 'S',
    0x2f00 + 'E',
    0x2f00 + ' ',
    0x2f00 + 'C',
    0x2f00 + 'o',
    0x2f00 + 'm',
    0x2f00 + 'p',
    0x2f00 + 'u',
    0x2f00 + 't',
    0x2f00 + 'e',
    0x2f00 + 'r',
    0x2f00 + ' ',
    0x2f00 + 'S',
    0x2f00 + 'y',
    0x2f00 + 's',
    0x2f00 + 't',
    0x2f00 + 'e',
    0x2f00 + 'm',
    0x2f00 + 's',
    0x2f00 + ' ',
    0x2f00 + 'E',
    0x2f00 + 'x',
    0x2f00 + 'p',
    0x2f00 + 'e',
    0x2f00 + 'r',
    0x2f00 + 't',
    0x2f00 + 'i',
    0x2f00 + 's',
    0x2f00 + 'e',
    0x2f00 + ' ',
    0x2f00 + ' '
};
#endif

/* [DOC] Function *********************************************
** Name:        pw_warning                                [API]
** SYNOPSIS:    void pw_warning(format,...)
**              char *format        Format string of warning
** Descr:       Opens a centered warning window and waits
**              for a key.
**              The 'pw_warning' function opens a warning window
**                in the middle of the screen  and displays a
**                warning message in it.
**              It uses the format string 'format' and its
**                parameters, just like the printf() function.
**              This function then waits for the user to press
**                a key, and returns this keycode.
** RETURNS:     Key code of the key pressed.
** SEE ALSO:    pw_error, printf
**************************************************************/
PWKEY pw_warning(char *format, ...)
{
    va_list arg_ptr;
    char temp[SCRATCH_LEN];

    va_start(arg_ptr, format);
    /* arg_ptr now points to argument after format string */
    vsprintf(temp, format, arg_ptr);
    va_end(arg_ptr);
    return pwg_message("Warning", temp, NULL, NULL);
}

/* [DOC] Function *********************************************
** Name:        pw_error                                  [API]
** SYNOPSIS:    void pw_error(format,...)
**              char *format        Format string of error
** Descr:       Displays a centered error window and waits
**              for a key.
**              The 'pw_error' function opens an error window in
**                the middle of the screen and displays an error
**                message in it.
**              It uses the format string 'format' and its
**                parameters, just like the printf() function.
**              This function then waits for the user to press
**                a key.
** RETURNS:     void
** SEE ALSO:    pw_warning, printf
**************************************************************/
void pw_error(char *format, ...)
{
    va_list arg_ptr;
    char temp[SCRATCH_LEN];

    va_start(arg_ptr, format);
    /* arg_ptr now points to argument after format string */
    vsprintf(temp, format, arg_ptr);
    va_end(arg_ptr);
    pwg_message("Error", temp, NULL, NULL);
}

PWKEY pwg_message(char *header, char *message, char *keyleft, char *keyright)
{
    int ncol;
    int nrow;
    PWWIN *pWin;
    PWKEY ret;

    pwg_MessageFormat(NULL, &nrow, &ncol, message); /* calculate nrow,ncol */
    pWin = pw_open(PWC_CENTER, PWC_CENTER, nrow, ncol, header, PWM_DEFAULT, PWW_ERROR);
    pw_keyinfo(pWin, keyleft, keyright);
    pwg_MessageFormat(pWin, &nrow, &ncol, message); /* take care of wrap */
    ret = pw_getkey();
    pw_close(pWin);
    return ret;
}

static void pwg_MessageFormat(PWWIN *pWin, int *nrow, int *ncol, char *ptr)
{
    int col = 1;
    int c;

    *ncol = 1;
    *nrow = 0;
    if (pWin) pw_putc(pWin, ' ');
    while ((c = *(ptr++)) != 0)
    {
        switch (c)
        {
        case '\t':
            while ((++col) % 8);
            if (pWin) pw_putc(pWin, c);
            break;
        case '\n':
            if (*ncol < col)
                *ncol = col;
            (*nrow)++;
            col = 1;
            if (pWin) { pw_putc(pWin, c); pw_putc(pWin, ' '); }
            break;
        default:
            if (col == _pwv_scrncol - 3)
            {
                (*nrow)++;
                *ncol = _pwv_scrncol - 3;
                col = 1;
                if (pWin) { pw_putc(pWin, '\n'); pw_putc(pWin, ' '); }
            }
            else
                col++;
            if (pWin) pw_putc(pWin, c);
            break;
        }
    }
    if (*ncol < col)
        *ncol = col;
    (*nrow)++;
    (*ncol)++;
}

void far *pwg_malloc(unsigned size)
{
#ifdef VERSION311
    void far *dest;

    dest = _fmalloc(size);
    if (!dest)
        pw_error("Not enough memory available."); */
    return dest;
#endif
    return _fmalloc(size);
}

void pwg_free(void far *source)
{
    _ffree(source);
}

static void pwg_openhelp(void)
{
    char helpfile[LEN_FNAME+LEN_EXT+2];

    pwg_strncpy(helpfile, _pwg_progname, sizeof(helpfile)-4);
    strcat(helpfile, ".PWH");
    if (*helpfile)
        _pw_helpfh = pwg_fopen(helpfile, "r", "HELP");
}

/* ***************************************************************
** Name:        pwg_fopen(name,type,place)
** SYNOPSIS:    FILE *pwg_fopen(name,type,place);
**              char *name          name of file (no path)
**              char *type          type of access
**              char *place         place to search
** Descr:       The 'pwg_tfopen' function opens the file specified
**                by 'pathname', searching on three directories.
**              The character string 'type' specifies the type of 
**                access requested for the file (see fopen).
**              The character string 'place' is the name of the
**                environment variable in which a default path
**                can be placed (using the DOS 'set' function).
**              The file will be searched in the following
**              directories in this order:
**                 - The current directory
**                 - The directory in environment variable 'place'
** RETURNS:     File pointer to the file opened.
**              NULL: Error in opening file
** SEE ALSO:    fopen
** **************************************************************/
static FILE *pwg_fopen(char *name, char *type, char *place)
{
    char *buf;
    char fname[LEN_PATH];
    char *tmp;
    FILE *fh;
    fh = fopen(name, type);                 /* first search the current dir */
    if (!fh)                                /* not in current dir */
    {                                       /* search through environment */
        if ((buf = getenv(place)) != NULL)  /* check if env.var. exists */
        {
            tmp = strtok(buf, ";");         /* first path */
            while ((!fh) && (tmp))          /* for all paths in var */
            {
                strcpy(fname, tmp);         /* get path */
                if (fname[strlen(fname)-1] != '\\')
                    strcat(fname, "\\");
                strcat(fname, name);        /* add filename */
                fh = fopen(fname, type);    /* try to open */
                tmp = strtok(NULL, ";");    /* next path */
            }
        }
    }
    return fh;
}

static void pwg_read_config(void)
{
    FILE *fh;
    unsigned int tmp[ATTRIBSIZE];
    char temp[LEN_FNAME+LEN_EXT+2];
    char id[13];
    int c;

    pwg_strncpy(temp, _pwg_progname, sizeof(temp)-4);
    strcat(temp, ".PWC");
    fh = pwg_fopen(temp, "r", "INIT");
    
    if (fh)
    {
        if (!fgets(id, 13, fh) || strncmp(id, "POWER Config", 12))
            fprintf(stderr, "\nFile '%s.PWC' is not a POWER config file", _pwg_progname);
        else
        {
            while (((c = fgetc(fh)) != EOF) && (c != '$'))
                ;
            if (fread(tmp, 1, sizeof(_pwcf_backgr), fh) == sizeof(_pwcf_backgr))
                memcpy(_pwcf_backgr, tmp, sizeof(_pwcf_backgr));
            if (fread(tmp, 1, sizeof(_pwcf_border), fh) == sizeof(_pwcf_border))
                memcpy(_pwcf_border, tmp, sizeof(_pwcf_border));
            if (fread(tmp, 1, sizeof(_pwcf_attrib), fh) == sizeof(_pwcf_attrib))
                memcpy(_pwcf_attrib, tmp, sizeof(_pwcf_attrib));
        }
        fclose(fh);
    }
}

static char *KwOption[] = {
    "MDA",
    "MONO",
    "COLOR",
    "NOSNOW",
#ifdef DEBUG
    "DEBUG",
#endif
    NULL
};
/* [DOC] Function *********************************************
** Name:        pw_options                                [API]
** SYNOPSIS:    int pw_options(argc, argv)
**              int argc            # of arguments
**              char *argv[]        Array of arguments
** Descr:       Parses command line for POWER options.
**              The 'pw_options' function parses the command line
**                parameters, given by 'argc' and 'argv' for
**                POWER options.  These parameters ususally are
**                the same onea as in the c function main().
**              The function scans the command line parameters
**                for an option symbol ('-' or '/'), followed
**                by one of the following keywords:
**                  MDA         Use MDA black&white colortable
**                  MONO        Use CGA/EGA/VGA Monochrome
**                  COLOR       Use CGA/EGA/VGA Color
**                  NOSNOW      Avoid snow on screen
**              If found, the option is placed at the end
**                of the command line parameter list 'argv',
**                and 'argc' is decreased by 1.
**              The function returns the thus corrected 'argc'.
** RETURNS:     The number of non-POWER option parameters.
** EXAMPLE:     void main(int argc, char **argv)
**              {
**                  if (pw_options(argc, argv) != 2) {
**                      fprintf(stderr, "\nMissing filename");
**                      exit(-1);
**                  }
**                  if (pw_init(argv[0], NULL) == ) {
**                      ....
**                      pw_quit();
**                  } else
**                      fprintf(stderr, "\nInsufficient memory");
**              }
** SEE ALSO:    pw_init
**************************************************************/
int pw_options(int argc, char **argv)
{
    int i, o;
    int skip;
    char *ptr;

    /* parse command line options */
    for (i=1; i<argc; i++)
    {
        ptr = argv[i];
        if ((*ptr == '/') || (*ptr == '-'))
        {
            skip = 1;
            o = 0;
            while (KwOption[o] && stricmp(KwOption[o], ptr+1))
                o++;
            switch (o)
            {
            case 0:
                _pwcf_mon = PWCF_MDAMONO;
                break;
            case 1:
                _pwcf_mon = PWCF_CGAMONO;
                break;
            case 2:
                _pwcf_mon = PWCF_CGACOLOR;
                break;
            case 3:
                _pwv_scrsnow = 1;
                break;
#ifdef DEBUG
            case 4:
                _pw_deblev = ptr[6] - '0';
                break;
#endif
            default:
                skip = 0;
                break;
            }
            if (skip)
            {
                for (skip=i+1; skip<argc; skip++)
                    argv[skip-1] = argv[skip];
                argc--;
                argv[argc-1] = ptr;
            }
        }
    }
    return argc;
}

/* [DOC] Function *********************************************
** Name:        pw_init                                   [API]
** SYNOPSIS:    PWWIN *pw_init(argv0, ExitFunction)
**              char *argv0                 Program name
**              int (*ExitFunction)(void)   Exit function
** Descr:       Initializes POWER environment.
**              The 'pw_init' function initializes the POWER
**                window environment.
**              The parameter 'argv0' must contain a program
**                name. Path and extension are optional.
**                Usually one uses 'argv[0]'.
**              Note: MSDOS 1.xx: argv[0] = ""
**                    MSDOS 2.xx:           ""
**                    MSDOS 3.xx:           "d:path\fname.ext"
**                    OS/2 v1.1:            "fname.ext"
**              POWER uses the program name for:
**                The help file          'name.PWH'.
**                The color file         'name.PWC'.
**              The parameter 'ExitFunction' may contain a
**                pointer to an exit function or NULL.
**                This function will be called when
**                  the user presses Shift-F3 or Ctrl-C.
** RETURNS:     0:  Successful call
**              -1: No text mode
**              -2: Text mode with less than 80 columns
**              -9: Not enough memory to initialize POWER
** EXAMPLE:     void main(int argc, char **argv)
**              {
**                  if (pw_options(argc, argv) != 2) {
**                      fprintf(stderr, "\nMissing filename");
**                      exit(-1);
**                  }
**                  pw_init(argv[0], 0);
**                    ....
**                  pw_exit(0);
**              }
** SEE ALSO:    pw_options, pw_exit, pw_quit
**************************************************************/
int pw_init(char *argv0, int (*ExitFunction)(void))
{
    int tmp = _pwcf_mon;
    char *ptr;
    char temp[LEN_FNAME+LEN_EXT+1];
    int ret = 0;
 
    pws_GetCursor(&_pwv_oldrow, &_pwv_oldcol);
    _pwv_oldbuf = NULL;
    _pwv_levbuf = NULL;
    if (!(ret = pwv_init()) &&              /* -1: notext -2:<80 cols */
        !(ret = pwg_keyinit()))             /* -9: insufficient mem for keys */
    {
        if ((_pwv_oldbuf = _fmalloc(_pwv_scrnrow * _pwv_scrncol * sizeof(PWCELL))) != NULL)
        {
            if ((_pwv_levbuf = _fmalloc(_pwv_scrnrow * _pwv_scrncol)) != NULL)
            {
                pws_mon();                  /* init pwcf_mon */
                if (tmp != -1)
                    _pwcf_mon = tmp;        /* option override */
                pwb_init1(_pwv_oldbuf);
                pwg_CursorOff();
                pwv_save(0, 0, _pwv_scrnrow, _pwv_scrncol, _pwv_oldbuf, _pwv_scrncol);
                /* fill _pwg_progname */
                if (!(ptr = strrchr(argv0, '\\')))
                    if (!(ptr = strrchr(argv0, ':')))
                        ptr = argv0-1;
                pwg_strncpy(temp, ptr+1, sizeof(temp));
                if ((ptr = strrchr(temp, '.')) != NULL)
                    *ptr = '\0';
                pwg_strncpy(_pwg_progname, temp, sizeof(_pwg_progname));

                pwg_read_config();
                pwg_openhelp();

                if ((ret = pwb_init2()) == 0)
                {
#ifdef MOUSE
                    pws_MouInit();
#endif
                    pw_ctrlc(ExitFunction); /*AK*/
#ifdef OLD
                    _pw_ExitFunction = ExitFunction;
                    signal(SIGINT, CtrlBrkHandler); /* Capture Ctrl-C */
#endif
                }
            } else
                ret = -9;                   /* insufficient memory */
        } else
            ret = -9;                       /* insufficient memory */
    }
    if (ret != 0)
    {
        if (_pwv_levbuf)
            _ffree(_pwv_levbuf);
        if (_pwv_oldbuf)
            _ffree(_pwv_oldbuf);
    }
    return ret;
}


/* *************************************************************
** RETURNS:     0:  Continue
**              !0: Quit with this return code
** ************************************************************/
int pwg_ExitFunc(void)
{

    if (_pw_ExitFunction != NULL)
        return _pw_ExitFunction();
    if (pw_question("Do you really want to quit? (N/y)","NY",PWW_ERROR)=='Y') {
        return PWR_QUIT;
    }
    return PWR_CONTINUE;
}


/* [DOC] Function *********************************************
** Name:        pw_ctrlc                                  [API]
** SYNOPSIS:    void pw_ctrlc(ExitFunc);
**              int (*ExitFunc)(void);
** Descr:       Installs a user defined ctrl-c handler.
**              if <ExitFunc> is NULL, a default ctrl-c
**                handler is used.
** RETURNS:     void
**************************************************************/
void pw_ctrlc(int (*ExitFunction)(void))  /*AK*/
{
    _pw_ExitFunction=ExitFunction;
    signal(SIGINT, CtrlBrkHandler);         /* Capture Ctrl-C */
}
/*AK*/


static void CtrlBrkHandler(int signum)
{
    int ret;

    signal(SIGINT, SIG_IGN);                /* disable Ctrl-C */
    pw_refresh();
    ret = pwg_ExitFunc();
    if (ret)
        pw_exit(ret);
    signal(SIGINT, CtrlBrkHandler);         /* Capture Ctrl-C */
}


/* [DOC] Function *********************************************
** Name:        pw_quit                                   [API]
** SYNOPSIS:    void pw_quit(void)
** Descr:       Terminates the POWER environment.
**              The 'pw_quit' function terminates the POWER
**                functions.  It frees all window memory and
**                restores the old screen environment.
**              Unlike 'pw_exit', this function does NOT
**                exit to DOS.
** RETURNS:     void
** EXAMPLE:       ....
**              if (fh = fopen(filename, "wb"))
**              {
**                ....
**              }
**              pw_quit();
** SEE ALSO:    pw_init, pw_exit
**************************************************************/
void pw_quit(void)
{
#ifdef EXITHEADER
    PWCELL far *fptr;
    int i;
#endif

    signal(SIGINT, SIG_DFL);                /* Restore Ctrl-C */
    if (_pwv_oldbuf)
    {
#ifdef EXITHEADER
        fptr = _pwv_oldbuf;
        if (_pwv_oldrow == _pwv_scrnrow-1)
            fptr += _pwv_scrncol;
        for (i=0; i<sizeof(_pwg_exitheader)/sizeof(PWCELL); i++)
            fptr[i] = _pwg_exitheader[i];
#endif
        pwv_display(0, 0, _pwv_scrnrow, _pwv_scrncol, _pwv_oldbuf, _pwv_scrncol);
        pws_cursor(_pwv_oldrow, _pwv_oldcol);
        _ffree(_pwv_oldbuf);
    }
    if (_pwv_levbuf)
        _ffree(_pwv_levbuf);
    if (_pw_helpfh)
        fclose(_pw_helpfh);
}

/* [DOC] Function *********************************************
** Name:        pw_exit                                   [API]
** SYNOPSIS:    void pw_exit(exitcode)
**              int  exitcode       Process return code
** Descr:       Terminates the POWER environment,
**              and returns to OS.
**              The 'pw_exit' function terminates the POWER
**                functions.  It frees all window memory,
**                restores the old screen environment,
**                and exits to DOS, returning the returncode
**                'exitcode'.
**              The function 'pw_quit' does the same, WITHOUT
**                exiting to DOS.
** RETURNS:     void
** EXAMPLE:       ....
**              if (fh = fopen(filename, "wb"))
**              {
**                ....
**              } else {
**                  pw_error("Can't write to '%s'", filename);
**                  pw_exit(-1);
**              }
** SEE ALSO:    pw_init, pw_quit
**************************************************************/
void pw_exit(int ExitCode)
{
    pw_quit();
    exit(ExitCode);
}

void pwg_strncpy(char *dest, char *src, int n)
{
    strncpy(dest, src, n);
    dest[n-1] = '\0';
}

void pwg_CursorOff(void)
{
    pws_cursor(_pwv_scrnrow, 0);
}

#ifdef HISTORY
void pwl_envsave(PWENV *pEnv)
{
    PWENV *pGlob = &_pw_glob.env;
    
/*    memcpy(pEnv, &_pw_glob.env, sizeof(PWENV)); */
    movedata(FP_SEG(pGlob), FP_OFF(pGlob),
             FP_SEG(pEnv),  FP_OFF(pEnv), sizeof(PWENV));
}

void pwl_envrestore(PWENV *pEnv)
{
    PWENV *pGlob = &_pw_glob.env;
    
/*     memcpy(&_pw_glob.env, pEnv, sizeof(PWENV)); */
    movedata(FP_SEG(pEnv),  FP_OFF(pEnv), 
             FP_SEG(pGlob), FP_OFF(pGlob), sizeof(PWENV));
}
#endif

/* todo:
    read_config: add _searchenv()
    openhelp: open file during F1 time 
*/


