/* [DOC] File ****************************************************
** MODULE INFORMATION*
**********************
**      FILE    NAME:       PWGGETK.C
**      SYSTEM  NAME:       POWER
**      VERSION NUMBER      3.00
**
** Descr:       Keyboard function for
**              Programmer Oriented Window Environment Routines.
**              (NoC) July 1989 - Ling Thio Software Productions.
******************************************************************
** ADMINISTRATIVE INFORMATION*
******************************
**      ORIGINAL AUTHOR:    Ling Thio
**      CREATION DATE:      89/07/10
**      CHANGES:            none
*****************************************************************/

#include <stdlib.h>                         /* for system() */
#include <string.h>                         /* for str*() */
#include <ctype.h>                          /* for toupper() */
#include <conio.h>                          /* for kbhit() */
#include "pwinc.h"                          /* main include file */

#define MAX_MACROS       8

struct _DEFKEY {
    PWKEY Key;
    PWKEY (*KeyFunction)(PWKEY key);
    PWKEY NewKey;
    struct _DEFKEY far *pNext;
};
#define DEFKEY struct _DEFKEY far

extern PWWIN *wintable[MAX_WINS];
extern int    _pwb_nlevels;

static PWKEY pwk_help(PWKEY key);
static PWKEY pwk_keyhelp(PWKEY key);
static PWKEY pwk_shell(PWKEY key);
static PWKEY pwk_exit(PWKEY key);
static PWKEY pwk_manipulate(PWKEY key);
static PWKEY pwk_macrodef(PWKEY key);
static void  pwg_macroopen(char *fname);
static PWKEY pwk_macro(PWKEY key);
static PWKEY pwk_alt(PWKEY key);
static PWKEY pwg_IsAltKey(PWKEY key);
static PWKEY pwg_IsCtrlKey(PWKEY key);

static DEFKEY *pFirstDefKey;
static char pw_AltTable[] = "QWERTYUIOP@@@@ASDFGHJKL@@@@@ZXCVBNM";
static char *WindowActionMenu[] = {
    "Move",
    "Pop",
    "Show",
    "Hide",
    NULL
};
PWKEY _pw_MouDefDefault[] = {
    PWK_LEFT,                               /* horizontal mouse movement*/
    PWK_RIGHT,
    PWK_UP,                                 /* vertical mouse movement  */
    PWK_DOWN,
    PWK_ENTER,                              /* left mouse button        */
    0,                                      /* middle button disabled   */
    PWK_ESC,                                /* right mouse button       */
};

static PWKEY pwg_info(void);
static PWKEY pwg_keyhelp(void);
static PWKEY pwg_help(char *key);
static PWKEY pwg_nohelp(char *key);
static char f1info[] = "F1=Info  F9=Keys";
static char pressany[] = "Press any key to continue";
static PWKEY *pw_mou_key = _pw_MouDefDefault;
static PWKEY (*pw_keyfunction)(void) = (PWKEY (*)())0;
static PWKEY pw_key = 0;
static FILE *pw_macrodef_fh;
static char pw_AltStr[] = "ALT@.PWM";

static int nmacros = 0;
static FILE *pw_macro_fh[MAX_MACROS] = {
NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL
};

static PWKEY pwk_help(PWKEY key)
{
    key = 0;
    if (!(_pwg_status & PW_IN_INFO))        /* don't help inside help */
    {
        if (_pwg_status & (PW_IN_HELP | PW_IN_KEYHELP))
            pwg_info();
        else
            pwg_help(((pwb_activewin()->header) ? pwb_activewin()->header : ""));
    }
    return key;
}

static PWKEY pwk_info(PWKEY key)
{
    key = 0;
    if (!(_pwg_status & PW_IN_INFO))        /* don't help inside help */
    {
        pwg_info();
    }
    return key;
}

static PWKEY pwk_macrodef(PWKEY key)
{
    PWWIN *pWin;
    char temp[17];
    int replace = -1;
    PWKEY k;

    key = 0;
    if (_pwg_status & PW_IN_MACRONAME)
        return key;
    if (!(_pwg_status & PW_IN_MACRODEF))
    {                                       /* begine of macro def */
        pWin = pw_open(_pwv_scrnrow-4, 3, 1, 14, "Macro Def", PWM_DEFAULT, PWW_ERROR);
        *temp = '\0';
        _pwg_status ^= PW_IN_MACRONAME;
        k = pw_input(pWin, 0, 0, "", 0, 1, 12, temp, 13, NULL, PWM_UPROMPT);
        _pwg_status ^= PW_IN_MACRONAME;
        if (k != PWK_ESC)
        {
            if (!strchr(temp, '.'))
                strcat(temp, pw_AltStr+4);
            if ((pw_macrodef_fh = fopen(temp, "rb")) != NULL)
            {
                fclose(pw_macrodef_fh);
                replace = (pw_question("Macro File exists, replace [Y/N]", "NY", PWW_ERROR) == 'Y');
            }
            if (((pw_macrodef_fh = fopen(temp, "wb")) != NULL) && replace)
            {
                fprintf(pw_macrodef_fh, "POWER Macro %s\n\26$", _pw_version);
                _pwg_status ^= PW_IN_MACRODEF;
                pw_putcellat(pw_win(PWID_BACKGROUND), _pwv_scrnrow-2, -1, (pwv_attrib(PWW_KEYINFO, PWT_SPECIAL) << 8)+'M');
            }
        }
        pw_close(pWin);
    }
    else
    {                                       /* end of macro def */
        fclose(pw_macrodef_fh);
        pw_putcellat(pw_win(PWID_BACKGROUND), _pwv_scrnrow-2, -1, (pwv_attrib(PWW_KEYINFO, PWT_NORMAL) << 8)+' ');
        _pwg_status ^= PW_IN_MACRODEF;
    }
    return key;
}

static void pwg_macroopen(char *fname)
{
    int c;

    if (!strchr(fname, '.'))
        strcat(fname, pw_AltStr+4);
    if ((nmacros < MAX_MACROS) && ((pw_macro_fh[nmacros] = fopen(fname, "rb")) != NULL))
    {
        while (((c = fgetc(pw_macro_fh[nmacros])) != EOF) && (c != '$'))
            ;
        nmacros++;
    }
}

static PWKEY pwk_macro(PWKEY key)
{
    PWWIN *pWin;
    char temp[17];

    if (key)
    {
        pWin = pw_open(_pwv_scrncol-4, 3, 1, 14, "Macro", PWM_DEFAULT, PWW_NORMAL);
        *temp = '\0';
        if (pw_input(pWin, 0, 0, "", 0, 1, 12, temp, 13, NULL, PWM_UPROMPT) != PWK_ESC)
            pwg_macroopen(temp);
        pw_close(pWin);
    }
    else
    {                                       /* end of macro  */
        fclose(pw_macro_fh[--nmacros]);
    }
    return 0;
}

static PWKEY pwk_alt(PWKEY altkey)
{
    if (!(_pwg_status & PW_IN_MACRONAME))
    {
        pw_AltStr[3] = (char)altkey;
        pwg_macroopen(pw_AltStr);
    }
    return 0;
}

static PWKEY pwk_shell(PWKEY key)
{
    key = 0;
    pw_system(NULL);
    return key;
}

static PWKEY pwk_exit(PWKEY key)
{
    int ret;

    ret = pwg_ExitFunc();
    if (ret)
        pw_exit(ret);
    return 0;
}

static int proc_move(PWWIN *win, int Action, int Element)
{
    int WindowAction;

    switch (Action)
    {
    case PWA_INIT:
        pw_keyinfo(win, NULL, NULL);
        WindowAction = 0;
        break;
    case PWA_ENTER:
        if (pw_choice(1, _pwv_scrncol-13, 0, 12, "Action", WindowActionMenu, &WindowAction) != PWK_ESC)
        {
            switch (WindowAction)
            {
            case 0:
                pwb_move(wintable[Element+1]);
                break;
            case 1:
                pwb_pop(wintable[Element+1]);
                Element = -3;               /* repaint entire menu window */
                break;
            case 2:
                pwb_show(wintable[Element+1]);
                break;
            case 3:
                pwb_hide(wintable[Element+1]);
                break;
            }
            pw_pop(win);
        }
        break;
    }
    return Element;
}

static PWKEY pwk_manipulate(PWKEY key)
{
    int element = 0;

    key = 0;
    if (!(_pwg_status & PW_IN_MOVE))        /* don't move inside move */
    {
        _pwg_status = _pwg_status ^ PW_IN_MOVE;
        pw_selectstruct(1, PWC_RIGHT, _pwb_nlevels, 12, "Windows", wintable+1, NULL, proc_move,
            sizeof(int)*7+sizeof(void far *)*4+sizeof(int)*1,
            &element);
        _pwg_status = _pwg_status ^ PW_IN_MOVE;
    }
#ifdef HISTORY
    key = 0;
    if (!(_pwg_status & PW_IN_MOVE))        /* don't move inside move */
    {
        _pwg_status = _pwg_status ^ PW_IN_MOVE;
        pwb_move();
        _pwg_status = _pwg_status ^ PW_IN_MOVE;
    }
#endif
    return key;
}

static PWKEY pwk_keyhelp(PWKEY key)
{
    key = 0;
    if (!(_pwg_status & PW_IN_KEYHELP))     /* don't keyhelp inside keyhelp */
        pwg_keyhelp();
    return key;
}

/* [DOC] Function *********************************************
** Name:        pw_defkey                                 [API]
** SYNOPSIS:    int pw_defkey(key, KeyFunction, NewKey)
** Descr:       Redefines a key to a C function or a new key.
**              The 'pw_defkey' function defines the key
**                'key' as the new scan code 'NewKey' if
**                'KeyFunction' is 0 or:
**              The 'pw_defkey' function calls the non-zero
**                function 'KeyFunction' each time the key
**                'key is pressed.
**              The 'pw_defkey' function restores key actions
**                of 'key' when both 'KeyFunction' and
**                'NewKey' are 0.
** RETURNS:     0:  Successful call.
**              -9: Insufficient memory.
**************************************************************/
int pw_defkey(PWKEY key, PWKEY (*KeyFunction)(PWKEY key), PWKEY NewKey)
{
    int ret = 0;
    DEFKEY *pDefKey;
    DEFKEY * far *ppNext;

    pDefKey = pFirstDefKey;
    ppNext = &pFirstDefKey;
    while ((pDefKey) && (pDefKey->Key != key))
    {
        ppNext = &(pDefKey->pNext);
        pDefKey = pDefKey->pNext;
    }
    if (pDefKey)
    {
        if ((KeyFunction) || (NewKey))
        {                                   /* change existing key def */
            pDefKey->KeyFunction = KeyFunction;
            pDefKey->NewKey = NewKey;
        }
        else
        {                                   /* remove from key def list */
            *ppNext = pDefKey->pNext;
            _ffree(pDefKey);
        }
    }
    else
    {
        if ((KeyFunction) || (NewKey))
        {                                   /* add new key def */
            if ((*ppNext = pDefKey = _fmalloc(sizeof(DEFKEY))) != NULL)
            {
                pDefKey->Key = key;
                pDefKey->KeyFunction = KeyFunction;
                pDefKey->NewKey = NewKey;
                pDefKey->pNext = NULL;
            }
            else
                ret = -9;                   /* insufficient memory */
        }
    }
    return ret;
}

int pwg_keyinit(void)
{
    int ret = 0;

    pFirstDefKey = NULL;
    ret |= pw_defkey(PWK_F8,      0,    PWK_PGDN);
    ret |= pw_defkey(PWK_F7,      0,    PWK_PGUP);
    ret |= pw_defkey(PWK_MOVE,     pwk_manipulate, 0);
    ret |= pw_defkey(PWK_EXIT,     pwk_exit,       0);
    ret |= pw_defkey(PWK_SHELL,    pwk_shell,      0);
    ret |= pw_defkey(PWK_F3,       0,        PWK_ESC);
    ret |= pw_defkey(PWK_MACRODEF, pwk_macrodef,   0);
    ret |= pw_defkey(PWK_MACRO,    pwk_macro,      0);
    ret |= pw_defkey(PWK_F9,       pwk_keyhelp,    0);
/*  ret |= pw_defkey(PWK_INFO,     pwk_info,       0); */
    ret |= pw_defkey(PWK_HELP,     pwk_help,       0);
    return ret;
}

/* [DOC] Function *********************************************
** Name:        pw_getkey                                 [API]
** SYNOPSIS:    PWKEY pw_getkey(void)
** Descr:       Gets a key from keyboard.
**              The 'pw_getkey' function waits for the user to
**                press a key and returns its key value.
** RETURNS:     The 'pw_getkey' function returns the key value
**                of the current key pressed.
**              PWKEY is of type unsigned int.
**              POWER keys are defined in POWER.H.
**                  PWK_... are normal keys,
**                  PWK_S... are <Shift> keys and
**                  PWK_C... are <Ctrl> keys.
** EXAMPLE:       ....
**              pw_putsat(win, 1, 1, "Press return to continue");
**              while (pw_getkey() != PWK_ENTER)
**                  ;
**                ....
**************************************************************/
PWKEY pw_getkey(void)
{
    PWKEY key = 0;
    PWKEY altkey;
    DEFKEY *pDefKey;

    do 
    {
        while (!key)
        {
            if (nmacros)
            {
                if (fread(&key, sizeof(PWKEY), 1, pw_macro_fh[nmacros-1]) != 1)
                {
                    pwk_macro(0);
                    key = pwl_getkey();
                }
            }
            else
                key = pwl_getkey();
            if ((altkey = pwg_IsAltKey(key)) != 0) /* PWK_ALT_ keys ? */
            {
                pwk_alt(altkey);
                key = 0;
            }
        }
        pDefKey = pFirstDefKey;
        if ((_pwg_status & PW_IN_MACRODEF) && (key != PWK_MACRODEF))
            fwrite(&key, sizeof(PWKEY), 1, pw_macrodef_fh);
        while ((pDefKey) && (pDefKey->Key != key))
            pDefKey = pDefKey->pNext;
        if (pDefKey)
        {
            if (pDefKey->KeyFunction)
                key = (*(pDefKey->KeyFunction))(key);
            else
                key = pDefKey->NewKey;
        }
    } while (key == 0);
    return key;
}

/* [DOC] Function *********************************************
** Name:        pw_question                               [API]
** SYNOPSIS:    PWKEY pw_question(question, answers)
**              char *question      Question text for user
**              char *answers       String of valid characters
**              int  wtype          Window type (PWW...)
** Descr:       Opens a centered question window
**              and waits for answer.
**              The 'pw_question' function opens a window of
**                type 'wtype' in the center of the screen and
**                displays the character string 'question'.
**              It then waits for the user to enter a ASCII key,
**                which must be either the escape key (PWK_ESC),
**                or any character present in the character
**                string 'answers'.
** RETURNS:     The 'pw_question' function returns PWK_ESC if
**                the escape key is pressed, or one of the
**                UPPER case characters in 'answers'.
**************************************************************/
PWKEY pw_question(char *question, char *answers, int wtype)
{
    int key;
    int l = strlen(question);
    PWWIN *tmp = pw_open(PWC_CENTER, PWC_CENTER, 1, l+2, "Question", PWM_DEFAULT, wtype);
    char _pwv_temp[SCRATCH_LEN];

    sprintf(_pwv_temp, "%s=Answer  Enter=%c", strupr(answers), *answers);
    pw_keyinfo(tmp, "F1=Help  F9=Keys  Esc=Cancel", _pwv_temp);
    pw_putsat(tmp, 0, 1, question);
    while ((key = pw_getkey()) > 255) 
        ;
    while ((key != PWK_ESC) && (key != PWK_ENTER) && 
           (!strchr(answers, toupper(key))))
        while ((key = pw_getkey()) > 255) 
            ;
    pw_close(tmp);
    if (key == PWK_ENTER)
        return (PWKEY)*answers;
    else
        return (PWKEY)toupper(key);
}

/* [DOC] Function *********************************************
** Name:        pw_refresh                              [API]
** SYNOPSIS:    void pw_refresh(void)
** Descr:       Refreshes the POWER screen.
**              Usually used after a pw_system() call, to
**                re-display the erased POWER screen.
** RETURNS:     0:   Successful call.
**              !=0: Error code.
** SEE ALSO:    pw_system.
**************************************************************/
void pw_refresh(void)
{
    pwb_refreshblock(127, 0, 0, _pwv_scrnrow, _pwv_scrncol);
}

/* [DOC] Function *********************************************
** Name:        pw_system                               [API]
** SYNOPSIS:    int pw_system(command)
**              char *command       Ptr to command or NULL
** Descr:       Executes an OS command or starts an OS shell.
**              The 'pw_system' function executes the command
**                'command' through an OS shell.
**              The statement pw_system(NULL) will
**                start a new DOS or OS2 shell. Typing 'exit'
**                at the DOS/OS2 prompt will cause
**                the OS to return to thhe power application.
** RETURNS:       0: successful call
**              !=0: Error code
**************************************************************/
int pw_system(char *command)
{
	int ret;

	pwb_shellinit();
    if (command)
    {
        if ((ret = system("cls")) == 0)
            ret = system(command);
    }
    else
        ret = pws_shell();
	pwb_shellend();
    pw_refresh();
	return ret;
}

static PWKEY pwg_info(void)
{
    unsigned int ret;
    PWWIN *temp = pw_open(PWC_CENTER, PWC_CENTER, 11, 55, "POWER Info", PWM_HIDE, PWW_HELP);

    _pwg_status |= PW_IN_INFO;
    pw_printfat(temp, 1, 21, "@hPOWER %s", _pw_version);
    pw_printfat(temp, 4, 4, "Programmer Oriented Window Environment Routines@p");
    pw_printfat(temp, 6,  4, "%s - %s", _pw_date, _pw_ltsp);
    pw_putsat(temp, 7,  4, "G. Gezelleln 49, 2624 KX Delft, The Netherlands");
    pw_putsat(temp,  3,  2, "The Screen Environment you are witnessing is using:");
    if (!(_pwg_status & PW_IN_KEYHELP))
        pw_putsat(temp,  9, 12, "Press F9 for help on POWER keys");
    pw_show(temp);
    pw_keyinfo(temp, (_pwg_status & PW_IN_KEYHELP) ? "" : f1info+9, (_pwg_status & PW_IN_KEYHELP) ? pressany : NULL);
    ret = pw_getkey();
    pw_close(temp);
    _pwg_status &= (~PW_IN_INFO);
    return ret;
}

static PWKEY pwg_keyhelp(void)
{
    unsigned int ret;
    PWWIN *temp = pw_open(PWC_CENTER, PWC_CENTER, 14, 68, "POWER keys help", PWM_HIDE, PWW_HELP);

    _pwg_status |= PW_IN_KEYHELP;
    pw_block(temp,  1,  2, 13, 10, PWT_HIGHLIGHT);
    pw_block(temp,  1, 35, 13, 12, PWT_HIGHLIGHT);
    pw_printfat(temp,  0,  4, "@sGlobal keys");
    pw_printfat(temp,  9,  4, "Menu keys");
    pw_printfat(temp,  0, 37, "Input keys");
    pw_printfat(temp, 10, 37, "Mouse events@n");
/*  pw_putsat(temp,  1,  2, "F1,F2      Window help,info"); */
    pw_putsat(temp,  1,  2, "F1         Window help");
    pw_putsat(temp,  2,  2, "F3,F7,F8   Esc, PgUp, PgDn");
    pw_putsat(temp,  3,  2, "Shift-F3   Exit program");
    pw_putsat(temp,  4,  2, "Ctrl -F3   DOS shell");
    pw_putsat(temp,  5,  2, "Shift-F4   Manipulate windows");
    pw_putsat(temp,  6,  2, "Ctrl -F10  Macro define");
    pw_putsat(temp,  7,  2, "Shift-F10  Macro replay");
    pw_putsat(temp,  8,  2, "F9         This key help");
    pw_putsat(temp, 10,  2, "Cursorkeys Select menu item");
    pw_putsat(temp, 11,  2, "Letter     Select menu item");
    pw_putsat(temp, 12,  2, "Enter      Activate selection");
    pw_putsat(temp, 13,  2, "Esc, F3    Exit menu");
    pw_putsat(temp,  1, 35, "Cursorkeys   Move cursor");
    pw_putsat(temp,  2, 35, "<Ctrl>Left   Move one word");
    pw_putsat(temp,  3, 35, "<Ctrl>Right  Move one word");
    pw_putsat(temp,  4, 35, "<Ctrl>End    Delete to end");
    pw_putsat(temp,  5, 35, "<Ctrl>Home   Delete all");
    pw_putsat(temp,  6, 35, "Insert       Insert/Overwrite");
    pw_putsat(temp,  7, 35, "Del          Delete at cursor");
    pw_putsat(temp,  8, 35, "Backspace    Delete before curs");
    pw_putsat(temp,  9, 35, "Esc, F3      Cancel");
    pw_putsat(temp, 11, 35, "Movement     Cursorkeys");
    pw_putsat(temp, 12, 35, "Left button  Enter key");
    pw_putsat(temp, 13, 35, "Right button Esc key");
    pw_show(temp);
    pw_keyinfo(temp, (_pwg_status & PW_IN_INFO) ? "" : "F1=Info", (_pwg_status & PW_IN_INFO) ?  pressany : NULL);
    ret = pw_getkey();
    pw_close(temp);
    _pwg_status &= (~PW_IN_KEYHELP);
    return ret;
}

unsigned int pwg_help(char *key)
{
    int c;
    char buf[160];
    char *tmp;
    int found = 0;
    int nrow = 0;
    int ncol = 0;
    int col = 0;
    long fptr;
    PWWIN * helpwin;
    unsigned int lastkey;
    int nchar = 0;
    int lastchar = 0;

    _pwg_status |= PW_IN_HELP;              /* set help bit */
    if ((!_pw_helpfh) || (!*key))
    {
        lastkey = pwg_nohelp(key);
        _pwg_status &= (~PW_IN_HELP);       /* reset help bit */
        return lastkey;
    }
    fseek(_pw_helpfh, 0L, SEEK_SET);        /* begin of file */
    while ((!feof(_pw_helpfh)) && (!found))
    {
        while (((c=fgetc(_pw_helpfh)) != EOF) &&  /* find marker '@' */
               (c != PW_HELP_CHAR))
            ;
        if (c == PW_HELP_CHAR)
        {
            fgets(buf, sizeof(buf), _pw_helpfh);
            tmp = strtok(buf, ",\n");       /* comp token list */
            while ((tmp) && (!found))
            {
                found = !stricmp(key, tmp);
                tmp = strtok(NULL, ",\n");
            }
        }
    }
    if (found)
    {
        fptr = ftell(_pw_helpfh);           /* mark begin of text */
        while (((c=fgetc(_pw_helpfh)) != EOF) &&
               (c != PW_HELP_CHAR))
        {                                   /* maxcol & maxrow of PWWIN * */
            nchar++;
            if (lastchar == PW_TYPE_CHAR) {
                col--;                      // %%
                if (strchr("pnhrs", c)) {
                    col--;                  // %p,%n,%h,%r,%s
                }
                lastchar = 0;
            } else {
                lastchar = c;
            }
            if (c == '\n')
            {
                if (col > ncol)
                    ncol = col;
                nrow++;
                col = 0;
            }
            else
                col++;
        }
        if (lastchar == '\n')               /* skip last '/n' */
            nchar--;
        if (ncol > _pwv_scrncol-2)          /* limit PWWIN * size */
            ncol = _pwv_scrncol-2;
        if (nrow > _pwv_scrnrow-2)
            nrow = _pwv_scrnrow-2;
        helpwin = pw_open(PWC_CENTER, PWC_CENTER, nrow, ncol, "Help", PWM_HIDE, PWW_HELP);
        fseek(_pw_helpfh, fptr, SEEK_SET);  /* goto begin of text */
        for (lastchar = 0; lastchar<nchar; lastchar++)
            pw_putc(helpwin, fgetc(_pw_helpfh));
        pw_show(helpwin);
        pw_keyinfo(helpwin, f1info, NULL);
        lastkey = pw_getkey();
        pw_close(helpwin);
    }
    else
        lastkey = pwg_nohelp(key);
    _pwg_status &= (~PW_IN_HELP);           /* reset help bit */
    return lastkey;
}

static PWKEY pwg_nohelp(char *key)
{
    char temp[SCRATCH_LEN];

    sprintf(temp, "No help available for window '%s'", key);
    return pwg_message("Message", temp, f1info, NULL);
}

PWKEY pwl_getkey(void)
{
    PWKEY key;

    if (!pw_key)
    {
#ifdef MOUSE
        pws_MouReset();
#endif
        while (!pwl_kbhit() && !pw_key)     /* wait for mouse or kbhit */
            ;
    }
    if (!(key = pw_key))
        key = pws_getkey();
    pw_key = 0;
    return key;
}

int pwl_kbhit(void)
{
    if (kbhit())
        return 1;                           /* keyboard hit */
    if (pw_key)
        return 2;                           /* mouse move or button */
#ifdef MOUSE
    pws_MouScan(&pw_key, pw_mou_key);
#endif
    if (pw_key)
        return 2;                           /* mouse move or button */
    if (pw_keyfunction)
        if ((pw_key = (*pw_keyfunction)()) != 0)
            return 3;                       /* kbwait function returned key */
    return 0;                               /* no mouse or kbhit */
}

void pwl_SetMouDef(PWKEY *MouDef)
{
    pw_mou_key = MouDef;
}

PWKEY *pwl_GetMouDef(void)
{
    return pw_mou_key;
}

void pwl_kbwait(PWKEY (*wait_function)(void))
{
    pw_keyfunction = wait_function;
}

static PWKEY pwg_IsAltKey(PWKEY key)
{
    PWKEY ret = 0;

    if ((key >= 0x1000) && (key <= 0x3200)) /* <Alt>-keys ? */
        ret = pw_AltTable[(key >> 8) - 0x10];
    return ret;
}

static PWKEY pwg_IsCtrlKey(PWKEY key)
{
    PWKEY ret = 0;

    if ((key >= 'A'-'@') && (key <= 'Z'-'@')) /* <Ctrl>-keys ? */
        ret = key + '@';
    return ret;
}

