/*
 * coalesce.c  -- program to coalesce several files into one file.  Reverse
 *                   of the fragment program.  Together these two programs
 *                   allow the user to transport files larger than the drives
 *		     of the involved machines.
 */

#include <stdio.h>
#include <string.h>

long filelength(int) ;			/* Wasn't declared in <stdio.h> ! */

#define	TRUE	1
#define FALSE	0

#define	DISK_SIZE_DESIRED	327680L
#define CHUNK_SIZE		16384
#define	MAX_FNAME_LEN		256

char buffer[CHUNK_SIZE] ;
int prompt = FALSE ;
char in_path[MAX_FNAME_LEN+1] ;
char in_template[MAX_FNAME_LEN+1] ;

void prompt_for_new_file(fragment_nbr)
int fragment_nbr ;
{
    fprintf(stderr, "Strike any key when ready with Fragment %0d.", fragment_nbr) ;
    fflush(stderr) ;
    fflush(stdin) ;
    getch() ;
}

FILE *next_input_file(in_template, suffix_ndx, fragment_nbr)
char *in_template ;
int suffix_ndx ;
int fragment_nbr ;
{
    char suffix[3+1] ;

    sprintf(suffix, "%0d", 100+fragment_nbr) ;
    strncpy(&in_template[suffix_ndx], &suffix[1], 2) ;

    return fopen(in_template, "rb") ;
}

void coalesce(big_stream, in_template, suffix_ndx)
FILE *big_stream ;
char *in_template ;
int   suffix_ndx ;
{
    long  bytes_written ;
    FILE *in_stream ;
    int   byte_cnt ;
    int   fragment_nbr ;
    int   chunk ;

    bytes_written = 0L ;
    for (fragment_nbr=0; fragment_nbr < 100; fragment_nbr++) {

	if (prompt)
	    prompt_for_new_file(fragment_nbr) ;

	in_stream = next_input_file(in_template, suffix_ndx, fragment_nbr) ;
	if (in_stream == NULL)
	    break ;

	for (chunk=0; chunk < DISK_SIZE_DESIRED/CHUNK_SIZE; chunk++) {
	    byte_cnt = fread(buffer, 1, CHUNK_SIZE, in_stream) ;
	    if (byte_cnt == 0)
		break ;
	    fprintf(stderr, "\rFragment %2d, Chunk %2d: %0d bytes Read     ",
			fragment_nbr, chunk, byte_cnt) ;
	    fflush(stderr) ;
	    if (fwrite(buffer, 1, byte_cnt, big_stream) != byte_cnt) {
		fprintf(stderr, "\n") ;
		fprintf(stderr, "Error Writing Fragment %2d, Chunk %2d.\n") ;
		fprintf(stderr, "Total of %0ld were successfully coalesced.\n",
				bytes_written) ;
		exit(1) ;
	    }
	    bytes_written += (long)byte_cnt ;
	    fprintf(stderr, "\rFragment %2d, Chunk %2d: %0d bytes Written    ",
			fragment_nbr, chunk, byte_cnt) ;
	    fflush(stderr) ;
	}

	fclose(in_stream) ;
	fprintf(stderr, "\rFragment %2d: Coalesced                                     \n", fragment_nbr) ;
    }

    fclose(big_stream) ;
    fprintf(stderr, "Coalescing complete.  %0d fragments were read, totally %0ld bytes\n",
		fragment_nbr, bytes_written) ;
}

void get_template(file_template, in_template, suffix_ndx)
char *file_template ;
char *in_template ;
int  *suffix_ndx ;
{
    char fname_template[8+1] ;
    char file_ext[1+3+1] ;
    char *file_name ;
    char *file_dot ;

    *in_path = '\0' ;
    file_name = file_template ;
    if (strchr(file_template, '\\') || strchr(file_template, ':')) {
	if (strchr(file_template, '\\'))
	    file_name = strrchr(file_template, '\\')+1 ;
        else
	    file_name = strrchr(file_template, ':')+1 ;
	if (file_name-file_template > MAX_FNAME_LEN) {
	    fprintf(stderr, "Directory path of %s is too long.\n", file_template) ;
	    exit(1) ;
	} else {
	    strncpy(in_path, file_template, file_name-file_template) ;
	    in_path[file_name-file_template+1] = '\0' ;
	}
    }

    *file_ext = '\0' ;
    file_dot = strchr(file_name, '.') ;
    if (file_dot)
	if (strlen(file_dot) > 4) {
	    fprintf(stderr, "The extension of '%s' is too long.\n", file_dot) ;
	    exit(1) ;
	} else {
	    strcpy(file_ext, file_dot) ;
	    *file_dot = '\0' ;
	}

    strcpy(fname_template, "________") ;
    if (strlen(file_name) > 6) {
	fprintf(stderr, "The filename of '%s' is too long.\n", file_name) ;
	exit(1) ;
    }
    strncpy(fname_template, file_name, strlen(file_name)) ;

    if (strlen(in_path)+strlen(fname_template)+strlen(file_ext) > MAX_FNAME_LEN) {
	fprintf("The resulting input pathname of '%s%s%s' is too long.\n",
		   in_path, fname_template,file_ext) ;
	exit(1) ;
    }

    strcpy(in_template, in_path) ;
    strcat(in_template, fname_template) ;
    *suffix_ndx = strlen(in_template) - 2 ;
    strcat(in_template, file_ext) ;
}

void main(argc, argv)
int argc ;
char *argv[] ;
{
    FILE *big_stream ;
    long  length ;
    int   suffix_ndx ;

    argc-- ; /* throw away "program name" argument */
    argv++ ;

    /* parse off any "prompt option" argument that is present */

    prompt = FALSE ;
    if (argc && (stricmp(argv[0], "-p") == 0)) {
	argc-- ;
	argv++ ;
	prompt = TRUE ;
    }

    if ((argc != 2) || (*argv[0] == '?')) {
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   coalesce [-p] <big fname> <input file template>\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   Where:\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       -p optionally causes the program to pause between input files.\n") ;
	fprintf(stderr, "           (allowing you to, say, change input disks)\n") ;
	fprintf(stderr, "       <big fname> is the full pathname to the file to be coalesced.\n") ;
	fprintf(stderr, "       <input file template> is the path to the input directory\n") ;
	fprintf(stderr, "          along with a six letter filename template which will\n") ;
	fprintf(stderr, "          be suffixed with the fragment nbr and the extension.\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "   Example:\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       coalesce -p c:\\pcip\\src\\doc.tar a:doc.frg\n") ;
	fprintf(stderr, "\n") ;
	fprintf(stderr, "       Coalesces the file c:\\pcip\\src\\doc.tar from the\n") ;
	fprintf(stderr, "          fragments named a:doc___00.frg, a:doc___01.frg, etc.\n") ;
	fprintf(stderr, "       Since -p is present, the program will pause between each file.\n") ;
	exit(1) ;
    }

    if ((big_stream = fopen(argv[0], "rb")) != NULL) {
	fprintf(stderr, "The output file '%s' already exists.\n", argv[0]) ;
	fprintf(stderr, "   Hit Ctrl-Break to abort, any other key to overwrite.\n") ;
	getch() ;
	fclose(big_stream) ;
    }
    if ((big_stream = fopen(argv[0], "wb")) == NULL) {
	fprintf(stderr, "can't open %s for input.\n", argv[0]) ;
	exit(1) ;
    }

    get_template(argv[1], in_template, &suffix_ndx) ;

    coalesce(big_stream, in_template, suffix_ndx) ;
}
