//===========================================================================
//
//      df.c - display free space on mounted disks
//
//      Author:  Larry Kollar <larryk@cton.com>
//               Basically a slight modification to drives.c by Jon Frievald
//
//
//===========================================================================

#pragma inline                  // start compilation via assembly to eliminate
                                // the error message generated
#include <stdio.h>
#include <stdlib.h>
#include <dos.h>
#include <dir.h>
#include "dfbits.h"

//===========================================================================
//
//      function declarations
//
//===========================================================================

void    main ( int, char ** );
int     check_disk_type ( int disk_number );
int     test_cd_driver ( void );
char *  get_label( int disk_number );
void    print_disk_info( int disk_number, int disk_type );
extern  void setCEH ( void );
extern  void unsetCEH ( void );
extern  unsigned short  number_floppy ( void );
extern  int	 found_an_option( int, char ** );
extern  void check_drive_args( int, char **, long * );

//===========================================================================
//
//      character arrays
//
//===========================================================================

char    *drive_type     [] = { "", "160K diskette", "180K diskette",\
        "320K diskette", "360K diskette", "1.2M diskette", "720K diskette",\
	"1.44M diskette", "unknown floppy", "fixed disk",\
	"network file service", 0 };

//===========================================================================
//
//      miscellany
//
//===========================================================================

// #define SMALL   0
// char    memory_model = SMALL;   // needed for numflopy.asm

//===========================================================================
//
//      main()
//
//              arguments:      currently none
//
//              returns:        0
//
//===========================================================================

void main ( int argc, char **argv )

{
    int  i, disk_type_found;
    static long drive_bits;


    SET_ALL( drive_bits );
    if( found_an_option(argc, argv) )
		exit( EXIT_FAILURE );

    check_drive_args( argc, argv, &drive_bits );

    setCEH();

    printf( "%s%s\n", "Drive  Label          Type                 ",
            "  K total   K used   K free" );

    /* check the floppies:  i=0 means Drive A:, etc. */

    for ( i=0; i < number_floppy(); i++ )
    {
		if( GET_BIT(drive_bits, i) )  /* do we want to look for it? */
		{
			disk_type_found = check_disk_type ( i+1 );
			if ( disk_type_found != 0 )
			    print_disk_info( i+1, disk_type_found );
		}
    }

    if ( i == 1 )
	i++;

    for ( ; i < 27; i++ )
	{
		if( GET_BIT(drive_bits, i) )  /* do we want to bother looking? */
        {
			disk_type_found = check_disk_type ( i+1 );
			if ( disk_type_found != 0 )
			    print_disk_info( i+1, disk_type_found );
		}
    }

    printf("\n");

    unsetCEH();

    exit( EXIT_SUCCESS );
}

//===========================================================================
//
//      check_disk_type()
//
//              determines whether disk is local, remote, or CD-ROM
//
//              arguments:      int     disk_number     number of disk as
//                                                      from getdisk()
//
//              returns:        int     disk type;
//                                                      0 = no disk
//                                                      1 = 160K floppy
//                                                      2 = 180K floppy
//                                                      3 = 320K floppy
//                                                      4 = 360K floppy
//                                                      5 = 1.2M floppy
//                                                      6 = 720K floppy
//                                                      7 = 1.44M floppy
//                                                      8 = unknown floppy
//                                                      9 = local
//                                                      10 = network
//                                                      11 = CD-ROM
//
//        uses global:    cd_driver       (0 if no driver or < 2.0, 1 if >= 2.0)
//
//===========================================================================

int check_disk_type ( int disk_number )

{
        union   REGS    regs;
        struct  SREGS   sregs;
        unsigned char far *des_byte;            // media descriptor byte
        int     rc = 1;

        regs.h.dl = disk_number;                // drive number to test (a: == 1)
        regs.h.ah = 0x1C;                       // "get drive data" function
        int86x ( 0x21, &regs, &regs, &sregs );

        if ( regs.h.al == 0xFF )                // Did an error occur?
                {
                return ( 0 );                   // Assume no disk present
                }
        des_byte = MK_FP ( sregs.ds, regs.x.bx );       // Make the far pointer to the descriptor byte

        if ( ( des_byte[0] != 0xF8 ) && ( des_byte[0] != 0x00 ) )       // Is it a fixed disk?

        /*  This is where the code varies from the documentation I can find on
            the subject -- it *SEEMS* that CD's return a descriptor byte of
            0x00, however, I cannot find that documented anywhere... (0xF8 is
            fixed disk).  I've had to include the test for 0x00 here so that we
            progress to the tests further down the line... */

                {
                regs.h.ah = 0x36;               // If not, get the number of sectors on it
                regs.h.dl = disk_number;        //   to determine the type of floppy it is
                int86 ( 0x21, &regs, &regs );   //   (the descriptor byte is not specific enough...)
                switch ( regs.x.dx * regs.x.ax )
                        {
                        case 313:               // 160K
                                return ( 1 );
                        case 351:               // 180K
                                return ( 2 );
                        case 630:               // 320K
                                return ( 3 );
                        case 708:               // 360K
                                return ( 4 );
                        case 2371:              // 1.2M
                                return ( 5 );
                        case 1426:              // 720K
                                return ( 6 );
                        case 2847:              // 1.44M
                                return ( 7 );
                        default:                // Unknown type
                                return ( 9 );
                        }
                }

        // If we get this far, it's a hard, net or CD-ROM disk

        rc = 9;                    // Set rc to 9 if we get this far because
                                   //   it's at least a local hard disk
        regs.h.ah = 0x44;          // Int 21, func 44, sub 9 is IOCTL, we
        regs.h.al = 0x9;           //   use it to diff local & net/cd (remote)
                                   //   This requires DOS 3.1 or higher
        regs.h.bl = (unsigned char) disk_number;
        int86 ( 0x21, &regs, &regs );

        if ( ( regs.x.dx & 0x1000 ) == 0x1000 ) // if bit 12 of DX is 1, disk is remote
        {
        	rc = 10;                    // if we're here, we're either net or cd
		}
        return ( rc );
}

// =============================================================================
// The rest of this file is from Larry Kollar, and is public domain (but if you
// use this code in your own projects, remember I are not a programmer).
// =============================================================================
// 
// get disk volume label
// 	input:  disk number (1='A', 2='B', etc.)
//	return: volume label or "(none)" if not found

char *get_label( int disk_number )
{
	struct ffblk ffblk;
	int done;
	char *root_dir = " :\*.*";
	static char vol_name[20];

	*vol_name = '\0';   					/* returns NULL if can't find vol. */
	root_dir[0] = '@' + disk_number;		/* (ugly, ain't it?) */
	done = findfirst( root_dir, &ffblk, FA_LABEL );
	while( !done )
	{
		if( ffblk.ff_attrib & FA_LABEL )	/* found the label? */
		{
			strcpy( vol_name, ffblk.ff_name );
			done = 1;
			break;
		}

      /* I doubt this loop is needed since the volume label gets written
         first, but the exe isn't that big anyway and I like to be safe
         when I'm mucking around with things I don't understand too well... */

		done = findnext( &ffblk );
	}

	if( *vol_name == '\0' )				/* no volume name found */
		strcpy( vol_name, "(none)" );

   	return vol_name;
}



// print disk information
// 	input:  disk number (1='A', 2='B', etc.)
//	return: none

void print_disk_info( int disk_number, int disk_type )
{
	long Ktotal, Kused, Kfree;
    long bytes_per_cluster;
    struct dfree disk_info;

    getdfree( (unsigned char)disk_number, &disk_info );

    bytes_per_cluster = disk_info.df_bsec * disk_info.df_sclus;
    Ktotal = disk_info.df_total * bytes_per_cluster / 1024;
    Kfree  = disk_info.df_avail * bytes_per_cluster / 1024;
    Kused  = Ktotal - Kfree;

    printf("   %c:  %-13s  %-20s   %7ld  %7ld  %7ld\n",
	   (char)('@'+disk_number), get_label(disk_number),
	   drive_type[disk_type], Ktotal, Kused, Kfree        );

}

