// PrnStr.H  copyright Brian Maxwell 1995
// This module defines the Prnstr and Printer classes as
// well as three specific classes for Epson, HP LaserJet and PostScript
// Printers
#define PrnstreamVersion	1.01

#ifndef __cplusplus
#error Must use C++ for the type prnstream.
#endif

#if !defined( __PRNSTR_H )
#define __PRNSTR_H

#if !defined( __OPRNSTR_H )
#include "oprnstr.h"
#endif

#if !defined( __IOSTREAM_H )
#include <iostream.h>
#endif

#if !defined( __IOMANIP_H )
#include <iomanip.h>
#endif

#if !defined( __STRNG_H)
#include <Strng.h>
#endif

#define B_size	1024

// This section of defines is needed to make the manipulators which have
// parameters work with oprnstream objects

// DO NOT ALTER

#define PMANIP(typ)     _Paste2(pmanip_, typ)
#define PAPP(typ)       _Paste2(papply_, typ)

#define PMANIPdeclare(typ)				\
class _CLASSTYPE PMANIP(typ) {                                          \
		oprnstream _FAR & (_Cdecl *_fn)(oprnstream _FAR &, typ);        \
		typ _ag;                                                        \
public:                                                                 \
		_Cdecl PMANIP(typ)(oprnstream _FAR & (_Cdecl *_f)(oprnstream _FAR &, typ), typ _z ) :            \
				_fn(_f), _ag(_z) { }                                    \
		friend oprnstream _FAR & _Cdecl operator<<(oprnstream _FAR & _s, const PMANIP(typ) _FAR & _f) {       \
				return(*_f._fn)(_s, _f._ag); }                          \
		};                                                              \
class _CLASSTYPE PAPP(typ) {                                            \
		oprnstream _FAR & (_Cdecl *_fn)(oprnstream _FAR &, typ);        \
public:                                                                 \
		_Cdecl PAPP(typ)(oprnstream _FAR & (_Cdecl *_f)(oprnstream _FAR &, typ)) : _fn(_f) { }           \
		PMANIP(typ) _Cdecl operator()(typ _z) {                         \
				return PMANIP(typ)(_fn, _z); }                          \
		};

enum PageSize { Letter = 0, Legal = 1 };

// B_size is the size of the default printer buffer. This size can be
// varied depending on the size of the built-in printer buffer. If the
// expected printer buffer is large, this value can be relatively small;
// if the printer has a small (or non-existent) buffer, 1K should work
// fairly well.

const	pfTimeout =		0x0001,
		pfIoError =		0x0008,
		pfSelected =    0x0010,
		pfOutOfPaper =	0x0020,
		pfAcknowledge =	0x0040,
		pfNotBusy =		0x0080;
// These constants are the bit masks for the printer status.

_CLASSDEF(prnbuf)
_CLASSDEF(prnstreambase)
_CLASSDEF(prnstream)

// ---------------------------------------------------------------------
// Class: prnbuf
// Derived from: streambuf
// Desc:
//	  Class prnbuf is the streambuf-derived class that will buffer the
//	stream output, and do the actual writes to the printer port. Uses
//	biosprint() RTL function to do actual printing, so print errors
//	(such as out-of-paper, etc) will NOT cause Dos INT 24h (error handler)
//	to be called (this is the 'Abort, Retry, ...' message). Instead, class
//	ios state flags can be used to determine printer errors.
// Note:
//	  A design decision was required on whether or not the constructors
//	should send the INIT signal to the printer to verify that a printer
//	is attached and operational. The benefit of sending the signal is that
//	the printer will be initialized when you 'open' the stream (ie any
//	pages will be ejected, defaults will be established). This is also
//	detrimental: there are many cases where you don't WANT the printer
//	settings mucked with. The first version was ambiguous on this point;
//	this version (1.01) NEVER initializes the printer in the ctor; if you
//	want the printer initialized, call the public method init(). All the
//	ctor does is send the STATUS signal to the printer to verify its
//	operational status - this does not effect printer settings.
//
// Methods:
//	  prnbuf()
//		Creates a buffer that is not attached to any printer port. A buffer
//		of size B_size is allocated.
//
//	  prnbuf( int f )
//		If f is a BIOS supported printer port (0=LPT1, 1=LPT2, etc) then
//		creates a buffer of size B_size and attaches it to this printer
//		port.
//
//	  prnbuf( int _f, char _FAR *buf, int sz )
//		If _f is a BIOS supported printer port (0=LPT1, 1=LPT2,etc) then
//		uses the sz size buffer buf and attaches it to this printer port.
//
//	  ~prnbuf()
//		Destroys the buffer, first flushing any data still remaining in
//		the buffer. If the ctors prnbuf(), prnbuf(int) were used to create
//		the buffer, or attach(int) was called and the buffer was empty,
//		then ~streambuf() will deallocate the buffer.
//
//    int fd()
//		Returns the 'file descriptor' (terminology used to conform to the
//		file streams); 0=LPT1, 1=LPT2, etc, EOF = non attached to a printer
//		port.
//
//    void init()
//		Initializes the printer (if fd() != EOF) by sending the INIT signal
//		to the printer. On most printers, this should 1) eject any pages
//		currently being 'partially' printed (on page printers like the
//		HP Laserjet/Deskjet), and 2) reset the printer settings such as
//		page size, current font, paper source, etc.
//
//    void setBinary( int state )
//      Sets/Clears the binary mode if state is Non-zero/zero. In binary
//		mode, every byte sent to the stream remains unchanged; in non-
//		binary mode (ie 'cooked' mode) every '\n' sent to the printer
//		(ie line feed) will have an additional '\r' sent by prnbuf
//		(ie carriage return).
//
//    prnbuf *attach( int f )
//      Attaches this prnbuf to a printer port. If currently attached to a
//		port, the buffer is first flushed. If an internal buffer has not
//		previously been allocated, a B_size buffer will be allocated. Sends
//		a STATUS request to verify a printer is attached.
//
//    int overflow( int ch = EOF )
//      Used internally to flush the buffer (ch = EOF) or when the buffer
//		is full and another char is written to it. Sends data to the port
//		using biosprint() RTL function. Returns EOF on error so that an
//		owning stream can set the proper ios flags.
//
//    int underflow()
//      Normally in a streambuf class, this method is used to extract chars
//		from the stream. Since printers are essentially an output stream
//		device, this method only returns the stream status. Therefore you
//		check an open prnbuf status by performing:
//			status = thisbuf.sgetc();
//
//    int sync()
//      Flushes the buffer to the printer.
//
// Members:
//    xfd
//		The attached printer port (0=LPT1, 1= LPT2, etc, EOF = none).
//
//    isBinary
//		Flag for binary mode setting (0='cooked', <>0 = 'raw').
// ---------------------------------------------------------------------

class  _CLASSTYPE prnbuf : public streambuf
	{
	public:

	// constructors, destructor
		_Cdecl prnbuf();   	// make a closed prnbuf
		_Cdecl prnbuf(int);    // make a prnbuf attached to fd
		_Cdecl prnbuf(int _f, char _FAR *, int); // same, with specified buffer
		_Cdecl ~prnbuf();

		int _Cdecl fd();        // what is the file descriptor
		void _Cdecl init();		// Printer initialization.
		void _Cdecl setBinary( int );	// set/clear binary mode.

		prnbuf _FAR * _Cdecl attach(int);  // attach this prnbuf to port
											// number.

/*
 * These perform the streambuf functions on a prnbuf
 * Get and Put pointers are kept together
 */
		virtual int _Cdecl overflow(int = EOF);
		virtual int _Cdecl underflow();
		virtual int _Cdecl sync();


	protected:
		int xfd;        // the file descriptor, EOF if closed
		short isBinary;	// flag.

	};

inline int  _Cdecl prnbuf::fd()        { return xfd; }
inline void _Cdecl prnbuf::setBinary( int st )
	{
	isBinary = st ? 1 : 0;
	}


// ---------------------------------------------------------------------
// Class: prnstreambase
// Derived from: ios (virtual)
// Desc:
//	  Method that contains the methods common to all printer streams.
// Methods:
//	prnstreambase()
//      Creates a stream base that is not attached to a printer.
//
//	prnstreambase( int p )
//		Creates a stream base attached to printer port p (0=LPT1, 1=LPT2,
//		etc).
//
//	prnstreambase( int _f, char *buf, int sz )
//		Creates a stream base attached to printer port _f using a sz-sized
//		buffer whose pointer is buf.
//
//	~prnstreambase()
//		Flushes the buffer, deallocates as necessary.
//
//	void attach( int p )
//		If currently attached to a port, flushes the buffer. Then attaches
//		to port p and allocates a buffer if one has not already been
//		created.
//
//	void init()
//		Calls prnbuf::init() to send the INIT signal to the printer.
//
//  void setBinary( int flag )
//      Calls prnbuf::setBinary(flag) to set/clear the binary mode.
//
//	void setbuf(char *buf, int sz )
//      Calls prnbuf::setbuf(buf,sz) (which calls streambuf::setbuf(buf,sz))
//		To change the buffer to use buf (sz-sized). If the old buffer was
//		allocated by prnbuf, it will be deleted.
//
//	prnbuf *rdbuf()
//      Returns a pointer to the prnbuf.
//
// Members:
//	buf
//		Prnbuf stream buffer.
// ---------------------------------------------------------------------

class _CLASSTYPE prnstreambase : virtual public ios
	{
	public:
		_Cdecl prnstreambase();
		_Cdecl prnstreambase( int);
		_Cdecl prnstreambase( int _f, char _FAR *, int );
		_Cdecl ~prnstreambase();

		void    _Cdecl attach(int);
		void	_Cdecl init();
		void	_Cdecl setBinary( int );
		void    _Cdecl setbuf(char _FAR *, int);
		prnbuf _FAR * _Cdecl rdbuf();

	private:
		prnbuf buf;
	};

inline prnbuf _FAR * _Cdecl prnstreambase::rdbuf() { return &buf; }
inline void prnstreambase::init()
	{	buf.init();	}
inline void prnstreambase::setBinary( int st )
	{	buf.setBinary(st);	}

// ---------------------------------------------------------------------
// Class: prnstream
// Derived from: prnstreambase, oprnstream
// Desc:
//	  Class prnstream implements a printer output stream (oprnstream) that
//	streams to a printer port. All of the public method of prnstreambase
//	are available, as well as the standard ostream methods, including
//	manipulators and overloaded operators.
// Methods:
//	prnstream()
//		Create a stream that is not attached to a printer port.
//
//	prnstream( int p )
//		Create a stream and attach to printer port p (0=LPT1, 1=LPT2, etc).
//
//  prnstream( int _f, char *buf, int sz )
//		Create a stream and attach to printer port _f(0=LPT1, 1=LPT2, etc).
//		Use the sz-sized buffer buf for buffered output.
//
//	~prnstream()
//		Flushes the stream and deallocated as necessary.
//
//	prnbuf *rdbuf()
//		Returns a pointer to the prnbuf stream buffer.
// ---------------------------------------------------------------------

class _CLASSTYPE prnstream : public prnstreambase, public oprnstream
	{
	public:
		_Cdecl prnstream(int psflag);
		_Cdecl prnstream(int port, int psflag);
		_Cdecl prnstream(int _f, char _FAR *, int, int psflag);
		_Cdecl ~prnstream();

		prnbuf _FAR * _Cdecl rdbuf();
	};

inline prnbuf _FAR * _Cdecl prnstream::rdbuf() { return prnstreambase::rdbuf(); }


// Generic Printer Base Class

class _CLASSTYPE Printer : public prnstream
	{

	public:
		_Cdecl	Printer(int psflag) : prnstream(psflag) { };
		_Cdecl  Printer(int f, int psflag) : prnstream(f, psflag) { };
		_Cdecl  Printer(int f, char _FAR *buf, int len, int psflag) :
					prnstream(f, buf, len, psflag) { };
		virtual void setPageSize(PageSize ps) = 0;
		virtual void setMargins(float tm, float bm, float lm, float rm) = 0;
		virtual void Reset() = 0;
		virtual void EndJob() = 0;       // Tell printer you're done
		virtual void setBold() = 0;
		virtual void setNormal() = 0;
		virtual void setItalics() = 0;
		virtual void setCompressed() = 0;
		virtual void setTabs(String& tabs) = 0;
		virtual void setFont(int font) = 0;
		virtual void setPitch(int pitch) = 0;
	};


class _CLASSTYPE EpsonPrinter : public Printer
	{
public:
		_Cdecl	EpsonPrinter();
		_Cdecl  EpsonPrinter(int f);
		_Cdecl  EpsonPrinter(int f, char _FAR *buf, int len);
		_Cdecl  ~EpsonPrinter( void );
		virtual void setPageSize(PageSize ps);
		virtual void setMargins(float tm, float bm, float lm, float rm);
		virtual void Reset();
		virtual void EndJob();
		virtual void setBold();
		virtual void setNormal();
		virtual void setItalics();
		virtual void setCompressed();
		virtual void setTabs(String& tabs);
		virtual void setFont(int font);
		virtual void setPitch(int pitch);
	};

class _CLASSTYPE HPPrinter : public Printer
	{
private:
		int fontsize;
		int currentFont;
public:
		_Cdecl	HPPrinter();
		_Cdecl  HPPrinter(int f);
		_Cdecl  HPPrinter(int f, char _FAR *buf, int len);
		_Cdecl  ~HPPrinter( void );
		virtual void setPageSize(PageSize ps);
		virtual void setMargins(float tm, float bm, float lm, float rm);
		virtual void Reset();
		virtual void EndJob();
		virtual void setBold();
		virtual void setNormal();
		virtual void setItalics();
		virtual void setCompressed();
		virtual void setTabs(String& tabs);
		virtual void setFont(int font);
		virtual void setPitch(int pitch);
	};

class _CLASSTYPE PSPrinter : public Printer
	{
private:
		int fontsize;
public:
		_Cdecl	PSPrinter();
		_Cdecl  PSPrinter(int f);
		_Cdecl  PSPrinter(int f, char _FAR *buf, int len);
		~PSPrinter(void);
		virtual void init();
		virtual void setPageSize(PageSize ps);
		virtual void setMargins(float tm, float bm, float lm, float rm);
		virtual void Reset();
        virtual void EndJob();
		virtual void setBold();
		virtual void setNormal();
		virtual void setItalics();
		virtual void setCompressed();
		virtual void setTabs(String& tabs);
		virtual void setFont(int font);
		virtual void setPitch(int pitch);
	};



// These are the manipulators for the Printer classes
// They will work with everything except the PostScript Printer

		oprnstream _FAR & _Cdecl bold(oprnstream _FAR & pr);
		oprnstream _FAR & _Cdecl normal(oprnstream _FAR & pr);
		oprnstream _FAR & _Cdecl italics(oprnstream _FAR & pr);
		oprnstream _FAR & _Cdecl compressed(oprnstream _FAR & pr);
		oprnstream _FAR & _Cdecl reset(oprnstream _FAR & pr);
		oprnstream _FAR & _Cdecl font(oprnstream _FAR & pr, int fnt);
		oprnstream _FAR & _Cdecl pitch(oprnstream _FAR & pr, int pit);
		oprnstream _FAR & _Cdecl tabs(oprnstream _FAR & pr, String& tabstr);


		typedef String& StrRef;
		PMANIPdeclare(StrRef);
		PMANIPdeclare(int);
		PMANIPdeclare(long);

		PMANIP(StrRef) tabs(String& tabstr);
		PMANIP(int) font(int fnt);
		PMANIP(int) pitch(int pit);


#endif
