/**********************************************************************************************
*  PROJECT NAME: Deadgame.                                                                    *
*  AUTHOR      : Mr. Meetul Kinarivala.                                                       *
*  START DATE  : 16th December 1995.                                                          *
*  END DATE    : 21st December 1995.                                                          *
*  DESCRIPTION : Deadgame is a board game. Here the computer plays against the human.         *
*  AIM         : To generate the deadgame engine.                                             *
*  FILENAME    : DEADGAME.H                                                                   *
*  FILE DESC.  : Header file for the deadgame engine.                                         *
**********************************************************************************************/

/*******************************************************************
* Important - Play the game properly. Then read the writeup        *
*             Then read DEADGAME.H and then DEADGAME.CPP           *
********************************************************************/


/**********************************************************************************************
Here we state the include files
**********************************************************************************************/

#include<iostream.h>
#include<mem.h>
#include<conio.h>
#include<stdio.h>
#include<stdlib.h>
#include<dos.h>

/**********************************************************************************************
Here we define some global independent constants.
**********************************************************************************************/

#define BOARD_SIZE_X 6 //Number of squares on the board horizontally.
#define BOARD_SIZE_Y 6 //Number of squares on the board vertically.

#define DEAD  0
#define ALIVE 1        //For each peg, there is a structure PEGGG in which
					   //there is a variable called status which is
					   //equal to ALIVE or DEAD

#define BLANK 0 	   //A two dimensional array board_layout[BOARD_SIZE_Y][BOARD_SIZE_6]
					   //maintains the data about board positions.
					   //board_layout[y][x] = BLANK if it is unoccupied.
					   //board_layout[y][x] = -ve if it is occupied by human
					   //board_layout[y][x] = +ve if it is occupied by computer

#define TRUE  1		   //for general use
#define FALSE 0		   //for general use

#define HUMAN -1	   //for general use to distinguish. Used as function arguments too.
#define COMPUTER 1	   //for general use to distinguish. Used as function arguments too.


#define PEGS 6         //initial no. of human/computer pegs. Cannot exceed BOARD_SIZE_X


int curposx,curposy; //Cursor positions on the board.

int level=0;
int MAXLEVEL=3;

/**********************************************************************************************
Here we define some global dependent constants. These vary as independent constants.
**********************************************************************************************/

#define BOARD_SQUARES 36 //Total no. of board squares.

/**********************************************************************************************
Here we define macros for keyboard events.
**********************************************************************************************/

#define U_ARW_PD (ascii==0 && scancode==72) //move up
#define D_ARW_PD (ascii==0 && scancode==80) //move down
#define R_ARW_PD (ascii==0 && scancode==77) //move right
#define L_ARW_PD (ascii==0 && scancode==75) //move left
#define ENTER (ascii==13 && scancode==28) //Pick/drop peg
#define ESC   (ascii==27 && scancode==1)  //Deselect peg
#define S_PD  (scancode==31) //Restart game
#define I_PD  (scancode==23) //Invert board
#define E_PD  (scancode==18) //Exchange pegs
#define Q_PD  (scancode==16) //Quit game
#define U_PD  (scancode==22) //Undo move
#define R_PD  (scancode==19) //Redo move
#define A_PD  (scancode==30) //Switch to arrange mode
#define C_PD  (scancode==46) //If C is pressed it is computer's turn to start
#define H_PD  (scancode==35) //If H is pressed it is human's turn to start
							 //Hint during normal mode
#define PLUS_PD  (scancode==78 || scancode==13)
#define MINUS_PD (scancode==74 || scancode==12)

/**********************************************************************************************
Here we define different colour attributes for screen display.
**********************************************************************************************/
#define SCREEN_ATTR 241
#define C_PEG_ATTR 249
#define H_PEG_ATTR 252
#define CURSOR_ATTR 247
#define MESSAGE_ATTR 252


#define signof(x) ((x)>>15)

/**********************************************************************************************
Here we define class BOARD.
**********************************************************************************************/

class BOARD
{
	private:

	char board_layout[BOARD_SIZE_Y][BOARD_SIZE_X];
	/*	board_layout is a two dimensional array
	which maintains positions of pegs on board.

	Computer pegs are numbered from 1 to 6.
	Human pegs are numbered from -1 to -6.

	board_layout[y][x] = BLANK if it is unoccupied.
	board_layout[y][x] = -ve peg number if it is occupied by human.
	board_layout[y][x] = +ve peg number if it is occupied by computer.
	board_layout[0][0] = top left corner of the board.

	Any board position can be represented by board_layout[y][x]
	where: 0<=x<=5 and 0<=y<=5
	*/

	struct PEGGG
	{
		int status;    //status of that piece = DEAD or ALIVE.
		int xpos,ypos; //(x,y) of a particular piece on board. Top left position = (0,0)
		int maxplaces; //number of squares on the board occupiable by the piece
		int mp[BOARD_SIZE_X+BOARD_SIZE_Y-2][2]; //set of coordinates
												 //of movable positions for that peg
	}	cp[PEGS+1],hp[PEGS+1];


/*    eg:              0 1 2 3 4 5
					  +-+-+-+-+-+-+
					 0| | |.| | | |0
					  +-+-+-+-+-+-+
					 1| | |.| | | |1
					  +-+-+-+-+-+-+
					 2|.|.|C|.|.|H|2
					  +-+-+-+-+-+-+
					 3| | |.| | | |3
					  +-+-+-+-+-+-+
					 4| | |.| | | |4
					  +-+-+-+-+-+-+
					 5| | |.| | | |5
					  +-+-+-+-+-+-+
					   0 1 2 3 4 5

	Here C = computer's peg numbered say, 4
	Here H = human's    peg numbered say, -2

	Therefore board_layout[2][2]=4
			  board_layout[5][2]=-2


	Hence looking at the figure,

	cp[4].status = ALIVE (ALIVE is defined as 1)
	cp[4].xpos   = 2
	cp[4].ypos   = 2
	cp[4].maxplaces = 9;(Denoted by dots)

	First movable position:
	( cp[4].mp[0][0] , cp[4].mp[0][1] ) = (2,1)

	Second movable position:
	( cp[4].mp[1][0] , cp[4].mp[1][1] ) = (2,0)

	Third movable position:
	( cp[4].mp[2][0] , cp[4].mp[2][1] ) = (2,3)

	Fourth movable position:
	( cp[4].mp[3][0] , cp[4].mp[3][1] ) = (2,4)

	Fifth movable position:
	( cp[4].mp[4][0] , cp[4].mp[4][1] ) = (2,5)

	Sixth movable position:
	( cp[4].mp[5][0] , cp[4].mp[5][1] ) = (1,2)

	Seventh movable position:
	( cp[4].mp[6][0] , cp[4].mp[6][1] ) = (0,2)

	Eighth movable position:
	( cp[4].mp[7][0] , cp[4].mp[7][1] ) = (3,2)

	Ninth movable position:
	( cp[4].mp[8][0] , cp[4].mp[8][1] ) = (4,2)

	hp[2].status = ALIVE (ALIVE is defined as 1)
	hp[2].xpos   = 5
	hp[2].ypos   = 2
	hp[2].maxplaces = 7;(NOT denoted by dots)



/********************************************************************************************/
/********************************************************************************************/

	public:

	int cq; //total number of movable positions for all computer pegs
	int hq; //total number of movable positions for all human pegs
	int cb; //number of computer's blocked pegs
	int hb; //number of human's blocked pegs
	int ca; //number of computer's alive pegs
	int ha; //number of human's alive pegs
	int cal;  //Net Computer's advantage level for this board configuration

	BOARD(void) //constructor with no arguments initializes board.
	{
		//BLANK entire board
		memset(board_layout,BLANK,BOARD_SQUARES);//blank entire board

		//initialize positions
		//i.e. assign all the pegs their individual starting positions

		for(int i=0; i<PEGS; ++i)
		{
			//computer takes positions at the top of the board
			board_layout[0][i] = i+1;
			//human takes positions at the bottom of the board
			board_layout[BOARD_SIZE_Y-1][i] = -(i+1);
		}

		//Now update the PEGGG structure for all the pegs on the board
		ups();
	}


	BOARD(char bl[BOARD_SIZE_Y][BOARD_SIZE_X]) //constructor initializes board_layout according
	{                                                 //to the argument passed to it.
		//copy bl to board_layout
		memcpy(board_layout,bl,BOARD_SQUARES);

		//does not update structure PEGGG
	}

	//Following function name: Update PEGGG Structure
	//This function updates PEGGG structure by looking at the board
	//It also updates variables: cq,hq,ca,ha
	//Thereafter It also calculates CAL(computer advantage level)

	void ups(void)
	{
		PEGGG *p;int piece,y,x,temp;

		//reset variables:
		cq=hq=cb=hb=ca=ha=0;

		//reset the all peg statuses to dead
		for(int i=1; i<=PEGS; ++i)	cp[i].status=hp[i].status=DEAD;

		//Now go on checking the board and simultaneously update the structs. cp and hp
		for(y=0;y<BOARD_SIZE_Y;++y)
			for(x=0;x<BOARD_SIZE_X;++x)
			{
				piece=board_layout[y][x];

				//if the (x,y) board position is blank then continue
				if (piece==BLANK) continue;
					//Continue because cannot update any structure as no piece found


				//if a piece is found at (x,y) then p is a pointer to its
				//corresponding PEGGG structure
				p = (piece>0) ? &cp[piece] : &hp[-piece];

				//Update xpos,ypos,status of the PEGGG structure of `piece'
				//via p because p is a pointer to it.
				p->xpos=x;
				p->ypos=y;
				p->status=ALIVE;

				//Accordingly as the piece is human's or computer's
				//update other variables.

				//movespossible(char piece) function returns the
				//number of occupiable positions for a particular piece
				//whether it may be human's or computer's and also
				//updates the mp[][] field in PEGGG structure as well as
				//PEGGG.maxplaces variable.
				if (piece>0) {++ca;cq += (temp=movespossible(piece)); cb+=(!temp);}//note down the number of possible moves for a peg
				else {++ha;hq += (temp=movespossible(piece));hb+=(!temp);};//note down the number of possible moves for a peg
			}
		//Now calculate computer advantage level cal
		cal = (ca+ha)*(ca-ha)*15 + (hb+cb)*(hb-cb)*(ca-ha)*5 + (cq-hq)*3;
	}


	//This function displays the board on the screen and the scores too
	void disp_board(void)
	{
		ups();
		//display board
		for(int y=0;y<BOARD_SIZE_Y;++y)
			for(int x=0;x<BOARD_SIZE_X;++x)
				print_piece(board_layout[y][x],x,y,0);

		//display scores;
		textattr(SCREEN_ATTR);
		gotoxy(75,20);cprintf("%1d",MAXLEVEL);
		gotoxy(75,21);cprintf("%1d",(PEGS-ca));
		gotoxy(75,22);cprintf("%1d",(PEGS-ha));
	}

	//Below function moves the `piece' `steps' steps in consecutive
	//movable positions checking for a kill.
	BOARD move(int piece,int steps)
	{
		PEGGG *p; p = (piece>0) ? &cp[piece] : &hp[-piece];

		int ynew=p->mp[--steps][0];
		int xnew=p->mp[steps  ][1];

		BOARD temp(board_layout);
		temp.board_layout[ p->ypos ][ p->xpos ] = BLANK;
		temp.board_layout[ ynew    ][ xnew    ] = piece;

		char *b,*c;

		//check for a kill
		if (ynew>=2)//check for a kill on the top
		{
			b = &temp.board_layout[ynew-1][xnew];
			c = &temp.board_layout[ynew-2][xnew];
			*b= ( (piece*(*b))<0 && ((*c)*(*b))<0 ) ? BLANK : *b;
		}
		if (ynew<=BOARD_SIZE_Y-3)//check for a kill on the bottom
		{
			b = &temp.board_layout[ynew+1][xnew];
			c = &temp.board_layout[ynew+2][xnew];
			*b= ( (piece*(*b))<0 && ((*c)*(*b))<0 ) ? BLANK : *b;
		}
		if (xnew>=2)//check for a kill on the left
		{
			b = &temp.board_layout[ynew][xnew-1];
			c = &temp.board_layout[ynew][xnew-2];
			*b= ( (piece*(*b))<0 && ((*c)*(*b))<0 ) ? BLANK : *b;
		}
		if (xnew<=BOARD_SIZE_X-3)//check for a kill on the right
		{
			b = &temp.board_layout[ynew][xnew+1];
			c = &temp.board_layout[ynew][xnew+2];
			*b= ( (piece*(*b))<0 && ((*c)*(*b))<0 ) ? BLANK : *b;
		}
		// update PEGGG structure for temp
		temp.ups();
		return(temp);
	}

	//This function moves a particular piece on the board at some
	//(x,y) MOVABLE position checking for a kill.
	BOARD move(int piece,int x,int y)
	{
		PEGGG *p;p = (piece>0) ? &cp[piece] : &hp[-piece];

		for(int i=0;i<p->maxplaces;++i)
			if (p->mp[i][0]==y && p->mp[i][1]==x) return (move(piece,i+1));
		return BOARD(board_layout);
	}


	//Below function moves the `piece' `steps' steps in consecutive
	//movable positions not checking for a kill.

	BOARD move_but_dont_kill(int piece,int steps)
	{
		PEGGG *p; p = (piece>0) ? &cp[piece] : &hp[-piece];

		int ynew=p->mp[--steps][0];
		int xnew=p->mp[steps  ][1];

		BOARD temp(board_layout);
		temp.board_layout[ p->ypos ][ p->xpos ] = BLANK;
		temp.board_layout[ ynew    ][ xnew    ] = piece;

		// update PEGGG structure for temp
		temp.ups();
		return(temp);
	}

	//This function moves a particular piece on the board at some
	//(x,y) MOVABLE position not checking for a kill.
	BOARD move_but_dont_kill(int piece,int x,int y)
	{
		PEGGG *p;p = (piece>0) ? &cp[piece] : &hp[-piece];

		for(int i=0;i<p->maxplaces;++i)
			if (p->mp[i][0]==y && p->mp[i][1]==x) return (move_but_dont_kill(piece,i+1));
		return BOARD(board_layout);
	}



	//returns by reference the data about a piece.
	void peg_info(int piece,int& status,int& maxplaces)
	{
		PEGGG *p;p = (piece>0) ? &cp[piece] : &hp[-piece];
		status = p->status;  maxplaces = p->maxplaces;
	}

	//This function updates PEGGG.maxplaces, PEGGG.mp for a piece.
	//Returns: PEGGG.maxplaces
	int movespossible(int piece)
	{
		int i,x,y,moves=0;
		PEGGG *p;p = (piece>0) ? &cp[piece] : &hp[-piece];
		x = p->xpos;         y = p->ypos;
		int *a=p->mp[0];

		//try to move upwards checking for an obstacle.
		for(i=y; (i-1)>=0 			&& board_layout[i-1][x]==BLANK;++moves)
		{*a++=--i;
		 *a++=x;}

		//try to move downwards checking for an obstacle.
		for(i=y; (i+1)<BOARD_SIZE_Y && board_layout[i+1][x]==BLANK;++moves)
		{*a++=++i;
		 *a++=x;}

		//try to move westwards checking for an obstacle.
		for(i=x; (i-1)>=0           && board_layout[y][i-1]==BLANK;++moves)
		{*a++=y;
		 *a++=--i;}

		//try to move eastwards checking for an obstacle.
		for(i=x; (i+1)<BOARD_SIZE_X && board_layout[y][i+1]==BLANK;++moves)
		{*a++=y;
		 *a++=++i;}

		return (p->maxplaces = moves);
	}



	//Prints a piece at any (x,y) location on the board.


	void print_piece(int piece,int x,int y,int blinks)
	{
		int px    = 7+(x)*8;
		int py    = 6+(y)*3;
		if (piece==BLANK) textattr(255);
		else textattr( (piece>0) ? C_PEG_ATTR : H_PEG_ATTR );

		for(int i=0;;)
		{
			gotoxy(px,py  );cprintf("");
			gotoxy(px,py+1);cprintf("");

			if ( (++i)>blinks) break;

			delay(125);
			gotoxy(px,py  );cprintf("");
			gotoxy(px,py+1);cprintf("");
			delay(125);
		}
	}
	//returns the number of the peg at the cursor positon-
	//(curposx,curposy) on board
	int look(void)
	{
		return(board_layout[curposy][curposx]);
	}

	//Moves 'piece' 'steps' step blinking it and checking for a kill.
	BOARD blinkmove(int piece,int steps) //Checks all possible moves of a piece and moves it 'steps' steps.
	{
		PEGGG *p; p = (piece>0) ? &cp[piece] : &hp[-piece];
		int y=p->ypos;
		int x=p->xpos;

		print_piece(piece,x,y,5);//Blink the piece at old position
		print_piece(BLANK,x,y,0);//Erase the piece at old position

		BOARD temp = move(piece,steps);

		p = (piece>0) ? &(temp.cp[piece]) : &(temp.hp[-piece]);
		y=p->ypos;
		x=p->xpos;

		temp.print_piece(piece,x,y,5);//Blink the piece at new position.
		temp.disp_board();

		return temp;
	}

	void invert_board(void)//inverts the boardlayout
	{
		BOARD temp(board_layout);

		for(int y=0;y<BOARD_SIZE_Y/2;++y)
		{
			memcpy(board_layout[y],temp.board_layout[BOARD_SIZE_Y-y-1],BOARD_SIZE_X);
			memcpy(board_layout[BOARD_SIZE_Y-y-1],temp.board_layout[y],BOARD_SIZE_X);
		}

		ups();
	}

	void exchange_pegs(void)//exchanges pegs
	{
		for(int i=0;i<BOARD_SQUARES;++i) (*(board_layout[0]+i))*=(-1);
		ups();
	}

	int compare_boards(BOARD b1)//compares b1 with this board
	{
		return(memcmp(board_layout,b1.board_layout,BOARD_SQUARES));
	}
};





//This function moves the cursor at some (x,y) position on board
void move_cursor(int x,int y,int show_flag)
{
	static char top_left[2]     = "";
	static char top_right[2]    = "";
	static char bottom_left[2]  = "";
	static char bottom_right[2] = "";

	//check whether the move is possible
	if (x>=BOARD_SIZE_X || x<0 || y>=BOARD_SIZE_Y || y<0) return;

	int curpossx = 5+(curposx)*8;
	int curpossy = 5+(curposy)*3;
	int sx    = 5+(x)*8;
	int sy    = 5+(y)*3;


	if (!(x==curposx && y==curposy))
	{
		textattr(SCREEN_ATTR);
		gotoxy(curpossx,curpossy  );cprintf("%c%c",*top_left,*top_right);
		gotoxy(curpossx,curpossy+1);cprintf("");gotoxy(curpossx+8,curpossy+1);cprintf("");
		gotoxy(curpossx,curpossy+2);cprintf("");gotoxy(curpossx+8,curpossy+2);cprintf("");
		gotoxy(curpossx,curpossy+3);cprintf("%c%c",*bottom_left,*bottom_right);

		gettext(sx,sy,sx,sy,top_left);
		gettext(sx+8,sy,sx+8,sy,top_right);
		gettext(sx  ,sy+3,sx  ,sy+3,bottom_left);
		gettext(sx+8,sy+3,sx+8,sy+3,bottom_right);

		curposx=x;curposy=y;
	}

	if(!show_flag) return;

	textattr(CURSOR_ATTR);

	gotoxy(sx,sy  );cprintf("");
	gotoxy(sx,sy+1);cprintf("");gotoxy(sx+8,sy+1);cprintf("");
	gotoxy(sx,sy+2);cprintf("");gotoxy(sx+8,sy+2);cprintf("");
	gotoxy(sx,sy+3);cprintf("");
}

